/* File: plhip.pl
 * Created: 4-Dec-93
 * Last Update:  10-Dec-93
 * Author: Afzal Ballim (C) 1993
 * Purpose: positive left head-corner island parser compiler.
 *			Rules can have multiple "heads" (known as "anchors"
 *			here), parsing is based on finding the anchors, then
 *			filling in the gaps, left-right. Anchors are searched
 *			for left to right, and depth-first.
 *			Oh, can also find unattached constituents afterwards.
 *
 * This version has user-configuarable thresholding.
 *
 * Needs: member/2,append/3
 */

% operators needed for plhip rules
:- op(1200,xfx,[ ~~> ]).
:- op(1150,xf,[ ? ]).
:- op(1151,fx,[ ? ]).
:- op(500,fy,[ ~ ]).
:- op(500,fy,[ * ]).
:- op(100,fx,[ @ ]).
:- op(500,yfx,[ # ]).
:- op(1000,xfy,[ : ]).

% dynamic clauses used for the chart and identifying plhip clauses
:- dynamic 
	plhip_known/9,
	plhip_known_fail/5,
	plhip_known_exhausted/5,
	plhip_ccount/1,
	plhip_threshold_value/1,
	plhip_success_counter/1.


% Default threshold value. For a rule to succeed 
%  Cover/Span > Threshold
plhip_threshold_value(0).

% counter for referencing successful rule firing
plhip_success_counter(0).

% need to distinguish clauses, otherwise plhip_known and plhip_fail
% cannot be used properly. Add an extra argument (the plhip clause 
% counter) to each clause to distinguish it.
plhip_ccount(0).


/************************************************************
 *
 * Term expansion code
 *
 ************************************************************/

/*===========================================================
 * Predicate: term_expansion/2
 * Modes: 
 * 
 * Purpose: File reading causes invokation of term_expansion
 *			under sicstus, which gives us a way of hooking
 *			in to compile the grammar. Term expansion "compiles"
 *			the plhip rules into prolog clauses, complete with
 *			memoization (which gives a "chart" like effect).
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update:  9-Dec-93
 *=========================================================*/

% a delete rule
term_expansion(
	(-X # T ~~>P),
	(Xin
	 :-	
	   copy_term(X,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		plhip_is_known(LKNOWN,S,E)
	   ;% if not known to have failed or be exhausted, try it
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		P1,
		C/(De-Ds) >= T,
		\+plhip_is_known(LKNOWN,S,E),
		plhip_decl_known(LKNOWN,Ref)
	   ;% have failed
		\+plhip_is_known(LKNOWN,S,E),
		\+plhip_has_failed(LFAIL),
		plhip_decl_fail(LFAIL),
		fail
	   ;% have exhausted successes
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		plhip_decl_exh(LEXH),
		fail)))
:-
	!,Xin = plhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL]-DL,Rf),
	LKNOWN=plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,-X),
	LFAIL=plhip_known_fail(Cntr,FS,S,E,Xfrz),
	LEXH=plhip_known_exhausted(Cntr,FS,S,E,Xfrz),
	plhip_body(P,P1,FS,S,E,Ds,De,C,Rf-[]),
	plhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(plhip_ccount(Cntr)),
	assert(plhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a normal rule
term_expansion(
	(X # T ~~>P),
	(Xin
	 :-
	   copy_term(X,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		plhip_is_known(LKNOWN,S,E)
	   ;% if not known to have failed or be exhausted, try it
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		P1,
		C/(De-Ds) >= T,
		\+plhip_is_known(LKNOWN,S,E),
		plhip_decl_known(LKNOWN,Ref)
	   ;% have failed
		\+plhip_is_known(LKNOWN,S,E),
		\+plhip_has_failed(LFAIL),
		plhip_decl_fail(LFAIL),
		fail
	   ;% have exhausted successes
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		plhip_decl_exh(LEXH),
		fail)))
:-
	!,plhip_clause(X,Xin,[FS,S,E,Ds,De,C,[Ref|DL]-DL,Rf]),
	LKNOWN=plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,X),
	LFAIL=plhip_known_fail(Cntr,FS,S,E,Xfrz),
	LEXH=plhip_known_exhausted(Cntr,FS,S,E,Xfrz),
	plhip_body(P,P1,FS,S,E,Ds,De,C,Rf-[]),
	plhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(plhip_ccount(Cntr)),
	assert(plhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a delete rule
term_expansion(
	(-X ~~> P),
	(Xin
	 :-	
	   plhip_threshold_value(T),
	   copy_term(X,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		plhip_is_known(LKNOWN,S,E)
	   ;% if not known to have failed or be exhausted, try it
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		P1,
		C/(De-Ds) >= T,
		\+plhip_is_known(LKNOWN,S,E),
		plhip_decl_known(LKNOWN,Ref)
	   ;% have failed
		\+plhip_is_known(LKNOWN,S,E),
		\+plhip_has_failed(LFAIL),
		plhip_decl_fail(LFAIL),
		fail
	   ;% have exhausted successes
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		plhip_decl_exh(LEXH),
		fail)))
:-
	!,Xin = plhip_delete(X,FS,S,E,Ds,De,C,[Ref|DL]-DL,Rf),
	LKNOWN=plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,-X),
	LFAIL=plhip_known_fail(Cntr,FS,S,E,Xfrz),
	LEXH=plhip_known_exhausted(Cntr,FS,S,E,Xfrz),
	plhip_body(P,P1,FS,S,E,Ds,De,C,Rf-[]),
	plhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(plhip_ccount(Cntr)),
	assert(plhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

% a normal rule
term_expansion(
	(X ~~> P),
	(Xin
	 :-
	   plhip_threshold_value(T),
	   copy_term(X,Xfrz),
	   numbervars(Xfrz,26000,_),
	   (% known to have succeeded already
		plhip_is_known(LKNOWN,S,E)
	   ;% if not known to have failed or be exhausted, try it
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		P1,
		C/(De-Ds) >= T,
		\+plhip_is_known(LKNOWN,S,E),
		plhip_decl_known(LKNOWN,Ref)
	   ;% have failed
		\+plhip_is_known(LKNOWN,S,E),
		\+plhip_has_failed(LFAIL),
		plhip_decl_fail(LFAIL),
		fail
	   ;% have exhausted successes
		\+plhip_has_failed(LFAIL),
		\+plhip_is_exhausted(LEXH),
		plhip_decl_exh(LEXH),
		fail)))
:-
	!,plhip_clause(X,Xin,[FS,S,E,Ds,De,C,[Ref|DL]-DL,Rf]),
	LKNOWN=plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,X),
	LFAIL=plhip_known_fail(Cntr,FS,S,E,Xfrz),
	LEXH=plhip_known_exhausted(Cntr,FS,S,E,Xfrz),
	plhip_body(P,P1,FS,S,E,Ds,De,C,Rf-[]),
	plhip_ccount(Cntr),
	NewCntr is Cntr+1,
	retract(plhip_ccount(Cntr)),
	assert(plhip_ccount(NewCntr)),
	format("Rule ~q: ~q~n",[Cntr,X]).

/*===========================================================
 * Predicate: plhip_body/9
 * Modes: plhip_body(+P,-Px,+FS,+S,+E,-Ds,-De,-C,+Rf-?Rl)
 * 
 * Purpose: performs the translation of the body of a plhip
 *			clause P, into Px, where FS is the sentence
 *			represented as a function, S is the start of the
 *			span over which the search occurs, E is the end
 *			boundary of the span (i.e., search occurs in S..E-1
 *			inclusive), Ds is the start of the found constituent
 *			De is the end boundary of it, and C is the number of
 *			tokens covered by the constituent in Ds..De-1.
 *			In addition, a diff-list is maintained as a store
 *			for tracking successful rule firing.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_body(P,Px,FS,S,E,Ds,De,C,Rf-Rl)
:-
	plhip_constrain(P,Knt,NonKnt,FS,Ds,De,E,S,S,_SNK,_SNI,[[],[]],
					Covers,Rf-Rl),
	!,plhip_unravel(Knt,NonKnt,Px,C,Covers).

plhip_body(P,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: could not compile plhip body ~q~n",[P]),
	!,fail.

/*===========================================================
 * Predicate: plhip_constrain/14
 * Modes: plhip_constrain(
 *			+Body,
 *			-Anchors,
 *			-NonAnchors,
 *			+Sent,
 *			+CvS,
 *			+CvE,
 *			+End,
 *			+EndPrevAnchor,
 *			+EndPrevItem,
 *			-StartNextAnchor,
 *			-StartNextItem,
 *			[+PrevCls,+PrevDCls],
 *			-Covers,
 *			+Rf-+Rl)
 * 
 * Purpose: 
 * 
 * Licensed Use: 
 * 
 * Created: 20-Nov-93
 * Last Update:  6-Dec-93
 *=========================================================*/

%                    +**ERROR** conditions+

%                    +No NEGATION+

plhip_constrain((~ P,_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: plhip does not support negation, use lhip ~q~n",
		[~ P]),
	!,fail.

plhip_constrain((~ P),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: plhip does not support negation, use lhip ~q~n",
		[~ P]),
	!,fail.

%                    -No NEGATION-

%					ILLEGAL HEADS

plhip_constrain(((*!),Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a head ~q~n",[((*!),Q)]),
	!,fail.

plhip_constrain((*[],_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a head~n",[]),
	!,fail.

plhip_constrain((*P,_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

plhip_constrain(((*!):Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a head ~q~n",[((*!),Q)]),
	!,fail.

plhip_constrain((*[]:_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a head~n",[]),
	!,fail.

plhip_constrain((*P:_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

plhip_constrain((*!),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a head ~q~n",[(*!)]),
	!,fail.

plhip_constrain(*[],_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a head~n",[]),
	!,fail.

plhip_constrain(*P,_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be heads ~q~n",[*P]),
	!,fail.

%					ILLEGAL SPECIFIC DELETE RULES

plhip_constrain(((-!),Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a delete rule ~q~n",[((-!),Q)]),
	!,fail.

plhip_constrain((-[],_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a delete rule~n",[]),
	!,fail.

plhip_constrain((-P,_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

plhip_constrain(((-!):Q),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a delete rule ~q~n",[((-!),Q)]),
	!,fail.

plhip_constrain((-[]:_),_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a delete rule~n",[]),
	!,fail.

plhip_constrain((-P:_),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

plhip_constrain((-!),_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	format("**ERROR**: cut cannot be a delete rule ~q~n",[(-!)]),
	!,fail.

plhip_constrain(-[],_,_,_,_,_,_,_,_,_,_,_,_,_)
:-
	format("**ERROR**: general delete cannot be a specific delete rule~n",[]),
	!,fail.

plhip_constrain(-P,_K,_NK,
			   _S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,_PC,_,_)
:-
	plhip_non_predicate(P),
	\+(P = @_),
	format("**ERROR**: only atomic rules can be delete rules ~q~n",[-P]),
	!,fail.

%				MUST HAVE AT LEAST ONE HEAD/NORMAL CLAUSE

plhip_constrain(!,_K,_NK,_S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule~n",[]),
	!,fail.

plhip_constrain({P},_K,_NK,_S,_CvS,_CvE,_End,_EPK,_EPI,_SNK,_SNI,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule ~q~n",[{P}]),
	!,fail.

plhip_constrain([],_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule~n",[]),
    !,fail.

plhip_constrain(-P,_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule ~q~n",[-P]),
    !,fail.

plhip_constrain(~P,_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule ~q~n",[~P]),
    !,fail.

plhip_constrain((?P?),_,_,_,_,_,_,_,_,_,_,[_,[]],_,_)
:-
	format("**ERROR**: rule body must contain a normal or head rule ~q~n",[(?P?)]),
    !,fail.

%                    -**ERROR** conditions-

%                    +Mult. Cls+

%                    +{}/!+

plhip_constrain((!,P),K,[{!}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl)
:-
	!,plhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl).

plhip_constrain(({E},P),K,[{E}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl)
:-
	!,plhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl).

plhip_constrain((!:P),K,[{!}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl)
:-
	!,plhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl).

plhip_constrain(({E}:P),K,[{E}|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl)
:-
	!,plhip_constrain(P,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,PC,Cov,Rf-Rl).

%                    -{}/!-

%                    +FCls+

%                    +Adjacency+

plhip_constrain((*P:Q),[Px|K],NK,S,CvS,CvE,End,EPK,EPK,CvS,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,CvS,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,De,[Px,Px],Cov,Rnf-Rl).

% adjacency does not transmit backwards through a delete or optional

plhip_constrain(([]:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[_,S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	Px = (Pdel,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

plhip_constrain((-P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[P,S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	Px = (Pdel,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

plhip_constrain(((?P?):Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPK,SNK,CvS,De,C,Rf-Rnf),!,
	Px = (Popt,SNI=De;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

% a negation followed by an adjacency means that the following thing
% must be treated as a head
plhip_constrain((~P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPK,SNK,CvS,SNI,C,_),!,
	Px = (\+Popt,CvS=SNI,C=0),
	(Q=(*Q1),Qn=Q;
	 Q=(*Q1,Qr),Qn=Q;
	 Q=(Q1,Qr),Qn=(*Q1,Qr);
	 Q=(*Q1 : Qr), Qn=Q;
	 Q=(Q1 : Qr), Qn=(*Q1 : Qr);
	 Qn= *Q),
	plhip_constrain(Qn,K,NK,S,CvS,CvE,End,EPK,EPK,SNK,SNI,[Px,[]],Cov,Rf-Rl).

plhip_constrain((P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,De,[Px,Px],Cov,Rnf-Rl).

%                    -Adjacency-

plhip_constrain((*P,Q),[Px|K],NK,S,CvS,CvE,End,EPK,EPK,CvS,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,CvS,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,_SNI,[Px,Px],Cov,Rnf-Rl).

plhip_constrain(([],Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[_,S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	Px = (Pdel;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

plhip_constrain((-P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[P,S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	Px = (Pdel;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

plhip_constrain(((?P?),Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPK,SNK,CvS,De,C,Rf-Rnf),!,
	Px = (Popt;CvS=SNI,De=EPK,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rnf-Rl).

plhip_constrain((~P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPK,SNK,CvS,De,C,_),!,
	Px = (\+Popt,CvS=SNI,De=EPK,C=0),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,[]],Cov,Rf-Rl).

plhip_constrain((P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPK,SNK,CvS,
			[[],[]],[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,SNK,CvS,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,_SNI,[Px,Px],Cov,Rnf-Rl).

%                    -FCls-

%                    +Adjacency+

plhip_constrain((*P:Q),[Px|K],NK,S,CvS,CvE,End,EPK,_EPI,Ds,Ds,
			_PC,[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,Ds,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,De,[Px,Px],Cov,Rnf-Rl).

plhip_constrain(([]:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[_,S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	Px = (Pdel,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain((-P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[P,S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	Px = (Pdel,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain(((?P?):Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPI,SNK,Ds,De,C,Rf-Rnf),!,
	Px = (Popt,SNI=De;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain((~P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPI,SNK,Ds,SNI,C,_),!,
	Px = (\+Popt,Ds=SNI,C=0),
	(Q=(*Q1),Qn=Q;
	 Q=(*Q1,Qr),Qn=Q;
	 Q=(Q1,Qr),Qn=(*Q1,Qr);
	 Q=(*Q1 : Qr), Qn=Q;
	 Q=(Q1 : Qr), Qn=(*Q1 : Qr);
	 Qn= *Q),
	plhip_constrain(Qn,K,NK,S,CvS,CvE,End,EPK,EPI,SNK,SNI,[Px,PdC],Cov,Rf-Rl).

plhip_constrain((P:Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			_PC,[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,De,[Px,Px],Cov,Rnf-Rl).

%                    -Adjacency-

plhip_constrain((*P,Q),[Px|K],NK,S,CvS,CvE,End,EPK,_EPI,Ds,Ds,
			_PC,[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,Ds,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,De,De,_SNK,_SNI,[Px,Px],Cov,Rnf-Rl).

plhip_constrain(([],Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[_,S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	Px = (Pdel;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain((-P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[P,S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	Px = (Pdel;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain(((?P?),Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPI,SNK,Ds,De,C,Rf-Rnf),!,
	Px = (Popt;Ds=SNI,De=EPI,C=0,Rf=Rnf),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rnf-Rl).

plhip_constrain((~P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,SNI,
			[_PC,PdC],[C|Cov],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPI,SNK,Ds,De,C,_),!,
	Px = (\+Popt,Ds=SNI,De=EPI,C=0),
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,SNI,[Px,PdC],Cov,Rf-Rl).

plhip_constrain((P,Q),K,[Px|NK],S,CvS,CvE,End,EPK,EPI,SNK,Ds,
			_PC,[C|Cov],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPI,SNK,Ds,De,C,Rf-Rnf,_]),!,
	plhip_constrain(Q,K,NK,S,CvS,CvE,End,EPK,De,SNK,_SNI,[Px,Px],Cov,Rnf-Rl).

%                    -Mult. Cls-

%                    +Last Cls+

%                    +CUT+

% last clause is cut, but there were other clauses, so previous clause
% is last clause
plhip_constrain(!,[],[{!}],_S,_CvS,CvE,End,_EPK,CvE,End,CvE,_PC,[],Rl-Rl).

%                    -CUT-

%                    +{P}+

% last clause is embedded code, but there were other clauses
plhip_constrain({P},[],[{P}],_S,_CvS,CvE,End,_EPK,CvE,End,CvE,_PC,[],Rl-Rl).

%                    -{P}-

%                    +Lc=Fc+

% last clause is first clause
plhip_constrain(*P,[Px],[],S,CvS,CvE,End,EPK,EPK,End,CvE,[[],[]],[C],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,CvS,CvE,C,Rf-Rl,_]),!.

plhip_constrain(P,[],[Px],S,CvS,CvE,End,EPK,EPK,End,CvE,[[],[]],[C],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,CvS,CvE,C,Rf-Rl,_]),!.

%                    -Lc=Fc-

% any other last clause
plhip_constrain(*P,[Px],[],S,_CvS,CvE,End,EPK,_EPI,Ds,Ds,_PC,[C],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPK,End,Ds,CvE,C,Rf-Rl,_]),!.

plhip_constrain([],[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[_,S,EPI,End,Ds,CvE,C,Rf-Rl,_]),!,
	Px = (Pdel;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

plhip_constrain(-P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf-Rl)
:-
	plhip_clause(plhip_delete,Pdel,[P,S,EPI,End,Ds,CvE,C,Rf-Rl,_]),!,
	Px = (Pdel;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

plhip_constrain((?P?),[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf-Rl)
:-
	plhip_body(P,Popt,S,EPI,End,Ds,CvE,C,Rf-Rl),
	Px = (Popt;Ds=EPI,CvE=EPI,C=0,Rf=Rl).

plhip_constrain(~P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rl-Rl)
:-
	plhip_body(P,Popt,S,EPI,End,Ds,CvE,C,_),
	Px = (\+Popt,Ds=EPI,CvE=EPI,C=0).

plhip_constrain(P,[],[Px],S,_CvS,CvE,End,_EPK,EPI,End,Ds,_PC,[C],Rf-Rl)
:-
	plhip_clause(P,Px,[S,EPI,End,Ds,CvE,C,Rf-Rl,_]),!.

%                    -Last Cls-


/*===========================================================
 * Predicate: plhip_unravel/5
 * Modes: plhip_unravel(+K,+N,-Cls,+C,+Covers)
 * 
 * Purpose: unravels the heads and non-heads to give the body
 * 
 * Licensed Use: 
 * 
 * Created: 20-Nov-93
 * Last Update: 
 *=========================================================*/

% there was only one clause
plhip_unravel([],[{P}],P,C,[C]) :- !.
plhip_unravel([],[P],P,C,[C]) :- !.
plhip_unravel([P],[],P,C,[C]) :- !.

% more than one clause, unravel
plhip_unravel([P|K],N,(P,Cls),C,Covers)
:-
	!,plhip_unravel(K,N,Cls,C,Covers).

plhip_unravel([],[],(C is Cls),C,Covers)
:-
	!,plhip_make_cover_cntr(Covers,Cls).

plhip_unravel([],[{P}],(P,C is Cls),C,Covers)
:-
	!,plhip_make_cover_cntr(Covers,Cls).

plhip_unravel([],[P],(P,C is Cls),C,Covers)
:-
	!,plhip_make_cover_cntr(Covers,Cls).

plhip_unravel([],[{P}|N],(P,Cls),C,Covers)
:-
	!,plhip_unravel([],N,Cls,C,Covers).

plhip_unravel([],[P|N],(P,Cls),C,Covers)
:-
	!,plhip_unravel([],N,Cls,C,Covers).

/*===========================================================
 * Predicate: plhip_make_cover_cntr/2
 * Modes: plhip_make_cover_cntr(+List,-Counter)
 * 
 * Purpose: Given a list of terms, make a term which "adds"
 *			them together
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/
plhip_make_cover_cntr([],_) :- !,fail.
plhip_make_cover_cntr([C],C).
plhip_make_cover_cntr([C|Cs],Counter+C)
:-
	plhip_make_cover_cntr(Cs,Counter).


/*===========================================================
 * Predicate: plhip_clause
 * Modes: plhip_clause(+LC,-PC,+XArgs)
 * 
 * Purpose: convert a plhip clause into a prolog one.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update:  8-Dec-93
 *=========================================================*/

plhip_clause(!,_,_)
:-
	format("**ERROR**: ! is not a clause, check that it does not",[]),
	format(" precede ! or embedded code~n",[]),
	!,fail.

plhip_clause([],_,_)
:-
	format("**ERROR**: [] is not a clause~n",[]),
	!,fail.

% disjunction, heads are local to the disjunction
plhip_clause((P;Q),
	(Px;Qx),
	[Fs,S,E,Ds,De,C,Rf-Rl,_])
:-
	!,plhip_body(P,Px,Fs,S,E,Ds,De,C,Rf-Rl),
	plhip_body(Q,Qx,Fs,S,E,Ds,De,C,Rf-Rl).

% terminal
plhip_clause((@X),X1,Args)
:-
	!,X1 =.. ['@',X|Args].

% Generic error handling.
plhip_clause(P,_,_)
:-
	plhip_non_predicate(P),
	format("**ERROR**: ~q is not a clause~n",[P]),
	!,fail.

plhip_clause(X,Y,Args) 
:-
	!,X =.. Xl,
	append(Xl,Args,Yl),
	Y =.. Yl.

/*===========================================================
 * Predicate: plhip_non_predicate/1
 * Modes: 
 * 
 * Purpose: succeeds if the argument is a prolog connective,
 *			such as ';' or ','.
 * 
 * Licensed Use: 
 * 
 * Created: 23-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_non_predicate(P)
:-
	functor(P,F,_),
	(F=',';
	 F=';';
	 F=':';
	 F='{}';
	 F='~';
	 F='-';
	 F='.';
	 F='*';
	 F='?';
	 F='@').

/*===========================================================
 * Predicate: @/9
 * Modes: 
 * 
 * Purpose: handle a terminal, allowing for alternation in
 *			the input signalled by embedded lists.
 * 
 * Licensed Use: 
 * 
 * Created: 17-Nov-93
 * Last Update: 
 *=========================================================*/

'@'(Term,Sent,SS,SE,FS,FE,C,[Ref|X]-X,[])
:-
	copy_term('@'(Term),COPY),
	numbervars(COPY, 26000, _),
	(plhip_is_known(
	  plhip_known(-1,Ref,[],Sent,FS,FE,C,COPY,'@'(Term)),SS,SE)
	;\+plhip_has_failed(plhip_known_fail(-1,Sent,SS,SE,COPY)),
	 \+plhip_is_exhausted(plhip_known_exhausted(-1,Sent,SS,SE,COPY)),
	 plhip_find_terminal(Term,Sent,SS,SE,FS,FE,C),
	 \+plhip_is_known(
		plhip_known(-1,Ref,[],Sent,FS,FE,C,COPY,'@'(Term)),SS,SE),
	 plhip_decl_known(plhip_known(-1,Ref,[],Sent,FS,FE,C,COPY,
				'@'(Term)),
				Ref)
	;\+plhip_has_failed(plhip_known_fail(-1,Sent,SS,SE,COPY)),
	 \+plhip_is_known(
		plhip_known(-1,Ref,[],Sent,FS,FE,C,COPY,'@'(Term)),SS,SE),
	 plhip_decl_fail(plhip_known_fail(-1,Sent,SS,SE,COPY)),
	 fail
	;\+plhip_has_failed(plhip_known_fail(-1,Sent,SS,SE,COPY)),
	 \+plhip_is_exhausted(plhip_known_exhausted(-1,Sent,SS,SE,COPY)),
	 plhip_decl_exh(plhip_known_exhausted(-1,Sent,SS,SE,COPY)),
	 fail).
	 
% if start has reached end, then fail as end is a far boundary
plhip_find_terminal(_Term,_Sent,Start,End,_F,_E,_C)
:-
	Start >= End,
	!,fail.

% found it
plhip_find_terminal(Term,Sent,Start,_End,Start,End0,1)
:-
	arg(Start,Sent,TermS),
	(Term=TermS;member(Term,TermS)),
	End0 is Start+1.

% not here, try next
plhip_find_terminal(Term,Sent,Start,End,Found,End0,C)
:-
	NewStart is Start+1,
	plhip_find_terminal(Term,Sent,NewStart,End,Found,End0,C).

/*===========================================================
 * Predicate: plhip_success_ref/1
 * Modes: plhip_success_ref(?T)
 * 
 * Purpose: the global success_counter value is T
 * 
 * Licensed Use: 
 *		plhip_success_ref(+T)	
 *			set success_counter value to T
 *		plhip_success_ref(-T)	
 *			T is the current success_counter value
 * 
 * Created: 25-Nov-93
 * Last Update:  6-Dec-93
 *=========================================================*/

plhip_success_ref(T)
:-
	var(T),
	plhip_success_counter(T).

plhip_success_ref(T)
:-
	nonvar(T),
	retract(plhip_success_counter(_)),
	assert(plhip_success_counter(T)).

/*===========================================================
 * Predicate: plhip_decl_known/2
 * Modes: plhip_decl_known(+Cls,-Ref)
 * 
 * Purpose: performs an assertz of Cls, having first bound
 *			Ref to the value of the success counter (which
 *			is then augmented by 1).
 * 
 * Licensed Use: 
 * 
 * Created:  6-Dec-93
 * Last Update: 
 *=========================================================*/

plhip_decl_known(Cls,Ref)
:-
	plhip_success_ref(Ref),
	assertz(Cls),
	NewRef is Ref+1,
	plhip_success_ref(NewRef).


/*===========================================================
 * Predicate: plhip_decl_fail/1
 * Modes: 
 * 
 * Purpose: checks to see if the current "fail" is the most general.
 *			If not, then it asserts it as being so
 * 
 * Licensed Use: 
 * 
 * Created: 10-Dec-93
 * Last Update: 
 *=========================================================*/

plhip_decl_fail(plhip_known_fail(Cntr,FS,S,E,Xfrz))
:-
	findall((Sa,Ea),
			plhip_known_fail(Cntr,FS,Sa,Ea,Xfrz),
			Extents),
	plhip_fail_org(Extents,Cntr,FS,S,E,Xfrz).

/*===========================================================
 * Predicate: plhip_fail_org/6
 * Modes: plhip(+Ext,+Cntr,+Sent,+S,+E,+Xfrz)
 * 
 * Purpose: check the existing failures of Xfrz over Sent wrt S,E
 *			to determine what's the maximal failure region
 * 
 * Licensed Use: 
 * 
 * Created: 11-Dec-93
 * Last Update: 
 *=========================================================*/

% S,E is the maximal region over which Xfrz fails
plhip_fail_org([],Cntr,Sent,S,E,Xfrz)
:-
	!,assertz(plhip_known_fail(Cntr,Sent,S,E,Xfrz)).


% an existing one is as large as it
plhip_fail_org([(Sa,Ea)|_Ex],_Cntr,_Sent,S,E,_Xfrz)
:-
	Sa =< S, E =< Ea,!.

% they are non-overlapping
plhip_fail_org([(Sa,Ea)|_Ex],_Cntr,_Sent,S,E,_Xfrz)
:-
	(Ea < S; E < Sa),!.

% will replace (Sa,Ea) with (min(S,Sa),max(E,Ea))
plhip_fail_org([(Sa,Ea)|Ex],Cntr,Sent,S,E,Xfrz)
:-
	retract(plhip_known_fail(Cntr,Sent,Sa,Ea,Xfrz)),
	(Sa =< S, Sn=Sa;Sn=S),
	(Ea >= E, En=Ea;En=E),!,
	plhip_fail_org(Ex,Cntr,Sent,Sn,En,Xfrz).

/*===========================================================
 * Predicate: plhip_decl_exh/1
 * Modes: 
 * 
 * Purpose: at the moment, exactly the same as plhip_decl_fail
 * 
 * Licensed Use: 
 * 
 * Created: 10-Dec-93
 * Last Update: 
 *=========================================================*/

plhip_decl_exh(plhip_known_exhausted(Cntr,FS,S,E,Xfrz))
:-
	findall((Sa,Ea),
			plhip_known_exhausted(Cntr,FS,Sa,Ea,Xfrz),
			Extents),
	plhip_exhausted_org(Extents,Cntr,FS,S,E,Xfrz).

/*===========================================================
 * Predicate: plhip_exhausted_org/6
 * Modes: plhip(+Ext,+Cntr,+Sent,+S,+E,+Xfrz)
 * 
 * Purpose: check the existing exhaustion of Xfrz over Sent 
 *			wrt S,E to determine what's the maximal exhaustion
 *			region.
 * 
 * Licensed Use: 
 * 
 * Created: 11-Dec-93
 * Last Update: 
 *=========================================================*/

% S,E is the maximal region over which Xfrz is exhausted
plhip_exhausted_org([],Cntr,Sent,S,E,Xfrz)
:-
	!,assertz(plhip_known_exhausted(Cntr,Sent,S,E,Xfrz)).


% an existing one is as large as it
plhip_exhausted_org([(Sa,Ea)|_Ex],_Cntr,_Sent,S,E,_Xfrz)
:-
	Sa =< S, E =< Ea,!.

% they are non-overlapping
plhip_exhausted_org([(Sa,Ea)|_Ex],_Cntr,_Sent,S,E,_Xfrz)
:-
	(Ea < S; E < Sa),!.

% will replace (Sa,Ea) with (min(S,Sa),max(E,Ea))
plhip_exhausted_org([(Sa,Ea)|Ex],Cntr,Sent,S,E,Xfrz)
:-
	retract(plhip_known_exhausted(Cntr,Sent,Sa,Ea,Xfrz)),
	(Sa =< S, Sn=Sa;Sn=S),
	(Ea >= E, En=Ea;En=E),!,
	plhip_exhausted_org(Ex,Cntr,Sent,Sn,En,Xfrz).

/*===========================================================
 * Predicate: plhip_is_known/3
 * Modes: 
 * 
 * Purpose: is there a known success within the range of the
 *			search?
 * 
 * Licensed Use: 
 * 
 * Created: 10-Dec-93
 * Last Update: 
 *=========================================================*/

plhip_is_known(plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,X),S,E)
:-
	plhip_known(Cntr,Ref,Rf,FS,Ds,De,C,Xfrz,X),
	S =< Ds, De =< E.

/*===========================================================
 * Predicate: plhip_has_failed/1
 * Modes: 
 * 
 * Purpose: is there a wider failure?
 * 
 * Licensed Use: 
 * 
 * Created: 10-Dec-93
 * Last Update: 
 *=========================================================*/
plhip_has_failed(plhip_known_fail(Cntr,FS,S,E,Xfrz))
:-
	plhip_known_fail(Cntr,FS,S1,E1,Xfrz),
	S1 =< S,E =< E1.

/*===========================================================
 * Predicate: plhip_is_exhausted/1
 * Modes: 
 * 
 * Purpose: is there a wider exhaustion of this item?
 * 
 * Licensed Use: 
 * 
 * Created: 10-Dec-93
 * Last Update: 
 *=========================================================*/
plhip_is_exhausted(plhip_known_exhausted(Cntr,FS,S,E,Xfrz))
:-
	plhip_known_exhausted(Cntr,FS,S1,E1,Xfrz),
	S1 =< S,E =< E1.

