/* Forming a 2-d topological feature map of the n-d input vector data:

   Square network of variable size
   Input vectors read from the file and cycled through
   Random initial input weights
   Response of a unit inversely proportional to the euclidian distance between
     the input vector and the weight vector of the unit
   Global selection of maximally responding unit
   Weights changed toward the input within a neighborhood of this unit
   Size of neighborhood and rate of change decreased linearly to zero
     in two phases

   For each unit, the best input vector is shown
   For each vector, the current maximally responding unit is shown
   Graphics can also be turned off and snapshots of the simulation
     saved into a file for later display

   1st parameter: simufile
   
   Risto Miikkulainen 8/21/1989

***********************************************************************/
   
#include <stdio.h>
#include <math.h>

#define notcray 1
#if notcray
#include <starbase.c.h>
#endif

#define cmapfile "/n/tetra/usr/users/ai/risto/screen/cmap/bryw64.cmap"
#define snapshotend 999999999

#define maxnets 20            /* maximum network size is n*n units */
#define maxsnaps 50           /* maximum number of snapshots */
#define maxvectors 200        /* maximum number of input vectors */
#define maxinps 200           /* maximum number of inputs */
#define maxlabell 30          /* maximum length of lables */
#define maxdim 50             /* maximum dimension of input vectors */
#define perr 0.15
#define serr 0.15

#define frame_color 1          /* 1 = green in color, black in bw */
#define bestvector_color 63    /* 63= white in color, black in bw */
#define bestunit_color 1       
#define assocunit_color 13
#define clear_color 0          /* 0 = black in color, white in bw */
#define plabelsize=(phght/3.0)
#define slabelsize=(shght/3.0)
#define expansion=0.5

long starttime;
int pnets, pt0, pt1, pt2, pnct0, pnct1, pbesti, pbestj, pnvectors, pdim,
    snets, st0, st1, st2, snct0, snct1, sbesti, sbestj, snvectors, sdim,
           at0, at1, at2, anct0, anct1, pnerr, snerr, pscorr, spcorr,
    pncorr, sncorr,
    seed, t, startt, tend, ninppairs, pninp, sninp, aninp,
    displaying, continuing, testing,
    nextsnapshot, snapshots[maxsnaps], fildes, cmapsize;
float colors[256][3];
int shuffletable[maxinps];
float palphat0, palphat1, salphat0, salphat1, aalphat0, aalphat1,
  phght, pwdth, plow, pleft, phigh, pright,
  shght, swdth, slow, sleft, shigh, sright, pdeltasum, sdeltasum;
char simufile[100], inpfile[100], pvectorfile[100], plabelfile[100],
  svectorfile[100], slabelfile[100];
FILE *fp, *fp2;

struct pair {
  int p,s;
}inppairs[maxinps];

struct inputvectors {
  char label[maxlabell+1];
  float comp[maxdim];
};

struct unitdata {
  int   labelcount;
  int   labels[maxvectors+1];
  float bestvalue;
  float value;
  float comp[maxdim];
};

float psassoc[maxnets][maxnets][maxnets][maxnets],
      spassoc[maxnets][maxnets][maxnets][maxnets];
struct inputvectors pvectors[maxvectors], svectors[maxvectors];
struct unitdata punits[maxnets][maxnets], sunits[maxnets][maxnets];

/*********************  main control ************************/

main(argc,argv)
int argc; char *argv[];
{
  init_params(argv);
  read_inputs();
#if notcray
  if (displaying)
    display_init();
#endif
  if (continuing) iterate_snapshots();
  else init_weights();
  training();
#if notcray
  if (displaying) gclose(fildes);
#endif
  exit(0);
}


iterate_snapshots()
/* go through the saved snapshots, displaying them on the screen */
{
  register int i, neti;
  int oldt, oldtesting, readfun(), randfun();

  fp=fopen(simufile,"r");
  read_params(fp);
  oldtesting=testing;
  testing=1;				/* we don't want to change weights */
  oldt=0;
  iterate_weights(randfun); /* just to update rand */
  while (fscanf(fp,"%d",&t)!=EOF) /* read the t */
    {
      /* update the shuffling and parameters */
      startt=t+1;
/*      for(i=0; i<startt-oldt; i++) shuffle();*/
      oldt=startt;
      
      /* read the current weights */
      iterate_weights(readfun);
#if notcray
      if (displaying)
	{
	  iterate_inputs();
/*	  while (getchar()!='\n');*/
	}
#endif
    }
  fclose(fp);
  for (nextsnapshot=0; t >= snapshots[nextsnapshot]; nextsnapshot++){}
  testing=oldtesting;
}

      
training()
{
  register int i;
  for(t=startt; t<=tend; t++)
    {
      iterate_inputs();
/*      shuffle();*/
      if (t >= snapshots[nextsnapshot]) save_current();
/*printf("%d: 0,5=%f, 5,5=%f  ",t, spassoc[0][5][0][5], psassoc[0][5][5][5]);
printf("%d: 0,5=%f, 3,2=%f\n",t, psassoc[0][5][0][5], psassoc[0][5][3][2]);*/
    }
}


iterate_inputs()
{
  register int i;
  float alpha();
  pdeltasum=sdeltasum=0.0;
  pnerr=snerr=pncorr=sncorr=spcorr=pscorr=0;
  init_units();
  for(i=0; i<ninppairs; i++)
    {
      compute_responses(shuffletable[i]);
      if (!testing)
	modify_weights(nc(t, pt0, pt1, pt2, pnct0, pnct1),
		       alpha(t, pt0, pt1, pt2, palphat0, palphat1),
		       nc(t, st0, st1, st2, snct0, snct1),
		       alpha(t, st0, st1, st2, salphat0, salphat1),
		       nc(t, at0, at1, at2, anct0, anct1),
		       alpha(t, at0, at1, at2, aalphat0, aalphat1),
		       shuffletable[i]);
    }
  print_stats();
#if notcray
  if (displaying)
/*    if(testing) display_assocs();*/
    if(testing) display_selected_assocs();
    else display_best();
#endif
}


print_stats()
{
  printf("Epoch %d:  p: %6.4f %5.1f %5.1f   s: %6.4f %5.1f %5.1f  ps: %5.1f  sp: %5.1f\n", t,
	 pdeltasum/(pdim*pninp), 100.0*pnerr/(pdim*pninp), 100.0*pncorr/pninp,
	 sdeltasum/(sdim*sninp), 100.0*snerr/(sdim*sninp), 100.0*sncorr/sninp,
	 100.0*pscorr/aninp, 100.0*spcorr/aninp);
}


compute_responses(inpv)
int inpv;
{
  int i,j,ii,jj,psbesti,psbestj,spbesti,spbestj;
  float distance();
  float pbest=999999999.9, sbest=999999999.9, psbest=(-1.0), second=(-1.0),
  spbest=(-1.0);

  if(t>=pt0 && inppairs[inpv].p > (-1))
    {
      second=(-1.0);
      for(i=0; i<pnets; i++)
	for(j=0; j<pnets; j++){
	  punits[i][j].value = distance(pvectors[inppairs[inpv].p].comp,
					punits[i][j].comp,pdim);
	  /* check if this unit's response is best so far encountered */
	  if (punits[i][j].value == pbest) second=pbest;
	  if (punits[i][j].value < pbest) {
	    pbesti=i; pbestj=j; pbest=punits[i][j].value;
	  }
	  if (displaying)
	    if (punits[i][j].value < punits[i][j].bestvalue){
	      punits[i][j].bestvalue = punits[i][j].value;
	      punits[i][j].labels[0] = inppairs[inpv].p;
	    }
	}
      for(i=0; i<pdim; i++)
	{
	  pdeltasum += fabs(punits[pbesti][pbestj].comp[i] -
			  pvectors[inppairs[inpv].p].comp[i]);
	  if(fabs(punits[pbesti][pbestj].comp[i] -
			  pvectors[inppairs[inpv].p].comp[i])<perr) pnerr++;
	}
      if(determine_nearest(punits[pbesti][pbestj].comp,pvectors,pnvectors,pdim)
	 == inppairs[inpv].p) pncorr++;
      if (second == pbest)
	printf("Warning: pbest for %s is not unique\n",
	       pvectors[inppairs[inpv].p].label);
      if (displaying)
	if(newlabel(punits[pbesti][pbestj].labels,
		    punits[pbesti][pbestj].labelcount,
		    inppairs[inpv].p))
	  {
	    punits[pbesti][pbestj].labelcount++;
	    punits[pbesti][pbestj].labels[ punits[pbesti][pbestj].labelcount ]=
	      inppairs[inpv].p;
	  }
      if(t>=at0)
	{
	  second=(-1.0);
	  for(ii=0; ii<snets; ii++)
	    for(jj=0; jj<snets; jj++)
	      {
		/*	if (pbesti==0 && pbestj==5)
			printf("%d,%d = %f best:%d,%d = %f\n",
			ii,jj,psassoc[pbesti][pbestj][ii][jj],
			psbesti,psbestj,psbest);*/
		if(psassoc[pbesti][pbestj][ii][jj]==psbest) second=psbest;
		if(psassoc[pbesti][pbestj][ii][jj]>psbest)
		  {
		    psbesti=ii;
		    psbestj=jj;
		    psbest=psassoc[pbesti][pbestj][ii][jj];
		  }
	      }
	  if (second == psbest)
	    printf("Warning: psbest for %s is not unique\n",
		   pvectors[inppairs[inpv].p].label);
	  if (displaying)
	    if(newlabel(sunits[psbesti][psbestj].labels,
			sunits[psbesti][psbestj].labelcount,
			inppairs[inpv].p+snvectors))
	      {
		sunits[psbesti][psbestj].labelcount++;
		sunits[psbesti][psbestj].labels[sunits[psbesti][psbestj].labelcount]=					      
		  inppairs[inpv].p+snvectors;    /* > snvectors means assoclabel */
	      }
	}
    }
  
  if (t>=st0 && inppairs[inpv].s > (-1))
    {
      second=(-1.0);
      for(i=0; i<snets; i++)
	for(j=0; j<snets; j++){
	  sunits[i][j].value=distance(svectors[inppairs[inpv].s].comp,
				      sunits[i][j].comp,sdim);
	  /* check if this unit's response is best so far encountered */
	  if (sunits[i][j].value == sbest) second=sbest;
	  if (sunits[i][j].value < sbest) {
	    sbesti=i; sbestj=j; sbest=sunits[i][j].value;
	  }
	  if (displaying)
	    if (sunits[i][j].value < sunits[i][j].bestvalue){
	      sunits[i][j].bestvalue = sunits[i][j].value;
	      sunits[i][j].labels[0] = inppairs[inpv].s;
	    }
	}
      for(i=0; i<sdim; i++)
	{
	  sdeltasum += fabs(sunits[sbesti][sbestj].comp[i] -
			  svectors[inppairs[inpv].s].comp[i]);
	  if(fabs(sunits[sbesti][sbestj].comp[i] -
			  svectors[inppairs[inpv].s].comp[i])<serr) snerr++;
	}
      if(determine_nearest(sunits[sbesti][sbestj].comp,svectors,snvectors,sdim)
	 == inppairs[inpv].s) sncorr++;
      if (second == sbest)
	printf("Warning: sbest for %s is not unique\n",
	       svectors[inppairs[inpv].s].label);
      if (displaying)
	if(newlabel(sunits[sbesti][sbestj].labels,
		    sunits[sbesti][sbestj].labelcount,
		    inppairs[inpv].s))
	  {
	    sunits[sbesti][sbestj].labelcount++;
	    sunits[sbesti][sbestj].labels[ sunits[sbesti][sbestj].labelcount ]=
	      inppairs[inpv].s;
	  }
      if(t>=at0)
	{
	  second=(-1.0);
	  for(i=0; i<pnets; i++)
	    for(j=0; j<pnets; j++)
	      {
		if(spassoc[sbesti][sbestj][i][j]==spbest) second=spbest;
		if(spassoc[sbesti][sbestj][i][j]>spbest)
		  {
		    spbesti=i;
		    spbestj=j;
		    spbest=spassoc[sbesti][sbestj][i][j];
		  }
	      }
	  if (second == spbest)
	    printf("Warning: spbest for %s is not unique\n",
		   svectors[inppairs[inpv].s].label);
	  if (displaying)
	    if(newlabel(punits[spbesti][spbestj].labels,
			punits[spbesti][spbestj].labelcount,
			inppairs[inpv].s+pnvectors))
	      {
		punits[spbesti][spbestj].labelcount++;
		punits[spbesti][spbestj].labels[punits[spbesti][spbestj].labelcount]=
		  inppairs[inpv].s+pnvectors; /* > pnvectors means assoclabel */
	      }
	}
    }
  if(t>=pt0 && inppairs[inpv].p>(-1) &&
     t>=st0 && inppairs[inpv].s>(-1) &&
     t>=at0)
    {
      if (psbesti==sbesti && psbestj==sbestj) pscorr++;
      if (spbesti==pbesti && spbestj==pbestj) spcorr++;
    }  
}


newlabel(labels, count, num)
int labels[], count, num;
{
  int i;
  for(i=1; i<=count; i++)
    if(labels[i]==num)
      return(0);
  return(1);
}


modify_weights(pnc, palpha, snc, salpha, anc, aalpha, inpv)
int pnc, snc, anc, inpv;
float palpha, salpha, aalpha;
{
  float sum, maxpdist, maxsdist, ddist();
  register int i,j,ii,jj,k;
  /* modify weighs toward the input in a neighborhood of the maximally 
     responding unit */

  if(t>=pt0 && t<=pt2 && inppairs[inpv].p > (-1))
    for(i=pbesti-pnc; i<=pbesti+pnc; i++)
      for(j=pbestj-pnc; j<=pbestj+pnc; j++)
	if (i>=0 && i<pnets && j>=0 && j<pnets)
	  for(k=0; k<pdim; k++)
	    punits[i][j].comp[k] += palpha*
	      (pvectors[inppairs[inpv].p].comp[k] - punits[i][j].comp[k]);
  
  if(t>=st0 && t<=st2 && inppairs[inpv].s > (-1))
    for(i=sbesti-snc; i<=sbesti+snc; i++)
      for(j=sbestj-snc; j<=sbestj+snc; j++)
	if (i>=0 && i<snets && j>=0 && j<snets)
	  for(k=0; k<sdim; k++)
	    sunits[i][j].comp[k] += salpha*
	      (svectors[inppairs[inpv].s].comp[k] - sunits[i][j].comp[k]);

  if(t>=pt0 && t>=st0 && t>=at0 && t<=at2
     && inppairs[inpv].p > (-1) && inppairs[inpv].s > (-1))
    {
      if (snc==0) anc=1;
      else anc=snc;
/*      maxpdist = ddist(pnc, pnc);
      maxsdist = ddist(anc, anc);*/
      maxpdist = pnc+1.0;
      maxsdist = anc+1.0;
      if(((int) maxpdist)==0) maxpdist = 1.0;
      for(i=pbesti-pnc; i<=pbesti+pnc; i++)
	for(j=pbestj-pnc; j<=pbestj+pnc; j++)
/*      i=pbesti;
      j=pbestj;*/
      if (i>=0 && i<pnets && j>=0 && j<pnets)
	{
	  for(ii=sbesti-anc; ii<=sbesti+anc; ii++)
	    for(jj=sbestj-anc; jj<=sbestj+anc; jj++)
	      if (ii>=0 && ii<snets && jj>=0 && jj<snets)
		psassoc[i][j][ii][jj] += aalpha*
		  (1.0-ddist(i-pbesti,j-pbestj)/maxpdist)*
		  (1.0-ddist(ii-sbesti,jj-sbestj)/maxsdist);
/*		  (pmax-punits[i][j].value)*(smax-sunits[ii][jj].value);*/
	  sum =0.0;
	  for(ii=0; ii<snets; ii++)
	    for(jj=0; jj<snets; jj++)
	      sum += psassoc[i][j][ii][jj]*psassoc[i][j][ii][jj];
	  sum = sqrt(sum);
	  for(ii=0; ii<snets; ii++)
	    for(jj=0; jj<snets; jj++)
	      psassoc[i][j][ii][jj] /= sum;
	}	  
      
      if (pnc==0) anc=1;
      else anc=pnc;
/*      maxpdist = ddist(anc, anc);
      maxsdist = ddist(snc, snc);*/
      maxpdist = anc+1.0;
      maxsdist = snc+1.0;
      if(((int) maxsdist)==0) maxsdist = 1.0;
      for(ii=sbesti-snc; ii<=sbesti+snc; ii++)
	for(jj=sbestj-snc; jj<=sbestj+snc; jj++)
/*      ii=sbesti;
      jj=sbestj;*/
      if (ii>=0 && ii<snets && jj>=0 && jj<snets)
	{
	  for(i=pbesti-anc; i<=pbesti+anc; i++)
	    for(j=pbestj-anc; j<=pbestj+anc; j++)
	      if (i>=0 && i<pnets && j>=0 && j<pnets)
		{
		  
		spassoc[ii][jj][i][j] += aalpha*
		  (1.0-ddist(i-pbesti,j-pbestj)/maxpdist)*
		  (1.0-ddist(ii-sbesti,jj-sbestj)/maxsdist);
/*		  (pmax-punits[i][j].value)*(smax-sunits[ii][jj].value);*/
/*		  if(t==50 || t==100 || t==150)
	  printf("%d %d %d %d %d %f\n",t,i, j, ii, jj, aalpha*
		  (1.0-ddist(i-pbesti,j-pbestj)/maxpdist)*
		  (1.0-ddist(ii-sbesti,jj-sbestj)/maxsdist));*/
/*		  (pmax-punits[i][j].value)*(smax-sunits[ii][jj].value));*/
	       }
	  sum =0.0;
	  for(i=0; i<pnets; i++)
	    for(j=0; j<pnets; j++)
	      sum += spassoc[ii][jj][i][j]*spassoc[ii][jj][i][j];
	  sum = sqrt(sum);
	  for(i=0; i<pnets; i++)
	    for(j=0; j<pnets; j++)
	      spassoc[ii][jj][i][j] /= sum;
	}	  
    }
}

float ddist(x, y)
int x, y;
{
  return( sqrt( (float) (x*x+y*y)));
}    

/*********************  initializations ******************************/

init_params(argv)
char *argv[1];
{
  register int i, neti;
  int c;

  sprintf(simufile, "%s", argv[1]);
  fp=fopen(simufile,"r");
  read_params(fp);
  if((c=getc(fp))==EOF)
    continuing=0;
  else
    continuing=1;
  fclose(fp);

  starttime=time(0);
  startt=0;
  srand(seed);				/* start random number sequence */
}

read_params(fp)
FILE *fp;
{
  char s[100];
  register int i;
  /* simulation parameters */
  fscanf(fp,"%s", pvectorfile); fgets(s,99,fp);
  fscanf(fp,"%s", svectorfile); fgets(s,99,fp);
  fscanf(fp,"%s", inpfile); fgets(s,99,fp);

  fscanf(fp,"%d %d %d %d", &displaying,&testing,&seed,&tend); fgets(s,99,fp);
  fscanf(fp,"%f %f %f %f", &pleft,&plow,&pright,&phigh); fgets(s,99,fp);
  fscanf(fp,"%f %f %f %f", &sleft,&slow,&sright,&shigh); fgets(s,99,fp);
  fscanf(fp,"%d %d %d %d %d %d %f %f", &pnets,&pt0,&pt1,&pt2,&pnct0,&pnct1,
	                            &palphat0,&palphat1); fgets(s,99,fp);
  fscanf(fp,"%d %d %d %d %d %d %f %f", &snets,&st0,&st1,&st2,&snct0,&snct1,
	                            &salphat0,&salphat1); fgets(s,99,fp);
  fscanf(fp,"%d %d %d %d %d %f %f",           &at0,&at1,&at2,&anct0,&anct1,
	                            &aalphat0,&aalphat1); fgets(s,99,fp);
  
  /* saving info */
  fscanf(fp,"%d", &snapshots[0]);
  for(i=0; i<maxsnaps && snapshots[i]<snapshotend; i++)
    fscanf(fp,"%d", &snapshots[i+1]);
  fgets(s,99,fp);
  nextsnapshot=0;
}


read_inputs()
{
  register int i,j,k;
  char s[100];
  float f01rnd();

  fp=fopen(pvectorfile,"r");
  /* read plabelfilename and pvector dimension */
  fscanf(fp,"%s", plabelfile); fgets(s,99,fp);
  fscanf(fp,"%d", &pdim); fgets(s,99,fp);

  /* read the labels */
  fp2=fopen(plabelfile,"r");
  for(i=0; i<maxvectors && fscanf(fp2,"%s", pvectors[i].label)!=EOF; i++){}
  pnvectors=i;
  fclose(fp2);

  /* read the vectors */
  for(i=0; i<pnvectors; i++)
    for(j=0; j<pdim; j++)
      fscanf(fp,"%f",&pvectors[i].comp[j]);
  fclose(fp);

  
  fp=fopen(svectorfile,"r");
  /* read slabelfilename and svector dimension */
  fscanf(fp,"%s", slabelfile); fgets(s,99,fp);
  fscanf(fp,"%d", &sdim); fgets(s,99,fp);

  /* read the labels */
  fp2=fopen(slabelfile,"r");
  for(i=0; i<maxvectors && fscanf(fp2,"%s", svectors[i].label)!=EOF; i++){}
  snvectors=i;
  fclose(fp2);

  /* read the vectors */
  for(i=0; i<snvectors; i++)
    for(j=0; j<sdim; j++)
      fscanf(fp,"%f",&svectors[i].comp[j]);
  fclose(fp);
  
  /* read the inputpairs */
  pninp=sninp=aninp=0;
  fp=fopen(inpfile,"r");
  for(i=0; i<maxinps && fscanf(fp,"%d", &inppairs[i].p)!=EOF; i++)
    {
      fscanf(fp,"%d", &inppairs[i].s);
      if(inppairs[i].p>(-1)) pninp++;
      if(inppairs[i].s>(-1)) sninp++;
      if(inppairs[i].p>(-1) && inppairs[i].s>(-1)) aninp++;
    }
  ninppairs=i;
  fclose(fp);
  
  for(i=0; i<ninppairs; i++)
    shuffletable[i]=i;
/*  shuffle();*/
}


init_weights()
{
  register int i,j,ii,jj;
  int randfun();
  float psvalue, spvalue;
  iterate_weights(randfun);
  psvalue = 1.0/sqrt((float) (snets*snets));
  spvalue = 1.0/sqrt((float) (pnets*pnets));
  for(i=0; i<pnets; i++)
    for(j=0; j<pnets; j++)
      for(ii=0; ii<snets; ii++)
	for(jj=0; jj<snets; jj++)
	  {
	    psassoc[i][j][ii][jj] = psvalue;
	    spassoc[ii][jj][i][j] = spvalue;
	  }
  if (snapshots[0]==(-1))
    save_current();			/* save the initial state */
}


iterate_weights(dofun)
int (*dofun)();
{
  register int i,j,k,ii,jj;

  for(i=0; i<pnets; i++)
    for(j=0; j<pnets; j++)
      for(k=0; k<pdim; k++)
	(*dofun)(&punits[i][j].comp[k]);

  for(i=0; i<snets; i++)
    for(j=0; j<snets; j++)
      for(k=0; k<sdim; k++)
	(*dofun)(&sunits[i][j].comp[k]);

  for(i=0; i<pnets; i++)
    for(j=0; j<pnets; j++)
      for(ii=0; ii<snets; ii++)
	for(jj=0; jj<snets; jj++)
	  {
	    (*dofun)(&psassoc[i][j][ii][jj]);
	    (*dofun)(&spassoc[ii][jj][i][j]);
	  }  
}  

init_units()
{
  register int i,j;
  for(i=0; i<pnets; i++)
    for(j=0; j<pnets; j++) 
      {
	punits[i][j].bestvalue = 999999999.9;
	punits[i][j].labelcount= 0;
      }
  for(i=0; i<snets; i++)
    for(j=0; j<snets; j++) 
      {
	sunits[i][j].bestvalue = 999999999.9;
	sunits[i][j].labelcount= 0;
      }
}


int determine_nearest(rep, vectors, nvectors,dim)
struct inputvectors vectors[maxvectors];
float rep[];
int nvectors,dim;
{
  int i,bestindex;
  float lbest, dist, distance();
  lbest=999999999.9;
  for(i=0; i<nvectors; i++)
    {
      dist=distance(vectors[i].comp, rep, dim);
      if(dist<lbest)
	{
	  bestindex=i;
	  lbest=dist;
	}
    }
  return(bestindex);
}


/***************   display routines   *******************************/
 
#if notcray
display_init()
{
  char s[40];
  register int i;

  fildes=gopen(getenv("SB_OUTDEV"),OUTDEV,getenv("SB_OUTDRIVER"),0);

/*  fp=fopen(cmapfile, "r");
  for(i=0; fscanf(fp,"%f%f%f",&colors[i][0],&colors[i][1],&colors[i][2])!=EOF; i++);
  fclose(fp);
  cmapsize=i;
  define_color_table(fildes,0,cmapsize,colors);*/

  mapping_mode(fildes,DISTORT);
  background_color_index(fildes,clear_color);
/*  clear_view_surface(fildes);*/
  perimeter_color_index(fildes,frame_color);
  interior_style(fildes, INT_HOLLOW, TRUE);

  text_alignment(fildes, TA_CENTER, TA_NORMAL_VERTICAL, 0.0, 0.0);
  text_path(fildes, PATH_RIGHT);

  pwdth=phght=1.0/pnets;
  swdth=shght=1.0/snets;
  character_expansion_factor(fildes, expansion);
}

display_best()
{
  char s[100];
  register int i,j,k;

  if(t>=pt0)
    {
      set_p1_p2(fildes, FRACTIONAL, pleft, plow, 0.0, pright, phigh, 0.0);
      character_height(fildes, plabelsize);
      for(i=0; i<pnets; i++)
	for(j=0; j<pnets; j++)
	  {
	    interior_style(fildes, INT_SOLID, TRUE);
	    fill_color_index(fildes,clear_color);
	    rectangle(fildes, i*pwdth, j*phght, (i+1)*pwdth, (j+1)*phght);
	    text_color_index(fildes, bestvector_color);
/*	    text2d(fildes, (i+0.5)*pwdth, (j+1)*phght-(0.75*plabelsize),
		   pvectors[punits[i][j].labels[0]].label, VDC_TEXT, FALSE);*/
	    for(k=1; k<punits[i][j].labelcount+1; k++)
	      if (punits[i][j].labels[k] <pnvectors)
		{
		  text_color_index(fildes, bestunit_color);
		  text2d(fildes,(i+0.5)*pwdth,
			 (j+1)*phght-(k+1)*(0.75*plabelsize),
			 pvectors[punits[i][j].labels[k]].label,
			 VDC_TEXT, FALSE);
		}
	      else
		{
		  text_color_index(fildes, assocunit_color);
		  text2d(fildes,(i+0.5)*pwdth,
			 (j+1)*phght-(k+1)*(0.75*plabelsize),
			 svectors[punits[i][j].labels[k]-pnvectors].label,
			 VDC_TEXT, FALSE);
		}
	  }
      character_height(fildes, 2*slabelsize);
      text_color_index(fildes, bestunit_color);
      sprintf(s,"%d",t);
      text2d(fildes, 0.5, 0.5, s, VDC_TEXT, FALSE);
      make_picture_current(fildes);
    }
  
  if(t>=st0)
    {
      set_p1_p2(fildes, FRACTIONAL, sleft, slow, 0.0, sright, shigh, 0.0);
      character_height(fildes, slabelsize);
      for(i=0; i<snets; i++)
	for(j=0; j<snets; j++)
	  {
	    interior_style(fildes, INT_SOLID, TRUE);
	    fill_color_index(fildes,clear_color);
	    rectangle(fildes, i*swdth, j*shght, (i+1)*swdth, (j+1)*shght);
	    text_color_index(fildes, bestvector_color);
/*	    text2d(fildes, (i+0.5)*swdth, (j+1)*shght-(0.75*slabelsize),
		   svectors[sunits[i][j].labels[0]].label, VDC_TEXT, FALSE);*/
	    for(k=1; k<sunits[i][j].labelcount+1; k++)
	      if (sunits[i][j].labels[k] <snvectors)
		{
		  text_color_index(fildes, bestunit_color);
		  text2d(fildes,(i+0.5)*swdth,
			 (j+1)*shght-(k+1)*(0.75*slabelsize),
			 svectors[sunits[i][j].labels[k]].label,
			 VDC_TEXT, FALSE);
		}
	      else
		{
		  text_color_index(fildes, assocunit_color);
		  text2d(fildes,(i+0.5)*swdth,
			 (j+1)*shght-(k+1)*(0.75*slabelsize),
			 pvectors[sunits[i][j].labels[k]-snvectors].label,
			 VDC_TEXT, FALSE);
		}
	  }
      character_height(fildes, 2*slabelsize);
      text_color_index(fildes, bestunit_color);
      sprintf(s,"%d",t);
      text2d(fildes, 0.5, 0.5, s, VDC_TEXT, FALSE);
      make_picture_current(fildes);
    }
}

display_assocs()
{
  float radius;
  int i,j,ii,jj;
  for(ii=0; ii<snets; ii++)
    for(jj=0; jj<snets; jj++)
      {
	display_best();
	set_p1_p2(fildes, FRACTIONAL, sleft, slow, 0.0, sright, shigh, 0.0);
	fill_color_index(fildes,frame_color);
	interior_style(fildes, INT_HOLLOW, TRUE);
	rectangle(fildes,
		  (ii+0.4)*swdth,(jj+0.4)*shght,(ii+0.6)*swdth,(jj+0.6)*shght);
	set_p1_p2(fildes, FRACTIONAL, pleft, plow, 0.0, pright, phigh, 0.0);
	for(i=0; i<pnets; i++)
	  for(j=0; j<pnets; j++)
	    {
	      radius = spassoc[ii][jj][i][j]/snets/2.0;
	      rectangle(fildes,
			(i+0.5)*pwdth-radius, (j+0.5)*phght-radius,
			(i+0.5)*pwdth+radius, (j+0.5)*phght+radius);
	    }
	make_picture_current(fildes);
	while (getchar()!='\n');
      }
  for(i=0; i<pnets; i++)
    for(j=0; j<pnets; j++)
      {
	display_best();
	set_p1_p2(fildes, FRACTIONAL, pleft, plow, 0.0, pright, phigh, 0.0);
	fill_color_index(fildes,frame_color);
	interior_style(fildes, INT_HOLLOW, TRUE);
	rectangle(fildes,
		  (i+0.4)*pwdth, (j+0.4)*phght, (i+0.6)*pwdth, (j+0.6)*phght);
	set_p1_p2(fildes, FRACTIONAL, sleft, slow, 0.0, sright, shigh, 0.0);
	for(ii=0; ii<snets; ii++)
	  for(jj=0; jj<snets; jj++)
	    {
	      radius = psassoc[i][j][ii][jj]/pnets/2.0;
	      rectangle(fildes,
			(ii+0.5)*swdth-radius, (jj+0.5)*shght-radius,
			(ii+0.5)*swdth+radius, (jj+0.5)*shght+radius);
	    }
	make_picture_current(fildes);
	while (getchar()!='\n');
      }
}
      
display_selected_assocs()
{
  int k,i,j,ii,jj,x,y;
  float radius;
  char s[100];
  display_best();
  for(k=0; scanf("%s", s)!=EOF; k++)
    {
      if (s[0]=='p')
	{
	  scanf("%d %d", &x, &y);
	  display_best();
	  set_p1_p2(fildes,FRACTIONAL, pleft, plow, 0.0, pright, phigh, 0.0);
	  fill_color_index(fildes,frame_color);
	  interior_style(fildes, INT_HOLLOW, TRUE);
	  rectangle(fildes,
		    (x+0.4)*pwdth,(y+0.4)*phght,(x+0.6)*pwdth,(y+0.6)*phght);
	  set_p1_p2(fildes,FRACTIONAL, sleft, slow, 0.0, sright, shigh, 0.0);
	  for(ii=0; ii<snets; ii++)
	    for(jj=0; jj<snets; jj++)
	      {
		radius = psassoc[x][y][ii][jj]/snets/2.0;
		rectangle(fildes,
			  (ii+0.5)*swdth-radius, (jj+0.5)*shght-radius,
			  (ii+0.5)*swdth+radius, (jj+0.5)*shght+radius);
	      }
	  make_picture_current(fildes);
	  printf("p: %d, %d -> s\n", x, y);
	  for(jj=snets-1; jj>=0; jj--)
	    {
	      for(ii=0; ii<snets; ii++)
		printf(" %.3f", psassoc[x][y][ii][jj]);
	      printf("\n");
	    }
	}
      else if (s[0]=='s')
	{
	  scanf("%d %d", &x, &y);
	  display_best();
	  set_p1_p2(fildes,FRACTIONAL, sleft, slow, 0.0, sright, shigh, 0.0);
	  fill_color_index(fildes,frame_color);
	  interior_style(fildes, INT_HOLLOW, TRUE);
	  rectangle(fildes,
		    (x+0.4)*swdth,(y+0.4)*shght,(x+0.6)*swdth,(y+0.6)*shght);
	  set_p1_p2(fildes,FRACTIONAL, pleft, plow, 0.0, pright, phigh, 0.0);
	  for(i=0; i<pnets; i++)
	    for(j=0; j<pnets; j++)
	      {
		radius = spassoc[x][y][i][j]/pnets/2.0;
		rectangle(fildes,
			  (i+0.5)*pwdth-radius, (j+0.5)*phght-radius,
			  (i+0.5)*pwdth+radius, (j+0.5)*phght+radius);
	      }
	  make_picture_current(fildes);
	  printf("s: %d, %d -> p\n", x, y);
	  for(j=pnets-1; j>=0; j--)
	    {
	      for(i=0; i<pnets; i++)
		printf(" %.3f", spassoc[x][y][i][j]);
	      printf("\n");
	    }
	}
    }
}
  
#endif

/*********************** I/O etc functions *****************/ 

save_current()
{
  int writefun();
  /*printf("%d\n", time(0)-starttime);*/
  fp=fopen(simufile,"a");
  fprintf(fp,"%d \n", t);
  iterate_weights(writefun);
  fclose(fp);
  nextsnapshot++;
}

shuffle()
{
  register int i;
  int temp1,temp2,temp3;
  for(i=0; i<ninppairs; i++)
    {
      temp1=rand()%ninppairs;
      temp2=rand()%ninppairs;
      temp3=shuffletable[temp1];
      shuffletable[temp1]=shuffletable[temp2];
      shuffletable[temp2]=temp3;
    }
}

readfun(place)
float *place;
{
  fscanf(fp,"%f", place);
} 

writefun(place)
float *place;
{
  fprintf(fp,"%f\n", *place);
} 

randfun(place)
float *place;
{
  float f01rnd();
  *place = f01rnd();
} 

      
/***********************  math stuff **************************/

float distance(v1, v2, ncomp)
int ncomp;
float v1[],v2[];
{
  float sum=0.0;
  register int i;
  /* unit response is equal to the euclidian distance of input and weight */
  for(i=0; i<ncomp; i++)
    sum=sum+(v1[i]-v2[i])*(v1[i]-v2[i]);
  return(sqrt(sum));
}

float alpha(t,t0,t1,t2,alphat0,alphat1)
int t,t0,t1,t2;
float alphat0,alphat1;
{
  /* decrease gain linearly from alphat0 to alphat1 during the period t0-t1
     and linearly from alphat1 to 0 during the period t1-t2 */
  if (t<t1) return(alphat0-(t-t0)*(alphat0-alphat1)/(t1-t0));
  else      return(alphat1-(t-t1)*alphat1/(t2-t1));
}

nc(t,t0,t1,t2,nct0,nct1)
int t,t0,t1,t2,nct0,nct1;
{
  /* decrease neighborhood size linearly from nct0 to nct1+1 during
     the period t0-t1 and from nct1 to 0 during the period t1-t2 
  return(nct0); */
  if (t<t1) return(floor(nct0+0.999999-(t-t0)*(0.0+nct0-nct1)/(t1-t0)));
  else      return(floor(nct1+0.999999-(t-t1)*(nct1+0.999999)/(t2-t1)));
}
 
float f01rnd()
{
  /* random float between [0,1] */
  return (rand()/32767.0);
}
