;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;     			     ANSIL
;;; 		      Advanced Network Simulator In Lisp
;;; 		      ----------------------------------
;;;
;;;				  Written By
;;;			       Peter J. Angeline
;;;			      Gregory M. Saunders
;;;
;;;			   The Ohio State University
;;;		       Laboratory for AI Research (LAIR)
;;;			     Columbus, Ohio, 43210
;;;
;;;			      Copyright (c) 1991
;;;
;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


;;;--------------------------------------------------------------------------
;;;
;;;                            Filename: random.lisp
;;;                            Authors: Franz Inc. 
;;;                                     Peter J Angeline
;;;
;;; Comments: Allows an implementation independent version of a random seed.
;;;
;;;---------------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  
;;  Copyright 1989, 1990 by Franz Inc.  This code may be
;;  distributed without charge and included in other products
;;  so long as:
;;
;;  1.  When distributed in source form, this copyright
;;      notice is preserved.
;;
;;  2.  An acknowledgement crediting Franz Inc. as the
;;      source of the code appears with other notices, copyrights,
;;      and acknowledgements in any documentation of the code
;;      (whether printed or online).
;;
;;  This file contains two function definitions and one
;;  variable definition.  It is designed to provide a faster
;;  random number generator for random integers between zero and
;;  a specific integer LESS THAN 2**27.  The defined functions
;;  are:
;;  
;;  RANDOM-INT:  This function takes an integer argument (num)
;;  between 2 and 2**27.  It returns a pseudo random integer
;;  between 0 and its argument - 1.  As a side effect it changes
;;  the value of *RANDOM-INT*.  It implements Algorithm A on
;;  page 27 of Knuth `Art of Computer Programming, Vol 2'.  That
;;  algorithm is used to generate a random integer between 0 and
;;  2**27 - 1.  The remainder of dividing that integer by the
;;  num is then returned UNLESS the random integer lies
;;  between (* n num) and 2**27 where n is the result of
;;  (truncate 2**27 num).  If the integer does lie between those
;;  values, a new random integer is generated.  For this reason,
;;  the efficiency of this algorithm decreases as the argument
;;  approaches 2**27.  Values over 2**26 as an argument are not
;;  recommended.  As written, no argument checking is done.  You
;;  may uncomment the argument checking lines in the function
;;  definition if desired.  This function must be compiled to
;;  run.  Note that the Lisp function RANDOM is *never* called
;;  by RANDOM-INT.
;;  
;;  MAKE-RANDOM-STATE:  A random-int-state is an a one
;;  dimensional array of type (simple-array fixnum 57).  The
;;  first 55 entries are integers between 0 and 2**27 - 1
;;  (inclusive) at least one of which is odd.  The final two
;;  entries are integers whose difference is 31 mod 55.  This
;;  function creates (and returns) such an object.  If you set
;;  the value of *RANDOM-SEED* to such an object, a new sequence
;;  of will be generated.  BE SURE TO SAVE THE NEW INITIAL VALUE
;;  OF *RANDOM-SEED*.  THE OBSERVED BEHAVIOR OF RANDOM-INT CAN
;;  NEVER BE REPEATED IF THE VALUE OF *RANDOM-SEED* IS NOT
;;  SAVED.
;;  
;;  *RANDOM-SEED*:  The value of this global variable must be a
;;  one dimensional array of type (simple-array fixnum 57)
;;  whose first 55 entries are integers between 0 and 2**27 - 1
;;  (inclusive) at least one of which is odd.  The final two
;;  entries are integers whose difference is 31 mod 55. 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; (pja) - This code was provided in its original form by Franz Inc as stated 
;;;         above.  I have adapted the code to return floating points between
;;;         0 and the optional number provided in the function call.  The
;;;         function random-number returns a value of type float.


;;; ----------------------------------------------------------------------
;;;
;;;          Common Lisp Independent Random Number Generator
;;;
;;; ----------------------------------------------------------------------

(in-package :mbp)
(export '(random-int *random-seed* make-random-state random-float))

(defun random-int (num)
  (declare (fixnum num) (optimize (speed 3)) (special *random-seed*))
  (declare (type (simple-array fixnum (*)) *random-seed*))
;;  Uncomment to enable argument error checking.
;;(if (or (null (fixnump num)) (< num 2) (> num #.(expt 2 27))) 
;;    (error "Out of range or bad value to RANDOM-SEED ~S~%" num))
  (let* ((bound 
	   (multiple-value-bind (q r)
	       (truncate #.(expt 2 27) num)
	     (if (zerop r) (1+ q) q)))
	 (j (aref *random-seed* 55))
	 (k (aref *random-seed* 56))
	 (j1 (1- j))
	 (k1 (1- k))
	 (y1 (aref *random-seed* j))
	 (y2 (aref *random-seed* k))
	 (x (logand #.(1- (expt 2 27)) (+ y1 y2))))
    (declare (fixnum j) (fixnum k) (fixnum j1) (fixnum k1)
	     (fixnum y1) (fixnum y2) (fixnum x) (fixnum q)
	     (fixnum r) (fixnum bound))
    (setf (aref *random-seed* k) x)
    (setf (aref *random-seed* 55) (if (minusp j1) 54 j1))
    (setf (aref *random-seed* 56) (if (minusp k1) 54 k1))
    (multiple-value-bind (q r)
        (truncate x num)
      (if (< q bound) r (random-int num)))))

(defvar *random-seed* 
  (make-array 57 :element-type 'fixnum :initial-contents
    '(93922509 93908763 108033456 56892205 73861679 72040972
    82409779 134009448 34406098 5536021 109836325 91137727
    48138720 37638482 112465523 13978365 79769853 64010871
    73065460 71476796 84272966 36597937 6065141 66844603
    103304320 74955022 50857000 113960830 22054126 103241462
    96380795 23146032 116906571 29268252 19321182 52070104
    121650798 111191178 94483939 30680265 101920795 87747622
    104487767 40038502 35482225 105974597 109691316 87184675
    31600524 58524102 117107845 118256475 26404552 70916151
    30976942 23 54)))

(defun make-random-state ()
  (let ((lis (list 23 54)))
    (dotimes (i 55) (setq lis (push (random #.(expt 2 27)) lis)))
    (if (evenp (first lis)) (setf (first lis) (1+ (first lis))))
    (make-array 57 :element-type 'fixnum :initial-contents lis)))

(defun random-float(&optional (max 1.0d0))
  (* (float max) (float (/ (random-int #.(expt 2 27)) #.(expt 2 27)))))

;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

