;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;     			     ANSIL
;;; 		      Advanced Network Simulator In Lisp
;;; 		      ----------------------------------
;;;
;;;				  Written By
;;;			       Peter J. Angeline
;;;			      Gregory M. Saunders
;;;
;;;			   The Ohio State University
;;;		       Laboratory for AI Research (LAIR)
;;;			     Columbus, Ohio, 43210
;;;
;;;			      Copyright (c) 1991
;;;
;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


;;; ---------------------------------------------------------------------- 
;;; File: raam2.lisp -  Raam maintenance... viewing, saving & analysis
;;;
;;; License: This code may be distributed free of charge.  This code may not be
;;; incorporated into any production software of any type or any code which is
;;; sold without the express written consent of the authors.  Modifications and
;;; inclusions for other types of software are permitted, provided that this
;;; header remains unchanged and included in all subsequent versions of the
;;; code.
;;;
;;; Disclaimer: The authors of this code make no claims as to its
;;; validity, and are in no way responsible for any errors contained within.
;;;
;;; Creation date: Feb 28, 1991
;;; Version: 0.95
;;; ----------------------------------------------------------------------
;;;
;;;
;;;                         Change Log
;;;                         ----------
;;;
;;; 02/28/91 - (gms) split off analysis code from the main raam code.
;;;
;;; 03/14/91 - (gms) added test for "same-structure" in decoding trees
;;; 
;;; 04/01/91 - (pja) altered write-raam so that the deltas, errors and
;;; activations are not written to the file.
;;;
;;; 04/04/91 - (pja) changed encode so that it handled unbalanced trees.  See
;;; new function encode3 which replaces encode2.
;;; ----------------------------------------------------------------------



;;; ----------------------------------------------------------------------
;;;
;;; Package info, exported functions, etc.
;;;
;;; ----------------------------------------------------------------------

(proclaim '(optimize speed))

(provide 'mraam)
(in-package 'mraam)

(export '(show-raam 
	  write-raam read-raam write-raam-symbol-table write-raam-reps
	  fix-raam-arrays
	  verify uncompress
	  reconstruct-tree reconstruct-trees 
	  reconstruct-stack-tree reconstruct-stack-trees
	  encode))


(require 'loop)
(require 'linear)
(require 'mbp)

(use-package 'loop)
(use-package 'linear)
(use-package 'mbp)


;;; ----------------------------------------------------------------------
;;;
;;; Printing out information to the user.
;;;
;;; ----------------------------------------------------------------------

(defun print-list (trees &optional (header nil))
  (if header (format t "~a~%" header))
  (format t "~{  ~a~%~}~%" trees))

(defun max-length (r &key (key #'identity))
  (if (null r) 0
    (apply #'max (loop for x in r collect
		       (length (princ-to-string (funcall key x)))))))

(defun to-string (x &optional (n nil))
  (let ((temp$ (princ-to-string x)))
    (if n (subseq temp$ 0 (min n (length temp$)))
      temp$)))
   
(defun show-raam-symbol-table (x &key
				 (digits 2) 
				 (width (+ digits 3))
				 (terminals-only nil)
				 (max-symbol-length
				  (max-length (raam-symbol-table x)
					      :key #'car))  
				 &allow-other-keys)
  (let* ((width$  (princ-to-string width))
	 (digits$ (princ-to-string digits))
	 (max$    (princ-to-string max-symbol-length))
	 (f-vec$  (concatenate 'string "#(~{~" width$ "," digits$ "f~})"))
	 (f$      (concatenate 'string "  ~" max$ "a = ")))
  (format t "Symbol table~%")
  (loop for sym-val in (raam-symbol-table x)
	for symbol = (car sym-val)
	when (or (not terminals-only)
		 (terminal? symbol))
	do
	(format t f$ (to-string symbol max-symbol-length))
	(show-vector (cdr sym-val) f-vec$)
	(terpri))
  (terpri)))

(defun show-raam-training-set (raam &key (verbose nil) (digits 2)
				         (width (+ digits 4))
				    (max-symbol-length
				     (max-length (raam-training-set raam))) 
				    &allow-other-keys)
  (let* ((symtab (raam-symbol-table raam))
	 (width$  (princ-to-string width))
	 (digits$ (princ-to-string digits))
	 (max$    (princ-to-string max-symbol-length))
	 (f-vec$  (concatenate 'string "#(~{~" width$ "," digits$ "f~})"))
	 (f$      (concatenate 'string "~%  ~" max$ "a = ")))
    (format t "Training set")
    (if verbose (format t " - format: (tree . index) = ( #(target - actual)  #(target - actual)  ...)"))
    (loop for x in (raam-training-set raam) do
	  (format t f$ (to-string x max-symbol-length))
	  (when (not (dummy-vector? (symbol-value-by-index symtab (cdr x))))
		(loop for v in (decode raam (cdr x) :difference t) do
		      (format t "(")
		      (show-vector v f-vec$)
		      (format t "    "))
		(format t ")")))
    (terpri)))


(defun show-raam (x &rest params &key (verbose nil) &allow-other-keys)
  "Print key fields of a raam.  &key :verbose t ===> prints information about
trees" 
  (unless (typep x 'raam) (error "show-raam called with non-raam"))
  (let* ((v (raam-valence x))
	 (net (raam-network x))
	 (parameter-list `("valence" ,v
			   "rep-width" ,(raam-rep-width x)
			   "terminal-width" ,(raam-terminal-width x)
			   "raam-width" ,(raam-raam-width x)))
	 (r1 (length (raam-fixed-trees x)))
	 (r2 (length (raam-org-trees x))))
    (format t "~%=====================================================~%")
    (format t "~a, created ~a" (raam-name x) (raam-creation-date x))
    (format t "~%=====================================================~%~%")
    (format t "Parameters~%~{  ~14a = ~a ~%~}~%" parameter-list)
    (if verbose 
	(progn
	  (print-list (raam-org-trees x) "Original trees")  
	  (print-list (raam-fixed-trees x)
		      (format nil "Trees after conversion to valence ~a" v))
	  (if (< r1 r2)
	      (format t "Warning:  ~a duplicate tree~:p after conversion to fixed valence~%~%"
		      (- r2 r1)))
	  (apply #'show-raam-network (cons x params))
	  (if (raam-history x)
	      (progn
		(format t "History list~%")
		(loop for x in (raam-history x) do
		      (format t "  At epoch ~4@a, ~a := ~a~%"
			      (history-record-epoch x)
			      (history-record-attribute x)
			      (history-record-value x)))
		(terpri)
		(terpri)))))
    (format t "Training information (*'s denote values user can set with \"set-raam\")~%")
    (format t "~{  ~23a= ~8,6f~%~}"
	    `("*Rate"                  ,(network-rate net)
	      "*Momentum"              ,(network-momentum net)
	      "*Terminal tolerance"    ,(raam-term-tol x)
	      "*Nonterminal tolerance" ,(raam-nonterm-tol x)
	      " Total error"           ,(network-total-error net)
	      " Max error"             ,(network-max-error net)))
    (format t "  ~23a= ~a~%~%" " Total epochs" (raam-total-epochs x))
    (apply #'show-raam-symbol-table (cons x params))
    (apply #'show-raam-training-set (cons x params))))

(defun show-raam-network (raam &rest params &key &allow-other-keys)
  (apply #'show-network (cons (raam-network raam) params))
  (format t "~%"))



;;; ----------------------------------------------------------------------
;;;
;;; Semi-analysis
;;;
;;; ----------------------------------------------------------------------

(defun decode (raam i &key (difference nil))
  (let* ((net (raam-network raam))
	 (symtab (raam-symbol-table raam))
	 (valence (raam-valence raam))
	 (temp-res ())
	 (act ())
	 (tree  (symbol-name-by-index symtab i)))
    (set-nodes net (symbol-value-by-index symtab i) *new-rep-level*)
    (forward-pass net :min-level *new-rep-level*)
    (setf act (pointer-to-activation-of-level (1+ *new-rep-level*) net))
    (setf temp-res 
	  (if difference 
	      (let* ((terminal-bits
		      (mapcar #'(lambda (x) (if (terminal? x) one zero)) tree))
		     (terminal-test
		      (make-array (list valence)
				  :element-type *element-type*
				  :initial-contents terminal-bits))
		     (codes (loop for x in tree collect
				  (symbol-value-by-name symtab x)))
		     (code (apply #'append-vector
				  (cons terminal-test codes))))
		(vector-apply+ nil code (vector-apply* nil act (- one))))
	    (copy-seq act)))
    (finish-output)
    (loop for subtree in tree
	  for len = (length (symbol-value-by-name symtab subtree))
	  for nodestart = valence then nodeend
	  for nodeend = (+ nodestart len)
	  collect (subseq temp-res nodestart nodeend))))



;;; ----------------------------------------------------------------------
;;;
;;; Saving & restoring 
;;;   Extension are added automatically when saving & restoring.  The 
;;;   default for the save name comes from the "name" field of the raam
;;;   structure (set by keyword in "create-raam").
;;;
;;; ---------------------------------------------------------------------- 

(defun write-raam (r &optional (filename (raam-name r)))
  (unless (typep r 'raam) (error "First argument must be of type \"raam\""))
  (let* ((network (raam-network r))
	 (deltas (network-deltas network))
	 (errors (network-errors network))
	 (acts (network-activations network)))
    (setf (network-deltas network) nil	; don't need to save these in file
	  (network-errors network) nil
	  (network-activations network) nil)
    (with-open-file (fi (concatenate 'string filename ".raam")
			:direction :output :if-exists :supersede)
		    (pprint r fi))
    (setf (network-deltas network) deltas ; restore current network's arrays
	  (network-errors network) errors
	  (network-activations network) acts))
  nil)

(defun read-raam (filename)
  (let ((tempraam 
	 (with-open-file (fi (concatenate 'string filename ".raam")
			     :direction :input)
			 (read fi))))
    (fix-raam-arrays tempraam)
    tempraam))

(defun write-raam-symbol-table (r &optional (filename (raam-name r)))
  (if (zerop (raam-total-epochs r)) (error "Raam hasn't been trained yet"))
  (let ((trees (loop for x in (raam-symbol-table r) collect (car x)))
	(vecs  (loop for x in (raam-symbol-table r)
		     collect (coerce (cdr x) 'list)))
	(tree-file (concatenate 'string filename ".trees")))
    (with-open-file (fi tree-file :direction :output :if-exists :supersede) 
		    (format fi "~{~a~%~}" trees))
    (with-open-file (fi (concatenate 'string filename ".vecs") 
			:direction :output :if-exists :supersede)
		    (loop for v in vecs do
			  (loop for a in v do (format fi "~e " a))
			  (terpri fi)))))

;;; ----------------------------------------------------------------------
;;;
;;; Ugly patch to fix a problem with saving/restoring raams.
;;; The problem is that when an array of type DOUBLE-FLOAT is written to
;;; disk, it will be read back in as an array of type TRUE.  The following
;;; routine alleviates this problem for raams by walking through the
;;; structure and effectively changing all arrays to be of type
;;; *element-type*.  Yuck.  
;;;
;;; See matrixbp.lisp for a similar fix for networks.
;;;
;;; ----------------------------------------------------------------------

(defun fix-raam-arrays (raam)
  (let ((bad-symtabs (list (raam-init-terminals raam)
			   (raam-symbol-table raam))))
    (loop for bad-symtab in bad-symtabs do
	  (loop for x in bad-symtab do
		(setf (cdr x) (mbp::fix-matrix (cdr x)))))
    (mbp::fix-network-arrays (raam-network raam))))


;;; ---------------------------------------------------------------------------
;;;
;;; Write the raam representations to a file.  The main use of this function is
;;; to show the user how to access the representations, since any output will
;;; probably need to be application-specific (example for a clustering, etc.) 
;;;
;;; ---------------------------------------------------------------------------

(defun write-raam-reps (raam &key (file$ (raam-name raam)) (subtrees? t))
  (unless (typep raam 'raam) (error "First argument must be of type \"raam\""))
  (let ((trees (if subtrees? (loop for x in (raam-symbol-table raam)
				   collect (car x))
		 (raam-fixed-trees raam)))
	(symtab (raam-symbol-table raam)))
  (with-open-file (fi (concatenate 'string file$ ".reps")
		      :direction :output :if-exists :supersede)
		  (loop for tr in trees do
			(format fi "~a~%~a~%" 
				tr
				(symbol-value-by-name symtab tr))))))


;;; ----------------------------------------------------------------------
;;;
;;; Uncompress a reduced representation
;;;
;;; ----------------------------------------------------------------------
(defun uncompress (raam i)
  (let* ((net (raam-network raam))
	 (symtab (raam-symbol-table raam)))
    (set-nodes net (symbol-value-by-index symtab i) *new-rep-level*)
    (forward-pass net :min-level *new-rep-level*)
    (car (last (network-activations (raam-network raam))))))


;;; ----------------------------------------------------------------------
;;;
;;; Reconstruction routines - The following functions reconstruct trees from
;;; the hidden unit representations.  Reconstruct tree takes a hidden unit
;;; representation (a vector) and unwraps it, i.e., it decodes it once, then
;;; looks at whether it has terminals or nonterminals.  Terminals are reported
;;; as such, and nonterminals are decoded recursively.  The other functions are
;;; conveniences, allowing for instance someone to reconstruct all of the trees
;;; of a raam.
;;;
;;; ----------------------------------------------------------------------

(defun same-structure? (t1 t2)
  (or (and (atom t1) (atom t2))
      (and (listp t1)
	   (listp t2)
	   (= (length t1) (length t2))
	   (same-structure? (car t1) (car t2))
	   (same-structure? (cdr t1) (cdr t2)))))

(defun reconstruct-tree (raam code &key (symbolp t) (max-depth 30))
  (let ((net (raam-network raam))
	(actv nil)
	(repwidth (raam-rep-width raam))
	(valence (raam-valence raam))
	(symtab (raam-symbol-table raam)))
    (unless (= repwidth (raam-terminal-width raam))
	    (error "raam reconstruction routine called with a stack"))
    (set-nodes net code *new-rep-level*)
    (forward-pass net :min-level *new-rep-level*)
    (setf actv (copy-seq 
		(cdr-vector
		 (car (last (network-activations net))))))
    (loop for i from 0 to (1- valence)
	  for start = valence then (+ start repwidth)
	  for end = (+ start repwidth -1)
	  for newvec = (extract-symbol actv start end)
	  collect
	  (if (> (extract-test actv i) half)
	      (if symbolp
		  (symbol-name-by-value symtab newvec :terminals-only t)
		newvec)
	    (if (zerop max-depth)
		(if symbolp (symbol-name-by-value symtab newvec)
		  newvec)
	      (reconstruct-tree raam newvec 
				:symbolp symbolp 
				:max-depth (1- max-depth)))))))

(defun reconstruct-index (raam i &key (symbolp t) (max-depth 30))
  (reconstruct-tree raam (symbol-value-by-index (raam-symbol-table raam) i)
		    :symbolp symbolp
		    :max-depth max-depth))

(defun reconstruct-trees (raam &key (symbolp t) (max-depth 30) (pprint t)
			       &aux (sametreecount 0) (samestructcount 0))
  (let* ((ts (raam-training-set raam))
	 (res (loop for tree-index in (raam-training-set raam)
		    for index = (cdr tree-index)
		    collect
		    (reconstruct-index raam index 
				       :symbolp symbolp
				       :max-depth max-depth))))
    (if pprint 
	(progn
	  (loop for x in res 
		for tree-index in ts 
		for tree        = (car tree-index)
		for sametree?   = (equalp tree x)
		for samestruct? = (same-structure? tree x) do
		(format t " ~3a,~3a: ~a~%      --> ~a~%~%"
			sametree?
			samestruct?
			(car tree-index)
			x)
		(if sametree?   (incf sametreecount))
		(if samestruct? (incf samestructcount)))
	  (format t "~a trees, ~a completely correct, ~a structurally correct"
		  (length ts) sametreecount samestructcount)
	  (= sametreecount (length ts)))
      res)))



;;; ----------------------------------------------------------------------
;;;
;;; Reconstruction routines II - analogous routines for stacks.
;;;
;;; ----------------------------------------------------------------------

(defun reconstruct-stack-tree (raam code &key (symbolp t) (max-depth 30))
  (let ((net (raam-network raam))
	(actv nil)
	(repwidth (raam-rep-width raam))
	(termwidth (raam-terminal-width raam))
	(valence (raam-valence raam))
	(symtab (raam-symbol-table raam)))
    (unless (= valence 2)               		;assume valence is 2
	    (error "Stack reconstruction assumes valence is 2"))
    (set-nodes net code *new-rep-level*)
    (forward-pass net :min-level *new-rep-level*)
    (setf actv (copy-seq 
		(cdr-vector
		 (car (last (network-activations net))))))
    (loop for i from 0 to (1- valence)
	  for start = valence then (+ start repwidth)
	  for end = (+ start repwidth -1) then (+ start termwidth -1)
	  for newvec = (extract-symbol actv start end)
	  collect
	  (if (> (extract-test actv i) half)
	      (if symbolp
		  (symbol-name-by-value symtab newvec :terminals-only t)
		newvec)
	    (if (zerop max-depth)
		(if symbolp (symbol-name-by-value symtab newvec)
		  newvec)
	      (reconstruct-stack-tree raam newvec 
				      :symbolp symbolp 
				      :max-depth (1- max-depth)))))))

(defun reconstruct-stack-index (raam i &key (symbolp t) (max-depth 30))
  (reconstruct-stack-tree raam 
			  (symbol-value-by-index (raam-symbol-table raam) i)
			  :symbolp symbolp
			  :max-depth max-depth))

(defun reconstruct-stack-trees (raam &key (symbolp t) (max-depth 30) (pprint t))
  (let* ((ts (raam-training-set raam))
	 (res (loop for tree-index in (raam-training-set raam)
		    for index = (cdr tree-index)
		    collect
		    (reconstruct-stack-index raam index 
					     :symbolp symbolp
					     :max-depth max-depth))))
    (if pprint (loop for x in res 
		     for tree-index in ts 
		     for tree = (car tree-index) do
		     (format t " ~3a: ~a~%  --> ~a~%~%"
			     (equalp tree x)
			     (car tree-index)
			     x))
      res)))


;;; ---------------------------------------------------------------------------
;;;
;;; Encode - encode a symbolic tree from scratch
;;;
;;; ---------------------------------------------------------------------------

(defun encode (raam tree &aux (valid-terminals (terminals-of-raam raam)))
  (unless (equal tree (fix-valence tree (raam-valence raam)))
	  (error "tree is not of proper valence"))
  (loop for term in (terminals tree)
	when (not (member term valid-terminals))
	do (error "invalid terminal detected"))
  (or (symbol-value-by-name (raam-symbol-table raam) tree)
      (encode3 raam tree)))

;; (pja) my version of encode2.  the real encode2 didn't like cases where the
;; input tree was unbalanced like '((a b) c).  This version simply looks up the
;; codes for each subtree in turn.  If a subtree is not a member of the orignal
;; training set, then it will not have a vector in the symbol table and must be
;; constructed.  We do this by recursing.  This function looks up each subtree,
;; creates the input vector, passes it through the raam and returns a copy of
;; the middle layer.
(defun encode3 (raam tree)
  (let* ((symtab (raam-symbol-table raam))
	 (codes (loop for subtree in tree
		      collect (or (symbol-value-by-name symtab subtree)
				  (encode3 raam subtree))))
	 (incode (apply #'append-vector codes))
	 (net (raam-network raam)))
    (set-nodes net incode)
    (forward-pass net)
    (copy-seq (pointer-to-activation-of-level *new-rep-level* net))))


;(defun encode2 (raam tree)
;  (let* ((net (raam-network raam)))
;   (cond ((and (vectorp (car tree))		  ;single vector means that
;		(= (length tree) 1))		  ;we're already done
;	   (car tree))
;	  ;;
;	  ;; Simple case - a list of vectors, just encode them
;	  ;;
;	  ((vectorp (car tree))		
;	   (progn
;	     (set-nodes net (apply #'append-vector tree))
;	     (forward-pass net)
;	     (copy-seq (pointer-to-activation-of-level *new-rep-level* net))))
;	  ;;
;	  ;; Another simple case - a list of terminals, just encode them
;	  ;; 
;	  ((terminal? (car tree))		
;	   (let* ((symtab (raam-symbol-table raam))
;		  (codes (loop for x in tree collect
;			       (symbol-value-by-name symtab x)))
;		  (incode (apply #'append-vector codes)))
;	     (set-nodes net incode)
;	     (forward-pass net)
;	     (copy-seq (pointer-to-activation-of-level *new-rep-level* net))))
;	  ;;
;	  ;; Finally, we have a list of lists (either vectors or symbols)
;	  ;; let's use some recursion...
;	  ;;
;	  (t (encode2 raam (loop for x in tree collect (encode2 raam x)))))))



    

