;;;    -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-     
;;;                                                                         
;;;       THE Ohio State University Laboratory for AI Research (LAIR)       
;;;                            Copyright (c) 1990                           
;;;                                                                         
;;;                            Filename: misc.cl
;;;                            Author: Greg Saunders
;;;
;;; Comments:
;;;  This file contains extra code not required for raam functionality, but 
;;;  that might be interesting to explore.
;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(proclaim '(optimize speed))

;;; ---------------------------------------------------------------------- 
;;;
;;; License: This code may be distributed free of charge.  This code may not be
;;; incorporated into any production software of any type or any code which is
;;; sold without the express written consent of the authors.  Modifications and
;;; inclusions for other types of software are permitted, provided that this
;;; header remains unchanged and included in all subsequent versions of the
;;; code.
;;;
;;; Disclaimer: The authors of this code make no claims as to its
;;; validity, and are in no way responsible for any errors contained within.
;;; 
;;; File description: This file contains implementations of linear algebra
;;; routines which are implemented in C routines and called by Franz Allegro
;;; Common Lisp.  This file is specifc to Franz Allegro Common Lisp and will
;;; not work with other common lisps.  The file Linear.lisp should be used as a
;;; replacement for this file when using a different common lisp. the file with
;;; the C routines, linear-lib.o, should be placed in the same directory.
;;;
;;; Creation date: Aug 5, 1990 Version: 0.9.0
;;; Modifications: none 
;;; 
;;; Interface/exported functions 
;;; Package info


;;; ----------------------------------------------------------------------
;;;                         Change Log
;;;                         ----------
;;;
;;; 11/9/90 - (pja) Added the license and change log.  Also added the functions
;;; within-tolerance and within-distance as terminal tests for raams.
;;;
;;; ----------------------------------------------------------------------

(in-package 'mraam)

(export '(bump-network bump-raam set-funny-weights cluster cleanup
	  stats group-stats random+- within-tolerance within-distance
))
	       
(use-package 'excl)            ; required for "shell" command

;;; ----------------------------------------------------------------------
;;;
;;; Bump-network (network jolt)
;;;   Adds a random number in the range of -jolt ---> +jolt to every
;;;   weight in the network
;;;
;;; ----------------------------------------------------------------------

(defun random+- (x)
  (if (< (random 1.0) .5)
      (- (random x))
    (random x)))

(defun bump-number (n jolt)
  (+ n (random+- jolt)))

(defun bump-network (net &optional (jolt .05))
  (loop for m in (network-weights net) do
	(loop for i from 0 to (- (array-dimension m 0) 1) do
	      (loop for j from 0 to (- (array-dimension m 1) 1) do
		    (setf (aref m i j)
			  (bump-number (aref m i j) jolt))))))

(defun bump-raam (raam &optional (jolt .05))
  (bump-network (raam-network raam) jolt))




;;; ----------------------------------------------------------------------
;;;
;;; Interface to Tony Plate's program
;;;   Tony's program is a version of backprop written in C.  To read in
;;;   weights from his program, first set up a network (or raam), and then
;;;   call set-funny-weights with the NETWORK and the filename.
;;;
;;; ----------------------------------------------------------------------

(defun read-funny-weights (file &aux x)
  (with-open-file (fi file :direction :input)
		  (loop while (setf x (read fi nil nil))
			collect x)))

(defun set-funny-weights (net file)
  (unless (typep net 'network) (error "First argument must be a network"))
  (let ((data (read-funny-weights file))
	(weights (network-weights net)))
    (loop for k from 0 to (1- (length weights)) 
	  for m = (nth k weights)
	  for d = (array-dimensions m)
	  for max-row = (1- (car d))
	  for max-col = (1- (cadr d)) do
	  (loop for i from 0 to max-col do
		(loop for j from 0 to max-row do
		      (setf (aref m j i)
			    (coerce (car data) *element-type*))
		      (setf data (cdr data)))))
    (format t  "Remaining-data = ~a" data)))





;;; ----------------------------------------------------------------------
;;;
;;; Clustering made simple
;;;   These routines act as an interface to the "cluster" program.  It is 
;;;   assumed that that program is somewhere in the user's path.
;;;
;;; ----------------------------------------------------------------------

;; Need to remove spaces from file since cluster program can't handle spaces.
;; Replace all spaces with hyphens.
(defun remove-spaces-from-file (fi newname)
  (shell (concatenate 'string "tr ' ' '-' <" fi ">" newname)))
		      
(defun pad-list (x n)
  (append x (loop for i from 1 to (- n (length x)) collect zero)))

;; see notes with "cluster" below for how terminals & nonterminals are
;; clustered together
(defun write-trees-for-clustering (r &optional (file$ (raam-name r)))
  (if (zerop (raam-total-epochs r)) (error "Raam hasn't been trained yet"))
  (let* ((pad-length (max (raam-rep-width r)
			  (raam-terminal-width r)))
	 (trees (loop for x in (raam-symbol-table r) collect (car x)))
	 (vecs  (loop for x in (raam-symbol-table r) 
		     collect (pad-list (coerce (cdr x) 'list) pad-length)))
	(tree-file (concatenate 'string file$ ".trees")))
    (with-open-file (fi tree-file :direction :output :if-exists :supersede) 
		    (format fi "~{~a~%~}" trees))
    (remove-spaces-from-file tree-file (concatenate 'string file$ 
						    ".trees-to-cluster"))
    (with-open-file (fi (concatenate 'string file$ ".vecs") 
			:direction :output :if-exists :supersede)
		    (loop for v in vecs do
			  (loop for a in v do
				(format fi "~f " (coerce a 'single-float)))
			  (terpri fi)))))


;; One simple command to do clustering
;;   Note:  the cluster program we use requires that all vectors be of the same
;;   length.  This is the default case for raams, but STACK raams will
;;   typically have a different length for terminal & nonterminal vectors.  The
;;   quick fix for this case adopted here is to simply pad the smaller of the
;;   two vectors with 0's.

(defun cluster (r &optional (file$ (raam-name r)))
  (write-trees-for-clustering r file$)
  (shell (concatenate 'string "~saunders/bin.sun4/cluster "
		      file$ ".vecs "
		      file$ ".trees-to-cluster "
		      "> " file$ ".cluster")))

(defun cleanup (r &optional (file$ (raam-name r)))
  (shell (concatenate 'string "rm " file$ ".vecs"))
  (shell (concatenate 'string "rm " file$ ".trees"))
  (shell (concatenate 'string "rm " file$ ".trees-to-cluster")))
  




;;; ----------------------------------------------------------------------
;;;
;;; Statistical analysis of trees -- pretty boring stuff
;;;
;;; ----------------------------------------------------------------------

(defun number-of-terminals (x)
  (length (terminals x)))
		
(defun height (x)
  (cond ((null x) 0)
	((atom x) 0)
	(t (1+ (apply #'max (loop for y in x collect (height y)))))))

(defun number-of-nodes (x)
  (cond ((null x) 0)
	((atom x) 1)
	(t (+ (number-of-nodes (car x))
	      (number-of-nodes (cdr x))))))

(defun number-of-branches (x)
  (if (atom x) 0
    (+ (length x)
       (loop for y in x sum (number-of-branches y)))))

(defun number-of-branch-points (x)
  (if (atom x) 0
    (1+ (loop for y in x 
	      when (listp y)
	      sum (number-of-branch-points y)))))

(defun branching-factor (x)
  (/ (number-of-branches x)
     (number-of-branch-points x)))

(defun avg (x)
  (/ (loop for i in x sum i) (length x)))

(defun stats (tree)
  (let ((nn (number-of-nodes tree))
	(nt (number-of-terminals tree))
	(ht (height tree))
	(bf (branching-factor tree)))
    (format nil "Averages...~{~%  ~19a = ~f~}"
	    (list "Number of nodes" nn
		  "Number of terminals" nt
		  "Height" ht
		  "Branching factor" bf))))

(defun group-stats (trees)
  (let ((nn (avg (loop for x in trees collect (number-of-nodes x))))
	(nt (avg (loop for x in trees collect (number-of-terminals x))))
	(ht (avg (loop for x in trees collect (height x))))
	(bf (avg (loop for x in trees collect (branching-factor x)))))
    (format nil "Averages...~{~%  ~19a = ~f~}"
	    (list "Number of nodes" nn
		  "Number of terminals" nt
		  "Height" ht
		  "Branching factor" bf))))


;;; ----------------------------------------------------------------------
;;;
;;; Terminal tests
;;;
;;; ----------------------------------------------------------------------
;; says whether the given vector is within tolerance for all nodes for some
;; terminal in the symbol table.  NOTE THAT THIS ONLY CONSIDERS TERMINALS IN
;; THE SYMBOL TABLE. 
(defun within-tolerance (raam stuff
			 &optional (tol (raam-term-tol raam))
			 &aux found)
  (unless (= (length stuff) (raam-rep-width raam))
	  (error "Within-Tolerance: Input not the correct length"))
  ;; need to make sure that the numbers in stuff are all of the right type.  So
  ;; take whatever we got apart and put it back together aafter being coerced
  (setf tol (coerce tol *element-type*))
  (when (vectorp stuff) (setf stuff (coerce stuff 'list)))
  (setf stuff (loop for x in stuff collect (coerce x *element-type*)))
  (setf tol (coerce tol *element-type*))
  (loop for symbol in (raam-symbol-table raam)
	while (not found)
	for term = (car symbol)
	for rep = (cdr symbol) do
    (unless (listp term)
       (setf found
	     (eval (cons 'and
			 (mapcar #'(lambda (x y) (< (abs (- x y)) tol))
				 (coerce rep 'list)
				 stuff))))))
  found)

;; says whether the given vector is within the euclidean distance specified 
;; to some terminal in the symbol table.  NOTE THAT THIS ONLY CONSIDERS
;; TERMINALS IN THE SYMBOL TABLE.
(defun within-distance (raam stuff
			 &optional (dist (raam-term-tol raam))
			 &aux found)
  (unless (= (length stuff) (raam-rep-width raam))
	  (error "Within-Distance: Input not the correct length"))
  ;; need to make sure that the numbers in stuff are all of the right type.  So
  ;; take whatever we got apart and put it back together aafter being coerced
  (setf dist (coerce dist *element-type*))
  (when (vectorp stuff) (setf stuff (coerce stuff 'list)))
  (setf stuff
	(make-array (list (length stuff))
		    :initial-contents
		       (loop for x in stuff collect (coerce x *element-type*))
		    :element-type *element-type*))
  (setf dist (coerce dist *element-type*))
  (loop for symbol in (raam-symbol-table raam)
	while (not found)
	for term = (car symbol)
	for rep = (cdr symbol) do
    (unless (listp term)
	    (setf found (< (abs (vector-distance stuff rep)) dist))))
  found)

