;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;;     			     ANSIL
;;; 		      Advanced Network Simulator In Lisp
;;; 		      ----------------------------------
;;;
;;;				  Written By
;;;			       Peter J. Angeline
;;;			      Gregory M. Saunders
;;;
;;;			   The Ohio State University
;;;		       Laboratory for AI Research (LAIR)
;;;			     Columbus, Ohio, 43210
;;;
;;;			      Copyright (c) 1991
;;;
;;; ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;;; ---------------------------------------------------------------------- 
;;; File: matrixbp.lisp - This file contains a matrix implementation of
;;; backprop.  Also, routines for implementing the pushprop algorithm are
;;; included.
;;;
;;; Author: Peter J Angeline
;;;
;;; License: This code may be distributed free of charge.  This code may not be
;;; incorporated into any production software of any type or any code which is
;;; sold without the express written consent of the authors.  Modifications and
;;; inclusions for other types of software are permitted, provided that this
;;; header remains unchanged and included in all subsequent versions of the
;;; code.
;;;
;;; Disclaimer: The authors of this code make no claims as to its
;;; validity, and are in no way responsible for any errors contained within.
;;; 
;;; File description: This file contains an implementation of back prop using
;;; linear algebra routines rather than symbolic means.  
;;;
;;; Creation date: Aug 5, 1990 Version: 0.95
;;; Modifications: none 
;;; ----------------------------------------------------------------------
;;;
;;;
;;;                         Change Log
;;;                         ----------
;;;
;;; 10/23/90 - (pja) changed set-network to do the keys correctly.
;;;
;;; 10/29/90 - (pja) fixed always-bp? to work correctly.  Added to the
;;; functionality of vector-binary-diff so that it returned a zero value for
;;; things within tolerance when always-bp? is nil.
;;;
;;; 11/5/90 - (pja) added function NETWORK-ZERO-ERRORS which zeros all of the
;;; errors in the network.
;;;
;;; 02/13/91 - (gms) made CREATE-NETWORK call RANDOMIZE-NETWORK instead of
;;; having the user do this.  Also, added slot NETWORK-TOTAL-EPOCHS to hold
;;; cumulative epochs of training.
;;;
;;; 02/28/91 - (gms) Updated the show-patterns routine, and moved vector
;;; display routines (show-vector, format-vector, format-vectors) from
;;; raam.lisp to this file, since they are now needed by show-patterns.  The
;;; change to show-patterns was to allow :digits & :width as keywords to
;;; control formatting of output.
;;;
;;; 03/26/91 - (pja) Changed around the randomization routines so a common lisp
;;; implementation independent random seed can be used as the representation
;;; for the initial conditions.  This should reduce the size of storing the
;;; structures in a file to be only half of their previous size.  Also added
;;; the routine REVERT-NETWORK which resets the network as it was prior to
;;; training with the same initial conditions.
;;;
;;; 04/01/91 - (pja) Altered the write-network routine so that the deltas,
;;; errors and activations were not written to the file.  Altered to
;;; read-network routine so it creates the delta, error and activation slots
;;; when reading in a network.  This should reduce the size of a network by
;;; about 3/4.
;;;
;;; 04/02/91 - (pja) Altered the compute-deltas routine to accept the key
;;; :repetitions so we can simulate a pattern appearing more than once in the
;;; training set.
;;;
;;; 04/04/91 - (pja) Added function get-nodes which is like set-nodes but makes
;;; a copy of the activation vector and returns it.  It takes a network and an
;;; optional level.  The default level is the output level.  Also added
;;; function use-network which takes a network and a vector and forward props
;;; the vector through the network and returns the output vector.
;;;
;;; 04/18/91 - (pja) Added the key min-level to compute-deltas so that the
;;; deltas for a subrange of a network could be computed.  Before, it was
;;; assumed the input layer was the start layer.
;;; ----------------------------------------------------------------------

(proclaim '(optimize speed))

(provide 'mbp)
(in-package 'mbp)

(export '(network-total-error network-max-error network-weights
	  network-initial-weights network-activations network-errors
	  network-deltas network-prev-deltas network-rate network-momentum
	  network-epsilon network-patterns create-network add-pattern
          forward-pass set-nodes compute-output-errors compute-hidden-errors
	  compute-deltas update-weights count-errors network-train
	  update-prev-deltas write-network read-network randomize-network
	  randomize-vector zero-deltas network sigmoid zero-prev-deltas
	  network-sp-add network-adjust? network-always-bp? network-zero-errors
	  invsigmoid set-network show-network read-network write-network
	  show-patterns show-vector revert-network get-nodes use-network))

;; modules needed to be loaded 
(require 'loop)	; the LOOP macros

(require 'linear)

(use-package 'loop)
(use-package 'linear)

;;; ----------------------------------------------------------------------
;;;
;;; Structures and constants
;;;
;;; ----------------------------------------------------------------------
;; definition of network structure and the constant *element-type*.

(eval-when (compile load)
 (defconstant *element-type* 'double-float) ; type of elements to have in the
					    ; matrices and vectors. 
 (defconstant zero (coerce 0 *element-type*))
 (defconstant  one (coerce 1 *element-type*))

 (eval	   
  `(defstruct network
     "Everything you ever wanted in a network, and then some."
     (weights nil :type list)		; one matrix per level of links
     (initial-weights nil :type array)	; random seed which gives init weights
     (activations nil :type list)	; one vector per level of nodes
     (errors nil :type list)		; one vector per level of nodes
     (deltas nil :type list)		; one matrix per level of links
     (prev-deltas nil :type list)	; one matrix per level of links
     (rate ,(coerce 0.7 *element-type*)
	   :type ,*element-type*)	; learning rate
     (momentum ,(coerce 0.5 *element-type*)
	       :type ,*element-type*)	; momentum term
     (epsilon ,(coerce 0.1 *element-type*)
	      :type ,*element-type*)	; acceptable error distance 
     (max-error ,zero
		:type ,*element-type*)	; maximum error found in last bp  
     (total-error ,zero :type ,*element-type*)	; total error from last bp 
     (depth 0 :type integer)		; the number of layers (of weights)
     (bias ,one :type ,*element-type*)	; the value for bias activation
     (adjust? nil :type symbol)		; adjust rate by pattern number
     (sp-add ,(coerce 0.0 *element-type*)
	     :type ,*element-type*)	; constant to add to sigmoid derivative
     (always-bp? t :type symbol)	; bp or not when within tolerence
     (patterns nil :type list)		; the patterns for this network
     (pushprop-patterns nil :type list)	; the patterns for pushprop assocs
     (total-epochs 0 :type integer)     ; total number of passes through
					; training set
     )))

;;; ----------------------------------------------------------------------
;;;
;;; Network and sytem set-up routines
;;;
;;; ----------------------------------------------------------------------
;; defines a new network and returns the structure.  The input list LEVELS
;; takes an arbitrary number of inputs where input i describes how many nodes
;; are in level i of the network.  All of the 1+'s below are for bias nodes
;; which will be designated as node number 0 in each vector or matrix.
(defun create-network (&rest levels &aux (net (make-network)))
  (loop for i in levels			; nodes which input into this level
	for j in (cdr levels)		; nodes which are output for level
	for lst = (list (1+ i) j) do
     ; create weight matrix for ith to jth level.  the weights go from the i+1
     ; nodes on the previous level to the j nodes on the next level.
    (setf (network-weights net)
	  (append (network-weights net) 
		  (list (make-array lst :element-type *element-type*
				    :initial-element
				    zero))))
     ; create delta matrix. one cell for each weight
    (setf (network-deltas net)
	  (append (network-deltas net) 
		  (list (make-array lst :element-type *element-type*
				    :initial-element
				    zero))))

     ; create prev-delta matrix. one cell for each weight
    (setf (network-prev-deltas net)
	  (append (network-prev-deltas net) 
		  (list (make-array lst :element-type *element-type*
				    :initial-element
				    zero))))
    ; vectors, first the activation.  there are i+1 (including bias) nodes in
    ; this row.  
    (setf (network-activations net)
	  (append (network-activations net)
		  (list (make-array (list (1+ i))
				    :initial-element
				    zero
				    :element-type *element-type*))))
     ; create error vector
    (setf (network-errors net)
	  (append (network-errors net)
		  (list (make-array (list (1+ j))
				    :initial-element
				    zero
				    :element-type *element-type*)))))
  ; Now do output layer's actv.  add a false bias node here just to keep all of
  ; the activation vectors consistenly constructed.
  (setf (network-activations net)
	(append (network-activations net)
		(list (make-array (1+ (car (last levels)))
				  :initial-element zero
				  :element-type *element-type*))))
  (setf (network-depth net) (- (length levels) 1))
  (randomize-network net)                          ;gms 02/13
  net)
  
;; Adds a pattern to be learned by the network to a list of patterns.  Each is
;; a list of the activation of the input nodes and the output nodes.  All
;; numbers should be floats or serious weirdness may result.  In and Out must
;; be a list
(defun add-pattern (net in out)
  (unless (= (length in) (1- (length (car (network-activations net)))))
	  (error "ADD-PATTERN: Input list wrong size."))
  (unless (= (length out) (1- (length (car (last (network-activations net))))))
	  (error "ADD-PATTERN: Output list wrong size."))
  
  (setf in (make-array (list (length in))
		       :element-type *element-type*
		       :initial-contents (coerce-list in *element-type*)))
  (setf out (make-array (list (length out))
			:element-type *element-type*
			:initial-contents
			(coerce-list out *element-type*)))
  (setf (network-patterns net) (cons (list in out) (network-patterns net))))

(defun coerce-list (lst el-type)
  (mapcar #'(lambda (x) (coerce x el-type)) (coerce lst 'list)))

;; Adds a pattern to be learned by the network to a list of patterns.  Each is
;; a list of the activation of the input nodes and the output nodes.  All
;; numbers should be floats or serious weirdness may result.
(defun add-pushprop-pattern (net in out)
  (when (listp in) (setf in (apply #'vector in)))
  (when (listp out) (setf out (apply #'vector out)))
  (setf (network-pushprop-patterns net)
	(cons (list in out) (network-pushprop-patterns net))))

;; function to set all of the setable things in a network.
(defun set-network (network &key (rate (network-rate network))
			         (momentum (network-momentum network))
				 (epsilon (network-epsilon network))
				 (bias (network-bias network))
				 (sp-add (network-sp-add network))
				 (adjust? (network-adjust? network))
				 (always-bp? (network-always-bp? network)))
  (setf (network-rate network) (coerce rate *element-type*))
  (setf (network-momentum network) (coerce momentum *element-type*))
  (setf (network-epsilon network) (coerce epsilon *element-type*))
  (setf (network-bias network) (coerce bias *element-type*))
  (setf (network-sp-add network) (coerce sp-add *element-type*))
  (setf (network-adjust? network) adjust?)
  (setf (network-always-bp? network) always-bp?)
  (format nil "~%Network Properties:~%~{   ~20a: ~a~%~}~%"
      `("Element Type" ,*element-type*
        "Adjust Rate? (adjust?)" ,(network-adjust? network)
	"Always BP? (always-bp?)" ,(network-always-bp? network)
	"Learning Rate (rate)" ,(network-rate network)
	"Momentum (momentum)" ,(network-momentum network)
	"Tolerance (epsilon)" ,(network-epsilon network)
	"Bias Activation (bias)" ,(network-bias network)
	"Sigmoid Prime Add (sp-add)" ,(network-sp-add network))))

;;; ----------------------------------------------------------------------
;;;
;;; Node activation code
;;;
;;; ----------------------------------------------------------------------

;; This function runs through a forward pass of the network by propagating the
;; activations.  and resetting the activation vector *actv*.  It is assumed
;; that the input nodes have already been actvated (thus the nodes in *actv*
;; which coorespond to input nodes have been set prior to entering this
;; routine).  The rest of the nodes should be 0.0.
;; are since they will be over written.
(defun forward-pass (net &key (min-level 0) (max-level (network-depth net)))
  (loop for ins = (car (nthcdr min-level (network-activations net))) then outs
	for outs in (nthcdr (1+ min-level) (network-activations net))
	for wts in (nthcdr min-level (network-weights net))
	for level from min-level to max-level do
     (matrix-multiply (row-array (cdr-vector outs)) (row-array ins) wts)
     (vector-apply-sigmoid (cdr-vector outs))
     (setf (aref outs 0) (network-bias net))))

;; sets the activation values for the input nodes from the given list.
(defun set-nodes (net pat &optional (level 0))
  (unless (= (length pat)
	     (1- (length (nth level (network-activations net)))))
	  (error "SET-NODES: Pattern of wrong length."))
  (when (listp pat) (setf pat (apply #'vector pat)))
  (vector-apply (nth level (network-activations net))
		#'identity
		(cons-vector (network-bias net) pat)))

;; sets the activation values for the input nodes from the given list.
(defun get-nodes (net &optional (level (network-depth net)))
  (cdr-vector (copy-vector (nth level (network-activations net)))))


;; sigmoid activation function
(defun sigmoid (x)
  (/ 1.0d0 (+ 1.0d0 (exp (- x)))))

;; inverse of sigmoid activation function
(defun invsigmoid (x)
  (when (>= x one) (setf x (coerce 0.9999 *element-type*)))
  (log (/ x (- one x))))

;;; ----------------------------------------------------------------------
;;;
;;; Backward Propagation code
;;;
;;; ----------------------------------------------------------------------
;; computes the output errors.  I made a specialized function so it would be
;; quicker.  max-level points to the output layer.  sp-add is a constant which
;; is added to the derivative of the sigmoid function. Falhman has reported
;; that when this constant is 0.1 backprop is significantly faster.  0.0 is the
;; default so that orginal backprop is the default.
(defun compute-output-errors (net pattern tol
			      &key (max-level (network-depth net))
			           (start 0) (end (1- (length pattern)))
			      &aux (sp-add (network-sp-add net))
			           (always-bp? (network-always-bp? net)))
  (vector-binary-diff
   (cdr-vector (nth (1- max-level) (network-errors net))) 
   pattern
   (cdr-vector (nth max-level (network-activations net)))
   sp-add
   (if always-bp? zero tol)
   start end))

;; This function sets the values of the output errors.  The cons'ing of err1 is
;; so the error vector has the phony node first.  This error computation
;; enforces a repelling force between the activations and the target rather
;; than an attractive one as in the backprop case.  The repelling force is
;; based on the global distance that the two vectors are from each other.
(defun pushprop-output-errors (net pattern tol)
  (let* ((actv (car (last (network-activations net))))
	 (dist (vector-distance (cdr-vector actv) pattern))
	 (epsilon (network-epsilon net)))
    (compute-output-errors net pattern (network-epsilon tol))
    (setf (car (last (network-errors net)))
	  (if (< dist epsilon)
	      (vector-apply* nil
			     (car (last (network-errors net)))
			     (/ (- dist epsilon) dist))
	    ; add a little error when we are further than epsilon away
	    (vector-apply* nil
			   (car (last (network-errors net)))
			   (coerce -0.001 *element-type*))))
    (setf (aref (car (last (network-errors net))) 0) (network-bias net))))

;; computes the error for each hidden unit.  No error computations is done for
;; the first or last activations vectors.  sp-add is a constant which is added
;; to each element of the derivative of the output of the node.  Fahlman
;; reports that when this is 0.1 that a fewer epochs are needed for backprop.
(defun compute-hidden-errors (net &aux (sp-add (network-sp-add net)))
  (loop for prev-err = (car (last (network-errors net))) then err
	for err in (cdr (reverse (network-errors net)))
	for actv in (cdr (reverse (cdr (network-activations net))))
	for wts in (reverse (cdr (network-weights net))) do
     (matrix-multiply (column-array err)
		      wts
		      (column-array (cdr-vector prev-err)))
     (vector-apply* err err (vector-diff nil actv sp-add))))

;; computes the delta for each link after a pattern presentation.  There are no
;; links going to the bias nodes on the next level so cdr the bias node off of
;; the actv vector.  :max-level determines how many layers from the input layer
;; to compute the deltas for.  :repetitions multiplies the error by the value
;; provided which is the same as backproping the same pattern that number of
;; times in the epoch.  :divisor alters the rate by dividing by the number
;; provided. 
(defun compute-deltas (net &key (min-level 0)
			        (max-level (network-depth net))
			        (divisor one)
				(repetitions one))
  (let ((rate (* (network-rate net) (/ repetitions divisor))))
    (loop for delta in (subseq (network-deltas net) min-level max-level)
	  for err in (subseq (network-errors net) min-level max-level)
	  for actv in (subseq (network-activations net)
			      min-level (1+ max-level))
	  for temp = (vector-product nil actv (cdr-vector err)) do
       (matrix-apply* temp temp rate)
       (matrix-apply+ delta delta temp))))

;; update the weights after an epoch
(defun update-weights (net &key (max-level (network-depth net))
			   &aux (momentum (network-momentum net)))
  ; first finish the computation of the deltas now that they are all computed
  (loop for delta in (subseq (network-deltas net) 0 max-level)
	for prev in (subseq (network-prev-deltas net) 0 max-level)
	for wts in (subseq (network-weights net) 0 max-level) do
      (matrix-apply* prev prev momentum)
      (matrix-apply+ delta delta prev)
      (matrix-apply+ wts wts delta)))

;; count the errors for the network.  Be sure not to count the phony bias node
;; in the activation matrix for the outputs of the network
(defun count-errors (net pattern 
 		     &key (max-level (network-depth net))
		          (display nil))
  (let ((epsilon (network-epsilon net))
	(outs (nth max-level (network-activations net)))) 
    (loop for pos from 0 to (1- (length pattern))
	  for actv = (aref outs (1+ pos))
	  for target = (aref pattern pos)
	  for err = (- target actv)
	  sum (* err err) into total-error 
	  maximize (abs err) into max-error 
	  count (> (abs err) epsilon) into num
	  finally
	  (when display (write-char (if (zerop num) #\. #\*)))
	  (incf (network-total-error net) total-error)
	  (setf (network-max-error net)
		(max max-error (network-max-error net)))
	  (return num))))

;; counts the errors for the pushprop algorithm.
(defun count-pushprop-errors (net pattern)
  (when (vectorp pattern) (setf pattern (coerce pattern 'list)))
  (let* ((actvs (cdr-vector (car (last (network-activations net)))))
	 (dist (vector-distance actvs pattern))
	 (epsilon (network-epsilon net))
	 (err? (< dist epsilon)))
    (loop for actv in (coerce actvs 'list)
	  for target in pattern
	  for err = (if (< dist epsilon)
			(- epsilon (abs (- target actv)))
		      zero)
	  sum (* err err) into total-error 
	  maximize (abs err) into max-error 
	  finally
  	    (write-char (if err? #\* #\.))
	    (incf (network-total-error net) total-error)
	    (setf (network-max-error net)
		  (max max-error (network-max-error net)))
	    (return (if err? 1 0)))))
  
;;; ----------------------------------------------------------------------
;;;
;;; Example Learning Routine 
;;;
;;; ----------------------------------------------------------------------
;; regular old backprop

(defun network-train (net &key (display 1) (num 10000) 
			       (max-level (network-depth net)))
  (unless (typep net 'network)
	  (error "First argument must be of type \"network\""))
  (let ((errors 1))
    (zero-prev-deltas net)
    (loop for iteration from 0 to (- num 1)
	  for output = (and (not (zerop display))
			    (zerop (mod iteration display))) 
	  until (= errors 0)
	  do
	  (incf (network-total-epochs net))
	  (zero-deltas net)
	  (setf (network-total-error net) zero
		(network-max-error net) zero
		errors 0)
	  (loop for pattern in (network-patterns net) do
		(set-nodes net (car pattern))
		(forward-pass net :max-level max-level)
		(compute-output-errors net (cadr pattern)
				       (network-epsilon net)
				       :max-level max-level)
		(compute-hidden-errors net)
		(incf errors (count-errors net (cadr pattern)
					   :max-level max-level
					   :display output))
		(compute-deltas net :max-level max-level))
	  (update-weights net)
	  (when output
		(format t "~%Epoch ~d: Err= ~d Tot=~,6f Max=~,6f~%"
			(network-total-epochs net)
			errors (network-total-error net)
			(network-max-error net)))
	  (update-prev-deltas net)
	  finally (format t "~%Epoch ~d: Err= ~d Tot=~,6f Max=~,6f~%"
			  (network-total-epochs net)
			  errors (network-total-error net)
			  (network-max-error net)))))

;; Pushprop and backprop combined.  Pushprop is run on all patterns stored in
;; (network-pushprop-patterns net) and bp is run on all patterns in the slot
;; (network-patterns net).
(defun pplearn (net &optional (display 1) &key (num 10000))
  (let ((errors 1)
	(tol (network-epsilon net)))
    (zero-prev-deltas net)
    (loop for iteration from 0 to (- num 1)
	  until (= errors 0)
	  do
	  (incf (network-total-epochs net))
	  (zero-deltas net)
	  (setf (network-total-error net) zero
		(network-max-error net) zero
		errors 0)
	; first compute deltas for bp patterns
	  (loop for pattern in (network-patterns net) do 
		(set-nodes net (car pattern))
		(forward-pass net)
		(compute-output-errors net (cadr pattern) tol)
		(compute-hidden-errors net)
		(compute-deltas net)
		(incf errors (count-errors net (cadr pattern))))
	; now compute deltas for pushprop patterns
	  (loop for pattern in (network-pushprop-patterns net) do 
		(set-nodes net (car pattern))
		(forward-pass net)
		(pushprop-output-errors net (cadr pattern) tol)
		(compute-hidden-errors net)
		(compute-deltas net)
		(incf errors (count-pushprop-errors net (cadr pattern))))
	  (update-weights net)
	  (when (zerop (mod iteration display))
		(format t "Epoch ~d: Err= ~d Tot= ~f Max= ~f~%"
			(network-total-epochs net) 
			errors (network-total-error net)
			(network-max-error net)))
	  (update-prev-deltas net)
	  finally (format t "Epoch ~d: Err= ~d Tot= ~f Max= ~f~%"
			  (network-total-epochs net)
			  errors (network-total-error net)
			  (network-max-error net)))))

;; switches the pointers to the delta matrices and prev-delta matrices in the
;; structure.  This is so the prev-deltas have the value of the correct values
;; in the next interation.  The deltas will be zeroed out at the start of the
;; next iteration.  This saves a lot of time consing up a new list of matrices
;; of the correct size.
(defun update-prev-deltas (net &aux (temp (network-deltas net)))
  (setf (network-deltas net) (network-prev-deltas net))
  (setf (network-prev-deltas net) temp))

;;; ----------------------------------------------------------------------
;;;
;;; Support Routines
;;;
;;; ----------------------------------------------------------------------
;; randomize the network and store the seed for the initial conditions
(defun randomize-network (net &optional (max 0.5))
  (setf (network-total-epochs net) 0)
  (setf *random-seed* (make-random-state))
  (setf (network-initial-weights net) (copy-seq *random-seed*))
  (mapc #'(lambda (mat) (randomize-matrix mat max)) (network-weights net))
  nil)

;; revert to the initial weghts before training
(defun revert-network (net &optional (max 0.5))
  (setf (network-total-epochs net) 0)
  (setf *random-seed* (copy-seq (network-initial-weights net)))
  (mapc #'(lambda (mat) (randomize-matrix mat max)) (network-weights net))
  nil)

;; 
(defun randomize-matrix (matrix &optional (max .5))
  (loop for i from 0 to (- (array-dimension matrix 0) 1)
	do (loop for j from 0 to (- (array-dimension matrix 1) 1)
		 do (setf (aref matrix i j)
			  (coerce (random-number max) *element-type*)))))

(defun randomize-vector (vector &optional (max .5))
  (loop for i from 0 to (- (length vector) 1)
	do (setf (aref vector i) (coerce (random-number max) *element-type*))))

(defun random-number (n)
  (* (if (zerop (random-int 2)) 1.0 -1.0) (random-float n)))

;; zero all error vectors in network
(defun network-zero-errors (net)
  (loop for vect in (network-errors net) do
	(vector-apply* vect vect zero)))

;; zero the previous deltas for a network
(defun zero-prev-deltas (net)
  (mapc #'zero-matrix (network-prev-deltas net)))

;; zero the deltas for a network
(defun zero-deltas (net)
  (mapc #'zero-matrix (network-deltas net)))

;; make all the elements in the matrix to be 0.0
(defun zero-matrix (matrix)
  (matrix-apply* matrix matrix zero))

;; write a network to a file
(defun write-network (network filename)
  (unless (typep network 'network)
	  (error "First argument must be of type \"network\""))
  (let ((deltas (network-deltas network))
	(errors (network-errors network))
	(acts (network-activations network)))
    (setf (network-deltas network) nil	; don't need to save these in file
	  (network-errors network) nil
	  (network-activations network) nil)
    (with-open-file (fi (concatenate 'string filename ".network")
			:direction :output :if-exists :supersede)
		    (pprint network fi))
    (setf (network-deltas network) deltas ; restore current network's arrays
	  (network-errors network) errors
	  (network-activations network) acts))
  nil)

; returns the read network
(defun read-network (filename)
  (let ((tempnet
	 (with-open-file (fi (concatenate 'string filename ".network")
			     :direction :input)
			 (read fi))))
    (fix-network-arrays tempnet)
    tempnet))

;;; ----------------------------------------------------------------------
;;; 
;;; Display and debugging routines
;;;
;;; ----------------------------------------------------------------------
(defun show-network (net &rest params &key &allow-other-keys)
  (unless (typep net 'network) (error "show-network called with non-network"))
  (format t "Network...~%")
  (loop for m in (network-weights net) do
	(apply #'print-matrix (cons m params))))

;;; Need to write better routine for this.
(defun show-actv (net)
  (mapcar #'(lambda (actv) (coerce actv 'list)) (network-activations net)))

(defun show-err (net)
  (mapcar #'(lambda (err) (coerce err 'list)) (network-errors net)))

(defun show-patterns (net &key (digits 3) (width (+ digits 3)))
  (let* ((width$  (princ-to-string width))
	 (digits$ (princ-to-string digits))
	 (f-vec$  (concatenate 'string "#(~{~" width$ "," digits$ "f~})")))
    (loop for pattern in (append (network-patterns net)
				 (network-pushprop-patterns net))
	  do
	  (set-nodes net (car pattern))
	  (forward-pass net)
	  (loop for x in pattern do
		(show-vector x f-vec$)
		(format t "   "))
	  (show-vector (car (last (network-activations net))) f-vec$)
	  (terpri))))

;;; ----------------------------------------------------------------------
;;;
;;; Ugly patch to fix a problem with saving/restoring networks.
;;; The problem is that when an array of type DOUBLE-FLOAT is written to
;;; disk, it will be read back in as an array of type TRUE.  The following
;;; routine alleviates this problem for networks by walking through the
;;; structure and effectively changing all arrays to be of type
;;; *element-type*.  Yuck.  
;;;
;;; See raam.lisp for a similar fix for raams.
;;;
;;; ----------------------------------------------------------------------

(defun fix-network-arrays (net)
  (unless (equal *element-type*
		 (type-of (network-rate net)))
	  (error (format nil "desired network not of type ~a" *element-type*)))
  (fix-matrix-list (network-weights net))
  (fix-matrix-list (network-prev-deltas net))
  (setf (network-deltas net) nil
	(network-activations net) nil
	(network-errors net) nil)
  (loop for lst in (network-patterns net) do (fix-matrix-list lst))
  (loop for mat in (network-weights net)
	do
	(setf (network-deltas net)
	      (append (network-deltas net)
		      (list (make-array (array-dimensions mat)
					:element-type *element-type*
					:initial-element zero))))
	(setf (network-activations net)
	      (append (network-activations net)
		      (list (make-array (list (array-dimension mat 0))
					:element-type *element-type*
					:initial-element zero))))
	(setf (network-errors net)
	      (append (network-errors net)
		      (list (make-array (list (1+ (array-dimension mat 1)))
					:element-type *element-type*
					:initial-element zero)))))
  (setf (network-activations net)
	(append (network-activations net)
		(list (make-array (1+ (array-dimension
				       (car (last (network-weights net)))
				       1))
				  :element-type *element-type*
				  :initial-element zero)))))

(defun fix-matrix-list (bad-matrix-list)
  (loop for i from 0 to (1- (length bad-matrix-list)) 
	for bad-matrix = (nth i bad-matrix-list) do
	(setf (nth i bad-matrix-list)
	      (make-array (array-dimensions bad-matrix)
			  :element-type *element-type*
			  :initial-contents (matrix->list bad-matrix)))))

(defun matrix->list (m)
  (let ((dim (array-dimensions m)))
    (cond ((= (length dim) 1)
	   (loop for i from 0 to (1- (car dim)) collect (aref m i)))
	  ((= (length dim) 2)
	   (loop for i from 0 to (1- (car dim)) collect
		 (loop for j from 0 to (1- (cadr dim)) collect
		       (aref m i j))))
	  (t (error "matrix->list supports only matrices of size <= 2")))))

(defun fix-matrix (bad-matrix)
  (make-array (array-dimensions bad-matrix)
	      :element-type *element-type*
	      :initial-contents (matrix->list bad-matrix)))
  
;;; ---------------------------------------------------------------------------
;;;
;;; Routines for displaying vectors, used in particular by raam.lisp
;;;
;;; ---------------------------------------------------------------------------

(defun show-vector (v &optional (f$ "#(~{~10,6f~})"))
  (format t f$ (coerce v 'list)))

;;; These functions work, but I don't think they're used anywhere. -gms
;;;
;;;(defun format-vector (v &optional (f$ "#(~{~10,6f~})"))
;;;  (format nil f$ (coerce v 'list)))
;;;
;;;(defun format-vectors (r &optional (f$ "#(~{~10,6f~})"))
;;;  (loop for v in r collect (format nil f$ (coerce v 'list))))


;;; ----------------------------------------------------------------------
;;;
;;;
;;;
;;; ----------------------------------------------------------------------
(defun use-network (net input)
  (set-nodes net input)
  (forward-pass net)
  (get-nodes net))