;;;       THE Ohio State University Laboratory for AI Research (LAIR)       
;;;                            Copyright (c) 1990                           
;;;                                                                         
;;;                            Filename: linear.lisp
;;;                            Author: Peter J Angeline
;;;                                    Greg Saunders
;;;
;;; Comments: Functions for linear algebra.
;;;
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(proclaim '(optimize speed))

;;; ---------------------------------------------------------------------- 
;;;
;;; License: This code may be distributed free of charge.  This code may not be
;;; incorporated into any production software of any type or any code which is
;;; sold without the express written consent of the authors.  Modifications and
;;; inclusions for other types of software are permitted, provided that this
;;; header remains unchanged and included in all subsequent versions of the
;;; code.
;;;
;;; Disclaimer: The authors of this code make no claims as to its
;;; validity, and are in no way responsible for any errors contained within.
;;; 
;;; File description: This file contains implementations of linear algebra
;;; routines which are implemented in C routines and called by Franz Allegro
;;; Common Lisp.  This file is specifc to Franz Allegro Common Lisp and will
;;; not work with other common lisps.  The file Linear.lisp should be used as a
;;; replacement for this file when using a different common lisp. the file with
;;; the C routines, linear-lib.o, should be placed in the same directory.
;;;
;;; Creation date: Aug 5, 1990 Version: 0.95
;;; Modifications: none 
;;; 
;;; Interface/exported functions 
;;; Package info

(require 'loop)

(use-package 'loop)

;; tell the lisp that this module provides the module "linear"
(provide 'linear)
(in-package 'linear)

(export '(matrix-multiply matrix-apply matrix-apply* matrix-apply+
	  row-array column-array row column vectorize cons-vector cdr-vector
	  append-vector vector-apply vector-apply* vector-apply+
	  vector-binary-diff vector-diff dot-product vector-product
	  set-row set-column set-diagonal front-vector vector-distance
	  append-vector copy-matrix copy-vector print-matrix
	  vector-midpoint))  

(eval-when (compile load)
   (defconstant *element-type* 'double-float)
   (defconstant one (coerce 1.0 *element-type*))
   (defconstant zero (coerce 0.0 *element-type*)))

;;; ----------------------------------------------------------------------
;;;
;;; Matrix Routines
;;;
;;; ----------------------------------------------------------------------
;; matrix-multiply algorithm.  If result is non-nil then it must be a matrix of
;; the correct size in which the computation will be done.  If result is
;; non-nil then the function gives no result but simply alters RESULT.
(defun matrix-multiply (result matrix1 matrix2)
  (unless (equal (array-dimension matrix1 1)
		 (array-dimension matrix2 0))
	  (error "MATRIX-MULTIPLY: Matrix sizes don't agree."))
  (when (and result
	     (not (= (array-dimension matrix1 0) (array-dimension result 0)))
	     (not (= (array-dimension matrix2 1) (array-dimension result 1))))
	  (error "MATRIX-MULTIPLY: Result Matrix not correct size."))
  (let ((answer (if result result
		  (let ((type (type-of (aref matrix1 0 0))))
		    (make-array (list (array-dimension matrix1 0)
				      (array-dimension matrix2 1))
				:element-type type))))
	(sum 0.0)
	(width (array-dimension matrix1 1))
	(rows (array-dimension matrix1 0))
	(cols (array-dimension matrix2 1)))
    (loop for i from 0 to (1- rows) do
	  (loop for j from 0 to (1- cols) do
	  (setf sum 0.0
		(aref answer i j)
		(dotimes (k width sum)
		   (incf sum (* (aref matrix1 i k) (aref matrix2 k j)))))))
    (if result nil answer)))


(defun matrix-apply* (result matrix &rest vals)
  (apply #'matrix-apply (append (list result #'* matrix) vals)))

(defun matrix-apply+ (result matrix &rest vals)
  (apply #'matrix-apply (append (list result #'+ matrix) vals)))

;; apply op to each element of MATRIX.  When val is a matrix, the matrices MUST
;; be the same size.  When val is a value, simple do op of this value to each
;; element in the matrix. Returns the finished product. Op should be any
;; binary op which returns the correct type of value.
(defun matrix-apply (result op matrix &rest vals)
  (let* ((rows (array-dimension matrix 0))
	 (cols (array-dimension matrix 1))
	 (type (type-of (aref matrix 0 0)))
	 (answer (if result result
		   (make-array (array-dimensions matrix)
			       :element-type type))))
  ; when vals is null, then do a single loop, otherwise move through the list
  ; stored in vals.  The (or vals '(t)) returns '(t) when vals is null so the
  ; loops does at least one iteration.
    (loop for val in (or vals '(nil))
	  for mat = matrix then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions mat)
				 (array-dimensions val))))
		(error "MATRIX-APPLY: Matrices must be the same size."))
	  (loop for i from 0 to (1- rows) do
		(loop for j from 0 to (1- cols) do
		      (setf (aref answer i j)
			    (coerce 
			     (apply op 
				    (cons (aref mat i j)
					  (cond ((null vals) nil)
						((listp (type-of val))
						 (list (aref val i j)))
						(t (list val)))))
			     type)))))
    (if result nil answer)))

;; takes a matrix, a row number and a vector and replaces sets the values in
;; that row of the matrix to be the same as the given vector.  No return value.
;; THIS IS A DESTRUCTIVE FUNCTION to the matrix.
(defun set-row (matrix row vector)
  (unless (= (array-dimension matrix 1) (length vector))
	  (error "SET-ROW: Vector is not the same length as row of matrix."))
  (dotimes (i (length vector) matrix)
	   (setf (aref matrix row i) (aref vector i))))

;; takes a matrix, a column number and a vector and replaces the values in
;; that column of the matrix to be the same as the given vector.  No return
;; value. THIS IS A DESTRUCTIVE FUNCTION to the matrix.
(defun set-column (matrix column vector)
  (unless (= (array-dimension matrix 0) (length vector))
	  (error "SET-COLUMN: Vector and matrix column different lengths."))
  (dotimes (i (length vector) matrix)
	(setf (aref matrix i column) (aref vector i))))

;; takes a square matrix, and a vector and replaces the values along
;; the diagonal of the matrix to be the same as the given vector.  No return
;; value. THIS IS A DESTRUCTIVE FUNCTION to the matrix. - gms
(defun set-diagonal (matrix vector)
  (unless (= (array-dimension matrix 0)
	     (array-dimension matrix 1)
	     (length vector))
	  (error "SET-DIAGONAL:  Lenghts not right."))
  (dotimes (i (length vector) matrix)
	(setf (aref matrix i i) (aref vector i))))

;; copies a matrix and it's contents into a distinct matrix.
(defun copy-matrix (matrix)
  (let ((result (make-array (array-dimensions matrix)
			    :element-type (type-of (aref matrix 0 0)))))
    (dotimes (i (array-dimension matrix 0) result)
       (dotimes (j (array-dimension matrix 1))
	  (setf (aref result i j) (aref matrix i j))))))

;;; ----------------------------------------------------------------------
;;;
;;; Operations on Vectors
;;;
;;; ----------------------------------------------------------------------
;; returns a vector which is equivalent to the nth row of matrix
(defun row (result matrix n)
  (unless (< n (array-dimension matrix 0))
	  (error "ROW: Matrix doesn't have enough rows."))
  (when (and result (not (= (length result) (array-dimension matrix 1))))
	(error "ROW: Result is wrong size."))
  (let ((answer (if result result
		  (make-array (list (array-dimension matrix 1))
			      :element-type (type-of (aref matrix 0 0))))))
    (dotimes (i (array-dimension answer 0))
	     (setf (aref answer i) (aref matrix n i)))
    (if result nil answer)))

;; returns a vector which is equivalent to the nth column of matrix
(defun column (result matrix n)
  (unless (< n (array-dimension matrix 1))
	  (error "COLUMN: Matrix doesn't have enough columns."))
  (when (and result (not (= (length result) (array-dimension matrix 0))))
	(error "COLUMN: Result is wrong size."))
  (let ((answer (if result result
		  (make-array (list (array-dimension matrix 0))
			      :element-type (type-of (aref matrix 0 0))))))
    (dotimes (i (array-dimension answer 0))
	     (setf (aref answer i) (aref matrix i n)))
    (if result nil answer)))

;; returns a row array which is equivalent to the vector. A copy is not made so
;; destructive operations on the result will change the original vector.
(defun row-array (vect)
  (make-array (cons 1 (array-dimensions vect)) :displaced-to vect :element-type (type-of (aref vect 0))))

;; returns a column array which is equivalent to the given vector.  A copy is
;; not made so destructive operations on the result will change the original
;; vector. 
(defun column-array (vect)
  (make-array (append (array-dimensions vect) '(1))
	      :displaced-to vect
	      :element-type (type-of (aref vect 0)))) 

;; takes either a row or column matrix and returns an equivalent vector.  A
;; copy is not made of the array, so modifications to the result of this
;; operation will appear in the original array.
(defun vectorize (matrix)
  (unless (and (= 2 (length (array-dimensions matrix)))
	       (= 1 (apply #'min (array-dimensions matrix))))
	  (error "VECTORIZE: Matrix can only be a Column or Row matrix."))
  (make-array (list (apply #'max (array-dimensions matrix)))
	      :element-type (type-of (aref matrix 0 0))           ; gms
	      :displaced-to matrix))
  
;; performs a dot (inner) product on the given vectors. Result is a single
;; value. Vectors must be of the same length.
(defun dot-product (vector1 vector2)
  (unless (= (array-dimension vector1 0) (array-dimension vector2 0))
	  (error "DOT-PRODUCT: Vectors not the same size."))
  (loop for i from 0 to (1- (length vector1)) 
	sum (* (aref vector1 i) (aref vector2 i))))

;; performs an outer product on the vectors.  Result is a (length vector1) x
;; (length vector2) matrix.
(defun vector-product (result vector1 vector2)
  (when (and result (not (equal (array-dimensions result)
				(list (length vector1) (length vector2)))))
	(error "VECTOR-PRODUCT: Result matrix worng size."))
  (let ((answer (if result result
		  (make-array (list (length vector1) (length vector2))
			    :element-type (type-of (aref vector1 0))))))
    (loop for i from 0 to (1- (length vector1)) 
	  for v1 = (aref vector1 i) do
	  (loop for j from 0 to (1- (length vector2)) do
		(setf (aref answer i j) (* v1 (aref vector2 j)))))
    (if result nil answer)))

(defun vector-apply* (result vect &rest vals)
  (apply #'vector-apply (append (list result #'* vect) vals)))

(defun vector-apply+ (result vect &rest vals)
  (apply #'vector-apply (append (list result #'+ vect) vals)))

(defun vector-diff (result vect sp-add)
  (vector-apply result #'(lambda (x) (+ sp-add (* x (- one x)))) vect))

;  (apply #'vector-apply
;	 (list result #'(lambda (x) (+ sp-add (* x (- one x))) vect))))

(defun vector-binary-diff (result vect1 vect2 sp-add tol start end)
  (let ((answer (if result result
		  (make-array (array-dimensions vect1)
			      :element-type (type-of (aref vect1 0))))))
    (loop for i from start to end 
	  for v1 = (aref vect1 i)
	  for v2 = (aref vect2 i) do
       (if (>= tol (abs (- v1 v2)))
	   (setf (aref result i) zero)
	 (setf (aref result i) (* (- v1 v2) (+ sp-add (* v2 (- one v2)))))))
    (if result nil answer)))

;; Applies the operator op to the input vector.  If vals is null, then the op
;; is considered to be unary.  Otherwise, the function applies the operator on
;; left to right pairs so that (vector-apply #'+ v1 v2 v3 v4 v5) => 
;; (+ (+ (+ (+ v1 v2) v3) v4) v5).  All vector must be the same length. When a
;; value in vals is a scaler, then that scaler is applied to each element of
;; the vector with op.  The function returns a new vector.
(defun vector-apply (result op vect &rest vals)
  (let* ((len (array-dimension vect 0))
	 (type (type-of (aref vect 0)))
	 (answer (if result result
		   (make-array (array-dimensions vect)
			       :element-type type))))
  ; when vals is null, then do a single loop, otherwise move through the list
  ; stored in vals.  The (or vals '(t)) returns '(t) when vals is null so the
  ; loops does at least one iteration.
    (loop for val in (or vals '(nil))
	  for ans = vect then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions ans)
				 (array-dimensions val))))
		(error "VECTOR-APPLY: Vectors must be the same size."))
	  (dotimes (i len)
		   (setf (aref answer i)
			 (coerce 
			  (apply op 
				 (cons (aref ans i)
				       (cond ((null vals) nil)
					     ((listp (type-of val))
					      (list (aref val i)))
					     (t (list val)))))
			  type))))
    (if result nil answer)))

;; computes the euclidean distance between two vectors.
(defun vector-distance (vector1 vector2)
  (unless (= (length vector1) (length vector2))
	  (error "VECTOR-DISTANCE: Vectors must be of same length."))
  (sqrt (apply #'+ (mapcar #'(lambda (v1 v2) (* (- v1 v2) (- v1 v2)))
			   (coerce vector1 'list)
			   (coerce vector2 'list)))))

;; copies a vector
(defun copy-vector (vect)
  (coerce (copy-seq (coerce vect 'list)) 'vector))

;;; ----------------------------------------------------------------------
;;;
;;; List operations on vectors
;;;
;;; ----------------------------------------------------------------------
;; returns a vector which is the given vector minus the first element.  the
;; returned vector is NOT a copy of the given vector.
(defun cdr-vector (vect)
  (make-array (list (1- (array-dimension vect 0)))
	      :displaced-to vect
	      :element-type (type-of (aref vect 0))
	      :displaced-index-offset 1))

;; returns a vector with the given item consed onto the front of it.
(defun cons-vector (item vect)
  (coerce (cons item (coerce vect 'list)) 'vector))

;; appends a vector to an other vector.
(defun append-vector (&rest params)
  (apply #'concatenate (cons 'vector params)))

;;; ----------------------------------------------------------------------
;;;
;;; Print routine
;;;
;;; ----------------------------------------------------------------------

(defun dashes (n)
  (loop for i from 1 to n do (princ "-"))
  (terpri))

(defun abstract (x threshold max min)
  (cond ((< x min) '-)
	((> x max) '+)
	((< (- x min) threshold) min)
	((< (- max x) threshold) max)
	(t x)))

;; &allow-other-keys is included so that other routines can call print-matrix
;; and pass paramters easily.  For example, see "show-raam"
(defun print-matrix (m &key (digits 3) (width (+ digits 4)) (threshold 0)
		       (max 100) (min -100) &allow-other-keys)
  "Prints a matrix, rounding entries to max/min if they're within threshold"
  (let* ((d (array-dimensions m))
	 (r (length d)))
    (cond ((= r 1) (format t "~a~%" m))
	  ((= r 2)
	   (let* ((max-row (1- (car d)))
		  (max-col (1- (cadr d)))
		  (width$  (princ-to-string width))
		  (digits$ (princ-to-string digits))
		  (count$  (concatenate 'string "~" width$ "@a"))
		  (array$  (concatenate 'string "~" width$ "," digits$ "f")))
	     (format t "~%f\\t| ")
	     (loop for i from 1 to (1+ max-col) do
		   (format t count$ i))
	     (terpri)
	     (princ "---\\-")
	     (dashes (* (1+ max-col) width))
	     (loop for i from 0 to max-row do
		   (format t "~2a | " i)
		   (loop for j from 0 to max-col do
			 (format t array$ (abstract (aref m i j)
						    threshold max min)))
		   (terpri))
	     (terpri)))
	  (t (error "Can only print matrices of dimension 2 or less")))))


(defun vector-midpoint (vect1 vect2)
  (unless (= (length vect1) (length vect2))
	  (error "Compute-Midpoint:: Vectors must be same length."))
  (make-array (list (length vect1))
	      :element-type *element-type*
	      :initial-contents
	       (loop for i from 0 to (1- (length vect1))
		     collect (+ (aref vect1 i)
				(/ (- (aref vect2 i) (aref vect1 i)) 2.0d0)))))
