;;;       THE Ohio State University Laboratory for AI Research (LAIR)       
;;;                            Copyright (c) 1990                           
;;;                                                                         
;;;                            Filename: fst.lisp
;;;                            Author: Peter J Angeline
;;;
;;; Comments:  This file implements the floating symbol table additions for
;;; raams and nets in general.  A floating symbol is a terminal who's value
;;; changes from the feedback of errors via backprop from the network inwhich
;;; the terminal is being used.  The terminals are stored in a table which gets
;;; updated after each pass through the network.  Each update consists of the
;;; integration of the errors from the network followed by the separation of
;;; the representations from each other.  The separation is necessary due to
;;; the tendency of some networks to force the terminals to be identical.
;;;
;;;
;;; Version 0.95 (ANSIL)
;;; ----------------------------------------------------------------------
;;;
;;;                               Change Log
;;;                               ----------
;;;
;;; 04/05/91 - (pja) updated to encompass changes to version 0.95 of ANSIL and
;;; generally cleaned things up a bit.
;;;
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
(proclaim '(optimize speed))

(provide 'mfst)
(in-package 'mfst)

(export '(fst fst-symbols fst-rep-width fst-max-diff fst-min-diff fst-diff
	  fst-delta fst-rate fst-momentum fst-amp fst-separate? fst-num-errors
	  fst-total-error fst-max-error create-fst fst-get-net-errors
	  fst-get-raam-errors fst-update-raam-terminals fst-zero-errors
	  fst-update-symbols randomize-fst fst-train fst-raam-train 
	  fst-set-rep fst-float? write-fst fst-newraam-train fst-schedule
	  fst-check-schedule set-fst show-reps show-distances)) 

(require 'loop)
(require 'linear)
(require 'mbp)
(require 'mraam)

(use-package 'loop)
(use-package 'linear)
(use-package 'mbp)
(use-package 'mraam)

(eval-when (compile load)
   (defconstant *element-type* 'double-float)
   (defconstant one (coerce 1.0 *element-type*))
   (defconstant zero (coerce 0.0 *element-type*)))

;;; ----------------------------------------------------------------------
;;;
;;; Structure Definitions
;;;
;;; ----------------------------------------------------------------------
;; the structure for a single floating symbol
(eval-when 
 (compile load)
 (eval	   
  `(defstruct floating-symbol
     (rep nil :type vector)		; rep for this symbol
     (errors nil :type vector)		; the error for this floating symbol
     (prev-errors nil :type vector)	; previous errors for this fs
     (num-bps ,(coerce 0.0 *element-type*)
	      :type ,*element-type*)	; number of errors added into above
     )))

;; The structure for the floating symbol table

(eval-when (compile load)
 (eval 
  `(defstruct fst
     (symbols nil :type list)		; the assoc list of floating symbols
     (rep-width 0 :type integer)		; rep width of the symbols
     (max-diff ,(coerce 1.0 *element-type*)
	       :type ,*element-type*)		; max separation between reps
     (min-diff ,(coerce 0.1 *element-type*)
	       :type ,*element-type*)	; min separation between reps
     (diff ,(coerce 0.5 *element-type*)
	   :type ,*element-type*)		; current separation difference
     (delta ,(coerce 0.05 *element-type*)
	    :type ,*element-type*)		; change for moving diff
     (rate ,(coerce 1.0 *element-type*)
	   :type ,*element-type*)	; learning rate for floating symbols
     (momentum ,(coerce 0.5 *element-type*)
	       :type ,*element-type*)	; momentum for floating symbols
     (amp ,(coerce 1.0 *element-type*)
	  :type ,*element-type*)	; error amplification from net to FST
     (separate? t :type symbol)		; do separation or not.
     (float? t :type symbol)
     (initial :type array)		; random-seed for initial conditions
     (schedule nil)			; schedule for momentum and rate
     (num-errors 0 :type integer)		; number of reps too close
     (total-error ,(coerce 0.0 *element-type*)
		  :type ,*element-type*) ; total error for reps last update
     (max-error ,(coerce 0.0 *element-type*)
		:type ,*element-type*)	; max error for reps last update
     )))

;;; ----------------------------------------------------------------------
;;;
;;; Creation Routine
;;;
;;; ----------------------------------------------------------------------
;; Creates the basic fst structure complete with the look-up table and returns
;; it to the user.  Terminals is a list of the names of the terminals to be
;; used as floating symbols in trees and rep-width is the representational
;; width of a symbol.
(defun create-fst (terminals rep-width &optional (sep 0.9))
  (let ((fst (make-fst :rep-width rep-width)))
    (setf (fst-symbols fst)
	  (loop for terminal in terminals
		collect (create-floating-symbol terminal rep-width)))
    (randomize-fst fst sep)
    fst))

(defun create-floating-symbol (terminal rep-width)
  (cons terminal
	(make-floating-symbol
	 :rep (make-array (list rep-width)
			  :initial-element zero
			  :element-type *element-type*)
	 :errors (make-array (list rep-width)
			     :initial-element zero
			     :element-type *element-type*)
	 :prev-errors (make-array (list rep-width)
				  :initial-element zero
				  :element-type *element-type*))))

;; allows anything setable in an fst to be set.
(defun set-fst (fst &key (max-diff (fst-max-diff fst))
		         (min-diff (fst-min-diff fst))
			 (diff (fst-diff fst))
			 (delta (fst-delta fst))
			 (amp (fst-amp fst))
			 (separate? (fst-separate? fst))
			 (float? (fst-float? fst))
			 (rate (fst-rate fst))
			 (momentum (fst-momentum fst)))
  (setf (fst-max-diff fst) (coerce max-diff *element-type*)
	(fst-min-diff fst) (coerce min-diff *element-type*)
	(fst-diff fst) (coerce diff  *element-type*)
	(fst-delta fst) (coerce delta *element-type*)
	(fst-separate? fst) separate?
	(fst-amp fst) (coerce amp *element-type*)
	(fst-float? fst) float?
	(fst-rate fst) (coerce rate  *element-type*)
	(fst-momentum fst) (coerce momentum *element-type*))
  (format nil "Current values are:~%~{   ~12a= ~a~%~}~%"
	  (list "Max Diff" max-diff "Min Diff" min-diff "Diff" diff
		"Delta" delta "Separate?" separate? "Float?" float?
		"Amp" amp "Rate" rate "Momentum" momentum)))

;;; ----------------------------------------------------------------------
;;;
;;; Symbol and Assoc List Code
;;;
;;; ----------------------------------------------------------------------
;; returns the floating symbol structure of the given from the given fst
(defun find-fs (fst terminal)
  (cdr (assoc terminal (fst-symbols fst))))

;; returns the vector which is the rep of the terminal 
(defun get-fst-rep (fst terminal)
  (floating-symbol-rep (find-fs fst terminal)))

;; new-rep must be a vector of size fst-rep-width.  Sets the represntation of
;; the given terminal in the fst to be new-rep
(defun fst-set-rep (fst terminal new-rep)
  (unless (= (length new-rep) (fst-rep-width fst))
	  (error "SET-FST-REP: Representation for terminal bad length."))
  (unless (equal (type-of new-rep) `(array ,*element-type*))
	  (setf new-rep
		(make-array (list (length new-rep))
			    :element-type *element-type*
			    :initial-contents (mapcar
					       #'(lambda (x)
						   (coerce x *element-type*))
					       (coerce new-rep 'list)))))
  (setf (floating-symbol-rep (find-fs fst terminal)) new-rep))

;; adds a symbol to the floating symbol table
(defun add-symbol (fst terminal
		   &optional (new-rep (make-array (list (fst-rep-width fst)))))
  (setf (fst-symbols fst) (acons terminal new-rep (fst-symbols fst))))

;; removes a symbol's fs structure from the fst
(defun remove-symbol (fst terminal)
  (setf (fst-symbols fst) (delete (assoc  fst terminal) (fst-symbols fst))))

;;; ----------------------------------------------------------------------
;;;
;;; Interface to Raams and Network Code
;;;
;;; ----------------------------------------------------------------------
;; this function computes the input error for a given network.  It is assumed
;; that the network has been forward-passed and the errors computed as in
;; matrixbp. 
(defun compute-input-errors (net)
  (cdr-vector
   (vector-apply* nil
		 (vectorize (matrix-multiply nil
			     (car (network-weights net))
			     (column-array
			      (cdr-vector (car (network-errors net))))))
		 (vector-diff nil (car (network-activations net))
			      (network-sp-add net)))))

;; This function computes and stores the errors for the given floating symbol. 
(defun fst-get-net-errors (net fs)
  (vector-apply+ (floating-symbol-errors fs)
		 (compute-input-errors net)
		 (floating-symbol-errors fs)))

;; Gets the error vector for each subtree of the tree and places it in the
;; floating symbol table.  fs is the floating symbol we are currently dealing
;; with.
(defun fst-get-raam-errors (raam fst tree)
  (when (fst-float? fst)
	(let ((errors (compute-input-errors (raam-network raam)))
	      (width (fst-rep-width raam))
	      (fserr nil)
	      (copies (if (raam-pattern-repeat raam)
			  (nth (symbol-index-by-name
				(raam-symbol-table raam)
				tree)
			       (raam-pattern-copies raam))
			one))
	      (fs nil))
	  (loop for subtree in tree
		for i = 0 then (1+ i) do
	     (when (symbolp subtree)
		(setf fs (find-fs fst subtree))
		(incf (floating-symbol-num-bps fs) copies)
		(setf fserr (floating-symbol-errors fs))
		(loop for j from 0 to (1- width)
		      for k from (* i width) to (1- (* (1+ i) width)) do
		   (incf (aref fserr j) (aref errors k))))))))

;; updates the terminals of the raam with the contents of the floating symbol
;; table 
(defun fst-update-raam-terminals (raam fst
				  &aux (symtab (raam-symbol-table raam)))
  (loop for term in (fst-symbols fst) do
	(setf (symbol-value-by-name symtab (car term))
	      (floating-symbol-rep (cdr term)))))

;; takes a floating symbol table and zeros out the error vector for each symbol
(defun fst-zero-errors (fst)
  (mapc #'(lambda (fs)
	    (vector-apply (floating-symbol-errors (cdr fs))
			  #'(lambda (x) (declare (ignore x)) zero)
			  (floating-symbol-errors (cdr fs)))
	    (setf (floating-symbol-num-bps (cdr fs)) zero))
	(fst-symbols fst)))

;; takes a floating symbol table and integrates the errors into the
;; represntations.
(defun fst-update-symbols (fst &optional display)
  (when (fst-float? fst)
	(update-reps fst)		; from the accumulated errors from net
	(update-prev-errors fst)	; update previous errors 
	(when (fst-separate? fst)
	      (fst-zero-errors fst)	; zero errors to collect for separation
	      (separate-terminals fst display); compute separation errors
	      (update-reps fst))))	; from the separation

;; takes the floating symbol table and separates the representations.
(defun separate-terminals (fst &optional display
			       &aux (last-total (fst-total-error fst)))
  (setf (fst-total-error fst) (coerce 0.0 *element-type*)
	(fst-max-error fst) (coerce 0.0 *element-type*))
  (loop for term1 in (fst-symbols fst) 
	for fs1 = (cdr term1) do
	(loop for term2 in (cdr (member term1 (fst-symbols fst)))
	      for fs2 = (cdr term2) do
	      (compute-separate-errors fst fs1 fs2 display)))
  (update-diff fst last-total))

;; Changes the fst's diff parameter according to how the total-error is
;; changing. 
(defun update-diff (fst last-tot)
  (setf (fst-diff fst)
	(max (fst-min-diff fst)
	     (min (fst-max-diff fst)
		  (+ (fst-diff fst)
		     (* (fst-delta fst)
			(if (<= (fst-total-error fst) last-tot) -1.0 1.0)))))))

;; compute the errors between the floating symbols fs1 and fs2.   
(defun compute-separate-errors (fst fs1 fs2 &optional display)
  (let* ((v1 (floating-symbol-rep fs1))
	 (v2 (floating-symbol-rep fs2))
	 (dist (vector-distance v1 v2))
	 (diff (fst-diff fst))
	 (errors1 (floating-symbol-errors fs1))
	 (errors2 (floating-symbol-errors fs2))
	 (err (abs (- diff dist))))
    (incf (floating-symbol-num-bps fs1))
    (when display (write-char (if (<= dist diff) #\- #\+)))
    (loop for i from 0 to (1- (length v1))
	  for actv1 = (aref v1 i)
	  for actv2 = (aref v2 i) do
	  (incf (aref errors1 i) (separation-error actv1 actv2 diff dist))
	  (incf (aref errors2 i) (separation-error actv2 actv1 diff dist)))
    (when (< dist diff) 
	  (incf (fst-total-error fst) (* err err))
	  (incf (fst-num-errors fst))
	  (setf (fst-max-error fst) err))))

;; computes the true error for the an activation of actv which is being pushed
;; away from the activation target.
;; difference between the activations.
(defun separation-error (actv target diff dist)
  (when (= dist zero)
	(setf dist (coerce 0.00001 *element-type*)))
  (if (> dist diff)
	 (* (coerce .001 *element-type*)
	    diff (or (and (plusp (- target actv)) (- one)) one))
       (* (- dist diff) (/ (- target actv) dist))))

;; updates the representation of the terminals 
;; *** Note: It would appear that in the last version of FSTs, the previous
;; *** deltas were mixed, in that when the separations were computed, the
;; *** prev-deltas from the feedback were used and vice versa.  This was an
;; *** unintensional "feature" %^).  For this code, rather than 
;; *** implementing a different version, this "feature" will be replicated.
;; *** However, the result of using separate prev-deltas should be
;; *** investigated as a possible algorithm alteration.
(defun update-reps (fst &aux (rate (fst-rate fst))
			     (amp (fst-amp fst))
			     (momentum (fst-momentum fst)))
  (loop for terminal in (fst-symbols fst) 
	for fs = (cdr terminal)
	for rep = (floating-symbol-rep fs)
	for err = (floating-symbol-errors fs)
	for prev = (floating-symbol-prev-errors fs)
	for num = (floating-symbol-num-bps fs) do
	(unless (zerop num)
		(loop for i from 0 to (1- (length rep)) do
		      ; when the err vector becomes the prev-err, it must be
		      ; adjusted by the number of bps and other stuff
		      (setf (aref err i)
			    (+ (* rate amp (/ (aref err i) num) (aref rep i))
			       (* momentum (aref prev i))))
		      (setf (aref rep i) (sigmoid
					  (+ (aref err i)
					     (invsigmoid (aref rep i)))))))))

;; randomizes the reps for the symbols
(defun randomize-fst (fst &optional (num 0.9))
  (when (fst-float? fst)
     (setf (fst-initial fst) (make-random-state))
     (setf *random-seed* (copy-seq (fst-initial fst)))
     (loop for fs in (fst-symbols fst) do
	   (randomize-vector (floating-symbol-rep (cdr fs)) num)
	   (setf (floating-symbol-rep (cdr fs))
		 (vector-apply nil #'sigmoid
			       (floating-symbol-rep (cdr fs)))))))


;; trades the vectors storing the prev-errors and errors for each terminal in
;; the fst.  The errors will be errored out later.
(defun update-prev-errors (fst &aux temp)
  (loop for term in (fst-symbols fst)
	for fs = (cdr term) do
      (setf temp (floating-symbol-errors fs))
      (setf (floating-symbol-errors fs) (floating-symbol-prev-errors fs))
      (setf (floating-symbol-prev-errors fs) temp)))

;;; ----------------------------------------------------------------------
;;;
;;; Example Floating Symbol Learning Routine
;;;
;;; ----------------------------------------------------------------------
;; it is assumed that the patterns stored in (network-patterns net) are interms
;; of the terminals, meaning that both the input and output are a single
;; terminal. For instance, ((a b) (c d) (e f))
(defun fst-train (fst net &key (display 1) (num 10000))
  (let ((errors 1))
    (zero-prev-deltas net)
    (loop for iteration from 0 to (- num 1)
	  for writer = (and (not (zerop display))
			    (zerop (mod iteration display)))
	  until (and (= errors 0) (= (fst-num-errors fst) 0))
	  do
	  (zero-deltas net)		; zero deltas in network
	  (fst-zero-errors fst)		; zero fs errors before this epoch
	  (setf (network-total-error net) 
		(coerce 0.0 *element-type*)  ; zero a bunch of other stuff
		(network-max-error net) (coerce 0.0 *element-type*)
		errors 0
		(fst-num-errors fst) 0)
	  (loop for pattern in (network-patterns net) 
		for input = (get-fst-rep fst (car pattern))
		for output = (cadr pattern) do 
		(set-nodes net input)
		(forward-pass net)
		(compute-output-errors net output (network-epsilon net))
		(compute-hidden-errors net)
		(compute-deltas net)
		; now feedback errors to inputs and move to fst
		(fst-get-net-errors net (find-fs fst (car pattern)))
		(incf errors (count-errors net output :display writer)))
	  (update-weights net)
	  (when writer
		(format t "~%Network: Num= ~d Err= ~f Max= ~f~%"
			errors (network-total-error net)
			(network-max-error net)))
	  (fst-update-symbols fst)
	  (update-prev-deltas net)
	  (fst-check-schedule fst net)
	  (when (and writer (fst-separate? fst))
		(format t
		   "~%FST: Epoch ~d Num=~d Err=~f Max=~f Sigma=~f R=~fM=~f~%" 
		   iteration (fst-num-errors fst) (fst-total-error fst)
		   (fst-max-error fst) (fst-diff fst) (fst-rate fst)
		   (fst-momentum fst)))
	  finally (format t "Epoch ~d: Num= ~d Err= ~f Max= ~f~%"
			  iteration errors (network-total-error net)
			  (network-max-error net)))))

;; Example learning routine for a RAAM with an FST.
(defun fst-raam-train (fst raam &key (display 1) (num 10000) (reconstruct 0)
			             (terminal-test t))
  (let* ((errors 1)
	 (t-set (raam-training-set raam))
	 (net (raam-network raam))
	 (numpats (if (network-adjust? net) (length t-set) one)))
    (zero-prev-deltas net)
    (loop for iteration from 1 to num
	  for output = (and (not (zerop display))
			    (zerop (mod iteration display)))
	  until (and (= errors 0) (= (fst-num-errors fst) 0))
	  do
	  (zero-deltas net)		; zero deltas in network
	  (fst-zero-errors fst)		; zero fs errors before this epoch
	  (setf (network-total-error net) zero
		(network-max-error net) zero
		errors 0 (fst-num-errors fst) 0)
	  (fst-update-raam-terminals raam fst)
	  (loop for tree-index in t-set do
		(incf errors
		      (raam-train-pattern raam tree-index numpats output
					  terminal-test))
		;; now feedback errors to inputs and move to fst
		(fst-get-raam-errors raam fst (car tree-index)))
	  (incf (raam-total-epochs raam))
	  (update-weights net)
	  (when (and output (fst-separate? fst)) (terpri))
	  (fst-update-symbols fst output)
	  (fst-check-schedule fst (raam-network raam))
	  (when output
	      (format t "~%Raam: Err=~d Tot=~,6f Max=~,6f~%"
		      errors (network-total-error net) (network-max-error net))
	      (format t "FST: Ep ~d N=~d Err=~,6f Max=~,6f S=~,6f R=~,6f~%"
		      iteration (fst-num-errors fst) (fst-total-error fst)
		      (fst-max-error fst) (fst-diff fst) (fst-rate fst)))
	  (update-prev-deltas net)
	  (when (and (zerop errors) (not (zerop reconstruct)))
		(setf errors 1))
	  (when (and (not (zerop reconstruct))
		     (or (zerop errors) (zerop (mod iteration reconstruct))))
		(if (eval (cons 'and (verify raam)))
		    (setf errors 0)
		  (setf errors 1)))
	  finally
	  (format t "~%Raam: Err=~d Tot=~,6f Max=~,6f~%"
		  errors (network-total-error net) (network-max-error net))
	  (format t 
		  "~%FST: Epoch ~d Err=~d Tot=~,6f Max=~,6f Sigma=~,6f~%"
		  iteration (fst-num-errors fst) (fst-total-error fst)
		  (fst-max-error fst) (fst-diff fst)))))

;; Example learning routine for a RAAM with an FST.
;(defun fst-newraam-train (fst raam &key (display 1) (num 10000))
;  (let* ((errors 1)
;	 (net (raam-network raam))
;	 (t-set (reverse (raam-training-set raam)))
;	 (numpats (if (network-adjust? net) (length t-set) one)))
;    (zero-prev-deltas net)
;    (loop for iteration from 0 to (- num 1)
;	  for output = (and (not (zerop display))
;			    (zerop (mod iteration display)))
;	  until (and (= errors 0) (= (fst-num-errors fst) 0))
;	  do
;	  (zero-deltas net)		; zero deltas in network
;	  (fst-zero-errors fst)		; zero fs errors before this epoch
;	  (zero-newraam-errors raam)
;	  (setf (network-total-error net) zero
;		(network-max-error net) zero
;		errors 0 (fst-num-errors fst) 0)
;	  (fst-update-raam-terminals raam fst)
;	  (loop for tree-index in t-set do
;		(update-symbol-table raam tree-index))
;	  (loop for tree-index in t-set do
;		(incf errors
;		      (raam-train-pattern raam tree-index numpats output))
;		;; now feedback errors to inputs and move to fst
;		(fst-get-raam-errors raam fst (car tree-index)))
;	  (update-weights net)
;	  (when (and output (fst-separate? fst)) (terpri))
;	  (fst-update-symbols fst output)
;	  (fst-check-schedule fst (raam-network raam))
;	  (when output
;	      (format t "~%NRaam: Num= ~d Err= ~8,6f Max= ~8,6f~%"
;		      errors (network-total-error net) (network-max-error net))
;	      (format t "FST: E=~d N=~d Err=~,6f Max=~,6f Sig=~,6f R=~,6f M=~,6f~%"
;		      iteration (fst-num-errors fst) (fst-total-error fst)
;		      (fst-max-error fst) (fst-diff fst) (fst-rate fst)
;		      (fst-momentum fst)))
;	  (update-prev-deltas net)
;	  finally
;	      (format t "~%Raam: Num= ~d Err= ~8,6f Max= ~8,6f~%"
;		      errors (network-total-error net) (network-max-error net))
;	      (format t "FST: E=~d N=~d Err=~,6f Max=~,6f Sig=~,6f R=~,6f M=~,6f~%"
;		      iteration (fst-num-errors fst) (fst-total-error fst)
;		      (fst-max-error fst) (fst-diff fst) (fst-rate fst)
;		      (fst-momentum fst)))))
 
;; checks the rate and momentum schedules to adjust the rate and momentum of
;; the network and the fst based
(defun fst-check-schedule (fst net
			       &aux (total-error (network-total-error net)))
  (loop for phase in (fst-schedule fst)
	for val = (car phase)
	for rate = (cadr phase)
	for momentum = (caddr phase) do
	(when (<= total-error (coerce val *element-type*))
	      (setf (fst-rate fst) (coerce rate *element-type*))
	      (setf (fst-momentum fst) (coerce momentum *element-type*))
	      (setf (network-rate net) (coerce rate *element-type*))
	      (setf (network-momentum net) (coerce momentum *element-type*)))))

;;
(defun fst-show-patterns (fst net)
  (loop for pattern in (network-patterns net)
	do
	(set-nodes net (get-fst-rep fst (car pattern)))
	(forward-pass net)
	(print (list pattern
		     (coerce (last (network-activations net)) 'list)))))
;; 
(defun show-distances (fst)
  (loop for fs1 in (fst-symbols fst) 
	for i = 1 then (1+ i) do
	(loop for fs2 in (nthcdr i (fst-symbols fst)) do
	      (print (vector-distance 
		      (floating-symbol-rep (cdr fs1))
		      (floating-symbol-rep (cdr fs2)))))))

(defun avg-dist (fst)
  (let ((sum (coerce 0.0 *element-type*))
	(count 0))
    (loop for fs1 in (fst-symbols fst)
	  for i = 1 then (1+ i) do
	  (loop for fs2 in (nthcdr i (fst-symbols fst)) do
		(incf sum (vector-distance 
			   (floating-symbol-rep (cdr fs1))
			   (floating-symbol-rep (cdr fs2))))
		(incf count)))
    (/ sum count)))

(defun show-reps (fst)
  (loop for fs in (fst-symbols fst) do
	(print (list (car fs) (floating-symbol-rep (cdr fs))))))

(defun show-errs (fst)
  (loop for fs in (fst-symbols fst) do
	(print (list (floating-symbol-errors (cdr fs))
		     (floating-symbol-prev-errors (cdr fs))))))
  
(defun zero-newraam-errors (raam)
  (loop for item in (raam-error-table raam)
	for err = (cadr item) do
	(setf (caddr item) (coerce 0.0 *element-type*))
	(vector-apply err #'* err (coerce 0.0 *element-type*))))

(defun write-fst (r filename)
  (unless (typep r 'fst) (error "First argument must be of type \"fst\""))
  (with-open-file (fi (concatenate 'string filename ".fst")
		      :direction :output :if-exists :supersede)
		  (pprint r fi))
  nil)

