;;;       THE Ohio State University Laboratory for AI Research (LAIR)       
;;;                            Copyright (c) 1990                           
;;;                                                                         
;;;                            Filename: clinear.lisp
;;;                            Author(s): Peter J Angeline 
;;;                                       Greg Saunders
;;;
;;; Comments: Functions for linear algebra.  these functions call the c library
;;; routines in linear-lib.o.  This file is loaded automatically if you are
;;; running FCL.
;;;
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(proclaim '(optimize speed))

;;; ---------------------------------------------------------------------- 
;;;
;;; License: This code may be distributed free of charge.  This code may not be
;;; incorporated into any production software of any type or any code which is
;;; sold without the express written consent of the authors.  Modifications and
;;; inclusions for other types of software are permitted, provided that this
;;; header remains unchanged and included in all subsequent versions of the
;;; code.
;;;
;;; Disclaimer: The authors of this code make no claims as to its
;;; validity, and are in no way responsible for any errors contained within.
;;; 
;;; File description: This file contains implementations of linear algebra
;;; routines which are implemented in C routines and called by Franz Allegro
;;; Common Lisp.  This file is specifc to Franz Allegro Common Lisp and will
;;; not work with other common lisps.  The file Linear.lisp should be used as a
;;; replacement for this file when using a different common lisp. the file with
;;; the C routines, linear-lib.o, should be placed in the same directory.
;;;
;;; Creation date: Aug 5, 1990 Version: 0.95
;;; Modifications: none 
;;; 
;;; Interface/exported functions 
;;; Package info

(require 'loop)				; use the nice loop macros
(require 'foreign)

(use-package 'loop)

(provide 'linear)			; define module linear
					
(in-package 'linear)			; define package linear

(export '(matrix-multiply matrix-apply matrix-apply+ matrix-apply*
	  row-array column-array row column vectorize cons-vector cdr-vector
	  vector-apply dot-product vector-product set-row set-column
	  vector-distance append-vector copy-matrix copy-vector print-matrix
	  set-diagonal vector-apply* vector-apply+ vector-midpoint
	  vector-binary-diff vector-diff vector-apply-sigmoid print-matrix))

(defvar user::*ansil-path*)

;; constants which are used by the compiler and the loader.
(eval-when (compile load)
  ;; the C routines expect things of type double float.
  (defconstant *element-type* 'double-float) ; size of the elements of
					      ; matrices in the code.
  (defconstant zero (coerce 0.0 *element-type*)))

;;; ----------------------------------------------------------------------
;;;
;;; Foreign Function Interface
;;;
;;; ----------------------------------------------------------------------
(ff:reset-entry-point-table)		; erase any entry points so don't
					; double load c code, which causes an
					; error.

;; add a path to the front of the file name here if you want to put
;; linear-lib.o in a different directory.
(load (concatenate 'string user::*ansil-path* "linear-lib.o") :system-libraries '("m"))
;(load "/n/music/0/nns/raamcl/ver0.9/linear-lib.o" :system-libraries '("m"))
;(load "linear-lib.o" :system-libraries '("m"))

;; define the foriegn functions.  All of these have lisp counterparts and will
;; be described there.
(ff:defforeign-list
 '((matrix_multiply
    :arguments ((array double-float) (array double-float) (array double-float)
		integer integer integer))

   (matrix_scalar_add 
    :arguments ((array double-float) (array double-float) double-float
		integer integer))

   (matrix_matrix_add 
    :arguments ((array double-float) (array double-float) (array double-float)
		integer integer))

   (matrix_scalar_mult
    :arguments ((array double-float) (array double-float) double-float
		integer integer))

   (matrix_matrix_mult
    :arguments ((array double-float) (array double-float) (array double-float)
		integer integer))

   (getrow
    :arguments ((array double-float) (array double-float) integer integer
		integer))

   (getcolumn
    :arguments ((array double-float) (array double-float) integer integer
		integer))

   (dot_product
    :arguments ((array double-float) (array double-float) integer)
    :return-type :double-float)

   (vector_product
    :arguments ((array double-float) (array double-float) (array double-float)
		integer integer))

   (set_row
    :arguments ((array double-float) integer (array double-float) integer
		integer)) 

   (set_column
    :arguments ((array double-float) integer (array double-float) integer
		integer))

   (vector_distance
    :arguments ((array double-float) (array double-float) integer)
    :return-type :double-float)

   (copy_matrix
    :arguments ((array double-float) (array double-float) integer integer))

   (vector_apply_sigmoid
    :arguments ((array double-float) integer))

   (vector_vector_mult
    :arguments ((array double-float) (array double-float) (array double-float)
		integer))

   (vector_vector_add
    :arguments ((array double-float) (array double-float) (array double-float)
		integer))

   (vector_scalar_mult
    :arguments ((array double-float) (array double-float) double-float
		integer)) 

   (vector_scalar_add
    :arguments ((array double-float) (array double-float) double-float
		integer)) 

   (vector_midpoint
    :arguments ((array double-float) (array double-float) (array double-float)
		integer))

   (vector_diff 
    :arguments ((array double-float) (array double-float) double-float
		integer))

   (vector_binary_diff
    :arguments ((array double-float) (array double-float) (array double-float)
		double-float double-float integer integer))))

;;; ----------------------------------------------------------------------
;;;
;;; Matrix Routines
;;;
;;; ----------------------------------------------------------------------
;; matrix-multiply algorithm.  If result is non-nil then it must be a matrix of
;; the correct size in which the computation will be done.  If result is
;; non-nil then the function gives no result but simply alters RESULT.  If
;; result is nil then a new matrix is created of the correct size and type and
;; becomes the return values of the function.
(defun matrix-multiply (result matrix1 matrix2)
  (unless (equal (array-dimension matrix1 1)
		 (array-dimension matrix2 0))
	  (error "MATRIX-MULTIPLY: Matrix sizes don't agree."))
  (when (and result
	     (not (= (array-dimension matrix1 0) (array-dimension result 0)))
	     (not (= (array-dimension matrix2 1) (array-dimension result 1))))
	  (error "MATRIX-MULTIPLY: Result Matrix not correct size."))
  (let ((answer (if result result
		  (let ((type (type-of (aref matrix1 0 0))))
		    (make-array (list (array-dimension matrix1 0)
				      (array-dimension matrix2 1))
				:element-type type))))
	(width (array-dimension matrix1 1))
	(rows (array-dimension matrix1 0))
	(cols (array-dimension matrix2 1)))
    (matrix_multiply answer matrix1 matrix2 rows width cols)
    (if result nil answer)))

;; apply op to each element of MATRIX.  If vals is null then the op must be
;; unary. When vals is not empty op must be binary. When the argument is a
;; matrix, the matrices MUST be the same size. When the argument is a value,
;; simply do op of this value to each element in the matrix. Returns the
;; finished product.
(defun matrix-apply (result op matrix &rest vals)
  (let* ((rows (array-dimension matrix 0))
	 (cols (array-dimension matrix 1))
	 (type (type-of (aref matrix 0 0)))
	 (answer (if result result
		   (make-array (array-dimensions matrix) :element-type type))))
  ; when vals is null, then do a single loop, otherwise move through the list
  ; stored in vals making val the next thing in vals and doing whatever needs
  ; to be done to answer.
    (loop for val in (or vals '(nil))
	  for mat = matrix then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions mat)
				 (array-dimensions val))))
		(error "MATRIX-APPLY: Matrices must be the same size."))
	  (loop for i from 0 to (1- rows) do
		(loop for j from 0 to (1- cols) do
		      (setf (aref answer i j)
			    (coerce 
			     (apply op 
				    (cons (aref mat i j)
					  (cond ((null vals) nil)
						((listp (type-of val))
						 (list (aref val i j)))
						(t (list val)))))
			     type)))))
    ; if we were given a matrix to alter then don't retunr anything. if we
    ; created a matrix then return it.
    (if result nil answer)))

;; same as matrix-apply but assigns op to be +.  This is so this can be
;; optimized in the C algorithm.
(defun matrix-apply+ (result matrix &rest vals)
  (when (null vals)
	(error "MATRIX-APPLY+: Must have at least one value to add."))
  (let ((rows (array-dimension matrix 0))
	(cols (array-dimension matrix 1))
	(answer (if result result
		  (let ((type (type-of (aref matrix 0 0))))
		    (make-array (array-dimensions matrix)
				:element-type type)))))
    (loop for val in (or vals '(nil))
	  for mat = matrix then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions mat)
				 (array-dimensions val))))
		(error "MATRIX-APPLY+: Matrices must be the same size."))
	  (if (listp (type-of val))
	      ; c routine to add two matrices elementwise.
	      (matrix_matrix_add answer mat val rows cols)
	    ; c routine to and a scalar to each element of a matrix.
	    (matrix_scalar_add answer mat val rows cols)))
    (if result nil answer)))

;; Same as matrix-apply but the op is bound to *.  This is so the function can
;; be optimized in the C algorithm.
(defun matrix-apply* (result matrix &rest vals)
  (when (null vals) 
	(error "MATRIX-APPLY*: Must have more than one value to multiply.")) 
 (let ((rows (array-dimension matrix 0))
	(cols (array-dimension matrix 1))
	(answer (if result result
		  (let ((type (type-of (aref matrix 0 0))))
		    (make-array (array-dimensions matrix)
				:element-type type)))))
    (loop for val in (or vals '(nil))
	  for mat = matrix then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions mat)
				 (array-dimensions val))))
		(error "MATRIX-APPLY*: Matrices must be the same size."))
	  (if (listp (type-of val))
	      (matrix_matrix_mult answer mat val rows cols)
	    (matrix_scalar_mult answer mat val rows cols)))
    (if result nil answer)))

;; takes a matrix, a row number and a vector and sets the values in
;; that row of the matrix to be the same as the given vector.  No return value.
;; THIS IS A DESTRUCTIVE FUNCTION to the matrix.
(defun set-row (matrix row vector)
  (unless (= (array-dimension matrix 1) (length vector))
	  (error "SET-ROW: Vector is not the same length as row of matrix."))
  (set_row matrix row vector (array-dimension matrix 0)
	   (array-dimension matrix 1))
  nil)
  
;; takes a matrix, a column number and a vector and replaces the values in
;; that column of the matrix to be the same as the given vector.  No return
;; value. THIS IS A DESTRUCTIVE FUNCTION to the matrix.
(defun set-column (matrix column vector)
  (unless (= (array-dimension matrix 0) (length vector))
	  (error "SET-COLUMN: Vector and matrix column different lengths."))
  (set_column matrix column vector (array-dimension matrix 0)
	      (array-dimension matrix 1))
  nil)

;;;delete?
;; takes a square matrix, and a vector and replaces the values along
;; the diagonal of the matrix to be the same as the given vector.  No return
;; value. THIS IS A DESTRUCTIVE FUNCTION to the matrix. - gms
(defun set-diagonal (matrix vector)
  (unless (= (array-dimension matrix 0)
	     (array-dimension matrix 1)
	     (length vector))
	  (error "SET-DIAGONAL:  Lenghts not right."))
  (dotimes (i (length vector) matrix)
	(setf (aref matrix i i) (aref vector i))))

;; copies a matrix and it's contents into a distinct matrix.  Returns the copy.
(defun copy-matrix (matrix)
  (let ((result (make-array (array-dimensions matrix)
			    :element-type (type-of (aref matrix 0 0)))))
    ; c routine to do this
    (copy_matrix result matrix (array-dimension matrix 0)
		 (array-dimension matrix 1))
    result))

;;; ----------------------------------------------------------------------
;;;
;;; Operations on Vectors
;;;
;;; ----------------------------------------------------------------------
;; returns a vector which is equivalent to the nth row of matrix
(defun row (result matrix n)
  (unless (< n (array-dimension matrix 0))
	  (error "ROW: Matrix doesn't have enough rows."))
  (when (and result (not (= (length result) (array-dimension matrix 1))))
	(error "ROW: Result is wrong size."))
  (let ((answer (if result result
		  (make-array (list (array-dimension matrix 1))
			      :element-type (type-of (aref matrix 0 0))))))
    ; c routine which is the meat of this function
    (getrow answer matrix n (array-dimension matrix 0)
	    (array-dimension matrix 1))
    (if result nil answer)))

;; returns a vector which is equivalent to the nth column of matrix
(defun column (result matrix n)
  (unless (< n (array-dimension matrix 1))
	  (error "COLUMN: Matrix doesn't have enough columns."))
  (when (and result (not (= (length result) (array-dimension matrix 0))))
	(error "COLUMN: Result is wrong size."))
  (let ((answer (if result result
		  (make-array (list (array-dimension matrix 0))
			      :element-type (type-of (aref matrix 0 0))))))
    ; speedy c routine to do this
    (getcolumn answer matrix n (array-dimension matrix 0)
	       (array-dimension matrix 1))
    (if result nil answer)))

;; returns a row array which is equivalent to the vector. A copy is not made so
;; destructive operations on the result will change the original vector.
(defun row-array (vect)
  (make-array (cons 1 (array-dimensions vect)) :displaced-to vect 
	      :element-type (type-of (aref vect 0)))) 

;; returns a column array which is equivalent to the given vector.  A copy is
;; not made so destructive operations on the result will change the original
;; vector. 
(defun column-array (vect)
  (make-array (append (array-dimensions vect) '(1)) :displaced-to vect
	      :element-type (type-of (aref vect 0)))) 

;; takes either a row or column matrix and returns an equivalent vector.  A
;; copy is not made of the array, so modifications to the result of this
;; operation will appear in the original array.
(defun vectorize (matrix)
  (unless (and (= 2 (length (array-dimensions matrix)))
	       (= 1 (apply #'min (array-dimensions matrix))))
	  (error "VECTORIZE: Matrix can only be a Column or Row matrix."))
  (make-array (list (apply #'max (array-dimensions matrix)))
	      :element-type (type-of (aref matrix 0 0))           ; gms
	      :displaced-to matrix))
  
;; performs a dot (inner) product on the given vectors. Result is a single
;; value. Vectors must be of the same length.
(defun dot-product (vector1 vector2)
  (unless (= (array-dimension vector1 0) (array-dimension vector2 0))
	  (error "DOT-PRODUCT: Vectors not the same size."))
  ; speedy c routine for this too!
  (dot_product vector1 vector2 (array-dimension vector1 0)))

;; performs an outer product on the vectors.  Result is a (length vector1) x
;; (length vector2) matrix.
(defun vector-product (result vector1 vector2)
  (when (and result (not (equal (array-dimensions result)
				(list (length vector1) (length vector2)))))
	(error "VECTOR-PRODUCT: Result matrix worng size."))
  (let ((answer (if result result
		  (make-array (list (length vector1) (length vector2))
			    :element-type (type-of (aref vector1 0))))))
    ; speedy c routine call
    (vector_product answer vector1 vector2
		    (array-dimension vector1 0) (array-dimension vector2 0))
    (if result nil answer)))

;; Applies the operator op to the input vector.  Works just like matrix-apply
;; so look at that for comments if you're confused about how this works.
(defun vector-apply (result op vect &rest vals)
  (let* ((len (array-dimension vect 0))
	 (type (type-of (aref vect 0)))
	 (answer (if result result
		   (make-array (array-dimensions vect) :element-type type))))
    (loop for val in (or vals '(nil))
	  for ans = vect then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions ans)
				 (array-dimensions val))))
		(error "VECTOR-APPLY: Vectors must be the same size."))
	  (dotimes (i len)
		   (setf (aref answer i)
			 (coerce 
			  (apply op 
				 (cons (aref ans i)
				       (cond ((null vals) nil)
					     ((listp (type-of val))
					      (list (aref val i)))
					     (t (list val)))))
			  type))))
    (if result nil answer)))

;; a version of vector-apply which assumes that the operation is multiplication
(defun vector-apply* (result vect &rest vals)
  (when (null vals) 
	(error "VECTOR-APPLY*: Must have more than one value to multiply."))
  (let ((len (array-dimension vect 0))
	(answer (if result result
		  (make-array (array-dimensions vect)
			      :element-type (type-of (aref vect 0))))))
    (loop for val in (or vals '(nil))
	  for ans = vect then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions ans)
				 (array-dimensions val))))
		(error "VECTOR-APPLY*: Vectors must be the same size."))
	  (if (listp (type-of val))
	      ; c routine to do elementwise multiply of vectors
	      (vector_vector_mult answer ans val len)
	    ; c routine to multiply elements of vector by scalar in val
	    (vector_scalar_mult answer ans val len)))
    (if result nil answer)))

;; a version of vector-apply which assumes that the operation is multiplication
(defun vector-apply+ (result vect &rest vals)
  (when (null vals) 
	(error "VECTOR-APPLY+: Must have more than one value to multiply."))
  (let ((len (array-dimension vect 0))
	(answer (if result result
		  (make-array (array-dimensions vect)
			      :element-type (type-of (aref vect 0))))))
    (loop for val in (or vals '(nil))
	  for ans = vect then answer do
	  (when (and (listp (type-of val))
		     (not (equal (array-dimensions ans)
				 (array-dimensions val))))
		(error "VECTOR-APPLY+: Vectors must be the same size."))
	  (if (listp (type-of val))
	      ; c routine
	      (vector_vector_add answer ans val len)
	    ; c routine
	    (vector_scalar_add answer ans val len)))
    (if result nil answer)))

;;; ----------------------------------------------------------------------
;;;
;;; Backprop specific entries
;;;
;;; These functions were included to make life in backprop a little more
;;; livable.  Having these functions here allows them to be optimized.
;;; ----------------------------------------------------------------------

;; Applies a sigmoid function to each element of the vector.  This function is
;; destructive of the input vector.  DESTRUCTIVE!
(defun vector-apply-sigmoid (vect)
  ; speedy c function
  (vector_apply_sigmoid vect (array-dimension vect 0))
  nil)

;; Applies the function (* (- vect1 vect2) vect1 (- 1.0 vect1)) to the vectors.
;; DESTRUCTIVE!
(defun vector-binary-diff (result vect1 vect2 sp-add tol start end)
  (let ((answer (if result result
		  (make-array (array-dimensions vect1)
			      :element-type (type-of (aref vect1 0))))))
    ; ye olde speedy c function
    (vector_binary_diff answer vect1 vect2 sp-add tol start end)
    (if result nil answer)))

;; Applies the function (* vect1 (- 1.0 vect1)) to the vectors.  DESTRUCTIVE!
(defun vector-diff (result vect &optional (sp-add zero))
  (let ((len (array-dimension vect 0))
	(answer (if result result
		  (make-array (array-dimensions vect)
			      :element-type (type-of (aref vect 0))))))
    ; Imagine that, another speedy c routine.
    (vector_diff answer vect sp-add len)
    (if result nil answer)))

;; Computes the midpoint between two vectors.  Returns the vector.
(defun vector-midpoint (vect1 vect2)
  (unless (= (length vect1) (length vect2))
	  (error "VECTOR-MIDPOINT:: Vectors different lengths."))
  (let ((result (make-array (list (length vect1))
			    :element-type *element-type*)))
    (vector_midpoint result vect1 vect2 (length vect1))
    result))

;; computes the euclidean distance between two vectors and returns it.
(defun vector-distance (vector1 vector2)
  (unless (= (length vector1) (length vector2))
	  (error "VECTOR-DISTANCE: Vectors must be of same length."))
  ; c function does this and returns the right thing.
  (sqrt (vector_distance vector1 vector2 (array-dimension vector1 0))))

;; copies a vector by using copy_matrix
(defun copy-vector (vect)
  (let ((result (make-array (array-dimensions vect)
			    :element-type (type-of (aref vect 0)))))
    (copy_matrix result vect (length vect) 1)
    result))

;;; ----------------------------------------------------------------------
;;;
;;; List operations on vectors
;;;
;;; Somtimes in the code we had a nagging urge to treat vectors as lists.
;;; rather than resist that urge, we wrote routines that made it obvious that
;;; we were doing this.  The routines do what you might think but to vectors
;;; rather than lists.
;;; ----------------------------------------------------------------------
;; returns a vector which is the given vector minus the first element.  the
;; returned vector is NOT a copy of the given vector.
(defun cdr-vector (vect)
  (make-array (list (1- (array-dimension vect 0)))
	      :displaced-to vect
	      :element-type (type-of (aref vect 0))
	      :displaced-index-offset 1))

;; returns a vector with the given item consed onto the front of it.
(defun cons-vector (item vect)
  (coerce (cons item (coerce vect 'list)) 'vector))

;; appends a vector to an other vector.
(defun append-vector (&rest params)
  (apply #'concatenate (cons `(vector ,*element-type*) params)))

;;; ----------------------------------------------------------------------
;;;
;;; Print routine for matrices
;;;
;;; ----------------------------------------------------------------------
(defun dashes (n)
  (loop for i from 1 to n do (princ "-"))
  (terpri))

(defun abstract (x threshold max min)
  (cond ((< x min) '-)
	((> x max) '+)
	((< (- x min) threshold) min)
	((< (- max x) threshold) max)
	(t x)))

;; &allow-other-keys is included so that other routines can call print-matrix
;; and pass paramters easily.  For example, see "show-raam"
(defun print-matrix (m &key (digits 3) (width (+ digits 4)) (threshold 0)
		       (max 100) (min -100) &allow-other-keys)
  "Prints a matrix, rounding entries to max/min if they're within threshold"
  (let* ((d (array-dimensions m))
	 (r (length d)))
    (cond ((= r 1) (format t "~a~%" m))
	  ((= r 2)
	   (let* ((max-row (1- (car d)))
		  (max-col (1- (cadr d)))
		  (width$  (princ-to-string width))
		  (digits$ (princ-to-string digits))
		  (count$  (concatenate 'string "~" width$ "@a"))
		  (array$  (concatenate 'string "~" width$ "," digits$ "f")))
	     (format t "~%f\\t| ")
	     (loop for i from 1 to (1+ max-col) do
		   (format t count$ i))
	     (terpri)
	     (princ "---\\-")
	     (dashes (* (1+ max-col) width))
	     (loop for i from 0 to max-row do
		   (format t "~2a | " i)
		   (loop for j from 0 to max-col do
			 (format t array$ (abstract (aref m i j)
						    threshold max min)))
		   (terpri))
	     (terpri)))
	  (t (error "Can only print matrices of dimension 2 or less")))))
