/* 

  ****************   NO WARRANTY  *****************

Since the Aspirin/MIGRAINES system is licensed free of charge,
the MITRE Corporation provides absolutley no warranty. Should
the Aspirin/MIGRAINES system prove defective, you must assume
the cost of all necessary servicing, repair or correction.
In no way will the MITRE Corporation be liable to you for
damages, including any lost profits, lost monies, or other
special, incidental or consequential damages arising out of
the use or inability to use the Aspirin/MIGRAINES system.

  *****************   COPYRIGHT  *******************

This software is the copyright of The MITRE Corporation. 
It may be freely used and modified for research and development
purposes. We require a brief acknowledgement in any research
paper or other publication where this software has made a significant
contribution. If you wish to use it for commercial gain you must contact 
The MITRE Corporation for conditions of use. The MITRE Corporation 
provides absolutely NO WARRANTY for this software.

   January, 1992 
   Russell Leighton
   The MITRE Corporation
   7525 Colshire Dr.
   McLean, Va. 22102-3481

*/
#include <stdio.h>

#include "ui.h"


/* extern */
extern UICONTEXT *current_context;
extern char *uiextract_word();

/* local */
static int p_x, p_y, p_header=1, p_format=P_ASCII, p_verbose=1;
static char *p_command, *p_name;
static UIPIPE *pipes = NULL;

static void pipe_write_data(p)
     UIPIPE *p;
{
  int dtype=P_FLOAT;

  if (p_verbose) fprintf(stderr, "\nWriting to data to %s\n", p->name);
  
  /* header? */
  if (p->header) {
    
    /* format */
    switch (p->format) {
    case P_ASCII :
      fprintf(p->stream, "ascii ");
      break;
    case P_BINARY :
      fprintf(p->stream, "binary ");
      break;
    }
    
    /* type */
    switch (dtype) {
    case P_FLOAT :
      fprintf(p->stream, "float ");
      break;
    case P_COMPLEX :
      fprintf(p->stream, "complex_float ");
      break;
    }
    
    /* end of header */
    fprintf(p->stream, "%d %d\n", p->data_xdim, p->data_ydim);
  }/* end if header */
  
  /* write it... */
  
  /* format */
  switch (p->format) {
  case P_ASCII : {
    float *d=p->data;
    int y = p->data_ydim;
    while(y--) {
      int x=p->data_xdim;
      while(x--) fprintf(p->stream, "%f ", *d++);
      fprintf(p->stream, "\n");
    }/* end while p->data_ydim */
    break;
  }
  case P_BINARY :
    fwrite(p->data, sizeof(float), p->data_xdim*p->data_ydim, p->stream);
    break;
  }
  
  fflush(p->stream);

}



static int pipe_node_index(input, type)
     char *input;
     int type;
{
  float val;
  int *index_ptr;

  if (type == 0) /* x */
    index_ptr = &p_x;
  else           /* y */
    index_ptr = &p_y;

  if ( sscanf(input, "%f", &val) != 1) return (1);

  if (val < 0) return(1);

  *index_ptr = val;

  return(0);
}


static int parse_pipe_args(type, buffer)
     int type;
     char *buffer;
{
  int arg=2;
  char input[MAX_INPUT_SIZE], *next;
  
  /* connection weights need node indices */
  switch (type) {
  case P_CONNECTION_WEIGHTS :
    /* x */
    if ( (next = uiextract_word(input, arg, buffer)) == NULL ||
	pipe_node_index(input,0) ) {
      fprintf(stderr, "\npopen: Unable to read %d argument (<x node index>)!\n", arg-1);
      return(1);
    }
    arg++;
    /* y */
    if ( (next = uiextract_word(input, arg, buffer)) == NULL ||
	pipe_node_index(input,1) ) {
      fprintf(stderr, "\npopen: Unable to read %d argument (<y node index>)!\n", arg-1);
      return(1);
    }
    arg++;
    break;
  }
  
  /* name */
  if ( (next = uiextract_word(input, arg, buffer)) == NULL) {
    fprintf(stderr, "\npopen: Unable to read %d argument (name)!\n", arg-1);
    return(1);
  }
  arg++;

  /* handle for pipe */
  p_name = am_alloc_mem(MAX_INPUT_SIZE);
  strcpy(p_name, input);

  /* Unix pipe */
  p_command = am_alloc_mem(MAX_INPUT_SIZE);
  strcpy(p_command, next);
  
  return(0);
}


int pipe_info(buffer)
     char *buffer;
{
  printf("\nMode: %s", (p_verbose)?"verbose":"silent");
  printf("\nPipe variables: header (%s) format (%s)",
	 (p_header)?"yes":"no",
	 (p_format==P_ASCII)?"ascii":"binary");

  /* open pipes? */
  if (pipes == NULL) return(1);
  printf("\nOpen pipes:");

  {
    UIPIPE *p=pipes;

    while(p != NULL) {
      printf("\n\t%s %10s...", p->name, p->command);
      p = p->next;
    }
  }

  return(1);

}

int pipe_verbose(buffer)
     char *buffer;
{
    p_verbose = 1; return(1);
}


int pipe_silent(buffer)
     char *buffer;
{
    p_verbose = 0; return(1);
}

int pipe_format_ascii(buffer)
     char *buffer;
{
    p_format = P_ASCII;  return(1);
}

int pipe_format_binary(buffer)
     char *buffer;
{
    p_format = P_BINARY;  return(1);
}


int pipe_noheader(buffer)
     char *buffer;
{
  p_header = 0;  return(1);
}

int pipe_header(buffer)
     char *buffer;
{
  p_header = 1;  return(1);
}

static UIPIPE *find_stream()
{
  UIPIPE *p=pipes;

  while(p != NULL) {
    if ( ! strcmp(p->name, p_name) ) break;
    p = p->next;
  }

  return(p);

}

static UIPIPE *open_stream()
{
  UIPIPE *p;

  /* is it in the list? */
  p = find_stream();
  if (p != NULL) {
    fprintf(stderr, "\nSorry, you already have a pipe by that name!\n");
    return(NULL);
  }

  /* push into list */
  p = (UIPIPE *)am_alloc_mem(sizeof(UIPIPE));

  p->name = p_name;
  p->command = p_command;
  p->info = (BB_INFO *)(current_context->data);
  p->x = p_x; p->y = p_y;
  p->header = p_header;
  p->format = p_format;
  p->data = NULL;
  p->next = NULL;
  p->stream = am_popen(p_command, "w");
  if ( p->stream == NULL) {
    fprintf(stderr, "\nUnable to open %s pipe to: %s\n", p_name, p_command);
    return(NULL);
  }

  if (pipes == NULL)
    pipes = p;
  else {
    p->next = pipes;
    pipes = p;
  }

  return (p);
}

static void close_stream(p)
     UIPIPE *p;
{
  
  if (p == NULL) return;

  if (p_verbose) fprintf(stderr, "\nClosing %s\n", p->name);

  /* empty the stream */
  fflush(p->stream);

  /* close stream */
  am_pclose(p->stream);
  
  /* free strings */
  am_free_mem(p->name);
  am_free_mem(p->command);

  /* free data */
  am_free_mem(p->data);
  
  /* remove from list */
  if (pipes == p)
    pipes = p->next;
  else  {
    UIPIPE *ptr=pipes;
    
    while(ptr->next != p ) ptr = ptr->next;
    
    ptr->next = p->next;
  }

  /* free */
  am_free_mem(p);
}

int pipe_open_nodes(buffer)
     char *buffer;
{
  UIPIPE *p;

  /* am i in a layer context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_LAYER) {
    fprintf(stderr, "\npopenNodes: Not in layer context!\n");
    return(1);
  }/* end if layer context */

  if ( parse_pipe_args(P_NODES, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, xdim, ydim;

    p->type = P_NODES;
    
    bb = p->info->bb;
    l = p->info->l;
    BPbb_layer_dim(bb, l, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_layer_values(bb, l, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenNodes: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}

int pipe_open_inputs(buffer)
     char *buffer;
{
  UIPIPE *p;

  /* am i in a layer context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_INPUTS) {
    fprintf(stderr, "\npopenInputs: Not in $INPUTS context!\n");
    return(1);
  }/* end if layer context */

  if ( parse_pipe_args(P_INPUTS, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, xdim, ydim;

    p->type = P_INPUTS;
    
    bb = p->info->bb;
    BPbb_input_dim(bb, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_input_values(bb, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenInputs: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}

int pipe_open_biases(buffer)
     char *buffer;
{
  UIPIPE *p;
  
  /* am i in a layer context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_LAYER) {
    fprintf(stderr, "\npopenBiases: Not in layer context!\n");
    return(1);
  }/* end if layer context */
  
  if ( parse_pipe_args(P_BIASES, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, xdim, ydim;

    p->type = P_BIASES;

    bb = p->info->bb;
    l = p->info->l;
    BPbb_layer_dim(bb, l, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_bias_values(bb, l, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenBiases: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}

int pipe_open_ar1(buffer)
     char *buffer;
{
  UIPIPE *p;
  
  /* am i in a layer context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_LAYER) {
    fprintf(stderr, "\npopenAr1: Not in layer context!\n");
    return(1);
  }/* end if layer context */
  
  if ( parse_pipe_args(P_AR1, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, xdim, ydim;

    p->type = P_AR1;

    bb = p->info->bb;
    l = p->info->l;
    BPbb_layer_dim(bb, l, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_ar_values(bb, l, 1, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenAr1: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}


int pipe_open_ar2(buffer)
     char *buffer;
{
  UIPIPE *p;
  
  /* am i in a layer context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_LAYER) {
    fprintf(stderr, "\npopenAr2: Not in layer context!\n");
    return(1);
  }/* end if layer context */
  
  if ( parse_pipe_args(P_AR2, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, xdim, ydim;

    p->type = P_AR2;

    bb = p->info->bb;
    l = p->info->l;
    BPbb_layer_dim(bb, l, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_ar_values(bb, l, 2, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenAr2: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}


int pipe_open_targets(buffer)
     char *buffer;
{
  BB_INFO *info;
  UIPIPE *p;

  info = (BB_INFO *)current_context->data;

  /* am i in a outout layer context? */
  if (info == NULL ||
      info->type != BB_LAYER ||
      info->l !=  BPbb_output_layer_index(info->bb) ) {
    fprintf(stderr, "\npopenTargets: Not in output layer context!\n");
    return(1);
  }/* end if layer context */

  if ( parse_pipe_args(P_TARGETS, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, xdim, ydim;

    p->type = P_TARGETS;
    
    bb = p->info->bb;
    l = p->info->l;
    BPbb_layer_dim(bb, l, &xdim, &ydim);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_target_values(bb, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenTargets: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }

  return(1);
}

int pipe_open_weights(buffer)
     char *buffer;
{
  UIPIPE *p;

  /* am i in a connection context? */
  if (current_context->data == NULL ||
      ((BB_INFO *)(current_context->data))->type != BB_LAYER_INPUT_CONNECTION) {
    fprintf(stderr, "\npopenWeights: Not in layer context!\n");
    return(1);
  }/* end if layer context */

  if ( parse_pipe_args(P_CONNECTION_WEIGHTS, buffer) ) return(1);

  /* get stream */
  p = open_stream();
  if (p == NULL) return(1);
  
  { /* output data */
    int bb, l, c, type, xdim, ydim, xoffset, yoffset;
    float *weights;
    
    p->type = P_CONNECTION_WEIGHTS;

    bb = p->info->bb;
    l = p->info->l;
    c = p->info->c;
    BPbb_connection_dim(bb, l, c, &type, &xdim, &ydim, &xoffset, &yoffset);

    p->data_xdim = xdim;
    p->data_ydim = ydim;

    p->data = (float *)am_alloc_mem(xdim*ydim*sizeof(float));

    BPrw_connection_values(bb, l, c, p->x, p->y, p->data, xdim*ydim, BP_DATA_READ);

    pipe_write_data(p);
  }

  /* wrap up */
  if ( ferror(p->stream) ) {
    fprintf(stderr, "\npopenWeights: error occurred writing to pipe: %s\n", p_command);
    close_stream(p);
  }
  
  return(1);
}

int pipe_close(buffer)
     char *buffer;
{
  UIPIPE *p;
  char input[MAX_INPUT_SIZE];
  
  /* name */
  if ( uiextract_word(input, 2, buffer) == NULL) {
    fprintf(stderr, "\npclose: Unable to read 1st argument (name)!\n");
    return(1);
  }

  p_name = input;

  p = find_stream();
  if (p == NULL) {
    fprintf(stderr, "\nCannot find stream: %s", p_name);
    return(1);
  }

  close_stream(p);

  return(1);
}

void update_pipes(type)
     int type;
{
  UIPIPE *p=pipes;
  
  while(p != NULL) {
    if (p->type == type) {
      switch(type) {
      case P_INPUTS :
	BPrw_input_values(p->info->bb,
			  p->data,
			  p->data_xdim*p->data_ydim,
			  BP_DATA_READ);
	break;
      case P_NODES :
	BPrw_layer_values(p->info->bb,
			  p->info->l,
			  p->data,
			  p->data_xdim*p->data_ydim,
			  BP_DATA_READ);
	break;
      case P_BIASES :
	BPrw_bias_values(p->info->bb,
			 p->info->l,
			 p->data,
			 p->data_xdim*p->data_ydim,
			 BP_DATA_READ);
	break;
      case P_AR1 :
	BPrw_ar_values(p->info->bb,
			 p->info->l,
			 1,
			 p->data,
			 p->data_xdim*p->data_ydim,
			 BP_DATA_READ);
	break;
      case P_AR2 :
	BPrw_ar_values(p->info->bb,
			 p->info->l,
			 2,
			 p->data,
			 p->data_xdim*p->data_ydim,
			 BP_DATA_READ);
	break;
      case P_TARGETS :
	BPrw_target_values(p->info->bb,
			 p->data,
			 p->data_xdim*p->data_ydim,
			 BP_DATA_READ);
	break;
      case P_CONNECTION_WEIGHTS :
	BPrw_connection_values(p->info->bb,
			       p->info->l,
			       p->info->c,
			       p->x, p->y,
			       p->data,
			       p->data_xdim*p->data_ydim,
			       BP_DATA_READ);
	break;
      }/* end switch */
      pipe_write_data(p);
    }/* end if */
    p = p->next;
  }
}
