/**
 ** YACC description of datafile reader
 **/


/*
 * Define parsers stack to be numbers, strings.
 */

%union {
	int i;    /* integer */
	float f;  /* float */
	char *s;  /* string */
}


%{
#include "BpDatafile.h"
#include "globals.h"

extern DATAPTR _file_init(); /* initialize files */
extern void  _readfile();    /* read in a datafile */

#define TRUE 1
#define FALSE 0

/* input stream */
static FILE *df_stream;

/* input buffer */
static unsigned char yytext[256];
static int yyleng;

DATAPTR data = NULL; /* root */

static int error_code = 0;

#define TARGET_LENGTH 50
#define N_PATTERN_NAMES 30


/* globals */
static char *user_label = NULL; /* user supplied label */  
static int genORpatterns = 0;   /* use a generator or patterns */
static int menu_defaults = TRUE;/* use default menu style */
static int target_length = 0;   /* length of user's target vector */
static float target_vector[TARGET_LENGTH];
static int self = FALSE; /* if target is same as data */
static char *target_file = NULL; /* name of target file (type5) */
static int n_pattern_names = 0;   /* number of user's pattern names */
static int switch_cycle = 1; /* when to switch between files (after how many patterns) */
static int scroll_step = 0; /* scroll pattern? if so, how much */
static int xoffset =0,yoffset =0,xlength=0,ylength=0; /* for clipping data */
static float range = -1.0, bias = 0.0, scalar=1.0; /* for rescaling data */
static int normalize = 0; /* for normalizing data */
static int submean = 0; /* for normalizing data */
static float threshold; /* for preprocessing data */
static int clear_delays = FALSE; /* TRUE if clear delay buffers before using */
static USER_NAME pattern_names[N_PATTERN_NAMES];
static int random_switch = FALSE;
static int swapbytes= FALSE; /* in case the data is from a machine w/opposite byte order */
static int math=DF_MATH_NONE;       /* for doing math on data points */

/* used for error messages */
int lineno = 1; 
static char *datafile_name;




/*****************************************************/
/*****************************************************/
/*****************************************************/
/*******************    lex    ***********************/



char *newstr (name, length)
  char *name;
  int length;
{
 char *temp;

 temp = am_alloc_mem(length + 1);
 *(temp + length) = 0;
 bcopy(name, temp, length+1);
 return(temp);
}

static void read_token()
{
  unsigned int c;


  yyleng=0;
  /* skip spaces, tabs , quotes, and newlines  */
  while((c=getc(df_stream)) != ' '
	&& !(iscntrl(c))
	&& c != '"'
	&& c != '{'
	&& c != '}'
	&& c != '('
	&& c != ')'
	&& c != '#'
	&& c != ','
	&& c != EOF) {
    yytext[yyleng++] = c;
  }
  ungetc((unsigned char)c,df_stream);
  yytext[yyleng] = '\0';
}

static void read_quoted_token()
{
  unsigned int c;


  yyleng=0;
  /* end at quote */
  while((c=getc(df_stream)) != '"' && c != EOF) {
    yytext[yyleng++] = c;
  }
  yytext[yyleng] = '\0';
}

static int input_equals(str)
     register char *str;
{
  register int counter;
  register unsigned char *tptr = yytext;

  counter = yyleng;
  if (strlen(str) > counter) counter = strlen(str);
  while(counter--) if (*str++ != *tptr++) return FALSE;;
  return TRUE;

}

#define CONTEXT_STACK_DEPTH 16
#define NO_CONTEXT 0
#define READFILE_CONTEXT 1
#define USEFILE_CONTEXT 2
#define KEYWORD_CONTEXT 3
#define MATH_CONTEXT 4
static int context_stack[CONTEXT_STACK_DEPTH], *cstackptr = context_stack;

#define switch_lex_context(ctx) *cstackptr = ctx;

static void push_lex_context(new_context)
     int new_context;
{
  if (cstackptr == context_stack + CONTEXT_STACK_DEPTH)
    yyerror("\nLex context stack full! (push_lex_context)\n");
  cstackptr++; *cstackptr = new_context;
}

static void pop_lex_context()
{
  if (cstackptr == context_stack)
    yyerror("\nLex context stack would be empty! (pop_lex_context)\n");
  /* pop */
  cstackptr--;
}

static int this_context(context)
     int context;
{
  return( *cstackptr == context );
}




/*****************************************************/
/*****************************************************/
/*****************************************************/





yywrap(){return(1);}

/*****************************************************/
/* warning                                           */
/*        Print warning message.                     */
/*****************************************************/
static void warning(s, t)
     char *s, *t;
{
  fprintf(stderr, "\n%s: %s", datafile_name, s);
  if (t != (char *)NULL) fprintf(stderr, "  %s", t);
}

/*****************************************************/
/* yyerror                                           */
/*        Called by yacc for syntax errors.          */
/*****************************************************/
yyerror (s)
     char *s;
{ extern unsigned char yytext[];
  extern int yyleng;
  char where[50];

  *where = '\0'; /* empty */
  if (yytext != (unsigned char *)NULL)
    {  char *token = am_alloc_mem(yyleng + 1);

       bcopy(yytext, token, yyleng);
       token[yyleng] = '\0'; /* end of string */
       sprintf(where, "(near \"%s\" somewhere around line %d)\n", yytext, lineno);
       am_free_mem(token);
    }/* end if */

  warning(s, where);
  error_code = 1;
  
}/* end yyerror */

/*****************************************************/
/* update_target_vector: Add numbers to target vector. */
/*****************************************************/
static void update_target_vector(x)
     float x;
{
  if (target_length == TARGET_LENGTH) yyerror("\nTarget vector too long!");
  target_vector[target_length++] = x;
}/* end update_target_vector */

/*****************************************************/
/* update_name_vector: Add names to pattern vector. */
/*****************************************************/
static void update_name_vector(name,x)
     char *name;
     int x;
{
  if (n_pattern_names == N_PATTERN_NAMES) yyerror("\nToo many pattern names!");
  pattern_names[n_pattern_names].index = x;
  pattern_names[n_pattern_names++].name = name;
}/* end update_target_vector */


/*****************************************************/
/* create_datafile: create datafile structure.
/*****************************************************/
static create_datafile(filename, type)
     char *filename;
     int type;
{
  DATAFILEPTR new_datafile;

  /* make new structure */
  new_datafile = (DATAFILEPTR)am_alloc_mem(sizeof(DATAFILE));
  new_datafile->filename = filename;

  new_datafile->type = type;

  new_datafile->usefile = FALSE;

  new_datafile->next = (DATAFILEPTR)NULL;

  new_datafile->genORpatterns = genORpatterns;
  genORpatterns = 0; /* reset */

  
  if (scroll_step) { /* scrolling data? */
    if (data->inputs_ydim == 1) /* 1D */
      new_datafile->scroll_step = scroll_step;
    else /* 2D */
      new_datafile->scroll_step = scroll_step * data->inputs_xdim;
    new_datafile->scroll_index = new_datafile->scroll_step; /* where to begin */
    new_datafile->genORpatterns = 0; /* scroll overrrides */
    menu_defaults = FALSE; /* scroll overrides */
    scroll_step = 0; /* reset */
    /* make a buffer for the scrolling data (if necessary) */
    if (data->input_buffer == (float *)NULL) {
      data->input_buffer = (float *)am_alloc_mem(data->inputs_size * sizeof(float)); 
    }/* end if */
    bzero((char *)data->input_buffer, data->inputs_size * sizeof(float)); 
  } else {
    new_datafile->size = data->inputs_size;
    new_datafile->xdim = data->inputs_xdim;
    new_datafile->ydim = data->inputs_ydim;
    new_datafile->scroll_step = 0;
    new_datafile->scroll_index = 0;
  }/* end if else scroll_step */

  new_datafile->switch_cycle = switch_cycle;
  switch_cycle = 1; /* reset */

  new_datafile->xoffset = xoffset;
  xoffset = 0; /* reset */
  new_datafile->yoffset = yoffset;
  yoffset = 0; /* reset */
  new_datafile->xlength = xlength;
  xlength = 0; /* reset */
  new_datafile->ylength = ylength;
  ylength = 0; /* reset */

  new_datafile->range = range;
  range = -1.0; /* reset */
  new_datafile->bias = bias;
  bias = 0.0; /* reset */
  new_datafile->scalar = scalar;
  scalar = 1.0; /* reset */
  new_datafile->normalize = normalize;
  normalize = 0; /* reset */
  new_datafile->threshold = threshold;
  threshold = AM_HUGE_VAL; /* reset */
  new_datafile->submean = submean;
  submean = 0; /* reset */
  new_datafile->math= math;
  math = 0; /* reset */
  new_datafile->swapbytes= swapbytes;
  swapbytes = 0; /* reset */

  new_datafile->clear_delays = clear_delays;
  clear_delays = FALSE; /* reset */

  new_datafile->menu_defaults = menu_defaults;
  menu_defaults = TRUE; /* reset */
  new_datafile->current_pattern = 0;

  new_datafile->user_label = user_label;
  user_label = (char *)NULL; /* reset */
  
  if (target_length) {
    new_datafile->user_target = (float *)am_alloc_mem(target_length * sizeof(float));
    /* copy */
    bcopy((char *)target_vector,
	  (char *)new_datafile->user_target,
	  (target_length * sizeof(float)));
    /* reset */
    target_length = 0;
  } else 
    new_datafile->user_target = (float *)NULL;


  new_datafile->self = self; self = FALSE;   /* use inputs as targets? */

  /* set the target file (type5) */
  new_datafile->target_file = target_file; target_file = NULL;

  if (n_pattern_names) {
    new_datafile->n_user_names = n_pattern_names;
    new_datafile->user_names = (USER_NAME_PTR)am_alloc_mem(n_pattern_names * sizeof(USER_NAME));
    /* copy */
    bcopy((char *)pattern_names,
	  (char *)new_datafile->user_names,
	  (n_pattern_names * sizeof(USER_NAME)));
    /* reset */
    n_pattern_names = 0;
  } else  {
    new_datafile->n_user_names = 0;
    new_datafile->user_names = (USER_NAME_PTR)NULL;
  }/* end else if */

  new_datafile->inputs = (float **)NULL;
  new_datafile->targets = (float **)NULL;
  new_datafile->input_labels = (char **)NULL;
  new_datafile->target_labels = (char **)NULL;

  /* push into list */
  data->n_datafiles += 1;
  if (data->datafiles == (DATAFILEPTR)NULL) {
    data->datafiles = new_datafile;    
    new_datafile->next = (DATAFILEPTR)NULL;
  } else {
    DATAFILEPTR temp;
    temp = data->datafiles;
    data->datafiles = new_datafile;
    new_datafile->next = temp;
  }/* end else */

}/* end create_datafile */

/*****************************************************/
/* find_datafile: Search and return named file     */
/*****************************************************/
DATAFILEPTR find_datafile(name)
     char *name;
{
  DATAFILEPTR ptr;

  ptr = data->datafiles;
  while(ptr != (DATAFILEPTR)NULL) {
    if (! ptr->usefile && ! strcmp(ptr->filename, name))
      return ptr;
    else
      ptr = ptr->next;
  }

  yyerror("Undefined file name for UseFile!");

}/* end find_datafile */

/*****************************************************/
/* use_datafile:   Just referenced a file            */
/*****************************************************/
static void use_datafile(name)
     char *name;
{
  DATAFILEPTR referenced_datafile, new_datafile;

  /* find name in list */
  referenced_datafile = find_datafile(name);

  /* make new structure */
  new_datafile = (DATAFILEPTR)am_alloc_mem(sizeof(DATAFILE));

  new_datafile->filename = name;

  /* mark so it won't be read */
  new_datafile->usefile = TRUE;

  /* push into list */
  data->n_datafiles += 1;
  {
    DATAFILEPTR temp;
    temp = data->datafiles;
    data->datafiles = new_datafile;
    new_datafile->next = temp;
  }/* end block */

}/* end use_datafile */

/*****************************************************/
/* rescale_data                                      */
/*****************************************************/
static void rescale_data(data, range, scalar, bias, n)
     register float *data;
     float range, scalar, bias;
     register int n;
{
  /* range */
  if (range > 0.0) {
    double max_val = -AM_HUGE_VAL, min_val = AM_HUGE_VAL;
    register int s = n;
    register float *ptr=data;
    
    /* max, min */
    while(s--) {
      if (*ptr > max_val) max_val = *ptr;
      if (*ptr < min_val) min_val = *ptr;
      ptr++;
    }
    
    max_val -= min_val;
    
    if (max_val != 0.0) {
      s = n;
      ptr = data;
      bias += range * -0.5;
      while(s--) *ptr++ = (range * ((*ptr - min_val)/max_val));
    } else  bzero(data, n * sizeof(float));
    
  } 

  /* scale */
  if (scalar != 1.0) {
    register int s=n;
    register float *ptr=data;

    while(s--) *ptr++ *= scalar;
  }


  /* bias */
  if (bias != 0.0) {
    register int s=n;
    register float *ptr=data;
    
      while(s--) *ptr++ += bias;
  }


}/* end rescale_data */

/*****************************************************/
/* swapbytes_data                                    */
/*****************************************************/
static void swapbytes_data(vector,n)
     register float *vector;
     register int n;
{
  
  /* for each value */
  while(n--) {
    union {
	float f;
	char c[sizeof(float)];
    } newval,oldval;
    int sz = sizeof(float);

    /* swap */
    oldval.f = *vector;
    while(sz--) newval.c[sz] = oldval.c[ sizeof(float) - 1 - sz ];
    *vector++ = newval.f;

  }/* end while */

}/* end swapbytes_data */


/*****************************************************/
/* math_data                                         */
/*****************************************************/
static void math_data(function,vector,n)
     int function;
     register float *vector;
     register int n;
{
  switch (function) {

  case DF_MATH_FABS :
    while (n--) {
      float x;
      
      x = *vector;
      *vector++ = AM_FABS(x);
    }
    break;

  case DF_MATH_SQUARE :
    while (n--) {
      float x;
      
      x = *vector;
      *vector++ = x *x;
    }
    break;

  case DF_MATH_LOG10 :
    while (n--) {
      float x;
      
      x = *vector;
      *vector++ = AM_LOG10(x);
    }
    break;

  case DF_MATH_PSEUDO_LOG10 :
    while (n--) {
      float x;
      
      x = *vector + 1.0;
      *vector++ = AM_LOG10(x);
    }
    break;

  }/* end switch */
  
}/* end math_data */

/*****************************************************/
/* submean_data                                      */
/*****************************************************/
static void submean_data(vector,n)
     register float *vector;
     register int n;
{
  register int counter=n;
  float mean=0.0, *ptr=vector;
  
  /* calc mean */
  while(counter--) mean += *ptr++;
  mean /= n;

  /* submean */
  counter = n;
  while(counter--)  *vector++ -= mean;

}/* end submean_data */

/*****************************************************/
/* threshold_data                                     */
/*****************************************************/
static void threshold_data(t,vector,n)
     float t;
     register float *vector;
     register int n;
{

  while(n--) {
    if (*vector < t) *vector = 0.0;
    vector++;
  }

}/* end threshold_data */

/*****************************************************/
/* normalize_data                                    */
/*****************************************************/
static void normalize_data(vector,n)
     register float *vector;
     register int n;
{
  register int counter=n;
  float power2=0.0, *ptr=vector;
  

  /* submean */
  submean_data(vector,n);
  
  /* calc power */
  counter = n;
  ptr = vector;
  while(counter--) {
    power2 += *ptr * *ptr;
    ptr++;
  }
  
  /* normalize power2 */
  if (power2 == 0.0) return;
  power2 = 1.0 / sqrt((double)power2);
  while(n--) *vector++ *= power2;
}/* end normalize_data */

/*****************************************************/
/* next_datafile: Goto next datafile. */
/*****************************************************/
static void next_datafile()
{
    data->current_datafile = data->current_datafile->next;
}/* end next_datafile */

/*****************************************************/
/* random_datafile: random next datafile. */
/*****************************************************/
static void random_datafile()
{
    data->current_datafile = *(data->datafile_table +
			       ( (int) (AM_RANDOM() * data->n_datafiles) ));
}/* end next_datafile */


%}

/*
 * Enumerate all tokens from yylex.
 */

/* This is used to reset the line counter */
%token NEWLINE

/* The following are the keyword tokens */
%token <f> NUMBER                 /* real */
%token <s> NAME                   /* string */

%token LCP                        /* left curly paren */
%token RCP                        /* right curly paren */
%token RP                         /* right paren */
%token LP                         /* left paren */

%token READFILE                   /* read file */
%token USEFILE                    /* reference a file */
%token RANDOM_SWITCH              /* randomly switch files */

%token ASCII                      /* file type */
%token MATLAB                     /* file type */
%token TYPE1                      /* file type */
%token TYPE2                      /* file type */
%token TYPE3                      /* file type */
%token TYPE4                      /* file type */
%token TYPE5                      /* file type */

%token LABEL                      /* key word */
%token GENERATOR                  /* key word */
%token PATTERNS                   /* key word */
%token TARGET                     /* key word */
%token SELF                       /* argument of TARGET */
%token PATTERN_NAME               /* key word */
%token SWITCH                     /* key word */
%token SCROLL                     /* key word */
%token XOFF                       /* key word */
%token XLENGTH                    /* key word */
%token YOFF                       /* key word */
%token YLENGTH                    /* key word */

%token RANGE                      /* key word */
%token NORMALIZE                  /* key word */
%token THRESHOLD                  /* key word */
%token SUBMEAN                    /* key word */
%token SCALE                      /* key word */
%token BIAS                       /* key word */
%token CLEAR_DELAYS               /* key word */
%token SWAPBYTES                  /* key word */
%token MATH                       /* key word */
%token <i> MATH_FUNCTION          /* key word set */

/*
 * Sub-expression (non-terminal) tokens.
 */

%type <i> type
%type <f> real
%type <i> integer
%type <i> optional_integer
%%


/*****************************************************/
/* Root production
/*****************************************************/
datafile: datafile_exp
         | datafile datafile_exp
         ;

/* these are the currently supported datafile expressions */
datafile_exp: files    /* new files */
              | newlines   /* cpp statements */
              ;

/* read a list of commands */
files: READFILE LP NAME type keys RP { create_datafile($3, $4); }
           | USEFILE LP NAME RP { use_datafile($3); }
           | RANDOM_SWITCH { random_switch = TRUE; }
           ;


/* what type? */
type : ASCII { $$ = ASCII_FILE; }
       | MATLAB { $$ = MATLAB_FILE; }
       | TYPE1 { $$ = TYPE1_FILE; }
       | TYPE2 { $$ = TYPE2_FILE; }
       | TYPE3 { $$ = TYPE3_FILE; }
       | TYPE4 { $$ = TYPE4_FILE; }
       | TYPE5 { $$ = TYPE5_FILE; }
       ;

/* list of key words */
keys : key
       | keys key

/* key words */
key: LP LABEL NAME RP { user_label = $3; }
      | LP GENERATOR RP { genORpatterns = 0; menu_defaults = FALSE; /* use generator */ }
      | LP PATTERNS RP { genORpatterns = 1; menu_defaults = FALSE; /* use patterns */ }
      | LP TARGET vector RP
      | LP TARGET SELF RP { self = TRUE ; }
      | LP TARGET NAME RP { target_file = $3; }
      | LP PATTERN_NAME integer NAME RP { update_name_vector($4,$3); }
      | LP SWITCH optional_integer RP { switch_cycle = $3; }
      | LP SCROLL optional_integer RP {
	if ($3 < 0) yyerror("You cannot have a negative scroll step!");scroll_step = $3; }
      | LP RANGE real RP { if ($3 <= 0.0) yyerror("Range must be postive!"); range = $3; }
      | LP XOFF integer RP { if ($3 < 0) yyerror("xoffset: must be postive!"); xoffset = $3; }
      | LP YOFF integer RP { if ($3 < 0) yyerror("yoffset: must be postive!"); yoffset = $3; }
      | LP XLENGTH integer RP { if ($3 < 0) yyerror("xlength: must be postive!"); xlength = $3; }
      | LP YLENGTH integer RP { if ($3 < 0) yyerror("ylength: must be postive!"); ylength = $3; }
      | LP BIAS real RP { bias = $3; }
      | LP SCALE real RP { scalar = $3; }
      | LP NORMALIZE RP { normalize = 1; }
      | LP THRESHOLD real RP { threshold = $3; }
      | LP SWAPBYTES RP { swapbytes = 1; }
      | LP MATH MATH_FUNCTION RP { math = $3; }
      | LP SUBMEAN RP { submean = 1; }
      | LP CLEAR_DELAYS RP {clear_delays = TRUE; }
      |
      ;

/* list of numbers */
 vector: real { update_target_vector($1); }
         | vector real { update_target_vector($2); }
         ;

/*****************************************************/
/* Update the line counter and current file          */
/* (used to parse cpp output)                        */
/*****************************************************/
newlines: newline
          | newlines newline
          ;

newline: NEWLINE integer NAME { lineno = $2;
 				datafile_name = $3;
			      } 
         | NEWLINE integer  NAME integer { 
	                                   lineno = $2;
 				           datafile_name = $3;
			                 }
         | NEWLINE NAME integer NAME { lineno = $3;
                                       datafile_name = $4;
                                     }
         | NEWLINE NAME NAME {}

         | NEWLINE integer { lineno = $2; }

         ;

/* numbers */
real: NUMBER { $$ = $1; }
     ;

optional_integer: integer { $$ = $1; }
                |  { $$ = 1; /* default */}
                ;

integer: NUMBER {$$ = (int)$1; }
         ; 
%%
  

yylex()
{
  unsigned int c;

  more :


  /* skip spaces and tabs */
  while((c=getc(df_stream)) == ' ' || c == '\t') ;

  if (c == '\n') {  ++lineno; goto more ; }

  if (iscntrl(c)) goto more ;
  
  if (c == EOF) return 0;

  if (c == '#') return(NEWLINE);

  if (c == '"') { /* quoted string */
    read_quoted_token();
    yylval.s = newstr(yytext, yyleng);
    return(NAME);
  }

  if (c == '(') {
    if (this_context(KEYWORD_CONTEXT))
      push_lex_context(KEYWORD_CONTEXT);
    return(LP);
  }/* end if */

  if (c == ')') {
    pop_lex_context();
    return(RP);
  }/* end if */


  /* ok, done with the single character stuff */
  ungetc((unsigned char)c, df_stream);
  read_token();

  if (isdigit((unsigned int)yytext[0]) ||
      (yyleng > 1 &&
       (yytext[0] == '-' || yytext[0] == '+' || yytext[0] == '.') &&
       isdigit((unsigned int)yytext[1])) ||
      (yyleng > 2 &&
       (yytext[0] == '-' || yytext[0] == '+') &&
       yytext[1] == '.' &&
       isdigit((unsigned int)yytext[2]))) { /* number */
    sscanf((char *)yytext, "%f", &(yylval.f)); 
    return(NUMBER);
  }/* end if number */ 

  if (this_context(NO_CONTEXT)) { /* global context */
    if (input_equals("RandomSwitch")) return(RANDOM_SWITCH);
  }

  if (this_context(NO_CONTEXT)) { /* global context */
    if (input_equals("UseFile")) {
      push_lex_context(USEFILE_CONTEXT);
      return(USEFILE);
    }
  }

  if (this_context(NO_CONTEXT)) { /* global context */
    if (input_equals("ReadFile")) {
      push_lex_context(READFILE_CONTEXT);
      return(READFILE);
    }
  }

  if (this_context(READFILE_CONTEXT)) {

    /* types */
    
    if (input_equals("Ascii")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(ASCII);
    }
    
    if (input_equals("Matlab")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(MATLAB);
    }
    
    if (input_equals("Type1")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(TYPE1);
    }

    if (input_equals("Type2")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(TYPE2);
    }

    if (input_equals("Type3")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(TYPE3);
    }

    if (input_equals("Type4")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(TYPE4);
    }

    if (input_equals("Type5")) {
      switch_lex_context(KEYWORD_CONTEXT);
      return(TYPE5);
    }
    
  }

  if (this_context(KEYWORD_CONTEXT)) {

    if (input_equals("Math->")) {
      push_lex_context(MATH_CONTEXT);
      return(MATH);
    }

    if (input_equals("ClearDelays->")) return(CLEAR_DELAYS);

    if (input_equals("SubMean->")) return(SUBMEAN);

    if (input_equals("Normalize->")) return(NORMALIZE);

    if (input_equals("Threshold->")) return(THRESHOLD);

    if (input_equals("Scale->")) return(SCALE);

    if (input_equals("Bias->")) return(BIAS);

    if (input_equals("Range->")) return(RANGE);

    if (input_equals("Yoffset->")) return(YOFF);

    if (input_equals("Xoffset->")) return(XOFF);

    if (input_equals("YLength->")) return(YLENGTH);

    if (input_equals("XLength->")) return(XLENGTH);

    if (input_equals("Scroll->")) return(SCROLL);

    if (input_equals("SwapBytes->")) return(SWAPBYTES);

    if (input_equals("Switch->")) return(SWITCH);

    if (input_equals("PatternName->")) return(PATTERN_NAME);

    if (input_equals("Target->")) return(TARGET);

    if (input_equals("Patterns->")) return(PATTERNS);

    if (input_equals("Generator->")) return(GENERATOR);

    if (input_equals("Label->")) return(LABEL);

    if (input_equals("All")) {yylval.f = -1.0; return(NUMBER);}

    if (input_equals("Self")) { return(SELF);}

  }

  if (this_context(MATH_CONTEXT)) {

    if (input_equals("abs[x]"))  yylval.i = DF_MATH_FABS;
    
    else if (input_equals("square[x]"))  yylval.i = DF_MATH_SQUARE;

    else if (input_equals("log10[x]"))  yylval.i = DF_MATH_LOG10;

    else if (input_equals("log10[x+1]"))  yylval.i = DF_MATH_PSEUDO_LOG10;

    else fprintf(stderr, "\nMath->: Unknown function!\n");

    pop_lex_context();

    return(MATH_FUNCTION);
  }


  /* names */
  {
    yylval.s = newstr(yytext, yyleng);
    return(NAME);
  }  

}/* end yylex */


/*****************************************************/
/* loaddata:       file = datafile specification from
                   which datafiles are read.
		   query_function = function
		   to interogate the network.
		   noise_type = type of noise to
		   corrupt input (if any)
		   mean = mean of noise distribution.
		   variance = variance of noise distribution.
 */
/*****************************************************/
int loaddata(file, query_function, noise_type, mean, variance)
     char *file;
     LB_PTR (*query_function)();
     int noise_type;
     float mean, variance;
{
  char *cpp_file;
  extern DATAPTR data;
  
  /******************************************/
  /* some basic init */
  threshold = AM_HUGE_VAL;
  datafile_name = file;
  
  /******************************************/
  /* run cpp on datafile */
  cpp_file = am_cpp(file,0);

  /******************************************/
  /* open  input from cpp */
  df_stream = am_fopen(cpp_file, "r");
  /* return on error */
  if (df_stream == NULL)  {
    fprintf(stderr, "\nUnable to open %s", cpp_file);
    am_perror("Reading .df file");
    return(1);
  }

  /******************************************/
  
  /******************************************/
  /* make the root */
  data = _file_init(query_function, noise_type, mean, variance);

  /******************************************/

  /******************************************/
  /* parse storing info in global variables */
  yyparse();
  if (error_code) return(error_code);
  /******************************************/

  /******************************************/
  /* close the stream */
  am_fclose(df_stream);
  /******************************************/

  /******************************************/
  /* read in the data                       */
  /******************************************/
  {
    DATAFILEPTR files = data->datafiles;

    if (files == (DATAFILEPTR)NULL) {
      fprintf(stderr, "\nNo data files!\n");
      return(1);
    }
    
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      if (!files->usefile) _readfile(files, data);
      files = files->next;
    }/* end while */

    
    /* set the first one */
    data->current_datafile = data->datafiles;

  }/* end block */
  
  /******************************************/
  /* Reference files w/UseFile              */
  /******************************************/
  {
    DATAFILEPTR files = data->datafiles;

    /* loop through each structure...reference (from UseFile) */
    while(files != (DATAFILEPTR)NULL) {

      if (files->usefile) { /* just a reference, don't read */
	DATAFILEPTR tmp_next = files->next; /* hold on */

	/* copy contents */
	*files = *(find_datafile(files->filename));

	/* fix list */
	files->next = tmp_next;

      }/* end if */

      files = files->next;
    }/* end while */

  }/* end block */

  /******************************************/  
  /* swapbytes                              */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      
      if (files->swapbytes && !files->usefile) {
	int n = files->npatterns;
	int size = files->size; /* size of a data pattern */

	while (n--) {
	  swapbytes_data(*(files->inputs + n), size);
	}/* end swap inputs */

	/* swap targets */
	switch (files->type) {
	case TYPE2_FILE :
	  /* size of a target pattern */
	  size = data->targets_xdim * data->targets_ydim;
	  swapbytes_data(*(files->targets), size);
	  break;
	  
	case TYPE3_FILE :
	case TYPE4_FILE : 
	case TYPE5_FILE : 
	  n = files->npatterns;
	  /* size of a target pattern */
	  size = data->targets_xdim * data->targets_ydim;
	  
	  while (n--) {
	    swapbytes_data(*(files->targets + n), size);
	  }/* end swap inputs */
	  break;
	  
	}/* end switch */

      }/* end if swapbytes */
      
      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/  
  
  /******************************************/  
  /* math                                   */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      
      if (files->math != DF_MATH_NONE && !files->usefile) {
	int n = files->npatterns;
	int size = files->size; /* size of a data pattern */
	
	while (n--) math_data(files->math, *(files->inputs + n), size);
      }/* end if submean */
      
      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/ 

  /******************************************/  
  /* threshold                              */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      
      if (files->threshold != AM_HUGE_VAL && !files->usefile) {
	int n = files->npatterns;
	int size = files->size; /* size of a data pattern */
	
	while (n--) threshold_data(files->threshold, *(files->inputs + n), size);
      }/* end if submean */
      
      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/  
  
  /******************************************/  
  /* submean                                */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      
      if (files->submean && !files->usefile) {
	int n = files->npatterns;
	int size = files->size; /* size of a data pattern */
	
	while (n--) submean_data(*(files->inputs + n), size);
      }/* end if submean */
      
      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/  
  
  
  /******************************************/  
  /* normalize                              */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      
      if (files->normalize && !files->usefile) {
	int n = files->npatterns;
	int size = files->size; /* size of a data pattern */
	
	while (n--) normalize_data(*(files->inputs + n), size);
      }/* end if normalize */
      
      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/  
  
  
  /******************************************/  
  /* rescale                                */
  /******************************************/  
  {
    DATAFILEPTR files = data->datafiles;
    /* loop through each structure...read */
    while(files != (DATAFILEPTR)NULL) {
      int n = files->npatterns;
      int size = files->size; /* size of a data pattern */
      float range = files->range;
      float scalar = files->scalar;
      float bias = files->bias;
      
      
      if (!files->usefile) {
	if (range > 0.0 || bias != 0.0 || scalar != 1.0) /* leave alone? */
	  while (n--)
	    rescale_data(*(files->inputs + n),
			 range, scalar, bias, size);
      }/* end if */

      files = files->next;
    }/* end while */
  }/* end block */
  /******************************************/  

  /******************************************/
  /* setup for the next_file function       */
  /******************************************/
  {
    /* set the next file function */
    if (random_switch) { /* random */
      DATAFILEPTR files = data->datafiles, *file_ptr;
      
      data->datafile_table = (DATAFILEPTR *)am_alloc_mem(data->n_datafiles * sizeof(DATAFILEPTR));
      file_ptr = data->datafile_table;
      
      /* loop through each structure...set into table */
      while(files != (DATAFILEPTR)NULL) {
	*file_ptr++ = files;
	files = files->next;
      }/* end while */
      
      data->next_datafile = random_datafile;
      
    } else { /* cyclic */
      
      /* set function */
      data->next_datafile = next_datafile;
      
    }/* end else */
    
  }/* end block */



  /******************************************/
  /* make circular list                     */
  /******************************************/
  {
    DATAFILEPTR files = data->datafiles, last_file;

    /* loop through each structure...find the end */
    while(files != (DATAFILEPTR)NULL) {
      last_file = files;
      files = files->next;
    }/* end while */
    
    /* make circular list */
    if (data->datafiles != (DATAFILEPTR)NULL) 
      last_file->next = data->datafiles;
    
  }/* end block */
  
  /******************************************/
  /* remove cpp file */
  am_remove_cpp_file(cpp_file);
  /******************************************/

  return(error_code);
}



