/* 

  ****************   NO WARRANTY  *****************

Since the Aspirin/MIGRAINES system is licensed free of charge,
the MITRE Corporation provides absolutley no warranty. Should
the Aspirin/MIGRAINES system prove defective, you must assume
the cost of all necessary servicing, repair or correction.
In no way will the MITRE Corporation be liable to you for
damages, including any lost profits, lost monies, or other
special, incidental or consequential damages arising out of
the use or inability to use the Aspirin/MIGRAINES system.

  *****************   COPYRIGHT  *******************

This software is the copyright of The MITRE Corporation. 
It may be freely used and modified for research and development
purposes. We require a brief acknowledgement in any research
paper or other publication where this software has made a significant
contribution. If you wish to use it for commercial gain you must contact 
The MITRE Corporation for conditions of use. The MITRE Corporation 
provides absolutely NO WARRANTY for this software.

   January, 1992 
   Russell Leighton
   The MITRE Corporation
   7525 Colshire Dr.
   McLean, Va. 22102-3481

*/
#include "aspirin_bp.h"


/* This file contains the support and numerical functions for
   the backprop (BP) simulations.
   
   By using conditional compilation 
   this file can be customized to run very efficiently on
   computers with vector processors by replacing key routines
   (e.g. BPvdot) with library calls to the corresponding vector
   routines.
*/

#ifdef MEIKO860
#  include "fortran_noops.h"
#endif

/* BLAS vector libraries */
#ifdef BLAS 
# define VANILLA 0
# undef BLAS 
# define BLAS 1
# ifdef ANSI_COMPILER
  extern float SDOT(int *n, float *x, int *incx, float *y, int *incy);
  extern float SAXPY(int *n, float *s, float *x, int *incx, float *y, int *incy);
  extern float SCOPY(int *n, float *x, int *incx, float *y, int *incy);
  extern float SGEMV(char *trans, int *m, int *n, float *alpha, float *matrix, int *lda, float *x, int *incx, float *beta, float *y, int *incy);
  extern float SSCAL(int *n, float *s, float *y, int *incy);

# else
  extern float SDOT();
  extern float SAXPY();
  extern float SCOPY();
  extern float SGEMV();
  extern float SSCAL();
# endif

#endif 

/* Sky SKYvec vector libraries */
#ifdef SKYVEC
# define VANILLA 0
# undef SKYVEC
# define SKYVEC 1
#endif 

/* Mecury SAL vector libraries */
#ifdef SAL
# define VANILLA 0
# undef SAL
# define SAL 1
#endif 

/* just use C code (no vector libraries) */
#ifndef VANILLA
# define VANILLA 1
#endif

/* clean */
#ifndef SAL
# define SAL 0
#endif

/* clean */
#ifndef SKYVEC
# define SKYVEC 0
#endif

/* clean */
#ifndef BLAS
# define BLAS 0
#endif 

/*** SUPPORT FUNCTIONS ***/

void BPfrandom_init(seed)
     long seed;
{
  AM_SEED_RANDOM(seed); /* init random number generator */
}

float BPfrandom(val)
     float val;
{
  val *= AM_RANDOM();
  return(val);
}/* end BPfrandom */

void BPread_string(fd, string)
     int fd;
     char *string;
{
  extern int read();
  do {
    read(fd, string, 1);
  }while (*string++ != '\0');
}/* end BPread_string */

int BPread_thresholds(fd, name, key, size, target_size, data,flag)
     char *name, *key;
     int fd, size, *flag;
     float *data;
{
  extern char *error_string;
  
  if(strcmp(name,key) == 0) {
    if(size != target_size) {
      sprintf(error_string, "\nBad layer size for %s\n", key);
      return(DFERROR);
    }/* end if */
    read(fd, data, size * sizeof(float));
    *flag = 0; /* set flag to 0 means found the thresholds! */
  }/* end if */
  return(0);
}/* end BPread_thresholds */


int BPread_reflection_weights(fd, name, key, size, target_size, data,flag)
     char *name, *key;
     int fd, size, *flag;
     float *data;
{
  extern char *error_string;
  
  if(strcmp(name,key) == 0) {
    if(size != target_size) {
      sprintf(error_string, "\nBad layer size for %s\n", key);
      return(DFERROR);
    }/* end if */
    read(fd, data, size * sizeof(float));
    *flag = 0; /* set flag to 0 means found the thresholds! */
  }/* end if */
  return(0);
}/* end BPread_reflection_weights */

int BPread_weights(fd, name1, key1, name2, key2, size, target_size, data,flag)
     char *name1, *name2, *key1, *key2;
     int fd, size, *flag;
     float *data;
{
  extern char *error_string;
  
  if(strcmp(name1,key1) == 0 && strcmp(name2,key2) == 0) {
    if(size != target_size) {
      sprintf(error_string, "\nBad layer size for %s\n", key1);
      return(DFERROR);
    }/* end if */
    read(fd, data, size * sizeof(float));
    *flag = 0; /* set flag to 0 means found the thresholds! */
  }/* end if */
  return(0);
}/* end BPread_weights */





/************************ vector stuff ****************************/

/* BPvsubsmulsum: Vector sub scalar mul sum */
void BPvsubsmulsum(result, new, old, scalar, n)
     float *result,*new,*old;
     float scalar;
     int n;
{
  
  /* the basic code Fortan Style and C Style */
# if FORLOOPS
  int i;
  
  for(i=0;i<n;i++) {
    result[i] += (new[i] - old[i]) * scalar;
  }/* for i to n */
# else
  while(n--) {
    *result++ += (*new++ - *old++) * scalar;
  }/* while n */
# endif
  
}/* end BPvsubsmulsum */



/* BPvmul: vector multiply  */
void BPvmul(to, v1, v2, n)
      float *to, *v1, *v2;
      int n;
{

#if SAL
  extern vmul();

  vmul(v1,1,v2,1,to,1,n);
#endif

#if SKYVEC
  skyvec_ = n;
  v$_rvvt0(v1,v2,to);
#endif

#if VANILLA || BLAS
/* the basic code Fortan Style and C Style */
# if FORLOOPS
   int i;
  
  for(i=0;i<n;i++) to[i] = v1[i] * v2[i];
# else
  while (n--) *to++ = *v1++ * *v2++;
# endif
#endif
}/* end BPvmul */

/* BPvmul_sum: vector multiply sum */
void BPvmul_sum(to, v1, v2, n)
      float *to, *v1, *v2;
      int n;
{
#if SAL
  extern void vma();
  vma(v1,1,v2,1,to,1,to,1,n);
#endif

#if VANILLA || BLAS || SKYVEC
/* the basic code Fortan Style and C Style */
# if FORLOOPS
   int i;
  
  for(i=0;i<n;i++) to[i] += v1[i] * v2[i];
# else
  while (n--) *to++ += *v1++ * *v2++;
# endif
#endif
}/* end BPvmul_sum */


/* BPvsmul: vector scalar multiply */
void BPvsmul(to, v, s, n)
      float *to, *v, s;
      int n;
{
#if SAL
  extern void vsmul();

  vsmul(v,1,&s,to,1,n);
#endif 

#if SKYVEC
  skyvec_ = n;
  v$_rsvt0(s, v, to);
#endif 

#if VANILLA || BLAS
/* the basic code Fortan Style and C Style */
# if FORLOOPS
   int i;
  
  for(i=0;i<n;i++) to[i] = s * v[i];
# else
  while(n--) *to++ = s * *v++;
# endif
#endif

}/* end BPvsmul */

/* BPvsmul_sum: vector scalar multiply sum */
void BPvsmul_sum(to, v, s, n)
      float *to, *v, s;
      int n;
{
#if SAL
  extern void vsma();
  
  vsma(v,1,&s,to,1,to,1,n);
#endif

#if SKYVEC
  skyvec_ = n;
  v$_rstvp0(s, v, to, to);
#endif 

#if VANILLA || BLAS
/* the basic code Fortan Style and C Style */
# if FORLOOPS
   int i;
  
  for(i=0;i<n;i++) to[i] += s * v[i];
# else
  while(n--) *to++ += s * *v++;
# endif
#endif
}/* end BPvsmul_sum */


/* BPlvsmul: looping vector scalar multiply  */
void BPlvsmul(to, v, vs, vsize, lpcount)
      float *to, *v, *vs;
      int vsize, lpcount;
{
  while(lpcount--) {
    BPvsmul(to, v, *vs, vsize);
    vs++;
    to += vsize;
    v += vsize;
  } /* end loop */
}/* end BPvsmul */

/* BPlvsmul_sum: looping vector scalar multiply increment */
void BPlvsmul_sum(to, v, vs, vsize, lpcount)
      float *to, *v, *vs;
      int vsize, lpcount;
{
  while(lpcount--) {
    BPvsmul_sum(to, v, *vs, vsize);
    vs++;
    to += vsize;
    v += vsize;
  } /* end loop */
}/* end BPvsmul_sum */



/* BPvdot: Vector dot product */
float BPvdot(v1, v2, n)
      float *v1,*v2;
      int n;
{
  float sum = 0.0;
  
#if BLAS 
  int stride=1;
  sum = SDOT(&n, v1, &stride, v2, &stride);
#endif

#if SAL
  extern void dotpr();
  dotpr(v1, 1, v2, 1, &sum, n);
#endif


#if SKYVEC
  skyvec_ = n;
  sum = v$_rvvdot0(v1,v2);
#endif
  
#if VANILLA
  /* the basic code Fortan Style and C Style */
# if FORLOOPS
  
  int i;
  
  for(i=0;i<n;i++) sum += v1[i] * v2[i];
  
# else
  
  while(n--) sum += *v1++ * *v2++; 
  
# endif
#endif
  
  return(sum);
  
}/* end BPvdot */


/* BPlvdot: Looping vector dot product */
void BPlvdot(weights, from, overlap, to, n, lc)
      float *weights,*from,*to;
      int overlap, n, lc;
{

  /* machine specific code */
#if  SKYVEC
  skyvec_ = n;
  if (overlap == n) { /* full connection */
    while(lc--) {
      
      *to++ = v$_rvvdot0(weights,from);
      weights += n;
    }
  } else { 

    while(lc--) {

      *to++ = v$_rvvdot0(weights,from);
      weights += n;
      from += n - overlap;  
    }

  }/* end else */
#endif


  /* machine specific code */
#if  SAL
  if (overlap == n) { /* full connection */
    while(lc--) {
      extern void dotpr();
      
      dotpr(weights, 1, from, 1, to, n);
      to++;
      weights += n;
    }
  } else { 

    while(lc--) {
      extern void dotpr();
      
      dotpr(weights, 1, from, 1, to, n);
      to++;
      weights += n;
      from += n - overlap;  
    }

  }/* end else */
#endif



#if BLAS 
  
  if (overlap == n) { /* full connection */
    char trans='t';
    int stride=1;
    float alpha=1,beta=0;
    
    SGEMV(&trans,&n,&lc,&alpha,weights,&n,
	  from,&stride,&beta,to,&stride); 

  } else {
    
    while(lc--) {
      int stride=1;
      
      *to++ = SDOT(&n, weights, &stride, from, &stride); 
      weights += n;
      from += n - overlap;  
    }/* end while */
    
  }/* end else */
#endif
  
#if VANILLA
  /* the basic code Fortan Style and C Style */
# if FORLOOPS
  
  int l_index,w_index=0,f_index=0;
  
  for(l_index=0;l_index<lc;l_index++) {
    int d_index;
    float sum = 0.0;
    
    for(d_index=0;d_index<n;d_index++) {
      sum += weights[w_index] * from[f_index];
      w_index++;
      f_index++;
    }/* end for */
    
    to[l_index]=sum;
    f_index -= overlap;
  }/* end for */
  
# else
  
  while(lc--) {
    *to++ = BPvdot(weights,from,n); 
    weights += n;
    from += n - overlap;
  }/* while lc */
  
# endif
#endif
  
}/* end BPlvdot */

/* BPlvdot_share: Looping vector dot product (shared weights) */
 void BPlvdot_share(weights, from, overlap, to, n, lc)
      float *weights,*from,*to;
      int overlap, n, lc;
{
  /* machine specific code */
#if SAL
  
  while(lc--) {
    extern void dotpr();
    
    dotpr(weights, 1, from, 1, to, n);
    to++;
    from += n - overlap;  
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n;
  while(lc--) {

      *to++ = v$_rvvdot0(weights,from);
      from += n - overlap;  
  }/* end while */
#endif

#if BLAS 
  
  while(lc--) {
    int stride=1;
    
    *to++ = SDOT(&n, weights, &stride, from, &stride); 
    from += n - overlap;  
  }/* end while */
  
#endif

#if VANILLA
  /* the basic code Fortan Style and C Style */
# if FORLOOPS

  int l_index,f_index=0;
  
  for(l_index=0;l_index<lc;l_index++) {
    int d_index;
    float sum = 0.0;
    
    for(d_index=0;d_index<n;d_index++) {
      sum += weights[d_index] * from[f_index];
      f_index++;
    }/* end for */
    
    to[l_index]=sum;
    f_index -= overlap;
  }/* end for */
  
# else
  
  while(lc--) {
    *to++ = BPvdot(weights,from,n);
    from += n - overlap;
  }/* while */
  
# endif
#endif
  
}/* end BPlvdot_share */

/* BPlvdot_sum: Looping vector dot product and sum */
 void BPlvdot_sum(weights, from, overlap, to, n, lc)
      float *weights,*from,*to;
      int overlap, n, lc;
{
  
  /* machine specific code */
#if SAL    
    while(lc--) {
      extern void dotpr();
      float sum;
      
      dotpr(weights, 1, from, 1, &sum, n);
      *to++ += sum;
      weights += n;
      from += n - overlap;  
    }/* end while */
#endif

#if SKYVEC
    skyvec_ = n;
    while(lc--) {
      *to++ += v$_rvvdot0(weights,from);
      weights += n;
      from += n - overlap;  
    }/* end while */
#endif


#if BLAS 
    while(lc--) {
      int stride=1;
      
      *to++ += SDOT(&n, weights, &stride, from, &stride); 
      weights += n;
      from += n - overlap;  
    }/* end while */
#endif

  
#if VANILLA
  /* the basic code Fortan Style and C Style */
# if FORLOOPS
  
  int l_index,w_index=0,f_index=0;
  
  for(l_index=0;l_index<lc;l_index++) {
    int d_index;
    float sum = 0.0;
    
    for(d_index=0;d_index<n;d_index++) {
      sum += weights[w_index] * from[f_index];
      w_index++;
      f_index++;
    }/* end for */
    
    to[l_index]+=sum;
    f_index -= overlap;
  }/* end for */
  
# else
  
  while(lc--) {
    *to++ += BPvdot(weights,from,n);
    weights += n;
    from += n - overlap;
  }/* while */
  
# endif
#endif
  
}/* end BPlvdot_sum */

/* BPlvdot_share_sum: Looping vector dot product and sum (shared weights) */
 void BPlvdot_share_sum(weights, from, overlap, to, n, lc)
      float *weights,*from,*to;
      int overlap, n, lc;
{
  /* machine specific code */
#if SAL
  
  while(lc--) {
    extern void dotpr();
    float sum;
    
    dotpr(weights, 1, from, 1, &sum, n);
    *to++ += sum;
    from += n - overlap;  
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n;
  while(lc--) {
    *to++ += v$_rvvdot0(weights,from);
    from += n - overlap;  
  }/* end while */
#endif


#if BLAS 
  
  while(lc--) {
    int stride=1;
    
    *to++ += SDOT(&n, weights, &stride, from, &stride); 
    from += n - overlap;  
  }/* end while */
#endif

  
#if VANILLA
  /* the basic code Fortan Style and C Style */
# if FORLOOPS
  
  int l_index,f_index=0;
  
  for(l_index=0;l_index<lc;l_index++) {
    int d_index;
    float sum = 0.0;
    
    for(d_index=0;d_index<n;d_index++) {
      sum += weights[d_index] * from[f_index];
      f_index++;
    }/* end for */
    
    to[l_index]+=sum;
    f_index -= overlap;
  }/* end for */
  
# else
  
  while(lc--) {
    *to++ += BPvdot(weights,from,n);
    from += n - overlap;
  }/* while */
  
# endif
#endif
  
}/* end BPlvdot_share_sum */

/* BPlvdot2d: For 2d connections. */
void BPlvdot2d(xdim, ydim, tess_xdim, tess_ydim,
	       weights, from, to, xoverlap,wstep,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *from, *to;
      int xoverlap,wstep,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
    while(counter--) {
      BPlvdot_sum(weights, from, xoverlap, to, tess_xdim, xdim);
      weights += wstep;
      from += fxstep;
    }/* end while tess_ydim */
    to += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPlvdot2d */

/* BPlvdot2d_share: For 2d connections (shared weights) */
void BPlvdot2d_share(xdim, ydim, tess_xdim, tess_ydim,
		     weights, from, to, xoverlap,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *from, *to;
      int xoverlap,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
     float *kernel=weights;
    while(counter--) {
      BPlvdot_share_sum(kernel, from, xoverlap, to, tess_xdim, xdim);
      kernel += tess_xdim;
      from += fxstep;
    }/* end while tess_ydim */
    to += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPlvdot2d_share */


/************************ node stuff ************************/

  
#define TABLE_SIZE 1024
static float sigmoid_array[TABLE_SIZE];             /* table lookup sigmoid */
static float *sigmoid_ptr;                          /* pointer to center of sigmoid */

void BPinit_sigmoid_table()
{
  int counter;
  /* init table lookup for sigmoid [-1/2,1/2] */
  for(counter = 0; counter<TABLE_SIZE; counter++)
    sigmoid_array[counter] /* domain (8,8) w/511.5 at center (for 1024 table) */
      = (1.0 / (1.0 + AM_EXP(((TABLE_SIZE-1.0) - 2.0 * counter) / (TABLE_SIZE/8.0)))) - 0.5;
  sigmoid_ptr = sigmoid_array + (TABLE_SIZE/2) - 1;
}/* end BPinit_sigmoid_table */

/* old sigmoid 
static float sigmoid(x)
     float x;
{
  if (AM_FABS(x) >= 8.0)
    if (x>0.0) return(0.499999); else return(-0.499999);
  else
    return(*(sigmoid_ptr + (int)((TABLE_SIZE/16.0)*x + 0.5)));

} end sigmoid  */

/* Sigmoid macros */
#define sigmoid(x) ( (AM_FABS((x)) >= 8.0)?(((x)>0.0)?(0.499999):(-0.499999)):(*(sigmoid_ptr + (int)((TABLE_SIZE/16.0)*(x) + 0.5))) )
#define sigmoid1(x) (sigmoid(x) + 0.5)
#define sigmoid2(x) sigmoid(x)
#define sigmoid3(x) (2.0 * sigmoid(x))
#define sigmoid1prime(x) ( (x) * (1.0 - (x) ))
#define sigmoid2prime(x) (0.25 - ( (x) * (x) ))
#define sigmoid3prime(x) (0.5 * (1.0 - ( (x) * (x) )))



/* addbias: add the bias to the node value */
static  void addbias(nodes,biases,n)
      float *nodes, *biases;
      int n;
{

#if SAL
  extern void vadd();

  vadd(nodes,1,biases,1,nodes,1,n);
#endif

#if SKYVEC
  skyvec_ = n;
  v$_rvvp0(nodes,biases,nodes);
#endif

#if BLAS 
  float scalar=1;
  int stride=1;

  SAXPY(&n, &scalar, biases, &stride, nodes, &stride);
#endif

#if VANILLA

/* the basic code Fortan Style and C Style */
# if FORLOOPS

  int i;

  for(i=0;i<n;i++) nodes[i] += biases[i];

# else

  while(n--) *nodes++ += *biases++;

# endif
#endif

}/* end addbias */

/* BPsig1: sigmoid1 transfer function on layer */
void BPsig1(nodes, biases, n)
      float *nodes, *biases;
      int n;
{
  addbias(nodes,biases,n);
  while(n--) {
    float val;

    val = *nodes;
    *nodes++ = sigmoid1(val);
  }/* end while */

}/* end BPsig1 */

/* BPsig2: sigmoid2 transfer function on layer */
void BPsig2(nodes, biases, n)
      float *nodes, *biases;
      int n;
{
  addbias(nodes,biases,n);
  while(n--) {
    float val;

    val = *nodes;
    *nodes++ = sigmoid2(val);
  }/* end while */

}/* end BPsig2 */

/* BPsig3: sigmoid3 transfer function on layer */
void BPsig3(nodes, biases, n)
      float *nodes, *biases;
      int n;
{
  addbias(nodes,biases,n);
  while(n--) {
    float val;

    val = *nodes;
    *nodes++ = sigmoid3(val);
  }/* end while */

}/* end BPsig3 */

/* BPlinear: linear transfer function on layer */
void BPlinear(nodes, biases, n)
      float *nodes, *biases;
      int n;
{
    addbias(nodes,biases,n);
}/* end BPlinear */

/* BPquadratic: quadratic transfer function on layer */
void BPquadratic(nodes, netinput, biases, n)
      float *nodes,*netinput,*biases;
      int n;
{
  /* Note: the netinput is accumulated in the node values vector,
           before being passed to this function.
   */
  addbias(nodes,biases,n);
  while(n--) {
    float sum;

    sum = *nodes;
    *netinput++ = sum;
    *nodes++ = sum * sum;
  }/* end while */
}/* end BPquadratic */

/* BPuser: user transfer function on layer */
void BPuser(nodes, netinput, biases, n, f)
      float *nodes, *netinput, *biases;
      int n;
      float (*f)();
{
  /* Note: the netinput is accumulated in the node values vector,
           before being passed to this function.
   */
  addbias(nodes,biases,n);
  while(n--) {
    *netinput = *nodes;
    *nodes++ = f(*netinput++);
  }/* end while */
}/* end BPuser */


/* BPderiv_sig1: derivative, update credit. */
void BPderiv_sig1(nodes,credit,n,bias)
      float *nodes,*credit;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *credit++ *= (sigmoid1prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPderiv_sig1 */


/* BPset_deriv_sig1: derivative */
void BPset_deriv_sig1(deriv,nodes,n,bias)
      float *deriv,*nodes;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *deriv++ = (sigmoid1prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPset_deriv_sig1 */


/* BPderiv_sig2: derivative, update credit. */
void BPderiv_sig2(nodes,credit,n,bias)
      float *nodes,*credit;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *credit++ *= (sigmoid2prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPderiv_sig2 */


/* BPset_deriv_sig2: derivative */
void BPset_deriv_sig2(deriv,nodes,n,bias)
      float *deriv,*nodes;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *deriv++ = (sigmoid2prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPset_deriv_sig2 */

/* BPderiv_sig3: derivative, update credit. */
void BPderiv_sig3(nodes,credit,n,bias)
      float *nodes,*credit;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *credit++ *= (sigmoid3prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPderiv_sig3 */


/* BPset_deriv_sig3: derivative */
void BPset_deriv_sig3(deriv,nodes,n,bias)
      float *deriv,*nodes;
      int n;
     float bias;
{
  while(n--) {
    /* Scott Fahlman suggests biasing the derivative (88 Connectionist Summer School) */
    *deriv++ = (sigmoid3prime(*nodes) + bias);
    nodes++;
  }/* end while */
}/* end BPset_deriv_sig3 */

/* BPderiv_quadratic: derivative, update credit. */
void BPderiv_quadratic(netinput,credit,n)
      float *netinput,*credit;
      int n;
{

  while(n--)  *credit++ *= (2.0 * *netinput++);

}/* end BPderiv_quadratic */

/* BPderiv_user: derivative, update credit. */
void BPderiv_user(netinput,credit,n,f_prime)
      float *netinput,*credit;
      int n;
      float (*f_prime)();
{

  while(n--) *credit++ *= f_prime(*netinput++);

}/* end BPderiv_user */


/************************ basic learning stuff ****************************/

float BPsum_squares(v, n)
     float *v;
     int n;
{
  float sum=0.0;

  while(n--) {
    float val;

    val = *v++;
    sum += val * val;
  }/* end while */

  return(sum);
}


/* BPoutput_error: Diff on the output, updates credit, returns MSE/2 */
float BPoutput_error(target,output,credit,scalar,n)
      float *target,*output,*credit;
      float scalar;
      int n; 
{
  float total_error;

#if SAL
  extern void vsub(), vsmul();

  vsub(output,1,target,1,credit,1,n);
  total_error = BPsum_squares(credit, n);
  vsmul(credit,1,&scalar,credit,1,n);
#endif


#if SKYVEC
  skyvec_ = n;
  v$_rvvm0(output,target,credit);
  total_error = BPsum_squares(credit, n);
  v$_rsvt0(scalar,credit,credit);
#endif

#if VANILLA || BLAS
/* the basic code Fortan Style and C Style */
# if FORLOOPS

  int i;

  for(i=0;i<n;i++) credit[i] = (target[i] - output[i]);
  total_error = BPsum_squares(credit, n);
  for(i=0;i<n;i++) credit[i] *= scalar;
# else

  
  { /* temp variables */
    float *cptr = credit;
    int m=n;
    
    while(m--) *cptr++ = (*target++ - *output++);
  }

  total_error = BPsum_squares(credit, n);
  while(n--) *credit++ *= scalar;

# endif
#endif

  return( total_error / 2.0 );
}/* end BPoutput_error */


/* BPaccum_biases: alter thresholds. */
 void BPaccum_biases(changes,credit,n)
      float *changes, *credit;
      int n;
{


#if SAL
  extern void vadd();

  vadd(changes,1,credit,1,changes,1,n);
#endif

#if SKYVEC
  skyvec_ = n;
  v$_rvvp0(changes,credit,changes);
#endif

#if BLAS
  float scalar=1;
  int stride=1;

  SAXPY(&n, &scalar, credit, &stride, changes, &stride);

#endif

#if VANILLA
/* the basic code Fortan Style and C Style */
# if FORLOOPS

   int i;

  for(i=0;i<n;i++) changes[i] += credit[i];

# else

  while(n--) *changes++ += *credit++;

# endif
#endif

}/* end BPaccum_biases */

/* BPaccum_weights_from_input: calc weight change. */
void BPaccum_weights_from_input(to_credit,from,changes,n_to,n_from,overlap)
      float *to_credit,*from,*changes;
      int n_to,n_from,overlap;
{

#if SAL
  
  while(n_to--) {
    extern void vsma();

    vsma(from,1,to_credit,changes,1,changes,1,n_from);
    to_credit++;
    from += n_from - overlap;
    changes += n_from;
  }/* end while */
#endif


#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {
    v$_rsvtvp0(*to_credit, from, changes, changes);
    to_credit++;
    from += n_from - overlap;
    changes += n_from;
  }/* end while */
#endif


#if BLAS

  while(n_to--) {
    int stride=1;
    
    SAXPY(&n_from, to_credit, from, &stride, changes, &stride);
    to_credit++;
    from += n_from - overlap;
    changes += n_from;
  } /* end while */

#endif


#if VANILLA
/* the basic code Fortan Style and C Style */
# if FORLOOPS

  int e_index,c_index=0,f_index=0;
  for(e_index=0;e_index<n_to;e_index++) {
     int counter;
    float error;

    error = to_credit[e_index];
    for(counter=0;counter<n_from;counter++) {
      changes[c_index] += from[f_index] * error;
      c_index++; f_index++;
    }/* end for */
    f_index -= overlap;
  }/* end for */

# else

  while(n_to--) {
    int counter=n_from;
    float error;
    error = *to_credit++;
    while(counter--) *changes++ += *from++ * error;
    from -= overlap;
  }/* end while */

# endif
#endif

}/* end BPaccum_weights_from_input */

/* BPaccum_weights_from_input_share: alter SHARED weights. */
void BPaccum_weights_from_input_share(to_credit,from,changes,n_to,n_from,overlap)
      float *to_credit,*from,*changes;
      int n_to,n_from,overlap;
{

#if SAL
  
  while(n_to--) {
    extern void vsma();

    vsma(from,1,to_credit,changes,1,changes,1,n_from);
    to_credit++;
    from += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {
    v$_rsvtvp0(*to_credit, from, changes, changes);
    to_credit++;
    from += n_from - overlap;
  }/* end while */
#endif

#if BLAS

  while(n_to--) {
    int stride=1;

    SAXPY(&n_from, to_credit, from, &stride, changes, &stride);
    to_credit++;
    from += n_from - overlap;
  }/* end while */
#endif


#if VANILLA
/* the basic code Fortan Style and C Style */
# if FORLOOPS

   int e_index=0,f_index=0;
  for(;e_index<n_to;e_index++) {
     int counter, c_index=0;
    float error;

    error = to_credit[e_index];
    for(counter=0;counter<n_from;counter++) {
      changes[c_index] += from[f_index] * error;
      c_index++; f_index++;
    }/* end for */
    f_index -= overlap;
  }/* end for */

# else

  while(n_to--) {
     int counter=n_from;
    float error, *kernel;
    
    kernel = changes;
    error = *to_credit++;
    while(counter--) *kernel++ += *from++ * error;
    from -= overlap;
  }/* end while */

# endif
#endif
}/* end BPaccum_weights_from_input_share */

/* BPaccum_weights_from_hidden: alter weights. */
void BPaccum_weights_from_hidden(to_credit,from,from_credit,
				      weights,changes,n_to,n_from,overlap)
      float *to_credit,*from,*from_credit,*weights,*changes;
      int n_to,n_from,overlap;
{

#if SAL
  
  while(n_to--) {
    extern void vsma();

    vsma(from,1,to_credit,changes,1,changes,1,n_from);
    vsma(weights,1,to_credit,from_credit,1,from_credit,1,n_from);
    to_credit++;
    changes += n_from;
    weights += n_from;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {
    v$_rsvtvp0(*to_credit, from, changes, changes);
    v$_rsvtvp0(*to_credit, weights, from_credit, from_credit);
    to_credit++;
    changes += n_from;
    weights += n_from;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if BLAS

  while(n_to--) {
    int stride=1;

    SAXPY(&n_from, to_credit, from, &stride, changes, &stride);
    SAXPY(&n_from, to_credit, weights, &stride, from_credit, &stride);
    to_credit++;
    changes += n_from;
    weights += n_from;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  } /* end while */

#endif

#if VANILLA
/* the basic code Fortan Style and C Style */
# if FORLOOPS

   int e_index,c_index=0,f_index=0,fc_index=0,w_index=0;
  for(e_index=0;e_index<n_to;e_index++) {
     int counter;
    float error;

    error = to_credit[e_index];
    for(counter=0;counter<n_from;counter++) {
      /* accumulate error */
      from_credit[fc_index] += error * weights[w_index];
      /* accumulate weight change */
      changes[c_index] += from[f_index] * error;
      fc_index++; w_index++; c_index++; f_index++;
    }/* end for */
    f_index -= overlap;
    fc_index -= overlap;
  }/* end for */

# else

  while(n_to--) {
    int counter=n_from;
    float error;
    error = *to_credit++;
    while(counter--) {
      /* accumulate error */
      *from_credit++ += error * *weights++;
      /* accumulate weight change */
      *changes++ += *from++ * error;
    }/* end while */
    from -= overlap;
    from_credit -= overlap;
  }/* end while */

# endif
#endif

}/* end BPaccum_weights_from_hidden */

/* BPaccum_weights_from_hidden_share: alter SHARED weights. */
void BPaccum_weights_from_hidden_share(to_credit,from,from_credit,
					    weights,changes,n_to,n_from,overlap)
      float *to_credit,*from,*from_credit,*weights,*changes;
      int n_to,n_from,overlap;
{


#if SAL
  while(n_to--) {
    extern void vsma();

    vsma(from,1,to_credit,changes,1,changes,1,n_from);
    vsma(weights,1,to_credit,from_credit,1,from_credit,1,n_from);
    to_credit++;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {

    v$_rsvtvp0(*to_credit, from, changes, changes);
    v$_rsvtvp0(*to_credit, weights, from_credit, from_credit);
    to_credit++;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if BLAS
  while(n_to--) {
    int stride=1;

    SAXPY(&n_from, to_credit, from, &stride, changes, &stride);
    SAXPY(&n_from, to_credit, weights, &stride, from_credit, &stride);
    to_credit++;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if VANILLA
/* the basic code Fortan Style and C Style */
# if FORLOOPS

   int e_index=0,f_index=0,fc_index=0;
  for(e_index=0;e_index<n_to;e_index++) {
     int counter,w_index=0,c_index=0;
    float error;

    error = to_credit[e_index];
    for(counter=0;counter<n_from;counter++) {
      /* accumulate error */
      from_credit[fc_index] += error * weights[w_index];
      /* accumulate weight change */
      changes[c_index] += from[f_index] * error;
      fc_index++; w_index++; c_index++; f_index++;
    }/* end for */
    f_index -= overlap;
    fc_index -= overlap;
  }/* end for */

# else

  while(n_to--) {
     int counter=n_from;
    float error, *kernel=weights,*ckernel=changes;
    error = *to_credit++;
    while(counter--) {
      /* accumulate error */
      *from_credit++ += error * *kernel++;
      /* accumulate weight changes */
      *ckernel++ += *from++ * error;
    }/* end while */
    from -= overlap;
    from_credit -= overlap;
  }/* end while */

# endif
#endif 

}/* end BPaccum_weights_from_hidden_share */

/* BPaccum2d_weights_from_input: For 2d connections. */
void BPaccum2d_weights_from_input(xdim, ydim, tess_xdim, tess_ydim,
				       changes, from, to_credit,
				       xoverlap,wstep,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *changes, *from, *to_credit;
      int xoverlap,wstep,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
    while(counter--) {
      BPaccum_weights_from_input(to_credit, from,
				      changes,
				      xdim, tess_xdim, xoverlap);
      changes += wstep;
      from += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPaccum2d_weights_from_input */

/* BPaccum2d_weights_from_input_share: For 2d connections (shared weights) */
void BPaccum2d_weights_from_input_share(xdim, ydim, tess_xdim, tess_ydim,
					     changes, from, to_credit,
					     xoverlap,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *changes, *from, *to_credit;
      int xoverlap,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
     float *kernel=changes;
    while(counter--) {
      BPaccum_weights_from_input_share(to_credit, from,
					    kernel, xdim, tess_xdim, xoverlap);
      kernel += tess_xdim;
      from += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPaccum2d_weights_from_input_share */

#ifndef GREENHILLS860  /* this piece of shit compiler pukes w/more than 12 args... */

/* BPaccum2d_weights_from_hidden: For 2d connections. */
void BPaccum2d_weights_from_hidden(xdim, ydim, tess_xdim, tess_ydim, 
				   weights, changes, from, from_credit, to_credit, 
				   xoverlap,wstep,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *changes, *from, *from_credit, *to_credit;
      int xoverlap,wstep,fxstep,fystep;  
{
  while(ydim--) {
     int counter=tess_ydim;
    while(counter--) {
      BPaccum_weights_from_hidden(to_credit, from, from_credit,
				       weights, changes,
				       xdim, tess_xdim, xoverlap);
      weights += wstep;
      changes += wstep;
      from += fxstep;
      from_credit += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    from -= fystep;
    from_credit -= fystep;
  }/* end while ydim */
}/* BPaccum2d_weights_from_hidden */


/* BPaccum2d_weights_from_hidden_share: For 2d connections (shared weights). */
void BPaccum2d_weights_from_hidden_share(xdim, ydim, tess_xdim, tess_ydim,
					 weights, changes, from, from_credit, to_credit,
					 xoverlap,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *changes, *from, *from_credit, *to_credit;
      int xoverlap,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
     float *kernel=weights,*ckernel=changes;
    while(counter--) {
      BPaccum_weights_from_hidden_share(to_credit, from, from_credit,
					     kernel, ckernel, xdim, tess_xdim, xoverlap);
      kernel += tess_xdim;
      ckernel += tess_xdim;
      from += fxstep;
      from_credit += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    from -= fystep;
    from_credit -= fystep;
  }/* end while ydim */
}/* BPaccum2d_weights_from_hidden_share */

#endif 

/* BPupdate_weights: change the weights by accumulated change */
void BPupdate_weights(inertia, changes, weights, delta_weights, n)
      float inertia;
      float *changes,*weights,*delta_weights;
      int n;
{
#if SAL
   extern void vadd(), vsmul();

   vsmul(delta_weights,1,&inertia,delta_weights,1,n);
   vadd(changes,1,delta_weights,1,delta_weights,1,n);
   vadd(delta_weights,1,weights,1,weights,1,n);
#endif

#if SKYVEC
   skyvec_ = n;
   v$_rsvt0(inertia, delta_weights, delta_weights);
   v$_rvvp0(changes, delta_weights, delta_weights);
   v$_rvvp0(delta_weights,weights,weights);
#endif

#if BLAS
  float scaler=1,i=inertia;
  int stride=1;

  SAXPY(&n,&i,delta_weights,&stride,changes,&stride);
  SAXPY(&n,&scaler,changes,&stride,weights,&stride);
  SCOPY(&n,changes,&stride,delta_weights,&stride); 

#endif 

#if VANILLA 
# if FORLOOPS

   int i;
  /* update */
  for(i=0;i<n;i++) {
    float change;
    change = changes[i] + (delta_weights[i] * inertia);
    weights[i] += change;
    delta_weights[i] = change;
  }/* end for */

# else  

  /* update */
  while(n--) {
    float change;
    change = *changes++ + (*delta_weights * inertia);
    *weights++ += change;
    *delta_weights++ = change;
  }/* end while n */

# endif
#endif

}/* BPupdate_weights */


/************************ autoregressive backprop stuff ****************************/


/* Finding the roots of the characterisitic equation:
   D(z) = z^n - a1*z^(n-1) - a2*z^(n-2) - a3,
   where the a's are the feedback weights.
 */

#define BPZERO(x) ( (AM_FABS(x) < 2.3283064e-10) )

/*    D(z) = z^1 - a1 */
void BPar1_poles(a,r)
     double a;
     AM_COMPLEX *r;

{
  r->real = a;    /* it's just the weight */
  r->imag = 0.0;
}

/*   D(z) = z^2 - a1*z^1 - a2 */
void BPar2_poles(a1,a2,r)
     double a1, a2;
     AM_COMPLEX *r;

{
  double d;

  a1 *= -1.0;
  a2 *= -1.0;

  /* use quadratic formula */
  d = (a1 * a1) - (4.0 * a2);
  if ( BPZERO(d) ) {  /* 2 equal real roots */
    double real;

    real = -a1 / 2.0;
    r->real = real;
    r->imag = 0.0;
    r++;
    r->real = real;
    r->imag = 0.0;

  } else if (d < 0.0) { /* complex conjugates */
    double real, imag;

    real = -a1 / 2.0;
    imag = AM_SQRT(-d) / 2.0;
    r->real = real;
    r->imag = imag;
    r++;
    r->real = real;
    r->imag = -imag;

  } else { /* 2 real */

    r->real = (-a1 + AM_SQRT(d)) / 2.0;
    r->imag = 0.0;
    r++;
    r->real = (-a1 - AM_SQRT(d)) / 2.0;
    r->imag = 0.0;

  }/* end if else */

}

/*   D(z) = z^3 - a1*z^2 - a2*z^1 - a3 */
void BPar3_poles(a1,a2,a3,r)
     double a1, a2, a3;
     AM_COMPLEX *r;

{
  double p,q, Q, A, B;

  a1 *= -1.0;
  a2 *= -1.0;
  a3 *= -1.0;

  p = ( - (a1 * a1) / 3.0 ) + a2;

  q = ( 2.0 * (a1/3.0) * (a1/3.0) ) - ( (a1*a2)/3.0 ) + a3;

  Q = ( (p/3.0) * (p/3.0) * (p/3.0) ) + ( (q * 0.5) * (q * 0.5) );

  if ( BPZERO(Q) ) { /* 3 real, 2 equal */

    A = B = AM_POW( (-q/2.0), (double)0.333333333333 );

    r->real = A + B;
    r->imag = 0.0;
    r++;

    r->real = (A + B)/2.0;
    r->imag = 0.0;
    r++;

    r->real = (A + B)/2.0;
    r->imag = 0.0;

  } else if (Q > 0.0) { /* 1 real, 2 conjugates */

    A = AM_POW( (-q/2.0) + AM_SQRT(Q) , (double)0.333333333333 );
    B = AM_POW( (-q/2.0) - AM_SQRT(Q) , (double)0.333333333333 );


    r->real = A + B;
    r->imag = 0.0;
    r++;

    r->real = (A + B)/2.0;
    r->imag = ( (A - B)/2.0 ) * AM_SQRT(3.0);
    r++;

    r->real = (A + B)/2.0;
    r->imag = ( (A - B)/2.0 ) * AM_SQRT(3.0);


  } else { /* 3 real , all different */
    double alpha;

    alpha = AM_ACOS( - q / (2.0 * AM_SQRT( - (p/3.0) * (p/3.0) * (p/3.0) )));

    r->real = 2.0 * AM_SQRT( - (p/3.0) ) * AM_COS( alpha/3.0 );
    r->imag = 0.0;
    r++;

    r->real = - 2.0 * AM_SQRT( - (p/3.0) ) * AM_COS( alpha/3.0 + (double)1.047197551);
    r->imag = 0.0;
    r++;

    r->real = - 2.0 * AM_SQRT( - (p/3.0) ) * AM_COS( alpha/3.0 - (double)1.047197551);
    r->imag = 0.0;
    

  }/* end if else */

}





/************* Stabilize by not allowing weights to *************/
/************* leave stable region...test with the  *************/
/************* Routh-Hurwitz criterion.             *************/


/* BPunstable_1st: true if unstable AR weight */
int BPunstable_1st(x)
     float x;
{
  return ( AM_FABS(x) >= (float)0.99 ) ;
}/* end BPunstable_1st */


/* BPTupdate_1st : update and stabilize first order reflection weights */
void BPTupdate_1st(inertia,changes,weights,delta_weights,n)
      float inertia,*changes,*weights,*delta_weights;
      int n;
{
  /* update, keep stable */
  while(n--) {
    float new_weight,change;
    
    change = *changes + (inertia * *delta_weights);
    new_weight = *weights + change;
    
    while ( BPunstable_1st(new_weight) ) { /* not stable if mag >= 1 */
      
      /* now contract to be within stability boundries */
      change *= 0.5; /* contract by 1/2 */
      /* new weight */
      new_weight = *weights + change;
      
    }/* end while */
    
    /* set */
    *weights = new_weight;
    *delta_weights = change;
    
    /* onward */
    weights++; delta_weights++; changes++;
    
  } /* end while */
  
}/* BPTupdate_1st */



/* BPunstable_2nd: true if unstable AR weights */
int BPunstable_2nd(x1,x2)
     float x1,x2;
{
  return ( (x2 + AM_FABS(x1) >= (float)0.99) || (x2 <= (float)-0.99) ) ;
}/* end BPunstable_2nd */


/* BPTupdate_2nd : update and stabilize second order reflection weights */
void BPTupdate_2nd(inertia,
		   changes_1,changes_2,
		   weights_1,weights_2,
		   delta_weights_1,delta_weights_2,
		   n)
     float inertia;
     float *changes_1,*changes_2;
     float *weights_1,*weights_2;
     float *delta_weights_1,*delta_weights_2;
     int n;
{
  /* update, keep stable */
  while(n--) {
    float new_weight_1, new_weight_2;
    float change_1,change_2;
    
    change_1 = *changes_1 + (inertia * *delta_weights_1);
    change_2 = *changes_2 + (inertia * *delta_weights_2);
    new_weight_1 = *weights_1 + change_1;
    new_weight_2 = *weights_2 + change_2;
    
    while ( BPunstable_2nd(new_weight_1,new_weight_2) ) { /* unstable! */
      
      /* now contract to be within stability boundries */
      change_1 *= 0.5; /* contract by 1/2 */
      change_2 *= 0.5; /* contract by 1/2 */
      /* new weights */
      new_weight_1 = *weights_1 + change_1;
      new_weight_2 = *weights_2 + change_2;
      
    }/* end while */
    
    /* set */
    *weights_1 = new_weight_1;
    *weights_2 = new_weight_2;
    *delta_weights_1 = change_1;
    *delta_weights_2 = change_2;
    
    /* onward */
    weights_1++; weights_2++;
    delta_weights_1++; delta_weights_2++;
    changes_1++; changes_2++;
    
  } /* end while */
  
}/* BPTupdate_2nd */



/* BPunstable_3rd: true if unstable AR weights */
int BPunstable_3rd(x1,x2,x3)
     float x1,x2,x3;
{
  return (x2 + AM_FABS(x1 + x3) >= (float)0.99 ||
	AM_FABS(x3) >= (float)0.99 ||
	AM_FABS(-x2 - (x1 * x3)) >= AM_FABS((float)0.99 - (x3 * x3)) ) ;
}/* end BPunstable_3rd */



/* BPTupdate_3rd : update and stabilize third order reflection weights */
void BPTupdate_3rd(inertia,
		   changes_1,changes_2,changes_3,
		   weights_1,weights_2,weights_3,
		   delta_weights_1,delta_weights_2,delta_weights_3,
		   n)
     float inertia;
     float *changes_1,*changes_2,*changes_3;
     float *weights_1,*weights_2, *weights_3;
     float *delta_weights_1,*delta_weights_2,*delta_weights_3;
     int n;
{
  /* update, keep stable */
  while(n--) {
    float new_weight_1, new_weight_2, new_weight_3;
    float change_1,change_2,change_3;
    
    change_1 = *changes_1 + (inertia * *delta_weights_1);
    change_2 = *changes_2 + (inertia * *delta_weights_2);
    change_3 = *changes_3 + (inertia * *delta_weights_3);
    new_weight_1 = *weights_1 + change_1;
    new_weight_2 = *weights_2 + change_2;
    new_weight_3 = *weights_3 + change_3;
    
    while (BPunstable_3rd(new_weight_1,new_weight_2,new_weight_3)) { /* unstable! */
      
      /* now contract to be within stability boundries */
      change_1 *= 0.5; /* contract by 1/2 */
      change_2 *= 0.5; /* contract by 1/2 */
      change_3 *= 0.5; /* contract by 1/2 */
      /* new weights */
      new_weight_1 = *weights_1 + change_1;
      new_weight_2 = *weights_2 + change_2;
      new_weight_3 = *weights_3 + change_3;
      
    }/* end while */
    
    /* set */
    *weights_1 = new_weight_1;
    *weights_2 = new_weight_2;
    *weights_3 = new_weight_3;
    *delta_weights_1 = change_1;
    *delta_weights_2 = change_2;
    *delta_weights_3 = change_3;
    
    
    /* onward */
    weights_1++; weights_2++; weights_3++;
    delta_weights_1++; delta_weights_2++; delta_weights_3++;
    changes_1++; changes_2++; changes_3++;
    
  } /* end while */
  
  
}/* BPTupdate_3rd */

/* BPTaccum_reflection_weights: Accumulate changes to reflection coef. */
void BPTaccum_reflection_weights(changes,partials,credit,n)
      float *changes, *partials, *credit;
      int n;
{
  BPvmul_sum(changes, partials, credit, n);
}/* end BPTaccum_reflection_weights */


/* BPTaccum_biases: thresholds. */
void BPTaccum_biases(changes,credit,to_deriv,partials,n)
     float *changes,*credit,*to_deriv,*partials; 
     int n;
{
#if SAL
  extern void vadd(), vma();

  vadd(partials,1,to_deriv,1,partials,1,n);
  vma(credit,1,partials,1,changes,1,changes,1,n);
#endif
  
#if SKYVEC
  skyvec_ = n;
  v$_rvvp0(partials, to_deriv, partials);
  while(n--) {
    *changes++ += *credit++ * *partials++;
  }/* end while */
#endif


#if VANILLA || BLAS
  while(n--) {
    *partials += *to_deriv++; /* from node value is 1 */
    *changes++ += *credit++ * *partials++; 
  }/* end while */
#endif

}/* end BPTaccum_biases */


/* BPTaccum_weights_from_input: calc weight change. */
void BPTaccum_weights_from_input(to_credit,to_deriv,from,
				      changes,partials,n_to,n_from,overlap)
      float *to_credit,*to_deriv,*from,*changes,*partials;
      int n_to,n_from,overlap;
{
#if SAL
  extern void vsma();

  while(n_to--) {
    vsma(from,1, to_deriv, partials,1, partials,1, n_from);
    vsma(partials,1, to_credit, changes,1, changes,1, n_from);
    to_deriv++; to_credit++;    /* scalars */
    partials += n_from;
    changes += n_from;
    from += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {
    v$_rsvtvp0(*to_deriv, from, partials, partials);
    v$_rsvtvp0(*to_credit, partials, changes, changes);
    to_deriv++; to_credit++;    /* scalars */
    partials += n_from;
    changes += n_from;
    from += n_from - overlap;
  }/* end while */
#endif

#if VANILLA || BLAS
  while(n_to--) {
    int counter=n_from;
    float error, deriv;

    /* scalars */
    error = *to_credit++;
    deriv = *to_deriv++;

    while(counter--) {
      *partials += deriv * *from++ ;         /* add in from value */
      *changes++ += error * *partials++;
    }/* end while */

    from -= overlap;
  }/* end while */
#endif

}/* end BPTaccum_weights_from_input */

/* BPTaccum_weights_from_input_share: alter SHARED weights. */
void BPTaccum_weights_from_input_share(to_credit,to_deriv,from,
					    changes,partials,n_to,n_from,overlap)
      float *to_credit,*to_deriv,*from,*changes,*partials;
      int n_to,n_from,overlap;
{
#if SAL
  extern void vsma();

  while(n_to--) {
    vsma(from,1, to_deriv, partials,1, partials,1, n_from);
    vsma(partials,1, to_credit, changes,1, changes,1, n_from);
    to_deriv++; to_credit++;    /* scalars */
    from += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {
    v$_rsvtvp0(*to_deriv, from, partials, partials);
    v$_rsvtvp0(*to_credit, partials, changes, changes);
    to_deriv++; to_credit++;    /* scalars */
    from += n_from - overlap;
  }/* end while */
#endif

#if VANILLA || BLAS
  while(n_to--) {
    int counter=n_from;
    float error, deriv, *kernel, *pkernel;
    
    kernel = changes;
    pkernel = partials;
    error = *to_credit++;
    deriv = *to_deriv++;
    while(counter--) {
      *pkernel += deriv * *from++;
      *kernel++ += error * *pkernel++;
    }/* end while */
    from -= overlap;
  }/* end while */
#endif

}/* end BPTaccum_weights_from_input_share */

/* BPTaccum_weights_from_hidden: alter weights. */
void BPTaccum_weights_from_hidden(to_credit,to_deriv,from,from_credit,
				       weights,changes,opartials,wpartials,n_to,n_from,overlap)
     float *to_credit,*to_deriv,*from,*from_credit,*weights,*changes,*opartials,*wpartials;
     int n_to,n_from,overlap;
{

#if SAL
  extern void vsma();

  while(n_to--) {
    /* accumulate error */
    vsma(weights,1, to_deriv, wpartials,1, wpartials,1, n_from);
    vsma(wpartials,1, to_credit, from_credit,1, from_credit,1, n_from);

    /* accumulate weight change */
    vsma(from,1, to_deriv, opartials,1, opartials,1, n_from);
    vsma(opartials,1, to_credit, changes,1, changes,1, n_from);

    /* update ptrs */
    to_deriv++; to_credit++;    /* scalars */
    weights += n_from;
    opartials += n_from;
    wpartials += n_from;
    changes += n_from;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif


#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {

    /* accumulate error */
    v$_rsvtvp0(*to_deriv, weights, wpartials, wpartials);
    v$_rsvtvp0(*to_credit, wpartials, from_credit, from_credit);

    /* accumulate weight change */
    v$_rsvtvp0(*to_deriv, from, opartials, opartials);
    v$_rsvtvp0(*to_credit, opartials, changes, changes);

    /* update ptrs */
    to_deriv++; to_credit++;    /* scalars */
    weights += n_from;
    opartials += n_from;
    wpartials += n_from;
    changes += n_from;
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif
  
#if VANILLA || BLAS
  while(n_to--) {
    int counter=n_from;
    float error,deriv;
    
    error = *to_credit++;
    deriv = *to_deriv++;
    
    while(counter--) {
      /* accumulate error */
      *wpartials += deriv * *weights++;
      *from_credit++ += error * *wpartials++;

      /* accumulate weight change */
      *opartials += deriv * *from++;
      *changes++ += error * *opartials++;
    }/* end while */
    from -= overlap;
    from_credit -= overlap;
  }/* end while */
#endif

}/* end BPTaccum_weights_from_hidden */

/* BPTaccum_weights_from_hidden_share: alter SHARED weights. */
void BPTaccum_weights_from_hidden_share(to_credit,to_deriv,from,from_credit,
					     weights,changes,opartials,wpartials,n_to,n_from,overlap)
     float *to_credit,*to_deriv,*from,*from_credit,*weights,*changes,*opartials,*wpartials;
     int n_to,n_from,overlap;
{
#if SAL
 extern void vsma();

  while(n_to--) {
    /* accumulate error */
    vsma(weights,1, to_deriv, wpartials,1, wpartials,1, n_from);
    vsma(wpartials,1, to_credit, from_credit,1, from_credit,1, n_from);

    /* accumulate weight change */
    vsma(from,1, to_deriv, opartials,1, opartials,1, n_from);
    vsma(opartials,1, to_credit, changes,1, changes,1, n_from);

    /* update ptrs */
    to_deriv++; to_credit++;    /* scalars */
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if SKYVEC
  skyvec_ = n_from;
  while(n_to--) {

    /* accumulate error */
    v$_rsvtvp0(*to_deriv, weights, wpartials, wpartials);
    v$_rsvtvp0(*to_credit, wpartials, from_credit, from_credit);

    /* accumulate weight change */
    v$_rsvtvp0(*to_deriv, from, opartials, opartials);
    v$_rsvtvp0(*to_credit, opartials, changes, changes);

    /* update ptrs */
    to_deriv++; to_credit++;    /* scalars */
    from += n_from - overlap;
    from_credit += n_from - overlap;
  }/* end while */
#endif

#if VANILLA || BLAS
  while(n_to--) {
    int counter=n_from;
    float *kernel=weights,*ckernel=changes,*opkernel=opartials,*wpkernel=wpartials;
    float error,deriv;
    
    error = *to_credit++;
    deriv = *to_deriv++;

    while(counter--) {
      /* accumulate error */
      *wpartials += deriv * *kernel++;
      *from_credit++ += error * *wpkernel++;

      /* accumulate weight changes */
      *opkernel += deriv * *from++;
      *ckernel++ += error * *opkernel++;
    }/* end while */
    from -= overlap;
    from_credit -= overlap;
  }/* end while */
#endif

}/* end BPTaccum_weights_from_hidden_share */

#ifndef GREENHILLS860  /* this piece of shit compiler pukes w/more than 13 args... */

/* BPTaccum2d_weights_from_input: For 2d connections. */
void BPTaccum2d_weights_from_input(xdim, ydim, tess_xdim, tess_ydim,
					changes, opartials, from, to_credit, to_deriv,
					xoverlap,wstep,fxstep,fystep)
     int xdim, ydim, tess_xdim, tess_ydim;
     float *changes, *opartials, *from, *to_credit,*to_deriv;
     int xoverlap,wstep,fxstep,fystep;
{
  while(ydim--) {
    int counter=tess_ydim;
    while(counter--) {
      BPTaccum_weights_from_input(to_credit, to_deriv, from,
				       changes, opartials,
				       xdim, tess_xdim, xoverlap);
      changes += wstep;
      opartials += wstep;
      from += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    to_deriv += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPTaccum2d_weights_from_input */

/* BPTaccum2d_weights_from_input_share: For 2d connections (shared weights) */
void BPTaccum2d_weights_from_input_share(xdim, ydim, tess_xdim, tess_ydim,
					      changes, opartials, from, to_credit,to_deriv,
					      xoverlap,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *changes, *opartials, *from, *to_credit,*to_deriv;
      int xoverlap,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
     float *kernel=changes,*pkernel=opartials;
    while(counter--) {
      BPTaccum_weights_from_input_share(to_credit,to_deriv,from,
					     kernel, pkernel,
					     xdim, tess_xdim, xoverlap);
      kernel += tess_xdim;
      pkernel += tess_xdim;
      from += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    to_deriv += xdim;
    from -= fystep;
  }/* end while ydim */
}/* BPTaccum2d_weights_from_input_share */

/* BPTaccum2d_weights_from_hidden: For 2d connections. */
void BPTaccum2d_weights_from_hidden(xdim, ydim, tess_xdim, tess_ydim,
					 weights, changes, opartials, wpartials, from, from_credit, to_credit,to_deriv,
					 xoverlap,wstep,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *changes, *opartials, *wpartials, *from, *from_credit, *to_credit,*to_deriv;
      int xoverlap,wstep,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
    while(counter--) {
      BPTaccum_weights_from_hidden(to_credit,to_deriv,  from, from_credit,
					weights, changes, opartials, wpartials,
					xdim, tess_xdim, xoverlap);
      weights += wstep;
      changes += wstep;
      opartials += wstep;
      wpartials += wstep;
      from += fxstep;
      from_credit += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    to_deriv += xdim;
    from -= fystep;
    from_credit -= fystep;
  }/* end while ydim */
}/* BPTaccum2d_weights_from_hidden */

/* BPTaccum2d_weights_from_hidden_share: For 2d connections (shared weights). */
void BPTaccum2d_weights_from_hidden_share(xdim, ydim, tess_xdim, tess_ydim,
					       weights, changes, opartials, wpartials,
					       from, from_credit, to_credit,to_deriv,
					       xoverlap,fxstep,fystep)
      int xdim, ydim, tess_xdim, tess_ydim;
      float *weights, *changes, *opartials, *wpartials, *from, *from_credit, *to_credit, *to_deriv;
      int xoverlap,fxstep,fystep;
{
  while(ydim--) {
     int counter=tess_ydim;
     float *kernel=weights,*ckernel=changes,*opkernel=opartials,*wpkernel=wpartials;
    while(counter--) {
      BPTaccum_weights_from_hidden_share(to_credit, to_deriv, from, from_credit,
					      kernel, ckernel, opkernel, wpkernel,
					      xdim, tess_xdim, xoverlap);
      kernel += tess_xdim;
      ckernel += tess_xdim;
      opkernel += tess_xdim;
      wpkernel += tess_xdim;
      from += fxstep;
      from_credit += fxstep;
    }/* end while tess_ydim */
    to_credit += xdim;
    to_deriv += xdim;
    from -= fystep;
    from_credit -= fystep;
  }/* end while ydim */
}/* BPTaccum2d_weights_from_hidden_share */

#endif 
