
/**
 **  Aspirin Compiler Grammer
 **/


/**
 ** YACC description of Aspirin, the neural network
 ** specification language parser for the Aspirin/Migraines System.
 **/


%{
#include "aspirin.h"

#define TRUE 1
#define FALSE 0

/* some macros for symbol manipulation */
#define sym_name(string) (((SYM_PTR)fetch_dictionary(string, network->symbol_table))->pname)
#define sym_bbd(string) (((SYM_PTR)fetch_dictionary(string, network->symbol_table))->bbd)
#define sym_dict(string) (((SYM_PTR)fetch_dictionary(string, network->symbol_table))->dict)

/* input buffer */
static unsigned char yytext[256];
static int yyleng;
  
/* holds network structures */
ND_PTR network;

/* holds current black box decriptor */
static  BD_PTR current_bbd = (BD_PTR)NULL;

/* name of current layer */
static char *to_layer;

/* integer code for type of nodes in layer */
static int type;

/* x and y dimensions for a layer */
static int layer_xdim, layer_ydim;

/* x and y dimensions for a tessellation */
static int tess_xdim, tess_ydim;

/* x and y start offsets for a tessellation */
static int tess_xoffset, tess_yoffset;

/* x and y start overlap */
static int tess_xoverlap, tess_yoverlap;

/* string w/name of C user function to initialize weights */
static char *C_init_function = NULL;

/* strings w/names of C user transfer and derivative functions */
static char *C_transfer = NULL;
static char *C_transfer_prime = NULL;

/* int var for order of node feedback */
static int layer_order = 0;

/* init vars for layer thresholds */
static int bias_init;
static float layer_bias;      /* constant */
static char *layer_bias_init; /* C function */

/* used to hold delay info */
DD delay_struct;

/* globals for the parser, used for error messages */
int lineno = 1; 
static char *aspirin_file;




/*****************************************************/
/* make_name:  Make a new string by appending        */
/*             the black box name to the name;       */
/*             a '!' means that this string refers   */
/*             to another black box.                 */
/*****************************************************/
static char *make_name(name, bbd)
     char *name;
     BD_PTR bbd;
{
  extern BD_PTR current_bbd;
  char *black_box_name = bbd->name;
  char *new_string;
  SYM_PTR sym_ptr; 
  
  /* if this is a reference to an extrernal black box
     then just return the black box name.
     */
  if (*name == '!')
    {
      new_string = ++name;
      /* record this name in the current black box
         as being an external input
       */
      insert_dictionary(new_string,
			(GENERIC_PTR)new_string,
			current_bbd->input_bbds); 
    }/* end if */
  /* those symbols prepended with '$' are special and should be returned */
  else if (*name == '$')
    {
      return(name);
    }/* end if */
  /* else this is a symbol in this black box, so
     prepend the black box name.
     */
  else
    {
      new_string = am_alloc_mem(strlen(name) + strlen(black_box_name) + 2); /* add 2 for ':' and \0 */
      sprintf(new_string, "%s:%s", black_box_name, name);
    }/* end else */

  /* record this symbol in global symbol table,
     uses structure like LISP atom...sort of
   */
  sym_ptr = (SYM_PTR)am_alloc_mem(sizeof(SYM_STRUCT));
  sym_ptr->black_box = current_bbd;
  sym_ptr->dict = current_bbd->lookup,
  sym_ptr->pname = name;
  insert_dictionary(new_string,
		    (GENERIC_PTR)sym_ptr,
		    network->symbol_table); 

  return(new_string);
}/* end  make_name */

/*****************************************************/
/* warning                                           */
/*        Print warning message.                     */
/*****************************************************/
static void warning(s, t)
     char *s, *t;
{
  fprintf(stderr, "\n%s: %s", aspirin_file, s);
  if (t != (char *)NULL) fprintf(stderr, "  %s", t);
  
}


/*****************************************************/
/* yyerror                                           */
/*        Called by yacc for syntax errors.          */
/*****************************************************/
int yyerror (s)
     char *s;
{
  char where[50];

  *where = '\0'; /* empty */
  if (yytext != (unsigned char *)NULL)
    {  char *token = am_alloc_mem(yyleng + 1);

       bcopy(yytext, token, yyleng);
       token[yyleng] = '\0'; /* end of string */
       sprintf(where, "(near \"%s\" somewhere around line %d)\n", yytext, lineno);
       am_free_mem(token);
    }/* end if */

  warning(s, where);
  exit(1);
}


/*****************************************************/
/* create_matrix_name                                */
/*        Make a string of the right size.           */
/*****************************************************/
static char *create_matrix_name(to,connection_number)
     char *to;
     int connection_number;
{
  int string_size;
  char *name;

  to = sym_name(to);

  /* because this is used in macro names for cpp and
     some cpp's puke if the names are too big 
  */
  if (strlen(to) > 14) 
    yyerror("\nLayer name is greater than 14 characters! (sorry)");

  string_size = 1; /* for cr */

  if (connection_number < 10) /* 1 character */
    string_size++;
  else
    string_size += (int)AM_CEIL(AM_LOG10((double)(connection_number)));

  string_size += strlen(to);

  name = (char *)am_alloc_mem(string_size);

  sprintf(name, "%s%d",to,connection_number);

  return(name);
  
}/* end create_matrix_name */

/*****************************************************/
/* copy_connection Return a copy of the connection.  */
/*****************************************************/
static CD_PTR copy_connection(connection)
     CD_PTR connection;
{
  CD_PTR new_connection;

  new_connection = (CD_PTR)am_alloc_mem(sizeof(CD));
  bcopy((char *)connection, (char *)new_connection, sizeof(CD));
  return(new_connection);
  
}/* end copy_connection */

/*****************************************************/
/* create_connection                                 */
/*        Create a connection structure              */
/*****************************************************/
static CD_PTR create_connection(from, to, type, shared,
			 xoffset, xrange, xoverlap,
			 yoffset, yrange, yoverlap,
			 init_function, delay, order)

     char *from, *to;
     int type, shared;
     int xoffset, yoffset; /* used for tessellation */
     int xrange, xoverlap; /* used for tessellation functions */
     int yrange, yoverlap; /* used for tessellation functions */
     int init_function;    /* code for type of weight initialization */
     DD_PTR delay;         /* connected at this time delay */
     int order;            /* the order of the connection */
{
  extern BD_PTR current_bbd;
  CD_PTR new_connection;

  /*** ERROR CHECKS ***/
  
  if ((xrange < xoverlap) || (xoverlap < 0))
    {
      yyerror("Your tessellation x-overlap is greater than your tessellation.");
    }/* end if */

  if ((yrange < yoverlap) || (xoverlap < 0))
    {
      yyerror("Your tessellation y-overlap is greater than your tessellation.");
    }/* end if */

  /* record */
  current_bbd->n_cds += 1;

  /* create a new connection structure */
  new_connection = (CD_PTR)am_alloc_mem(sizeof(CD));
  new_connection->from = from;
  new_connection->to = to;
  if (delay->n) network->temporal = TRUE;  /* there is state info */
  if (delay->start > 0 || delay->end > 0 )
    yyerror("Your delay time cannot be a positive integer!");
  new_connection->delay.start = -1 * delay->start;
  new_connection->delay.end = -1 * delay->end;
  new_connection->delay.n = -1 * delay->n;
  new_connection->delay.next = (DD_PTR)NULL;
  new_connection->lookup = create_dictionary();
  new_connection->type = type; 
  new_connection->shared = shared;
  new_connection->xoffset = xoffset;
  new_connection->xrange = xrange;
  new_connection->xoverlap = xoverlap;
  new_connection->yoffset = yoffset;
  new_connection->yrange = yrange;
  new_connection->yoverlap = yoverlap;
  new_connection->next = (CD_PTR)NULL;
  new_connection->array_name = create_matrix_name(to,current_bbd->n_cds);
  new_connection->size = 0; /* reset */
  new_connection->first_order_size = 0; /* reset */
  new_connection->second_order_size = 0; /* reset */

  /* order of connection */
  if (order == 0 || order > 1)
    yyerror("Only 1st order connections are supported!");
  new_connection->order = order;

  /* function to initialize weights */
  new_connection->init_function = init_function;

  if (init_function == C_INIT)
    {
      new_connection->C_init = C_init_function; /* set the name of user function */
    }/* end else if */
  else if (init_function == CONSTANT_INIT) /* not presently used */
    {
      new_connection->constant = 0.0;
    }/* end else if */      

  return(new_connection);
  
} /* end create_connection */

/*****************************************************/
/* insert_TDNN_connection:                           */
/*        Inserts connections with unique [start,end]*/
/*****************************************************/
static DD_PTR insert_TDNN_connection(list, delay)
     DD_PTR list, delay;
{
  DD_PTR ptr = list;

  if (delay->end == delay->start) return(list);       /* is  a TDNN style ? */

  if (list == (DD_PTR)NULL) return(delay); /* first one */

  while(1) {

    if (ptr->start == delay->start &&  /* not unique, no insert */
	ptr->end == delay->end)
      break;

    if (ptr->next == (DD_PTR)NULL) {
      ptr->next = delay;
      break;
    }
      
    ptr = ptr->next;
  }
  
  return(list);
}

/*****************************************************/
/* create_layer                                      */
/*        Create a layer structure                   */
/*****************************************************/
static LD_PTR create_layer (name, size, xdim, ydim, input_connections)
     char *name;
     int size, xdim, ydim;
     CD_PTR input_connections;
{
  extern BD_PTR current_bbd;
  extern int type;
  LD_PTR new_layer;
  DICTIONARYPTR d;


  new_layer = (LD_PTR)am_alloc_mem(sizeof(LD));
  new_layer->name = name;
  new_layer->type = type; /* this was set in a production */
  if (size <= 0) yyerror("\nThe number of nodes is <= 0.");
  new_layer->n_nodes = size;
  new_layer->xdim = xdim;
  new_layer->ydim = ydim;

  if (layer_order) network->temporal = TRUE; /* time varying */
  new_layer->last_delay = layer_order;
  new_layer->layer_order = layer_order;
  /* little hack...probably doesn't belong here */
  if (layer_order < 0 || layer_order > 2)               
    yyerror("\nLayer order must be between 0 and 2.");
  layer_order = 0; /* reset */

  new_layer->TDNN_connections = (DD_PTR)NULL;
  new_layer->bbd = current_bbd; /* back pointer */
  new_layer->inputs_from = input_connections;
  new_layer->next = (LD_PTR)NULL;
  new_layer->previous = (LD_PTR)NULL;

  new_layer->init_function = bias_init;
  new_layer->C_transfer = C_transfer;
  new_layer->C_transfer_prime = C_transfer_prime;
  C_transfer_prime = C_transfer = (char *)NULL;   /* reset */
  if (bias_init == CONSTANT_INIT) new_layer->bias = layer_bias;
  else if (bias_init == C_INIT) new_layer->C_bias_init = layer_bias_init;

  new_layer->lookup = create_dictionary();
  /* add the new nodes to the total */
  current_bbd->n_nodes += size;
  /* record a new layer */
  new_layer->number = current_bbd->n_layers;
  current_bbd->n_layers += 1;
  d = current_bbd->lookup;
  if(fetch_dictionary(name, d) != (GENERIC_PTR)NULL)
    yyerror("\nRedefined layer\n"); 
  insert_dictionary(name, (GENERIC_PTR)new_layer, d); 
  /* this is the current to_layer */
  return(new_layer);
  
}/* end create_layer */

/*****************************************************/
/* create_black_box                                  */
/*             create a black box descriptor and     */
/*             store it in the network dictionary    */
/*****************************************************/
static void create_black_box(name)
     char *name;
{
  extern ND_PTR network;
  extern BD_PTR current_bbd;
  DICTIONARYPTR d;
  BD_PTR bbd;

  d = network->lookup; 
  bbd = (BD_PTR)am_alloc_mem(sizeof(BD));
  if(fetch_dictionary(name, d) != (GENERIC_PTR)NULL)
    yyerror("\nRedefined black box\n");
  insert_dictionary(name, (GENERIC_PTR)bbd, d);
  current_bbd = bbd;
  bbd->name = name;
  /* assign an index to black box */
  bbd->number = network->n_bbds;
  network->n_bbds++;
  bbd->lookup = create_dictionary();
  bbd->input_bbds = create_dictionary();
  bbd->output_bbds = create_dictionary();
  bbd->n_nodes = 0;
  bbd->n_layers = 0;
  bbd->n_cds = 0;
  bbd->n_inputs = 0;
  bbd->clear_delay = 0;
  bbd->update_interval = 1; /* default */
  bbd->last_input_delay = 0;
  bbd->TDNN_connections = (DD_PTR)NULL;
  bbd->n_outputs = 0;
  bbd->dynamic = BBDYNAMIC;
  bbd->ar_learning = TRUE;
  bbd->ff_learning = TRUE;
  bbd->input_connections = (CD_PTR)NULL;
  bbd->output_layer_name = (char *)NULL;
  bbd->output_layer = (LD_PTR)NULL;
  bbd->input_filter = (char *)NULL;
  bbd->output_filter = (char *)NULL;
  bbd->error_function = (char *)NULL;
  bbd->efferent = 1; /* assume no one connects until someone does */
  bbd->layers = (LD_PTR)NULL;
  bbd->network = network; /* back pointer to root */
  bbd->n_bb_inputs = 0;
  bbd->n_bb_outputs = 0;

}/* end create_black_box */

/*****************************************************/
/* Create_network                                    */
/*        Create a network structure and store       */
/*        into a global variable.                    */
/*****************************************************/
static void create_network()
{
  extern char *aspirin_file;
  extern ND_PTR network;

  network = (ND_PTR)am_alloc_mem(sizeof(ND));

  network->name = aspirin_file; /* name of the aspirin file */

  network->temporal = FALSE; /* assume no feedback */

  network->line_search = NO_LINE_SEARCH; /* assume no line search */
  network->line_search_timeout = -1;
  network->line_search_update = -1;

  network->conjugate_gradient = NO_CONJ_GRAD; /* assume no conjugate gradient */

  network->n_bbds = 0; /* how many black box descriptiors */

  /* symbol tables */
  network->lookup = create_dictionary();
  network->symbol_table = create_dictionary(); /* maps symbol => dictionary */
    
}/* end create_network */

/*****************************************************/
/* update_connection: Error check, and keep sure all data is in place. */
/*****************************************************/
static void update_connection(connection, to_layer, from_layer, bbd)
     CD_PTR connection;
     LD_PTR to_layer, from_layer;
     BD_PTR bbd;
{
  char error_string[50];
  int xresidue, xretina_size, yresidue, yretina_size;
  int to_size, from_size;
  int to_xdim, from_xdim, to_ydim, from_ydim;
  
  to_size = to_layer->n_nodes;
  to_xdim = to_layer->xdim;
  to_ydim = to_layer->ydim;
  
  if (from_layer == (LD_PTR)NULL) { /* must be from the bb's input */
    from_xdim = bbd->inputs_xdim;
    from_ydim = bbd->inputs_ydim;
    from_size = bbd->n_inputs;
  } else {
    from_xdim = from_layer->xdim;
    from_ydim = from_layer->ydim;
    from_size = from_layer->n_nodes;
  }/* end if */
  
  
  /* fix the type (there is an ambiguity for 1D) */
  if (connection->type == TESS_1D_CONNECTION_TYPE &&
      from_ydim != 1)
    connection->type = TESS_2D_CONNECTION_TYPE; 
  
  connection->to_layer = to_layer;
  connection->from_layer = from_layer;
  connection->bbd = bbd;
  
  /* Check the connection structures */
  switch (connection->type)    {
  case NXM_CONNECTION_TYPE : {
    /* need to set the xrange, yrange here because
       this is the first time you have to info to do so.
       */
    connection->xrange = from_size;
    connection->yrange = 1;
    connection->xoverlap = from_size;
    connection->yoverlap = 0;
    break;
  }/* end case */
  case TESS_1D_CONNECTION_TYPE : {
    int x_unconnected = 0;
    
    /* check size */
    if (connection->xrange > from_xdim)
      {
	fprintf(stderr,
		"\nError: Connection from %s to %s has too large tessellation.\n",
		connection->to,
		connection->from);
	exit(1);
      }/* end if */
    /* check the mapping of receptive fields */
    xretina_size = ((to_xdim *
		     (connection->xrange -
		      connection->xoverlap))
		    + connection->xoverlap);
    xresidue = xretina_size - (from_xdim - connection->xoffset);
    if (xresidue < 0)
      {
	/* in this case the to_nodes do NOT connect to every 
	   from_node, but all are connected.
	   */
	printf("\nWarning: Connections of %s do not fully cover %s in the x direction.",
	       connection->to,
	       connection->from);
	x_unconnected = (-1 * xresidue);
	printf("\n         %s has %d nodes unconnected.",
	       connection->from,
	       x_unconnected);
	if (connection->xoffset > 0)
	  printf(" (plus %d from offset)", connection->xoffset);
      }/* end if */
    else if (xresidue > 0)
      {
	/* in this case the to_nodes may not connect
	   to a from node*/
	printf("\nError: Incomplete connections to %s from %s.\n",
	       connection->to,
	       connection->from);
	if ((xresidue / connection->xrange) > 0)
	  printf("\n      %d nodes in %s are not connected to %s in the x direction.\n",
		 (xresidue / connection->xrange),
		 connection->to,
		 connection->from);
	exit(1);
      }/* end else */
    /* record for future use */
    connection->xresidue = x_unconnected;
    
    break;
  }/* end case */
  case TESS_2D_CONNECTION_TYPE : {
    int xrange = connection->xrange;
    int yrange = connection->yrange;
    int xoverlap = connection->xoverlap;
    int yoverlap = connection->yoverlap;
    int x_unconnected = 0, y_unconnected = 0;
    
    /* check size */
    if (xrange > from_xdim) {
      fprintf(stderr,
	      "\nError: Connection from %s to %s has too large x tessellation.\n",
	      connection->to,
	      connection->from);
      exit(1);
    }/* end if */
    if (yrange > from_ydim) {
      fprintf(stderr,
	      "\nError: Connection from %s to %s has too large y tessellation.\n",
	      connection->to,
	      connection->from);
      exit(1);
    }/* end if */
    
    /* check the mapping of receptive fields in x dimension */
    xretina_size = ((to_xdim * (xrange - xoverlap)) + xoverlap);
    xresidue = xretina_size - (from_xdim - connection->xoffset);
    if (xresidue < 0)  {
      /* in this case the to_nodes do NOT connect to every 
	 from_node, but all are connected.
	 */
      printf("\nWarning: Connections of %s do not fully cover %s in the x dimension.",
	     connection->to,
	     connection->from);
      x_unconnected = (-1 * xresidue);
      printf("\n\t%d under shoot in the x dimension.",
	     x_unconnected);
      if (connection->xoffset > 0)
	printf(" (plus %d from offset)", connection->xoffset);
    }/* end if */
    else if (xresidue > 0) {
      /* in this case the to_nodes may not connect
	 to a from node*/
      printf("\nError: Incomplete connections to %s from %s.\n",
	     connection->to,
	     connection->from);
      if ((xresidue / connection->xrange) > 0)
	printf("\n      %d over shoot in x dimension.",
	       (xresidue / connection->xrange));
      exit(1);
    }/* end else */
    /* record for future use */
    connection->xresidue = x_unconnected;
    
    /* check the mapping of receptive fields in y dimension */
    yretina_size = ((to_ydim * (yrange - yoverlap)) + yoverlap);
    yresidue = yretina_size - (from_ydim - connection->yoffset);
    if (yresidue < 0)  {
      /* in this case the to_nodes do NOT connect to every 
	 from_node, but all are connected.
	 */
      printf("\nWarning: Connections of %s do not fully cover %s in the y dimension.",
	     connection->to,
	     connection->from);
      y_unconnected = (-1 * yresidue);
      printf("\n\t%d under shoot in the y dimension.",
	     y_unconnected);
      if (connection->yoffset > 0)
	printf(" (plus %d from offset)", connection->yoffset);
    }/* end if */
    else if (yresidue > 0) {
      /* in this case the to_nodes may not connect
	 to a from node*/
      printf("\nError: Incomplete connections to %s from %s.\n",
	     connection->to,
	     connection->from);
      if ((yresidue / connection->yrange) > 0)
	printf("\n      %d over shoot in y dimension.",
	       (yresidue / connection->yrange));
      exit(1);
    }/* end else */
    /* record for future use */
    connection->yresidue = y_unconnected;
    
    break;
  }/* end case */
    default :	{
      sprintf(error_string,
	      "Unknown connection type from %s to %s",
	      connection->from,
	      connection->to);
      warning(error_string, (char *)NULL);
    }/* end default */
  } /* end switch */  
  
}/* end check connection */

/*****************************************************/
/* update_io_dictionaries: Replace the string with
                            the named structure pointer.
			    Update output dictionary.
*/
/*****************************************************/
static void update_io_dictionaries(bbd_name)
     char *bbd_name;
{
  extern ND_PTR network;
  extern BD_PTR current_bbd;
  BD_PTR input_bbd;

  input_bbd = (BD_PTR)fetch_dictionary(bbd_name, network->lookup);

  /* inputs */
  insert_dictionary(bbd_name,
		    (GENERIC_PTR)input_bbd,
		    current_bbd->input_bbds);
  /* outputs */
  insert_dictionary(current_bbd->name,
		    (GENERIC_PTR)current_bbd,
		    input_bbd->output_bbds);

}/* end update_io_dictionaries */

/*****************************************************/
/* update_black_box:  Fill the slots, back pointers, etc. */
/*****************************************************/
static void update_black_box(bbd)
     BD_PTR bbd;
{
  extern ND_PTR network;
  DICTIONARYPTR d = bbd->lookup; /* holds the layers */
  LD_PTR out_layer, to_layer, from_layer; 
  CD_PTR input_connection, new_connection;
  BD_PTR from_bbd;
  int to_size;
  char error_string[64];

  /* add the outputs_to for every layer */
  to_layer = bbd->layers;
  while (to_layer != (LD_PTR)NULL)
    {
      /* for every input find the connected layer and update outputs */
      input_connection = to_layer->inputs_from;
      to_size = to_layer->n_nodes;
      while (input_connection != (CD_PTR)NULL)
	{
	  /* is from layer is $INPUTS?, then push connection into
	     input_connections.
	   */
	  if (strcmp(input_connection->from, "$INPUTS") == 0)
	    {
	      /* you cannot connect to $INPUTS if they don't exist */
	      if (bbd->n_inputs == 0)
		{
		  sprintf(error_string,
			  "\nLayer %s in black box %s needs InputSize-> declared.\n",
			  to_layer->name,
			  bbd->name);
		  fprintf(stderr, error_string);
		  fflush(stderr);
		  exit(1);
		}/* end if */

	      /* check connection structure */
	      update_connection(input_connection,
				to_layer,
				(LD_PTR)NULL,
				bbd);
	      new_connection = copy_connection(input_connection);
	      new_connection->next = bbd->input_connections;
	      bbd->input_connections = new_connection;
	      /* update the delay */
	      if (new_connection->delay.end > bbd->last_input_delay) {
		bbd->last_input_delay = new_connection->delay.end;
	      }
	      bbd->TDNN_connections = insert_TDNN_connection(bbd->TDNN_connections, &(new_connection->delay));

	    }/* end if */
	  else
	    {
	      from_layer = (LD_PTR)fetch_dictionary(input_connection->from, d);
	      /* if we cannot find the layer in this black box then
		 look for a black box with that name */
	      if (from_layer == (LD_PTR)NULL)
		 {
		   /* see if connection is to another black box */
		   from_bbd = (BD_PTR)fetch_dictionary(input_connection->from,
						       network->lookup);
 		   if (from_bbd == (BD_PTR)NULL)
		     {
		       sprintf(error_string,
			       "\nUnknown layer or black box name: %s\n",
			       input_connection->from);
		       fprintf(stderr, error_string);
		       exit(1);
		     }/* end if */
		   else
		     {
		       /* mark this black box as internal to the system */
		       from_bbd->efferent = 0;
		       /* inc inter bb connection counters */
		       bbd->n_bb_inputs++;
		       from_bbd->n_bb_outputs++;
		       /* update the layer outputs */
		       if (from_bbd->output_layer_name == (char *)NULL) 
			 {
			   fprintf(stderr,
				  "\nNo output layer declared for black box.");
			   exit(1);
			 }/* end if */
		       from_layer =
			 ((LD_PTR)
			  fetch_dictionary(from_bbd->output_layer_name,
					   from_bbd->lookup));
		       /* check connection structure */
		       update_connection(input_connection,
					 to_layer,
					 from_layer,
					 bbd);
                       new_connection = copy_connection(input_connection);
		       new_connection->next = from_layer->outputs_to;
		       from_layer->outputs_to = new_connection; 
		       /* update the delay */
		       if (new_connection->delay.end > from_layer->last_delay) {
			 from_layer->last_delay = new_connection->delay.end;
		       }
		       from_layer->TDNN_connections = insert_TDNN_connection(from_layer->TDNN_connections,
									     &(new_connection->delay));
		     }/* end else */
		 }/* end if */
	      else
		{
		  /* check connection structure */
		  update_connection(input_connection,
				    to_layer,
				    from_layer,
				    bbd);
		  new_connection = copy_connection(input_connection);
		  new_connection->next = from_layer->outputs_to;
		  from_layer->outputs_to = new_connection;
		  /* update the delay */
		  if (new_connection->delay.end > from_layer->last_delay) {
		    from_layer->last_delay = new_connection->delay.end;
		  }
		  from_layer->TDNN_connections = insert_TDNN_connection(from_layer->TDNN_connections,
									&(new_connection->delay));
		}/* end else */
	    }/* end else */

	  /* next */
	  input_connection = input_connection->next;
	}/* end while */
      /* next */
      to_layer = to_layer->next;
    }/* end while */

  /* add the output layer */
  if (bbd->output_layer_name == (char *)NULL) 
    {
      fprintf(stderr, "\nNo output layer declared for black box.");
      exit(1);
    }/* end if */
  out_layer = (LD_PTR)fetch_dictionary(bbd->output_layer_name, d);
  if (out_layer == (LD_PTR)NULL)
    {
      fprintf(stderr, "\nUnknown output layer declared for black box.");
      exit(1);
    }/* end if */
  bbd->output_layer = out_layer;
  bbd->n_outputs = out_layer->n_nodes;

  /* error check */
  if(bbd->n_outputs <= 0)
    {
      fprintf(stderr, "Number of outputs to black box <= 0");
      exit(1);
    }/* end if */
  
  /* update the dictionary holding the input
     black box names with an actual structure
     pointer.
   */
  current_bbd = bbd;
  map_dictionary(update_io_dictionaries, bbd->input_bbds);

}/* end update_black_box */

/*****************************************************/
/* update     Link all the data structures.
 */
/*****************************************************/
static void update()
{
  extern ND_PTR network;

  /* update the connection structures */
  map_dictionary(update_black_box, network->lookup);
  
}/* end update */

/*****************************************************/
/* mark_static_black_boxes Change a BBSTATIC -> BBSTATIC_DEPEND.
*/
/*****************************************************/
static mark_static_black_boxes(bbd)
     BD_PTR bbd;
{
  if ( bbd->dynamic == BBSTATIC)  bbd->dynamic = BBSTATIC_DEPEND;
}/* end mark_static_black_boxes */

/*****************************************************/
/* depend_black_box  for every BBDYNAMIC or BBSTATIC_DEPEND black box,
                     if it outputs to another black box,
                     then check all black boxes that
                     depend on this black box and
                     mark static bbs as BBSTATIC_DEPEND. 
*/
/*****************************************************/
static void depend_black_box(bbd)
     BD_PTR bbd;
{
  if ( bbd->dynamic == BBSTATIC) return;

  if ( bbd->n_bb_outputs ) { /* if I output to another bb */

    map_dictionary(mark_static_black_boxes, bbd->output_bbds);

  }/* end if */

}/* end depend_black_box */


/*****************************************************/
/* check_dependencies   Make sure static bbs are
                        marked if there is a dependence.
 */
/*****************************************************/
static void check_dependencies()
{
  extern ND_PTR network;
  int n_bbds = network->n_bbds;

  /* looping for n_bbds guarantees all dependencies
     have been checked (brush fire search).
   */
  while(n_bbds--)
    map_dictionary(depend_black_box, network->lookup);
  
}/* end check_dependencies */


%}

/*
 * Define parsers stack to be numbers, strings, connection_descriptors or layer_descriptors
 */


%union {
	int i;    /* integer                */
	float f;  /* float                  */
	char *s;  /* string                 */
	DD_PTR d; /* delay struct  ptr      */
	CD_PTR c; /* connection struct ptr  */
	LD_PTR l; /* layer struct ptr       */
}

/*
 * Enumerate all tokens for yylex.
 */

/* This is used to reset the line counter */
%token NEWLINE

/* The following are the keyword tokens */
%token  INPUT_FIELDS 
%token  INPUT_SIZE
%token  COMPONENTS
%token  OUTPUT_LAYER
%token  INPUTS_FROM
%token  STATIC
%token  CLRDELAYS
%token  UPDATE_INTERVAL
%token  DISABLE_FEEDBACK_LEARNING
%token  DISABLE_FEEDFORWARD_LEARNING
%token  INPUT_FILTER
%token  OUTPUT_FILTER
%token  ERROR_FUNCTION

%token  BBD                        /* 'black-box-definition' */

%token  ORDER                

%token  PDPNODE1                   /* type of node in network */
%token  PDPNODE2                   /* type of node in network */
%token  PDPNODE3                   /* type of node in network */
%token  LINEAR_NODE                /* type of node in network */
%token  QUAD_NODE                  /* type of node in network */
%token  USER_NODE                  /* type of node in network */

%token  CONJ                      /* Conjuntive for connection description */
%token  DIMENSION                 /* 'x' */

%token  LOAD_DATA                 /* LoadData into (<black box>) from (<black box>) in (<filename>) */
%token  INTO
%token  FROM
%token  IN

%token  LINE_SRCH                 /* LineSearch */
%token  LINE_SRCH_VERBOSE         /* LineSearchVerbose */
%token  CONJ_GRAD                 /* ConjugateGradient */

%token <f> NUMBER                 /* real */
%token <s> NAME                   /* string */

%token QUOTE                      /* double quotation mark */
%token COMMA                      /* , */
%token LCP                        /* left curly paren */
%token RCP                        /* right curly paren */
%token RP                         /* right paren */
%token LP                         /* left paren */
%token LB                         /* left bracket */
%token RB                         /* right bracket */

%token TESSELLATION               /* tessellation */
%token SHARED                     /* shared tessellation */
%token WITH                       /* conj for tessellations */
%token USING                      /* conj for tessellations */
%token XOVERLAP                   /* token indicates xoverlap for tess */
%token YOVERLAP                   /* token indicates yoverlap for tess */
%token XOFFSET                    /* token indicates where tess begins */
%token YOFFSET                    /* token indicates where tess begins */

%token AT                         /* @ */
%token TIME                       /* time of delay */

%token INIT                       /* initialize weights with function */

%token BIAS                       /* initialize bias */
%token EQUALS                     /* = */


/*
 * Sub-expression (non-terminal) tokens.
 */

%type <l> layer
%type <l> layers
%type <s> component
%type <c> args
%type <c> arg
%type <i> shared
%type <i> layer_dimensions
%type <c> connection_specs
%type <c> connection_spec
%type <i> initialize
%type <f> real
%type <i> integer
%type <i> connection_order
%type <i> node_order
%type <d> delay

%%


/*****************************************************/
/* Root production
/*****************************************************/
aspirin: aspirin_exp
         | aspirin aspirin_exp
         ;
/* these are the currently supported aspirin expressions */
aspirin_exp: bbdefs        /* black box definitions */
              | load_bbs   /* load data from file */
              | newlines   /* cpp statements */
              | LINE_SRCH integer integer {
                 network->line_search = LINE_SEARCH;
                 network->line_search_update = $2;
                 network->line_search_timeout = $3;
              }
              | LINE_SRCH_VERBOSE integer integer {
                 network->line_search = LINE_SEARCH_VERBOSE;
                 network->line_search_update = $2;
                 network->line_search_timeout = $3;
              }
              | CONJ_GRAD {
                network->conjugate_gradient = CONJ_GRAD_SIMPLE;
              }
              ;

/*****************************************************/
/* load_bbs   Load a black box at init time with     */
/*            data from a dump_file.                 */
/*****************************************************/
load_bbs: load_bb
          | load_bbs load_bb
          ;

load_bb: LOAD_DATA INTO NAME FROM NAME IN NAME  {BD_PTR black_box;
  	                           black_box = (BD_PTR)fetch_dictionary($3, network->lookup);
				   if (black_box == (BD_PTR)NULL)
				     {
				       yyerror("Unknown Black Box name.");
				     }/* end if */
				   else
				     {
				       black_box->load_key = $5;
				       black_box->datafile = $7;
				     }/* end else */
			        }
         ;

/*****************************************************/
/* Update the line counter and current file          */
/* (used to parse cpp output)                        */
/*****************************************************/
newlines: newline
          | newlines newline
          ;

newline: NEWLINE integer  NAME  { lineno = $2;
 				  aspirin_file = $3;
			        } 
         | NEWLINE integer  NAME integer { lineno = $2;
 				           aspirin_file = $3;
			                 }
         | NEWLINE NAME integer NAME { lineno = $3;
                                       aspirin_file = $4;
                                     }
         | NEWLINE integer { lineno = $2; }
         | NEWLINE NAME NAME {}
         ;

/*****************************************************/
/* Black Box Definition Language Specification       */
/*****************************************************/


/* A file may contain number of  black box definitions.
   Check to make sure required fields are filled.
*/
bbdefs:  bbdef
         | bbdefs bbdef
	 ;

bbdef:  BBD bbname LCP fields RCP

/* Make a bbd structure and store into global dictionary */
bbname: NAME  { create_black_box($1); }

/* List of fields */
fields: field
        | fields field 
        ;

/* fields */
field:     INPUT_SIZE layer_dimensions {
            if (current_bbd->n_inputs == 0)
	      {
		/* black box */
		current_bbd->n_inputs = $2;
		current_bbd->inputs_xdim = layer_xdim;
		current_bbd->inputs_ydim = layer_ydim;
	      }/* end if */
	    else
	      {
		/* already specified # of inputs */
		yyerror("Sorry, number of inputs already specified. ");
	      }/* end else */
	   }
	   | COMPONENTS  LCP layers RCP { current_bbd->layers = $3; }
           | STATIC { current_bbd->dynamic = BBSTATIC; }
           | DISABLE_FEEDBACK_LEARNING { current_bbd->ar_learning = FALSE; }
           | DISABLE_FEEDFORWARD_LEARNING { current_bbd->ff_learning = FALSE; }
           | CLRDELAYS integer { 
	        if ($2 < 0) {
		  yyerror("The argument to ClearDelays-> must be a positive integer ");
		}
	        current_bbd->clear_delay = $2; 
	      }
	   | OUTPUT_LAYER  NAME { current_bbd->output_layer_name =
				    make_name($2, current_bbd); }
           | UPDATE_INTERVAL integer { 
	        if ($2 <= 0) {
		  yyerror("The argument to UpdateInterval-> must be a positive integer ");
		}
                current_bbd->update_interval = $2; 
             }
           | INPUT_FILTER NAME { current_bbd->input_filter = $2; }
           | OUTPUT_FILTER NAME { current_bbd->output_filter = $2; }
           | ERROR_FUNCTION NAME { current_bbd->error_function = $2; }
	   ;


/* Components of a black box.    */
 layers: layer  { $$ = $1; }
         | layers layer { ($2)->next = $1; /* push into list (double links) */
			  ($1)->previous = $2;
			  $$ = $2;
			}
	 ;

layer: component layer_dimensions option LCP args RCP { 
                                                      $$ = create_layer($1,   /* name */
							   	        $2,   /* size */
								        layer_xdim, /* x size */
								        layer_ydim, /* y size */
								        $5);  /* connections */
	    			                    }
	;

/* component: <node type> <name> */
component: PDPNODE1 NAME       { /* set global for access by
			           connection_specs
		      	         */
                                 to_layer =
			            make_name($2, current_bbd);
			         /* set for use when creating a new layer */
			         type = PDP_LAYER_TYPE1;

			         $$ = to_layer;
			       }
          | PDPNODE2 NAME       { /* set global for access by
			           connection_specs
		      	         */
                                 to_layer =
			            make_name($2, current_bbd);
			         /* set for use when creating a new layer */
			         type = PDP_LAYER_TYPE2;

			         $$ = to_layer;
			       }
          | PDPNODE3 NAME       { /* set global for access by
			           connection_specs
		      	         */
                                 to_layer =
			            make_name($2, current_bbd);
			         /* set for use when creating a new layer */
			         type = PDP_LAYER_TYPE3;

			         $$ = to_layer;
			       }
           | LINEAR_NODE NAME  { /* set global for access by
				    connection_specs
				  */
	                         to_layer =
				   make_name($2, current_bbd);
				 /* set for use when creating
						   a new layer
						   */
				 type = LINEAR_LAYER_TYPE;

				 $$ = to_layer; 
			       }
           | QUAD_NODE NAME  { /* set global for access by
				    connection_specs
				  */
	                         to_layer =
				   make_name($2, current_bbd);
				 /* set for use when creating
						   a new layer
						   */
				 type = QUAD_LAYER_TYPE;

				 $$ = to_layer; 
			       }
           | USER_NODE NAME NAME NAME { /* set global for access by
				            connection_specs
				          */
	                         to_layer =
				   make_name($4, current_bbd);
				 /* set for use when creating
						   a new layer
						   */
				 type = USER_LAYER_TYPE;
				 /* user must supply the name
				    of 2 C functions, a tansfer
				    function and the derivative
				    of the transfer function.
				  */
				 C_transfer = $2;
				 C_transfer_prime = $3;

				 $$ = to_layer; 
			       }
	   | NAME              { yyerror("Unknown component type."); }
	   ;


/* args Right now only one argument is allowed */
args: arg   { $$ = $1; }
       ;

arg: INPUTS_FROM connection_specs { $$ = $2; }
     ;

/* connection_spec is a list of components to connect to 
 */
connection_specs: connection_spec                          { $$ = $1; }
                  | connection_specs CONJ connection_spec  {
		                                             ($3)->next = $1;
		                                              $$ = $3;
		                                           }
		  ;

/* Create a connection of the appropiate type. The default is a full
   connection. 1d and 2d tessellations are also allowed.
 */
connection_spec: NAME connection_order delay initialize {
                 $$ = create_connection(make_name($1, current_bbd),
					to_layer,
					NXM_CONNECTION_TYPE, /* default */
					0, /* default */
					0, /* default */
					0, /* default */
					0, /* default */
					0, /* default */
					0, /* default */
					0, /* default */
				        $4,
					$3,/* delay */
					$2 /* order */
					); 
	         }
                  /* 1d and 2d Tessellation */
                 | NAME connection_order delay LP WITH shared tess_dimensions TESSELLATION tess_overlap tess_offsets initialize RP {
                 $$ = create_connection(make_name($1, current_bbd),
					to_layer,
					/* type of tessellation */
					((tess_ydim == 1) && (layer_ydim == 1)) 
					?TESS_1D_CONNECTION_TYPE
					:TESS_2D_CONNECTION_TYPE,
					$6,                /* equals 1 if shared */
					tess_xoffset,      /* where begin in x */
					tess_xdim,         /* xrange of tessellation */
					tess_xoverlap,     /* xoverlap */
					tess_yoffset,      /* where to begin in y*/
					tess_ydim,         /* yrange */
					tess_yoverlap,     /* yoverlap */
					$11,               /* type of initialize function for weights */
					$3,                /* delay */
					$2                 /* order */
					);
	         }
		 ;

/* optional order */
connection_order :                              { $$ = 1; /* default 1st order */ }
                 | ORDER EQUALS integer         { $$ = $3; }
                 ;

/* optional delay */
delay :                              { 
                                       delay_struct.n = 0;
                                       delay_struct.start = 0;
				       delay_struct.end = 0;
				       $$ = &delay_struct; /* default */}
       | AT LP TIME EQUALS integer RP { 
                                       delay_struct.n = 1;
                                       delay_struct.start = $5;
				       delay_struct.end = $5;
                                       $$ = &delay_struct;}
       | AT LP TIME EQUALS LB integer COMMA integer RB RP {
                                       delay_struct.start = $6;
				       delay_struct.end = $8;
                                       delay_struct.n = delay_struct.end - delay_struct.start - 1;
                                       $$ = &delay_struct;}
       ;

/* optional shared tessellation (defaults to 0) */
shared :                             { $$ = 0; /* default */}
       |
       SHARED                        { $$ = 1; }
       ;

/* optional speficfied init function for the weights */
initialize:                                 { $$ = RANDOM_INIT; } /* default random init */                           
          |
          INIT WITH NAME { C_init_function = $3; $$ = C_INIT; }
          ;




/* optional dimensions (defaults to [1]). 2 DIMS MAX(for now) */
layer_dimensions: integer             {
                                        layer_xdim = $1; /* set global to record */
			                layer_ydim = 1; /* set global to record */
                                        $$ = $1;
			              }
            |
            LB integer  RB            {  layer_xdim = $2; /* set global to record */
			                 layer_ydim = 1; /* set global to record */
                                         $$ = $2;
			              }
            | LB integer DIMENSION integer RB {  /* set global to record */
                                                 layer_xdim = $2;
                                                 /* set global to record */
				                 layer_ydim = $4; 
				                 $$ = layer_xdim * layer_ydim;
				              }
	    ;            

/* optional dimensions (defaults to [1]). 2 DIMS MAX(for now) */
tess_dimensions: integer                  { tess_xdim = $1; /* set global to record */
			                    tess_ydim = 1; /* set global to record */
			                  }
            |
            LB integer  RB                { tess_xdim = $2; /* set global to record */
			                    tess_ydim = 1; /* set global to record */
			                  }
            | LB integer DIMENSION integer RB {  /* set global to record */
	                                         tess_xdim = $2;
					         /* set global to record */
				                 tess_ydim = $4;
				              }
	    ;            
/* optional offsets for tessellation */
tess_overlap:  /* default to nothing */         { tess_xoverlap = 0;
					          tess_yoverlap = 0;
					        }
            |
            USING integer XOVERLAP CONJ integer YOVERLAP { tess_xoverlap = $2;
					                   tess_yoverlap = $5;
					                 }
            |
            USING integer XOVERLAP              { tess_xoverlap = $2;
					          tess_yoverlap = 0;
			  		        }
            |
            USING integer YOVERLAP              { tess_xoverlap = 0;
					          tess_yoverlap = $2;
					        }
            ;

tess_offsets:  /* default to nothing */        { tess_xoffset = 0;
					         tess_yoffset = 0;
					       }
            |
            WITH integer XOFFSET CONJ integer YOFFSET  { tess_xoffset = $2;
					                 tess_yoffset = $5;
					               }
            |
            WITH integer XOFFSET               { tess_xoffset = $2;
					         tess_yoffset = 0;
					       }
            |
            WITH integer YOFFSET               { tess_xoffset = 0;
					         tess_yoffset = $2;
					       }
            ;
      
/* node options */        
option:                   { 
                           bias_init = RANDOM_INIT;  /* default to random            */
		          }
      |
      BIAS EQUALS real    {
                           bias_init = CONSTANT_INIT;
                           layer_bias = $3; 
                          }
      |
      BIAS EQUALS NAME    {
                           bias_init = C_INIT;
                           layer_bias_init = $3; 
                          }
     |
      node_order          {
                           layer_order = $1;
			  }
     ;

node_order :                                    { $$ = 0; /* default 0th order */ }
                 | ORDER EQUALS integer         { $$ = $3; }
                 ;
            

/* numbers */
real: NUMBER { $$ = $1; }
     ;
integer: NUMBER {$$ = (int)$1; }
         ; 
%%

/*****************************************************/
/*****************************************************/
/*****************************************************/
/*******************    lex    ***********************/



char *newstr (name, length)
  char *name;
  int length;
{
 char *temp;

 temp = am_alloc_mem(length + 1);
 *(temp + length) = 0;
 bcopy(name, temp, length+1);
 return(temp);
}

static void read_token()
{
  unsigned int c;


  yyleng=0;
  /* skip spaces, tabs , and newlines  */
  while((c=getchar()) != ' '
	&& !(iscntrl(c))
	&& c != '['
	&& c != ']'
	&& c != '{'
	&& c != '}'
	&& c != '('
	&& c != ')'
	&& c != '@'
	&& c != '='
	&& c != '#'
	&& c != ','
	&& c != EOF) {
    yytext[yyleng++] = c;
  }
  ungetc((unsigned char)c,stdin);
  yytext[yyleng] = '\0';
}

static int input_equals(str)
     register char *str;
{
  register int counter;
  register unsigned char *tptr = yytext;

  counter = yyleng;
  if (strlen(str) > counter) counter = strlen(str);
  while(counter--) if (*str++ != *tptr++) return FALSE;;
  return TRUE;

}


#define CONTEXT_STACK_DEPTH 16
#define NO_CONTEXT 0
#define DIMENSION_CONTEXT 1
#define CONNECTION_CONTEXT 2
#define TIME_CONTEXT 3
#define BLACK_BOX_CONTEXT 4
#define COMPONENTS_CONTEXT 5
static int context_stack[CONTEXT_STACK_DEPTH], *cstackptr = context_stack;

static void push_lex_context(new_context)
     int new_context;
{
  if (cstackptr == context_stack + CONTEXT_STACK_DEPTH)
    yyerror("\nLex context stack full! (push_lex_context)\n");
  cstackptr++; *cstackptr = new_context;
}

static void pop_lex_context()
{
  if (cstackptr == context_stack)
    yyerror("\nLex context stack would be empty! (pop_lex_context)\n");
  /* pop */
  cstackptr--;
}

static int this_context(context)
     int context;
{
  return( *cstackptr == context );
}

int yylex()
{
  unsigned int c;

  more :

  /* single character stuff */

  /* skip spaces and tabs */
  while((c=getchar()) == ' ' || c == '\t') ;

  /* jump on carrage return */
  if (c == '\n') {  ++lineno; goto more ; }

  /* jump on control chars */
  if (iscntrl(c)) goto more ;

  if (c == EOF) return 0;

  if (c == '#') return(NEWLINE);

  if (c == '{') {
    return(LCP);
  }/* end if */

  if (c == '}') {
    pop_lex_context();
    return(RCP);
  }/* end if */

  if (c == '(') {
    if (this_context(CONNECTION_CONTEXT))
      push_lex_context(CONNECTION_CONTEXT);
    return(LP);
  }/* end if */

  if (c == ')') {
    pop_lex_context();
    return(RP);
  }/* end if */

  if (c == '[') {
    push_lex_context(DIMENSION_CONTEXT);
    return(LB);
  }/* end if */

  if (c == ']') {
    pop_lex_context();
    return(RB);
  }/* end if */

  /* dimension context */
  if(this_context(DIMENSION_CONTEXT))
     if (c == 'x' || c == 'X') return(DIMENSION);

  if (this_context(DIMENSION_CONTEXT))
      if (c == ',') return(COMMA);

  if (c == '=') return(EQUALS);

  if (c == '@') {
    push_lex_context(TIME_CONTEXT);
    return(AT);
  }

  /* ok, done with the single character stuff, names  */
  ungetc((unsigned char)c, stdin);  read_token();

  if (isdigit((unsigned int)yytext[0]) ||
      (yyleng > 1 &&
       (yytext[0] == '-' || yytext[0] == '+' || yytext[0] == '.') &&
       isdigit((unsigned int)yytext[1])) ||
      (yyleng > 2 &&
       (yytext[0] == '-' || yytext[0] == '+') &&
       yytext[1] == '.' &&
       isdigit((unsigned int)yytext[2]))) { /* number */
    sscanf((char *)yytext, "%f", &yylval.f);
    return(NUMBER);
  }/* end if number */ 

  if (this_context(NO_CONTEXT)) { /* global context */
    if (input_equals("DefineBlackBox")) {
      push_lex_context(BLACK_BOX_CONTEXT);
      return(BBD);
    }
  }

  if (this_context(BLACK_BOX_CONTEXT)) {
    
    if (input_equals("InputSize->")) return(INPUT_SIZE);
    
    if (input_equals("OutputLayer->")) return(OUTPUT_LAYER);
    
    if (input_equals("Static->")) return(STATIC);
    
    if (input_equals("ClearDelays->")) return(CLRDELAYS);
    
    if (input_equals("InputFilter->")) return(INPUT_FILTER);

    if (input_equals("OutputFilter->")) return(OUTPUT_FILTER);
    
    if (input_equals("ErrorFunction->")) return(ERROR_FUNCTION);

    if (input_equals("UpdateInterval->")) return(UPDATE_INTERVAL);

    if (input_equals("DisableFeedbackLearning->")) return(DISABLE_FEEDBACK_LEARNING);

    if (input_equals("DisableFeedforwardLearning->")) return(DISABLE_FEEDFORWARD_LEARNING);

    if (input_equals("Components->")) {
      push_lex_context(COMPONENTS_CONTEXT);
      return(COMPONENTS);
    }
  }



  if (this_context(COMPONENTS_CONTEXT)) {
    
    if (input_equals("PdpNode")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(PDPNODE1);
    }
    
    if (input_equals("PdpNode1")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(PDPNODE1);
    }
    
    if (input_equals("PdpNode2")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(PDPNODE2);
    }
    
    if (input_equals("PdpNode3")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(PDPNODE3);
    }
    
    if (input_equals("LinearNode")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(LINEAR_NODE);
    }
    
    if (input_equals("QuadraticNode")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(QUAD_NODE);
    }
    
    if (input_equals("UserNode")) {
      push_lex_context(CONNECTION_CONTEXT);
      return(USER_NODE);
    }
    
  }

  if (this_context(TIME_CONTEXT))
    if (input_equals("Time")) return(TIME);


  /* connection description context */
  if (this_context(CONNECTION_CONTEXT)) {
    if (input_equals("InputsFrom->")) return(INPUTS_FROM);

    if (input_equals("and")) return(CONJ);

    if (input_equals("Tessellation")) return(TESSELLATION);
    
    if (input_equals("Xoffset")) return(XOFFSET);
    
    if (input_equals("Yoffset")) return(YOFFSET);
    
    if (input_equals("Xoverlap")) return(XOVERLAP);
    
    if (input_equals("Yoverlap")) return(YOVERLAP);
    
    if (input_equals("Shared")) return(SHARED);
    
    if (input_equals("with")) return(WITH);
    
    if (input_equals("using")) return(USING);
    
    if (input_equals("initialized")) return(INIT);

    if (input_equals("Order")) return(ORDER);

    if (input_equals("Bias")) return(BIAS);
    
    /* skip articles */
    if (input_equals("a")) goto more ;
  }


  if (this_context(NO_CONTEXT)) { /* global context */
    
    /* LoadData context */
    if (input_equals("LoadData")) return(LOAD_DATA);

    if (input_equals("into")) return(INTO);
    
    if (input_equals("in")) return(IN);
    
    if (input_equals("from")) return(FROM);

    /* LineSearch */
    if (input_equals("LineSearch")) return(LINE_SRCH);
    if (input_equals("LineSearchVerbose")) return(LINE_SRCH_VERBOSE);

    /* ConjugateGradient */
    if (input_equals("ConjugateGradient")) return(CONJ_GRAD);

  }
  
  /* names */
  {
    extern char *strchr();
    char *hyphen_ptr;
    yylval.s = newstr(yytext, yyleng);
    /* NO HYPHENS ALLOWED IN NAMES PASSED TO C! '-' => '_' */ 
    hyphen_ptr = strchr(yylval.s, (int)'-');
    while (hyphen_ptr != (char *)NULL) {
      *hyphen_ptr = '_';
      hyphen_ptr = strchr(yylval.s, (int)'-');
    }/* end while */
    return(NAME);
  }  

}/* end yylex */

/*****************************************************/
/*****************************************************/
/*****************************************************/





/*****************************************************/
/* parse       file = aspirin spefication from
               which a network of black box
	       descriptors is created.
 */
/*****************************************************/
ND_PTR parse(file)
     char *file;
{
  extern char *aspirin_file;
  extern ND_PTR network;
  char *cpp_file;
  FILE *in_stream;


  printf("\nAspirin Parser"); /* say hi! */
  printf("\n%s", BASIC_COPYRIGHT);

  
  /******************************************/
  /* run cpp on aspirin file */
  cpp_file = am_cpp(file,0);
  /******************************************/

  /******************************************/
  /* open  input from cpp */
  in_stream = freopen(cpp_file, "r", stdin);
  /* exit on error */
  if (in_stream == NULL)
    {
      fprintf(stderr, "\nUnable to open %s", cpp_file);
      am_perror("Aspirin file");
      exit(1);
    }
  aspirin_file = file;
  /******************************************/
  
  /******************************************/
  /* make the root */
  create_network();
  /******************************************/

  /******************************************/
  /* parse storing info in global variables */
  yyparse();
  /******************************************/

  /******************************************/
  /* update pointers, etc. */
  update(); 
  /******************************************/

  /******************************************/
  /* check dependencies for static bbs */
  check_dependencies(); 
  /******************************************/

  /******************************************/
  /* remove cpp file */
  am_remove_cpp_file(cpp_file);
  /******************************************/

  /******************************************/
  /* return */
  return(network);
  /******************************************/
}


    
