/* 

  ****************   NO WARRANTY  *****************

Since the Aspirin/MIGRAINES system is licensed free of charge,
the MITRE Corporation provides absolutley no warranty. Should
the Aspirin/MIGRAINES system prove defective, you must assume
the cost of all necessary servicing, repair or correction.
In no way will the MITRE Corporation be liable to you for
damages, including any lost profits, lost monies, or other
special, incidental or consequential damages arising out of
the use or inability to use the Aspirin/MIGRAINES system.

  *****************   COPYRIGHT  *******************

This software is the copyright of The MITRE Corporation. 
It may be freely used and modified for research and development
purposes. We require a brief acknowledgement in any research
paper or other publication where this software has made a significant
contribution. If you wish to use it for commercial gain you must contact 
The MITRE Corporation for conditions of use. The MITRE Corporation 
provides absolutely NO WARRANTY for this software.

   January, 1992 
   Russell Leighton
   The MITRE Corporation
   7525 Colshire Dr.
   McLean, Va. 22102-3481

*/
#include <stdio.h>
#include <math.h>
  
#include "aspirin_bp.h"

extern LB_PTR network_query();

static LB_PTR get_layer_buffer(bbindex, lindex)
     int bbindex, lindex;
{
  LB_PTR l;

  l = network_query(bbindex,lindex);

  if (l == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    exit(1);
  }/* end if */

  return(l);
}

int BPn_bbs()
{
  return ( (get_layer_buffer(0,0))->network_info.n_black_boxes );
}/* end BPn_bbs */


int BPbb_dynamic(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.dynamic );
}/* end BPbb_dynamic */

int BPbb_efferent(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.efferent );
}/* end BPbb_efferent */

char *BPbb_name(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.bb_name );
}/* end BPbb_name */

char *BPbb_layer_name(bbindex,lindex)
     int bbindex,lindex;
{
  return ( (get_layer_buffer(bbindex,lindex))->layer_name );
}/* end BPbb_layer_name */

char *BPbb_connection_from_name(bbindex,lindex,cindex)
     int bbindex,lindex,cindex;
{
  return ( ((get_layer_buffer(bbindex,lindex))->connections + cindex)->from_layer_name );
}/* end BPbb_connection_from_name */

int BPn_bb_layers(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.n_layers );
}/* end BPn_bb_layers */

int BPbb_input_size(bbindex)
     int bbindex;
{
  LB_PTR l;
  BBB_PTR bb;

  l = get_layer_buffer(bbindex,0);

  bb = & ( l->black_box_info );

  return ( bb->inputs_xdim * bb->inputs_ydim );
}/* end BPbb_input_size */

int BPbb_input_delays(bbindex)
     int bbindex;
{
 return ( (get_layer_buffer(bbindex,0))->black_box_info.inputs_delay );
}/* end BPbb_input_size */

void BPbb_input_dim(bbindex, xdim, ydim)
     int bbindex, *xdim, *ydim;
{
  register BBB_PTR bb;
  
  bb = & ( (get_layer_buffer(bbindex,0))->black_box_info );
  *xdim = bb->inputs_xdim;
  *ydim = bb->inputs_ydim;

}/* end BPbb_input_dim */

int BPbb_output_layer_index(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.output_layer_index );
}/* end BPbb_output_layer_index */


int BPn_bb_inputs(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.n_bb_inputs );
}/* end BPn_bb_inputs */

int BPn_bb_outputs(bbindex)
     int bbindex;
{
  return ( (get_layer_buffer(bbindex,0))->black_box_info.n_bb_outputs );
}/* end BPn_bb_outputs */

void BPbb_layer_dim(bbindex, lindex, xdim, ydim)
     int bbindex, lindex, *xdim, *ydim;
{
  register LB_PTR l;
  
  l = get_layer_buffer(bbindex,lindex);
  *xdim = l->xdim;
  *ydim = l->ydim;

}/* end BPbb_layer_dim */

void BPbb_layer_type(bbindex, lindex, type)
     int bbindex, lindex, *type;
{
  *type = get_layer_buffer(bbindex,lindex)->layer_type;
}/* end BPbb_layer_dim */

void BPbb_connection_dim(bbindex, lindex, cindex, type, xdim, ydim, xoffset, yoffset)
     int bbindex, lindex, cindex, *type, *xdim, *ydim, *xoffset, *yoffset;
{
  register CB_PTR c;
  
  c = get_layer_buffer(bbindex,lindex)->connections + cindex;

  *type = c->type;
  *xoffset = c->tess_xoffset;
  *yoffset = c->tess_yoffset;

  /* if full connection to 2d layer then display as 2d */
  if (*type == NXM_CONNECTION_TYPE && c->from_ydim != 1) {
    *xdim = c->from_xdim;
    *ydim = c->from_ydim;
  } else {
    *xdim = c->tess_xdim;
    *ydim = c->tess_ydim;
  }/* end if else */

}/* end BPbb_connection_dim */

int BPbb_connection_delay(bbindex, lindex, cindex)
     int bbindex, lindex, cindex;
{
  register CB_PTR c;
  
  c = get_layer_buffer(bbindex,lindex)->connections + cindex;

  return( c->delay );

}/* end BPbb_connection_delay */

int BPn_layer_inputs(bbindex, lindex)
     int bbindex, lindex;
{
  return ( (get_layer_buffer(bbindex,lindex))->n_inputs );
}/* end BPn_layer_inputs */

int BPn_layer_outputs(bbindex, lindex)
     int bbindex, lindex;
{
  return ( (get_layer_buffer(bbindex,lindex))->n_outputs );
}/* end BPn_layer_outputs */

void BPis_bb_connected(to_bb_index, from_bb_index, connect_ftn)
     int to_bb_index, from_bb_index;
     void (*connect_ftn)();
{
  LB_PTR layer_buffer;
  int lindex=0;
  

  /* apply connect_ftn to the connections into layers of to_bb_index
     for from_bb_index */
  while ( (layer_buffer = get_layer_buffer(to_bb_index, lindex)) !=  NULL) {
    int ninputs;
    CB_PTR connections;
    
    ninputs = layer_buffer->n_inputs;
    connections = layer_buffer->connections;
    while(ninputs--) {
      if ((connections + ninputs)->from_bb_index == from_bb_index) {
	connect_ftn(to_bb_index, lindex, ninputs, 
		    from_bb_index, connections->from_layer_index);
      }
    }/* end while */
    lindex++;
  }/* end while */
  
}/* end BPis_bb_connected */


void BPis_input_connected(to_bb_index, connect_ftn)
     int to_bb_index;
     void (*connect_ftn)();
{
  LB_PTR layer_buffer;
  int lindex=0;
  
  /* apply connect_ftn to the connections from $INPUTS is
     to_bb_index */
  while ( (layer_buffer = get_layer_buffer(to_bb_index, lindex)) !=  NULL) {
    int ninputs;
    CB_PTR connections;
    
    ninputs = layer_buffer->n_inputs;
    connections = layer_buffer->connections;
    while(ninputs--) {
      if ((connections + ninputs)->from_layer_index == -1) /* $INPUTS */
	connect_ftn(to_bb_index, lindex, ninputs, 
		    connections->from_bb_index, connections->from_layer_index);
    }/* end while */
    lindex++;
  }/* end while */
  
}/* end BPis_input_connected */

void BPis_layer_intra_connected(to_bb_index, lindex, connect_ftn)
     int to_bb_index, lindex;
     void (*connect_ftn)();
{
  LB_PTR layer_buffer;
  
  /* apply connect_ftn to the connections to lindex in to_bb_index */
  if ( (layer_buffer = get_layer_buffer(to_bb_index, lindex)) !=  NULL) {
    int ninputs;
    CB_PTR connections;
    
    ninputs = layer_buffer->n_inputs;
    connections = layer_buffer->connections;
    while(ninputs--)
      if ( (connections + ninputs)->from_layer_index != -1 /* not $INPUTS */
	  && (connections + ninputs)->from_bb_index == to_bb_index) /* within this bb */
	connect_ftn(to_bb_index, lindex, ninputs,
		    (connections + ninputs)->from_bb_index,
		    (connections + ninputs)->from_layer_index);
  }/* end if */
  
}/* end BPis_layer_intra_connected */


void BPis_layer_inter_connected(to_bb_index, lindex, connect_ftn)
     int to_bb_index, lindex;
     void (*connect_ftn)();
{
  LB_PTR layer_buffer;
  
  /* apply connect_ftn to the connections to lindex in to_bb_index */
  if ( (layer_buffer = get_layer_buffer(to_bb_index, lindex)) !=  NULL) {
    int ninputs;
    CB_PTR connections;
    
    ninputs = layer_buffer->n_inputs;
    connections = layer_buffer->connections;
    while(ninputs--)
      if ( (connections + ninputs)->from_layer_index != -1 /* not $INPUTS */
	  && (connections + ninputs)->from_bb_index != to_bb_index) /* NOT within this bb */
	connect_ftn(to_bb_index, lindex, ninputs,
		    (connections + ninputs)->from_bb_index,
		    (connections + ninputs)->from_layer_index);
  }/* end if */
  
}/* end BPis_layer_inter_connected */


void BPrw_input_values(bbindex, d, n, rw)
     register int bbindex;
     register float *d;
     register int n, rw;
{
  LB_PTR l;
  float *nn_d;

  l = get_layer_buffer(bbindex,0);

  nn_d = l->black_box_info.get_input();

  if (nn_d == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */

  switch(rw) {
  case BP_DATA_READ :
    bcopy((char *)nn_d, (char *)d, n * sizeof(float));
    break;
  case BP_DATA_WRITE :
    bcopy((char *)d, (char *)nn_d, n * sizeof(float));
    break;
  }/* end switch */

}/* end BPrw_input_values */

void BPrw_target_values(bbindex, d, n, rw)
     register int bbindex;
     register float *d;
     register int n, rw;
{
  LB_PTR l;
  float *nn_d;

  l = get_layer_buffer(bbindex,0);

  nn_d = l->black_box_info.get_target_output();

  if (nn_d == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */

  switch(rw) {
  case BP_DATA_READ :
    bcopy((char *)nn_d, (char *)d, n * sizeof(float));
    break;
  case BP_DATA_WRITE :
    bcopy((char *)d, (char *)nn_d, n * sizeof(float));
    break;
  }/* end switch */

}/* end BPrw_target_values */

void BPr_mag_diff_target_values(bbindex, d, n)
     register int bbindex;
     register float *d;
     register int n;
{
  float *nn_targets, *nn_outputs;

  nn_targets = get_layer_buffer(bbindex,0)->black_box_info.get_target_output();

  if (nn_targets == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */

  nn_outputs = get_layer_buffer(bbindex,0)->black_box_info.get_output();

  while(n--) *d++ = AM_FABS( *nn_targets++ - *nn_outputs++ );

}/* end BPr_mag_diff_target_values */

void BPrw_layer_values(bbindex, lindex, d, n, rw)
     register int bbindex, lindex;
     register float *d;
     register int n, rw;
{
  LB_PTR l;
  float *nn_d;

  l = get_layer_buffer(bbindex,lindex);

  nn_d = l->values;

  switch(rw) {
  case BP_DATA_READ :
    bcopy((char *)nn_d, (char *)d, n * sizeof(float));
    break;
  case BP_DATA_WRITE :
    bcopy((char *)d, (char *)nn_d, n * sizeof(float));
    break;
  }/* end switch */

}/* end BPrw_layer_values */

void BPrw_bias_values(bbindex, lindex, d, n, rw)
     register int bbindex, lindex;
     register float *d;
     register int n, rw;
{
  LB_PTR l;
  float *nn_d;

  l = get_layer_buffer(bbindex,lindex);

  nn_d = l->thresholds;

  switch(rw) {
  case BP_DATA_READ :
    bcopy((char *)nn_d, (char *)d, n * sizeof(float));
    break;
  case BP_DATA_WRITE :
    bcopy((char *)d, (char *)nn_d, n * sizeof(float));
    break;
  }/* end switch */

}/* end BPrw_bias_values */

int BPbb_layer_order(bbindex, lindex)
     register int bbindex, lindex;
{
  return( get_layer_buffer(bbindex,lindex)->order );
}

void BPrw_ar_values(bbindex, lindex, delay, d, n, rw)
     int bbindex, lindex, delay;
     float *d;
     int n, rw;
{
  LB_PTR l;
  float *nn_d;

  l = get_layer_buffer(bbindex,lindex);

  switch (delay) {
  case 1 :
    nn_d = l->ar1;
    break;
  case 2 :
    nn_d = l->ar2;
    break;
  case 3 :
    nn_d = l->ar3;
    break;
  }/* end switch */

  if (nn_d == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */

  switch(rw) {
  case BP_DATA_READ :
    bcopy((char *)nn_d, (char *)d, n * sizeof(float));
    break;
  case BP_DATA_WRITE :
    bcopy((char *)d, (char *)nn_d, n * sizeof(float));
    break;
  }
  
}/* end BPrw_ar_values */

void BPrw_connection_values(bbindex, lindex, cindex, x, y, d, n, rw)
     register int bbindex, lindex, cindex, x, y;
     register float *d;
     register int n, rw;
{
  LB_PTR l;
  CB_PTR c;
  float *nn_d;
  int xmatrices, ymatrices;


  l = get_layer_buffer(bbindex,lindex);

  c = l->connections + cindex;

  BPbb_layer_dim(bbindex, lindex, &xmatrices, &ymatrices);

  if (x < 0 || y < 0 || x+1 > xmatrices || y+1> ymatrices) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */

  /* find the data */
  switch(c->type) {
  case NXM_CONNECTION_TYPE :
  case TESS_1D_CONNECTION_TYPE : 
    if (c->shared)
      nn_d = c->weights;
    else {
      nn_d = c->weights + ( (x + (y * xmatrices)) * n);
    }/* end else */
    break;
  case TESS_2D_CONNECTION_TYPE :
    if (c->shared)
      nn_d = c->weights;
    else 
      nn_d = c->weights + (x * c->tess_xdim) + (y * xmatrices * n);
    break;
  }/* end switch */

  if (nn_d == NULL) { /* not there */
    fprintf(stderr, "\nCannot find requested neural network data!\n");
    bzero((char *)d, n * sizeof(float));
    return;
  }/* end if */
  
  if (c->type == TESS_2D_CONNECTION_TYPE && !c->shared) {  /* 2d is interleaved! */
    register int width = c->tess_xdim, height = c->tess_ydim, stride;

    stride = xmatrices * width;
    while(height--) {
      switch(rw) {
      case BP_DATA_READ :
	bcopy((char *)nn_d, (char *)d, width * sizeof(float));
	break;
      case BP_DATA_WRITE :
	bcopy((char *)d, (char *)nn_d, width * sizeof(float));
	break;
      }/* end switch */
      d += width;
      nn_d += stride;
    }/* end while */
  } else {
    switch(rw) {
    case BP_DATA_READ :
      bcopy((char *)nn_d, (char *)d, n * sizeof(float));
      break;
    case BP_DATA_WRITE :
      bcopy((char *)d, (char *)nn_d, n * sizeof(float));
      break;
    }/* end switch */
  }/* end if */

}/* end BPrw_connection_values */
