/* 

  ****************   NO WARRANTY  *****************

Since the Aspirin/MIGRAINES system is licensed free of charge,
the MITRE Corporation provides absolutley no warranty. Should
the Aspirin/MIGRAINES system prove defective, you must assume
the cost of all necessary servicing, repair or correction.
In no way will the MITRE Corporation be liable to you for
damages, including any lost profits, lost monies, or other
special, incidental or consequential damages arising out of
the use or inability to use the Aspirin/MIGRAINES system.

  *****************   COPYRIGHT  *******************

This software is the copyright of The MITRE Corporation. 
It may be freely used and modified for research and development
purposes. We require a brief acknowledgement in any research
paper or other publication where this software has made a significant
contribution. If you wish to use it for commercial gain you must contact 
The MITRE Corporation for conditions of use. The MITRE Corporation 
provides absolutely NO WARRANTY for this software.

   January, 1992 
   Russell Leighton
   The MITRE Corporation
   7525 Colshire Dr.
   McLean, Va. 22102-3481

*/
#include <stdio.h>
#include "ntalk.h"

extern char *am_alloc_mem(); /* from os.a - error checked malloc */

typedef char *CHARPTR;
typedef float *FLOATPTR;

#define TRUE 1
#define FALSE 0

#define NPHONEME_TOKENS 54
#define NPHONEME_UNITS 21
#define NSTRESS_TOKENS 5
#define NSTRESS_UNITS 5

#define NTOKENS 29   /* # of types of letters */
#define WINDOWSIZE 7 /* # of letters */
#define NINPUTS (WINDOWSIZE * NTOKENS)

#define NOUTPUTS 26 /* # of features */

/* Position in mouth- */
#define   FRONT1() *(target_vector + 0) = 1.0
#define   LABIAL() *(target_vector + 0) = 1.0
#define   FRONT2() *(target_vector + 1) = 1.0
#define   DENTAL() *(target_vector + 1) = 1.0
#define   CENTRAL1() *(target_vector + 2) = 1.0
#define   ALVEOLAR() *(target_vector + 2) = 1.0
#define   CENTRAL2() *(target_vector + 3) = 1.0
#define   PALATAL() *(target_vector + 3) = 1.0
#define   BACK1() *(target_vector + 4) = 1.0
#define   VELAR() *(target_vector + 4) = 1.0
#define   BACK2() *(target_vector + 5) = 1.0
#define   GLOTTAL() *(target_vector + 5) = 1.0
/* Phonemetype- */
#define   STOP() *(target_vector + 6) = 1.0
#define   NASAL() *(target_vector + 7) = 1.0
#define   FRICATIVE() *(target_vector + 8) = 1.0
#define   AFFRICATIVE() *(target_vector + 9) = 1.0
#define   GLIDE() *(target_vector + 10) = 1.0
#define   LIQUID() *(target_vector + 11) = 1.0
#define   VOICED() *(target_vector + 12) = 1.0
#define   UNVOICED() *(target_vector + 12) = 0.0
#define   TENSED() *(target_vector + 13) = 1.0
/* Vowel Height- */
#define   HIGH() *(target_vector + 14) = 1.0
#define   MEDIUM() *(target_vector + 15) = 1.0
#define   LOW() *(target_vector + 16) = 1.0
/* Punctuation- */
#define   SILENT() *(target_vector + 17) = 1.0
#define   ELIDE() *(target_vector + 18) = 1.0
#define   PAUSE() *(target_vector + 19) = 1.0
#define   FULLSTOP() *(target_vector + 20) = 1.0

/*
Stress and Syllabel Boundries-
	'>'	indicates a consonant prior to a syllable nucleus.
	'<'	indicates a consonant or vowel following the first
		vowel of the syllable nucleus.
	'0'	indicates the first vowel in the nucleus of an unstressed
		syllable.
	'2'	indicates the first vowel in the nucleus of a syllable
		receiving secondary stress.
	'1'	indicates the first vowel in the nucleus of a syllable
		receiving primary stress.
*/
#define STRESS_A() *(target_vector + 21)=1.0
#define STRESS_B() *(target_vector + 22)=1.0
#define STRESS_C() *(target_vector + 23)=1.0
#define STRESS_D() *(target_vector + 24)=1.0
#define STRESS_E() *(target_vector + 25)=1.0
   

/* input & target buffers */
static float input_vector[ NINPUTS ], target_vector[ NOUTPUTS ];
static float *output_vector;

static int pattern_counter = -1;
static int char_counter = -1;

static int nwords;
static int nsymbols = 0;

/* data (arrays of strings) */
static char *(*words), *(*phonemes), *(*stresses), *lengths;

/* table of phoneme features */
static float *(*phoneme_features), *phoneme_mags;

/* table of stress features */
static float *(*stress_features);

/* the phoneme tokens */
static char phoneme_tokens[NPHONEME_TOKENS] = {'a',
				  'b',
				  'c', 
				  'd', 
				  'e', 
				  'f', 
				  'g', 
				  'h', 
				  'i', 
				  'k', 
				  'l', 
				  'm', 
				  'n', 
				  'o', 
				  'p', 
				  'r', 
				  's', 
				  't', 
				  'u', 
				  'v', 
				  'w', 
				  'x', 
				  'y', 
				  'z', 
				  'A', 
				  'C', 
				  'D', 
				  'E', 
				  'G', 
				  'I', 
				  'J', 
				  'K', 
				  'L', 
				  'M', 
				  'N', 
				  'O', 
				  'Q', 
				  'R', 
				  'S', 
				  'T', 
				  'U', 
				  'W', 
				  'X', 
				  'Y', 
				  'Z', 
				  '@', 
				  '!', 
				  '#', 
				  '*', 
				  '|', 
				  '^', 
				  '-', 
				  '_', 
				  '.'};

/* the stress tokens */
static char stress_tokens[NSTRESS_TOKENS] = {'>','<','0','2','1'};


static char *string_copy(s,n)
     char *s;
     int n;
{
  char *new;

  new = am_alloc_mem(n);
  bcopy(s,new,n);
  return(new);
}

static float mag(ptr, n)
     register float *ptr;
     register int n;
{
  float sum = 0.0;
  
  while(n--) {
    sum += *ptr * *ptr;
    ptr++;
  }
  return( sqrt((double)sum) );
}

static float vdot(ptr1, ptr2, n)
     register float *ptr1, *ptr2;
     int n;
{
  float sum = 0.0;
  while(n--) {
    sum += *ptr1++ * *ptr2++;
  }
  return( sum );
}

/* return the cosine of the angle btwn ptr1 & ptr2 */
static float angle_cos(ptr1, ptr2, mag1, mag2, n)
     register float *ptr1, *ptr2;
     float mag1, mag2;
     int n;
{
  return (  vdot(ptr1, ptr2, n)/(mag1 * mag2));
}

/* return the square of the distance between vectors */
static float dist_squared(ptr1,ptr2,n)
     register float *ptr1, *ptr2;
     register int n;
{
  register float result = 0.0;

  while(n--) {
    register float diff;

    diff = *ptr1++ - *ptr2++;
    result += diff * diff;
  }/* while n */

  return(result);
}

static void readdata(datafile, nwords)
     char *datafile;
     int nwords;
{
  FILE *fp;
  char word[128],phoneme[128],stress[128],weird[128];

  printf("\nUsing datafile: %s", datafile);

  fp = fopen(datafile, "r");
  if (fp == (FILE *)NULL) {
    fprintf(stderr, "\nUnable to open %s!\n", datafile);
    exit(1);
  }

  /* get some memory */
  words = (char *(*))am_alloc_mem(nwords * sizeof(CHARPTR));
  phonemes = (char *(*))am_alloc_mem(nwords * sizeof(CHARPTR));
  stresses = (char *(*))am_alloc_mem(nwords * sizeof(CHARPTR));
  lengths = (CHARPTR)am_alloc_mem(nwords * sizeof( char ));
  
  /* read the fields into strings */
  while(nwords--) {
    fscanf(fp,"%s %s %s %s",
	   word, phoneme, stress, weird);
    lengths[nwords] = strlen(word);
    nsymbols += lengths[nwords];
    words[nwords] = string_copy(word, lengths[nwords]);
    phonemes[nwords] = string_copy(phoneme, lengths[nwords]);
    stresses[nwords] = string_copy(stress, lengths[nwords]);
  }/* end loop */
  
  fclose(fp);
}



static void set_letter(ptr, word_index, character, letter)
     float *ptr;
     int word_index, character, letter;
{
  int char_index;

  char_index = character + 3 - letter;
  if ( char_index >= 0 && char_index < lengths[word_index]) {
    char symbol;

    symbol = *(*(words + word_index) + char_index);
    switch(symbol) {/* maps symbol => input */
    case '.' :
      *(ptr + 26) = 1.0;
      break;
    case ',' :
      *(ptr + 27) = 1.0;
      break;
    case ' ' :
      *(ptr + 28) = 1.0;
      break;
    default:
      symbol -= 65;
      if (symbol > 27) symbol -= 32;
      *(ptr + symbol) = 1.0;
    }

  }/* end if */

}

static set_input(index, character)
     int index, character;
{
  register int counter = WINDOWSIZE;

  /* Clear */
  bzero((CHARPTR)input_vector, NINPUTS * sizeof(float)); 

  /* there are 7 input characters */
  while(counter--) {
    set_letter(input_vector + (counter * NTOKENS),
	       index,
	       character,
	       counter);
  }
}

static void set_stress( stress )
     char stress;
{
  switch( stress ) {
  case '>' :
    STRESS_A();
    break;
  case '<' :
    STRESS_B();
    break;
  case '0' :
    STRESS_C();
    break;
  case '2' :
    STRESS_D();
    break;
  case '1' :
    STRESS_E();
    break;
  }
}

static void set_phoneme(phoneme)
     char phoneme;
{

  switch( phoneme ) {
  case 'a' :
    LOW();
    TENSED();
    CENTRAL2();
    break;
  case 'b' :
    VOICED();
    LABIAL();
    STOP();
    break;
  case 'c' :
    MEDIUM();
    VELAR();
    break;
  case 'd' :
    VOICED();
    ALVEOLAR();
    STOP();
    break;
  case 'e' :
    MEDIUM();
    TENSED();
    FRONT2();
    break;
  case 'f' :
    UNVOICED();
    LABIAL();
    FRICATIVE();
    break;
  case 'g' :
    VOICED();
    VELAR();
    STOP();
    break;
  case 'h' :
    UNVOICED();
    GLOTTAL();
    GLIDE();
    break;
  case 'i' :
    HIGH();
    TENSED();
    FRONT1();
    break;
  case 'k' :
    UNVOICED();
    VELAR();
    STOP();
    break;
  case 'l' :
    VOICED();
    DENTAL();
    LIQUID();
    break;
  case 'm' :
    VOICED();
    LABIAL();
    NASAL();
    break;
  case 'n' :
    VOICED();
    ALVEOLAR();
    NASAL();
    break;
  case 'o' :
    MEDIUM();
    TENSED();
    BACK2();
    break;
  case 'p' :
    UNVOICED();
    LABIAL();
    STOP();
    break;
  case 'r' :
    VOICED();
    PALATAL();
    LIQUID();
    break;
  case 's' :
    UNVOICED();
    ALVEOLAR();
    FRICATIVE();
    break;
  case 't' :
    UNVOICED();
    ALVEOLAR();
    STOP();
    break;
  case 'u' :
    HIGH();
    TENSED();
    BACK2();
    break;
  case 'v' :
    VOICED();
    LABIAL();
    FRICATIVE();
    break;
  case 'w' :
    VOICED();
    LABIAL();
    GLIDE();
    break;
  case 'x' :
    MEDIUM();
    CENTRAL2();
    break;
  case 'y' :
    VOICED();
    PALATAL();
    GLIDE();
    break;
  case 'z' :
    VOICED();
    TENSED();
    FRICATIVE();
    break;
  case 'A' :
    MEDIUM();
    TENSED();
    FRONT2(); CENTRAL1();
    break;
  case 'C' :
    UNVOICED();
    PALATAL();
    AFFRICATIVE();
    break;
  case 'D' :
    VOICED();
    DENTAL();
    FRICATIVE();
    break;
  case 'E' :
    MEDIUM();
    FRONT1(); FRONT2();
    break;
  case 'G' :
    VOICED();
    VELAR();
    NASAL();
    break;
  case 'I' :
    HIGH();
    FRONT1();
    break;
  case 'J' :
    VOICED();
    VELAR();
    NASAL();
    break;
  case 'K' :
    UNVOICED();
    PALATAL();
    FRICATIVE(); VELAR();
    AFFRICATIVE();
    break;
  case 'L' :
    VOICED();
    ALVEOLAR();
    LIQUID();
    break;
  case 'M' :
    VOICED();
    DENTAL();
    NASAL();
    break;
  case 'N' :
    VOICED();
    PALATAL();
    NASAL();
    break;
  case 'O' :
    MEDIUM();
    TENSED();
    CENTRAL1(); CENTRAL2();
    break;
  case 'Q' :
    VOICED();
    LABIAL(); VELAR();
    AFFRICATIVE();
    STOP();
    break;
  case 'R' :
    VOICED();
    VELAR();
    LIQUID();
    break;
  case 'S' :
    UNVOICED();
    PALATAL();
    FRICATIVE();
    break;
  case 'T' :
    UNVOICED();
    DENTAL();
    FRICATIVE();
    break;
  case 'U' :
    HIGH();
    BACK1();
    break;
  case 'W' :
    HIGH(); MEDIUM();
    TENSED();
    CENTRAL2(); BACK1();
    break;
  case 'X' :
    UNVOICED();
    FRICATIVE();
    FRONT2(); CENTRAL1();
    break;
  case 'Y' :
    HIGH();
    TENSED();
    FRONT2(); CENTRAL1();
    break;
  case 'Z' :
    VOICED();
    PALATAL();
    FRICATIVE();
    break;
  case '@' :
    LOW();
    FRONT2();
    break;
  case '!' :
    UNVOICED();
    LABIAL(); DENTAL();
    AFFRICATIVE();
    break;
  case '#' :
    VOICED();
    PALATAL(); VELAR();
    AFFRICATIVE();
    break;
  case '*' :
    VOICED();
    GLIDE();
    FRONT1(); LOW(); 
    CENTRAL1();
    break;
  case '|' :
    HIGH();
    FRONT1(); FRONT2();
    break;
  case '^' :
    LOW();
    CENTRAL1();
    break;
  case '-' :
    SILENT();
    ELIDE();
    break;
  case '_' :
    PAUSE();
    ELIDE();
    break;
  case '.' :
    PAUSE();
    FULLSTOP();
    break;
    default : {
      fprintf(stderr,"\nUnknown phoneme: %c",
	      phoneme);
      exit(1);
    }
  }
}

static void set_target(word_index, character)
     int word_index, character;
{
  /* each phoneme has an associated unique output vector */

  /* clear */
  bzero((CHARPTR)target_vector, NOUTPUTS * sizeof(float));

  /* set phoneme features */
  set_phoneme( *(*(phonemes + word_index) + character) );
  
  /* set stress */
  set_stress( *(*(stresses + word_index) + character) );


}


static void input_character()
{
  char_counter = (char_counter + 1)% lengths[pattern_counter];
  if (!char_counter) {
    pattern_counter = (pattern_counter + 1) % nwords;
    /* printf("\n"); */
  }

  set_input(pattern_counter, char_counter);
  set_target(pattern_counter, char_counter); 
  
}


static void build_table()
{
  register int counter;

  /* get some memory */
  phoneme_mags = (float *)am_alloc_mem(NPHONEME_TOKENS * sizeof(float));
  phoneme_features = (float *(*)) am_alloc_mem(NPHONEME_TOKENS * sizeof(FLOATPTR));
  stress_features = (float *(*)) am_alloc_mem(NSTRESS_TOKENS * sizeof(FLOATPTR));
  
  /* now fill the tables with vectors */
  for(counter=0; counter<NPHONEME_TOKENS; counter++) { /* phonemes */
    /* clear */
    bzero((CHARPTR)target_vector, NOUTPUTS * sizeof(float));
    /* set vector */
    set_phoneme(phoneme_tokens[counter]);
    /* copy vector into table */
    phoneme_features[counter] = (float *)am_alloc_mem(NPHONEME_UNITS * sizeof(float));
    bcopy((CHARPTR)target_vector,
	  phoneme_features[counter],
	  NPHONEME_UNITS * sizeof (float));
    /* record magnitude */
    phoneme_mags[counter] = mag(phoneme_features[counter], NPHONEME_UNITS);
  }

  for(counter=0; counter<NSTRESS_TOKENS; counter++) { /* stresses */
    /* clear */
    bzero((CHARPTR)target_vector, NOUTPUTS * sizeof(float));
    /* set vector */
    set_stress(stress_tokens[counter]);
    /* copy vector into table */
    stress_features[counter] = (float *)am_alloc_mem(NSTRESS_TOKENS * sizeof(float));
    bcopy((CHARPTR)(target_vector + NPHONEME_UNITS),
	  stress_features[counter],
	  NSTRESS_TOKENS * sizeof (float));
    /* no need to record magnitude  (== 1) */
  }
}

static char closest_angle_stress()
{
  register int counter;
  float angle, largest_cos = -2.0;
  char largest_token;

  for(counter=0; counter<NSTRESS_TOKENS; counter++) {
    angle = angle_cos(output_vector + NPHONEME_UNITS,
		      stress_features[counter],
		      mag(output_vector + NPHONEME_UNITS, NSTRESS_UNITS),
		      1.0,
		      NSTRESS_UNITS);
    if (angle  > largest_cos) {
      largest_cos = angle;
      largest_token = stress_tokens[counter];
    }
  }

  return(largest_token);
}

static char closest_dist_stress()
{
  register int counter;
  float d;
  double smallest_d = AM_HUGE_VAL;
  char smallest_token;

  for(counter=0; counter<NSTRESS_TOKENS; counter++) {
    d = dist_squared(output_vector + NPHONEME_UNITS,
		      stress_features[counter],
		      NSTRESS_UNITS);
    if (d  < smallest_d) {
      smallest_d = d;
      smallest_token = stress_tokens[counter];
    }
  }

  return(smallest_token);
}


static char closest_angle_phoneme()
{
  register int counter;
  float angle, largest_cos = -2.0;
  char largest_token;
  
  for(counter=0; counter<NPHONEME_TOKENS; counter++) {
    angle = angle_cos(output_vector,
		      phoneme_features[counter],
		      mag(output_vector, NPHONEME_UNITS),
		      phoneme_mags[counter],
		      NPHONEME_UNITS);
    if (angle  > largest_cos) {
      largest_cos = angle;
      largest_token = phoneme_tokens[counter];
    }
  }
/*  printf("%c", largest_token); */
  return(largest_token);
}

static char closest_dist_phoneme()
{
  register int counter;
  float d;
  double smallest_d = AM_HUGE_VAL;
  char smallest_token;

  
  for(counter=0; counter<NPHONEME_TOKENS; counter++) {
    d = dist_squared(output_vector,
		      phoneme_features[counter],
		      NPHONEME_UNITS);
    if (d  < smallest_d) {
      smallest_d = d;
      smallest_token = phoneme_tokens[counter];
    }
  }
/*  printf("%c", smallest_token); */
  return(smallest_token);
}

#define current_stress() *(*(stresses + pattern_counter) + char_counter)
#define current_phoneme() *(*(phonemes + pattern_counter) + char_counter)

/* true if closest phoneme is correct phoneme */
static int classify_angle_phoneme()
{
  
  if (closest_angle_phoneme() == current_phoneme()) return(TRUE);
  return(FALSE);
}

/* true if closest stress is correct */
static int classify_angle_stress()
{
  
  if (closest_angle_stress() == current_stress()) return(TRUE);
  return(FALSE);
}


/* true if closest phoneme is correct phoneme */
static char word_buffer[128];
static int word_buffer_length =0;
static int classify_dist_phoneme()
{
  word_buffer[char_counter] = closest_dist_phoneme();
  word_buffer_length++;

  if (word_buffer[char_counter] == current_phoneme()) return(TRUE);
  return(FALSE);
}

/* true if closest stress is correct */
static int classify_dist_stress()
{

  if (closest_dist_stress() == current_stress()) return(TRUE);
  return(FALSE);
}

static int count_phones(s1,s2,s1n,s2n)
     register char *s1,*s2;
     int s1n, s2n;
{
  register n = s1n;
  register int count = 0;

  if (s2n < n) n = s2n; /* min */

  while(n--) {
    if (*s1 == *s2) count++;
    s1++; s2++;
  }

  return(count);
}


static int string_compare(s1,s2,s1n,s2n)
     register char *s1,*s2;
     int s1n, s2n;
{
  register n = s1n;
  register int count = 0;

  if (s2n < n) n = s2n; /* min */

  while(n--) {
    if (*s1 != *s2) return(FALSE);
    s1++; s2++;
  }

  return(TRUE);
}


static int closest_phon_word(phon_word,phon_length)
     char *phon_word;
     int phon_length;
{
  int closest_phon_word;
  int new_match, best_match = -1;
  register int word_counter = nwords;

  closest_phon_word = 0;

  /* loop thru all the words and test...brute force */
  while(word_counter--) {
    if (lengths[word_counter] == phon_length) { /* only check same length */
      new_match = count_phones(phonemes[word_counter],word_buffer,lengths[word_counter],phon_length);
      if (new_match > best_match) {
	best_match = new_match;
	closest_phon_word = word_counter;
      }/* if */
    }/* if length*/
  }/* end while */

  return(closest_phon_word);
}

/* true if closest word (one with most same phonemes) is correct */
static int first_time = TRUE;
static int classify_word()
{

  if (!char_counter && !first_time) {
    int closest_phon_word_index, this_pattern;

    if (pattern_counter) this_pattern = pattern_counter -1;
    else this_pattern = nwords -1;

    closest_phon_word_index = closest_phon_word(word_buffer,word_buffer_length-1);

/*
    {
      int n;

      word_buffer_length--;
      n = word_buffer_length;
      printf("\n");
      while(n--) printf("%c", *(word_buffer + word_buffer_length - n -1));
      printf(" ");
      n = lengths[this_pattern];
      while(n--) printf("%c",
			*(*(phonemes + this_pattern) + lengths[this_pattern] - n -1));
      printf(" ");
      n = lengths[closest_phon_word_index];
      while(n--) printf("%c",
			*(*(phonemes + closest_phon_word_index) + lengths[closest_phon_word_index] - n -1));
    }
*/

    word_buffer_length = 1; /* reset , new word */

    if (string_compare(*(phonemes + closest_phon_word_index),
			*(phonemes + this_pattern),
			lengths[closest_phon_word_index],
			lengths[this_pattern])) 
      return(TRUE);

  }
  first_time = FALSE;
  return(FALSE);
}

/* uses "best guess"-
   1.  the phoneme/stress vector with the smallest angle to output 
   2.  the phoneme/stress vector with the smallest disatance from output */
static void evaluate()
{
  register int symbol_counter;
  register int pcorrect_sm = 0, scorrect_sm = 0;
  register int pcorrect_d = 0, scorrect_d = 0, wcorrect = 0;

  char_counter = -1; pattern_counter = 0;

  for(symbol_counter=0; symbol_counter<nsymbols; symbol_counter++) {

    network_forward(1,input_character); /* forward once */

    /* look up the phoneme and the stress */
    if (classify_angle_phoneme()) pcorrect_sm++;
    if (classify_angle_stress()) scorrect_sm++;
    if (classify_dist_phoneme()) pcorrect_d++;
    if (classify_dist_stress()) scorrect_d++;
    /* look up word */
    if (classify_word()) wcorrect++;
  }

  printf("\n\nPhonemes Best Guess Stats (Smallest Angle)");
  printf("\nPd: %d/%d = %f",
	 pcorrect_sm, nsymbols, (float)pcorrect_sm/(float)nsymbols);
  printf("\nPfa: %d/%d = %f\n",
	 nsymbols - pcorrect_sm,
	 nsymbols, (float)(nsymbols - pcorrect_sm)/(float)nsymbols);
  printf("\n\nStresses Best Guess Stats");
  printf("\nPd: %d/%d = %f",
	 scorrect_sm, nsymbols, (float)scorrect_sm/(float)nsymbols);
  printf("\nPfa: %d/%d = %f\n",
	 nsymbols - scorrect_sm,
	 nsymbols, (float)(nsymbols - scorrect_sm)/(float)nsymbols);



  printf("\n\nPhonemes Best Guess Stats (Euclidian Distance)");
  printf("\nPd: %d/%d = %f",
	 pcorrect_d, nsymbols, (float)pcorrect_d/(float)nsymbols);
  printf("\nPfa: %d/%d = %f\n",
	 nsymbols - pcorrect_d,
	 nsymbols, (float)(nsymbols - pcorrect_d)/(float)nsymbols);
  printf("\n\nStresses Best Guess Stats");
  printf("\nPd: %d/%d = %f",
	 scorrect_d, nsymbols, (float)scorrect_d/(float)nsymbols);
  printf("\nPfa: %d/%d = %f\n",
	 nsymbols - scorrect_d,
	 nsymbols, (float)(nsymbols - scorrect_d)/(float)nsymbols);


  printf("\n\nWords Best Guess Stats (Largest Match)");
  printf("\nPd: %d/%d = %f",
	 wcorrect, nwords, (float)wcorrect/(float)nwords);
  printf("\nPfa: %d/%d = %f\n",
	 nwords - wcorrect,
	 nwords, (float)(nwords - wcorrect)/(float)nwords);

}

/* useage: Performace <dumpfile> <datafile> <number of items in datafile> */
main(argc, argv)
     int argc;
     char *(*argv);
{
#ifdef MCOS
  extern int krn_fte_disable();
  (void)krn_fte_disable();
#endif
#ifdef SUNOS
  extern void abrupt_underflow_();
  abrupt_underflow_();
#endif

  if (argc != 4) {
    fprintf(stderr, "\nusage: %s <dumpfile> <datafile> <number of items in datafile>\n",
	    *argv);
    exit(1);
  }
  
  /* initialize the network */
  (void)network_initialize(*(argv + 1), 1);
  ntalk_set_input(input_vector);
  ntalk_set_target_output(target_vector);
  output_vector = ntalk_get_output();

  /* read in the data */
  sscanf(*(argv + 3), "%d", &nwords);
  readdata(*(argv + 2), nwords);

  /* build phoneme table */
  build_table(); 

  /* evaluate */
  evaluate();
}
    
