;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;Trap.Lisp Version 2.0
;;
;;Copyright 1990, Apple Computer, Inc.
;;
;;
;;   This file contains trap definitions for most Macintosh traps provided
;;   by the Macintosh Plus, and some additional traps from later Macintoshes
;;   (e.g. color Quickdraw traps).
;;
;;  Additional trap definitions can be added easily.  All you need is the trap
;;  number, from Inside Macintosh.
;;
;;  To access traps from MACL, load this file.  The traps can then be
;;  called using the format described in chapter 12 of the MACL User's
;;  Guide.
;;
;;  Note that traps are defined as macros, so this file must be loaded before
;;  compiling functions which use the traps.
;;
;;  If memory is short, you may want to comment-out unused traps.
;;  Loading this entire file will use of about 60K of memory.
;;
;;

;;
;;The following mechanism is used to define traps:
;;
;;   For each trap _Foo, this file defines a lisp constant _Foo (which has
;;   as its value the trap number) and a macro _Foo which expands into a call
;;   to the function STACK-TRAP or the function REGISTER-TRAP.
;;
;;   The file contains four functions which generate these constants and macros.
;;   Each function produces trap calls of a slightly different format (to
;;   support stack-traps, register-traps, hfs-traps, and package-traps).
;;
;;   Each of these functions is mapped over an array of
;;   trap-names/trap-numbers.
;;
;;   To add trap definitions, add trap-name/trap-number pairs to the appropriate
;;   array definition.  To remove trap definitions, comment-out
;;   trap-name/trap-number pairs from the array definition.
;;
;;



(in-package "CCL")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;stack-trap-macro-function
;;
;;function used for defining stack-traps
;;
;;it transforms a call that looks like this
;;
;;                              (_Foo [:check-error] . args)
;;
;;into one that looks like this
;;
;;                              (stack-trap [:check-error] _Foo . args)


(defun stack-trap-macro-function (call env &aux (name (car call)))
  (declare (ignore env))
  (setq call (cdr call))
  (cons 'stack-trap
        (if (memq (car call) *error-check-keywords*)
          (list* (car call) name (cdr call))
          (cons name call))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;register-trap-macro-function
;;
;;function used for defining register-traps
;;
;;it transforms a call that looks like this
;;
;;                              (_Foo [:check-error] . args)
;;
;;into one that looks like this
;;
;;                              (register-trap [:check-error] _Foo . args)

  
(defun register-trap-macro-function (call env &aux (name (car call)))
  (declare (ignore env))
  (setq call (cdr call))
  (cons 'register-trap
        (if (memq (car call) *error-check-keywords*)
          (list* (car call) name (cdr call))
          (cons name call))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;hfs-trap-macro-function
;;
;;function used for defining hfs-traps
;;
;;it transforms a call that looks like this
;;
;;                (_Foo [:check-error] . args)
;;
;;into one that looks like this
;;
;;                (register-trap [:check-error] _HFSDispatch :d0 _Foo . args)
;;

(defun hfs-trap-macro-function (call env &aux (name (car call)))
  (declare (ignore env))
  (setq call (cdr call))
  (cons 'register-trap
        (if (memq (car call) *error-check-keywords*)
          (list* (car call) #xA260 :d0 name (cdr call))
          (list* #xA260 :d0 name call))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;dispatch-stack-trap-macro-function
;;
;;function used for defining dispatching stack-traps
;;
;;it transforms a call that looks like this
;;
;;                      (_Foo [:check-error] . args )
;;
;;into one that looks like this
;;
;;                      (stack-trap [:check-error] trap# ..args.. :word _Foo)
;;

(defvar *dispatching-traps* (make-hash-table :test #'eq))

(defun dispatch-stack-trap-macro-function (call env
                                                &aux (name (car call))
                                                (trap (gethash (car call) *dispatching-traps*)))
  (declare (ignore env))
  (setq call (reverse (cdr call)))
  (setq call
        (reverse
         (if (and call (assq (car call) *stack-trap-output-keywords*))
           (list* (car call) name :word (cdr call))
           (list* name :word call))))
  (cons 'stack-trap
        (if (memq (car call) *error-check-keywords*)
          (list* (car call) trap (cdr call))
          (cons trap call))))

(defun long-dispatch-stack-trap-macro-function (call env
                                                     &aux (num (symbol-value (car call)))
                                                     (trap (gethash (car call) *dispatching-traps*)))
  (declare (ignore env))
  (setq call (reverse (cdr call)))
  (setq call
        (reverse
         (if (and call (assq (car call) *stack-trap-output-keywords*))
           (list* (car call) (ash num -16) :word (logand num #xffff) :word (cdr call))
           (list* (ash num -16) :word (logand num #xffff) :word call))))
  (cons 'stack-trap
        (if (memq (car call) *error-check-keywords*)
          (list* (car call) trap (cdr call))
          (cons trap call))))

;; ####################################################################
;; ########################## Stack Traps #############################
;; ####################################################################

(let ((macro-fn #'stack-trap-macro-function)
      (traps '#(
                ;**************************************************
                ;Control Manager traps:

                (_DisposControl . #xA955)
                (_DragControl . #xA967)
                (_DrawControls . #xA969)
                (_FindControl . #xA96C)
                (_Draw1Control . #xA96D)
                (_GetCRefCon . #xA95A)
                (_GetCTitle . #xA95E)
                (_GetCtlAction . #xA96A)
                (_GetCtlValue . #xA960)
                (_GetMaxCtl . #xA962)
                (_GetMinCtl . #xA961)
                (_GetNewControl . #xA9BE)
                (_HideControl . #xA958)
                (_HiliteControl . #xA95D)
                (_KillControls . #xA956)
                (_MoveControl . #xA959)
                (_NewControl . #xA954)
                (_SetCRefCon . #xA95B)
                (_SetCTitle . #xA95F)
                (_SetCtlAction . #xA96B)
                (_SetCtlValue . #xA963)
                (_SetMaxCtl . #xA965)
                (_SetMinCtl . #xA964)
                (_ShowControl . #xA957)
                (_SizeControl . #xA95C)
                (_TestControl . #xA966)
                (_TrackControl . #xA968)

                (_UpdtControls . #xA953)

                ;**************************************************
                ;Desk Accessory traps:

                (_CloseDeskAcc . #xA9B7)
                (_OpenDeskAcc . #xA9B6)
                (_SysEdit . #xA9C2)
                (_SystemClick . #xA9B3)
                (_SystemEvent . #xA9B2)
                (_SystemMenu . #xA9B5)
                (_SystemTask . #xA9B4)


                ;**************************************************
                ;Dialog Manager traps:

                (_Alert . #xA985)
                (_CautionAlert . #xA988)
                (_CloseDialog . #xA982)
                (_CouldAlert . #xA989)
                (_CouldDialog . #xA979)
                (_DialogSelect . #xA980)
                (_DisposDialog . #xA983)
                (_DrawDialog . #xA981)
                (_ErrorSound . #xA98C)
                (_FreeAlert . #xA98A)
                (_FreeDialog . #xA97A)
                (_GetDItem . #xA98D)
                (_GetIText . #xA990)
                (_GetNewDialog . #xA97C)
                (_InitDialogs . #xA97B)
                (_IsDialogEvent . #xA97F)
                (_ModalDialog . #xA991)
                (_NewDialog . #xA97D)
                (_NoteAlert . #xA987)
                (_ParamText . #xA98B)
                (_SelIText . #xA97E)
                (_SetDItem . #xA98E)
                (_SetIText . #xA98F)
                (_StopAlert . #xA986)

                (_FindDItem . #xA984)
                (_HideDItem . #xA827)
                (_ShowDItem . #xA828)
                (_UpdtDialog . #xA978)

                ;**************************************************
                ;Event Manager traps:

                (_Button . #xA974)
                (_EventAvail . #xA971)
                (_GetKeys . #xA976)
                (_GetMouse . #xA972)
                (_WaitNextEvent . #xA860)
                (_GetNextEvent . #xA970)
                (_StillDown . #xA973)
                (_TickCount . #xA975)
                (_WaitMouseUp . #xA977)

                ;**************************************************
                ;Font Manager traps:

                (_FMSwapFont . #xA901)
                (_GetFName . #xA8FF)
                (_GetFNum . #xA900)
                (_InitFonts . #xA8FE)
                (_RealFont . #xA902)
                (_SetFontLock . #xA903)

                (_FontMetrics . #xA835)
                (_SetFScaleDisable . #xA834)

                ;**************************************************
                ;Menu Manager traps:

                (_AddResMenu . #xA94D)
                (_AppendMenu . #xA933)
                (_CalcMenuSize . #xA948)
                (_CheckItem . #xA945)
                (_ClearMenuBar . #xA934)
                (_CountMItems . #xA950)
                (_DeleteMenu . #xA936)
                (_DisableItem . #xA93A)
                (_DisposMenu . #xA932)
                (_DrawMenuBar . #xA937)
                (_EnableItem . #xA939)
                (_FlashMenuBar . #xA94C)
                (_GetItem . #xA946)
                (_GetItmIcon . #xA93F)
                (_GetItmMark . #xA943)
                (_GetItmStyle . #xA941)
                (_GetMenuBar . #xA93B)
                (_GetMHandle . #xA949)
                (_GetNewMBar . #xA9C0)
                (_GetRMenu . #xA9BF)
                (_HiliteMenu . #xA938)
                (_InitMenus . #xA930)
                (_InsertMenu . #xA935)
                (_InsertResMenu . #xA951)
                (_MenuKey . #xA93E)
                (_MenuSelect . #xA93D)
                (_NewMenu . #xA931)
                (_SetItem . #xA947)
                (_SetItmIcon . #xA940)
                (_SetItmMark . #xA944)
                (_SetItmStyle . #xA942)
                (_SetMenuBar . #xA93C)
                (_SetMFlash . #xA94A)
                (_DelMenuItem . #xA952)
                (_InsMenuItem . #xA826)


                ;**************************************************
                ;Package Manager traps:

                (_InitAllPacks . #xA9E6)
                (_InitPack . #xA9E5)
                (_Pack0 . #xA9E7)
                (_Pack1 . #xA9E8)
                (_Pack2 . #xA9E9)
                (_dskInit . #xA9E9)  ;Non-standard name
                (_Pack3 . #xA9EA)
                (_STDFile . #xA9EA)  ;Non-standard name
                (_Pack4 . #xA9EB)
                (_fp68k . #xA9EB)
                (_Pack5 . #xA9EC)
                (_elems68k . #xA9EC)
                (_Pack6 . #xA9ED)
                (_intUtil . #xA9ED) ;Non-standard name
                (_Pack7 . #xA9EE)
                (_DecStr68K . #xA9EE)
                (_Pack8 . #xA816)
                (_Pack9 . #xA82B)
                (_Pack10 . #xA82C)
                (_Pack11 . #xA82D)
                (_Pack12 . #xA82E)
                (_Pack13 . #xA82F)
                (_Pack14 . #xA830)
                (_Pack15 . #xA831)

                ;**************************************************
                ;QuickDraw traps:

                (_AddPt . #xA87E)
                (_ANGLEFROMSLOPE . #xA8C4)
                (_BackColor . #xA863)
                (_BackPat . #xA87C)
                (_CharWidth . #xA88D)
                (_ClipRect . #xA87B)
                (_ClosePgon . #xA8CC)
                (_ClosePicture . #xA8F4)
                (_ClosePort . #xA87D)
                (_CloseRgn . #xA8DB)
                (_ColorBit . #xA864)
                (_CopyBits . #xA8EC)
                (_CopyRgn . #xA8DC)
                (_DiffRgn . #xA8E6)
                (_DisposRgn . #xA8D9)
                (_DrawChar . #xA883)
                (_DrawPicture . #xA8F6)
                (_DrawString . #xA884)
                (_DrawText . #xA885)
                (_EmptyRect . #xA8AE)
                (_EmptyRgn . #xA8E2)
                (_EqualPt . #xA881)
                (_EqualRect . #xA8A6)
                (_EqualRgn . #xA8E3)
                (_EraseArc . #xA8C0)
                (_EraseOval . #xA8B9)
                (_ErasePoly . #xA8C8)
                (_EraseRect . #xA8A3)
                (_EraseRgn . #xA8D4)
                (_EraseRoundRect . #xA8B2)
                (_FillArc . #xA8C2)
                (_FillOval . #xA8BB)
                (_FillPoly . #xA8CA)
                (_FillRect . #xA8A5)
                (_FillRgn . #xA8D6)
                (_FillRoundRect . #xA8B4)
                (_ForeColor . #xA862)
                (_FrameArc . #xA8BE)
                (_FrameOval . #xA8B7)
                (_FramePoly . #xA8C6)
                (_FrameRect . #xA8A1)
                (_FrameRgn . #xA8D2)
                (_FrameRoundRect . #xA8B0)
                (_GetClip . #xA87A)
                (_GetFontInfo . #xA88B)
                (_GetPen . #xA89A)
                (_GetPenState . #xA898)
                (_GetPixel . #xA865)
                (_GetPort . #xA874)
                (_GlobalToLocal . #xA871)
                (_GrafDevice . #xA872)
                (_HideCursor . #xA852)
                (_HidePen . #xA896)
                (_InitCursor . #xA850)
                (_InitGraf . #xA86E)
                (_InitPort . #xA86D)
                (_InsetRect . #xA8A9)
                (_InsetRgn . #xA8E1)
                (_InverRect . #xA8A4)
                (_InverRgn . #xA8D5)
                (_InverRoundRect . #xA8B3)
                (_InvertArc . #xA8C1)
                (_InvertOval . #xA8BA)
                (_InvertPoly . #xA8C9)
                (_KillPicture . #xA8F5)
                (_KillPoly . #xA8CD)
                (_Line . #xA892)
                (_LineTo . #xA891)
                (_LocalToGlobal . #xA870)
                (_MapPoly . #xA8FC)
                (_MapPt . #xA8F9)
                (_MapRect . #xA8FA)
                (_MapRgn . #xA8FB)
                (_MOOV . #xA894) 
                (_Move . #xA894)
                (_MovePortTo . #xA877)
                (_MoveTo . #xA893)
                (_NewRgn . #xA8D8)
                (_ObscureCursor . #xA856)
                (_OffsetPoly . #xA8CE)
                (_OffsetRect . #xA8A8)
                (_OFSETRGN . #xA8E0) 
                (_OffsetRgn . #xA8E0)
                (_OpenPicture . #xA8F3)
                (_OpenPoly . #xA8CB)
                (_OpenPort . #xA86F)
                (_OpenRgn . #xA8DA)
                (_PackBits . #xA8CF)
                (_PaintArc . #xA8BF)
                (_PaintOval . #xA8B8)
                (_PaintPoly . #xA8C7)
                (_PaintRect . #xA8A2)
                (_PaintRgn . #xA8D3)
                (_PaintRoundRect . #xA8B1)
                (_PenMode . #xA89C)
                (_PenNormal . #xA89E)
                (_PenPat . #xA89D)
                (_PenSize . #xA89B)
                (_PicComment . #xA8F2)
                (_PortSize . #xA876)
                (_Pt2Rect . #xA8AC)
                (_PtInRect . #xA8AD)
                (_PtInRgn . #xA8E8)
                (_PtToAngle . #xA8C3)
                (_Random . #xA861)
                (_RectInRgn . #xA8E9)
                (_RectRgn . #xA8DF)
                (_ScalePt . #xA8F8)
                (_ScriptUtil . #xA8B5)
                (_ScrollRect . #xA8EF)
                (_SectRect . #xA8AA)
                (_SectRgn . #xA8E4)
                (_SetClip . #xA879)
                (_SetCursor . #xA851)
                (_SetEmptyRgn . #xA8DD)
                (_SetOrigin . #xA878)
                (_SetPBits . #xA875)
                (_SetPenState . #xA899)
                (_SetPort . #xA873)
                (_SetPt . #xA880)
                (_SetRecRgn . #xA8DE)
                (_SetRect . #xA8A7)
                (_SetStdProcs . #xA8EA)
                (_ShowCursor . #xA853)
                (_ShowPen . #xA897)
                (_ShutDown . #xA895)
                (_SLOPEFROMANGLE . #xA8BC)
                (_SpaceExtra . #xA88E)
                (_StdArc . #xA8BD)
                (_StdBits . #xA8EB)
                (_StdComment . #xA8F1)
                (_StdGetPic . #xA8EE)
                (_StdLine . #xA890)
                (_StdOval . #xA8B6)
                (_StdPoly . #xA8C5)
                (_StdPutPic . #xA8F0)
                (_StdRect . #xA8A0)
                (_StdRgn . #xA8D1)
                (_StdRRect . #xA8AF)
                (_StdText . #xA882)
                (_StdTxMeas . #xA8ED)
                (_StringWidth . #xA88C)
                (_StuffHex . #xA866)
                (_SubPt . #xA87F)
                (_TextFace . #xA888)
                (_TextFont . #xA887)
                (_TextMode . #xA889)
                (_TextSize . #xA88A)
                (_TextWidth . #xA886)
                (_Unimplemented . #xa89f)
                (_UnionRect . #xA8AB)
                (_UnionRgn . #xA8E5)
                (_UnpackBits . #xA8D0)
                (_XOrRgn . #xA8E7)
                (_CalcMask . #xA838)
                (_CopyMask . #xA817)
                (_GetMaskTable . #xA836)
                (_MeasureText . #xA837)
                (_SeedFill . #xA839)

                ;**************************************************
                ;Color traps:
                
                (_OPENCPORT . #xAA00) 
                (_INITCPORT . #xAA01) 
                (_CLOSECPORT . #xA87D) 
                (_NEWPIXMAP . #xAA03) 
                (_DISPOSPIXMAP . #xAA04) 
                (_COPYPIXMAP . #xAA05) 
                (_SETCPORTPIX . #xAA06) 
                (_NEWPIXPAT . #xAA07) 
                (_DISPOSPIXPAT . #xAA08) 
                (_COPYPIXPAT . #xAA09) 
                (_PENPIXPAT . #xAA0A) 
                (_BACKPIXPAT . #xAA0B) 
                (_GETPIXPAT . #xAA0C) 
                (_MAKERGBPAT . #xAA0D) 
                (_FILLCRECT . #xAA0E) 
                (_FILLCOVAL . #xAA0F) 
                (_FILLCROUNDRECT . #xAA10) 
                (_FILLCARC . #xAA11) 
                (_FILLCRGN . #xAA12) 
                (_FILLCPOLY . #xAA13) 
                (_RGBFORECOLOR . #xAA14) 
                (_RGBBACKCOLOR . #xAA15) 
                (_SETCPIXEL . #xAA16) 
                (_GETCPIXEL . #xAA17) 
                (_GETCTABLE . #xAA18) 
                (_GETFORECOLOR . #xAA19) 
                (_GETBACKCOLOR . #xAA1A) 
                (_GETCCURSOR . #xAA1B) 
                (_SETCCURSOR . #xAA1C) 
                (_ALLOCCURSOR . #xAA1D) 
                (_GETCICON . #xAA1E) 
                (_PLOTCICON . #xAA1F) 
                (_OPCOLOR . #xAA21) 
                (_HILITECOLOR . #xAA22) 
                (_CHAREXTRA . #xAA23) 
                (_DISPOSCTABLE . #xAA24) 
                (_DISPOSCICON . #xAA25) 
                (_DISPOSCCURSOR . #xAA26) 
                (_SEEDCFILL . #xAA50) 
                (_CALCCMASK . #xAA4F) 
                (_GETMAXDEVICE . #xAA27) 
                (_GETCTSEED . #xAA28) 
                (_GETDEVICELIST . #xAA29) 
                (_GETMAINDEVICE . #xAA2A) 
                (_GETNEXTDEVICE . #xAA2B) 
                (_TESTDEVICEATTRIBUTE . #xAA2C) 
                (_SETDEVICEATTRIBUTE . #xAA2D) 
                (_INITGDEVICE . #xAA2E) 
                (_NEWGDEVICE . #xAA2F) 
                (_DISPOSGDEVICE . #xAA30) 
                (_SETGDEVICE . #xAA31) 
                (_GETGDEVICE . #xAA32) 
                (_COLOR2INDEX . #xAA33) 
                (_INDEX2COLOR . #xAA34) 
                (_INVERTCOLOR . #xAA35) 
                (_REALCOLOR . #xAA36) 
                (_GETSUBTABLE . #xAA37) 
                (_MAKEITABLE . #xAA39) 
                (_ADDSEARCH . #xAA3A) 
                (_ADDCOMP . #xAA3B) 
                (_SETCLIENTID . #xAA3C) 
                (_PROTECTENTRY . #xAA3D) 
                (_RESERVEENTRY . #xAA3E) 
                (_SETENTRIES . #xAA3F) 
                (_QDERROR . #xAA40) 
                (_SAVEENTRIES . #xAA49) 
                (_RESTOREENTRIES . #xAA4A) 
                (_DELSEARCH . #xAA4C) 
                (_DELCOMP . #xAA4D) 
                (_SETSTDCPROCS . #xAA4E) 
                (_STDOPCODEPROC . #xABF8) 
                (_SETWINCOLOR . #xAA41) 
                (_GETAUXWIN . #xAA42) 
                (_SETCTLCOLOR . #xAA43) 
                (_GETAUXCTL . #xAA44) 
                (_NEWCWINDOW . #xAA45) 
                (_GETNEWCWINDOW . #xAA46) 
                (_SETDESKCPAT . #xAA47) 
                (_GETCWMGRPORT . #xAA48) 
                (_GETCVARIANT . #xA809) 
                (_GETWVARIANT . #xA80A) 
                (_DELMCENTRIES . #xAA60) 
                (_GETMCINFO . #xAA61) 
                (_SETMCINFO . #xAA62) 
                (_DISPMCINFO . #xAA63) 
                (_GETMCENTRY . #xAA64) 
                (_SETMCENTRIES . #xAA65) 
                (_MENUCHOICE . #xAA66) 
                (_SETFRACTENABLE . #xA814) 
                (_NEWCDIALOG . #xAA4B)
                (_InitPalettes . #xAA90)
                (_NewPalette	 . #xAA91)
                (_GetNewPalette . #xAA92)
                (_DisposePalette . #xAA93)
                (_ActivatePalette . #xAA94)
                (_SetPalette	 . #xAA95)
                (_GetPalette	 . #xAA96)
                (_PmForeColor . #xAA97)
                (_PmBackColor . #xAA98)
                (_AnimateEntry . #xAA99)
                (_AnimatePalette . #xAA9A) 
                (_GetEntryColor . #xAA9B)
                (_SetEntryColor . #xAA9C)
                (_GetEntryUsage . #xAA9D) 
                (_SetEntryUsage . #xAA9E)
                (_CTab2Palette . #xAA9F) 
                (_Palette2CTab . #xAAA0)


                ;**************************************************
                ;Resource Manager traps:

                ;_AddReference	trapw	#xA9AC	removed in Mac+
                (_AddResource . #xA9AB)
                (_ChangedResData . #xA9AA)
                (_CloseResFile . #xA99A)
                (_CountResources . #xA99C)
                (_CountTypes . #xA99E)
                (_CreateResFile . #xA9B1)
                (_CurResFile . #xA994)
                (_DetachResource . #xA992)
                (_GetIndResource . #xA99D)
                (_GetIndType . #xA99F)
                (_GetNamedResource . #xA9A1)
                (_GetResAttrs . #xA9A6)
                (_GetResFileAttrs . #xA9F6)
                (_GetResInfo . #xA9A8)
                (_GetResource . #xA9A0)
                (_HomeResFile . #xA9A4)
                (_InitResource . #xA995)
                (_LoadResource . #xA9A2)
                (_OpenResFile . #xA997)
                (_ReleaseResource . #xA9A3)
                (_ResError . #xA9AF)
                ;_RmveReference	trapw	#xA9AE	removed in Mac+
                (_RmveResource . #xA9AD)
                (_RsrcZoneInit . #xA996)
                (_SetResAttrs . #xA9A7)
                (_SetResFileAttrs . #xA9F7)
                (_SetResInfo . #xA9A9)
                (_SetResLoad . #xA99B)
                (_SetResPurge . #xA993)
                (_SizeResource . #xA9A5)
                (_UniqueID . #xA9C1)
                (_UpdateResFile . #xA999)
                (_UseResFile . #xA998)
                (_WriteResource . #xA9B0)
                (_Count1Resources . #xA80D)
                (_Count1Types . #xA81C)
                (_Get1IndResource . #xA80E)
                (_Get1IndType . #xA80F)
                (_GET1IXRESOURCE . #xA80E) 
                (_GET1IXTYPE . #xA80F)
                (_Get1NamedResource . #xA820)
                (_Get1Resource . #xA81F)
                (_MaxSizeRsrc . #xA821)
                (_OpenRFPerm . #xA9C4)
                (_RsrcMapEntry . #xA9C5)
                (_Unique1ID . #xA810)
                (_XMUNGER . #xA819) 

                ;**************************************************
                ;Scrap Manager traps:

                (_GetScrap . #xA9FD)
                (_InfoScrap . #xA9F9)
                (_LodeScrap . #xA9FB)
                (_PutScrap . #xA9FE)
                (_UnlodeScrap . #xA9FA)
                (_ZeroScrap . #xA9FC)

                ;**************************************************
                ;Segment Loader traps:

                (_ExitToShell . #xA9F4)
                (_GetAppParms . #xA9F5)
                (_LoadSeg . #xA9F0)
                (_UnloadSeg . #xA9F1)

                ;**************************************************
                ;Text Edit traps:

                (_TEActivate . #xA9D8)
                (_TECalText . #xA9D0)
                (_TEClick . #xA9D4)
                (_TECopy . #xA9D5)
                (_TECut . #xA9D6)
                (_TEDeactivate . #xA9D9)
                (_TEDelete . #xA9D7)
                (_TEDispose . #xA9CD)
                (_TEGetText . #xA9CB)
                (_TEIdle . #xA9DA)
                (_TEInit . #xA9CC)
                (_TEInsert . #xA9DE)
                (_TEKey . #xA9DC)
                (_TENew . #xA9D2)
                (_TEPaste . #xA9DB)
                (_TEScroll . #xA9DD)
                (_TESetJust . #xA9DF)
                (_TESetSelect . #xA9D1)
                (_TESetText . #xA9CF)
                (_TEUpdate . #xA9D3)
                (_TextBox . #xA9CE)
                (_TEAutoView . #xA813)
                (_TEPinScroll . #xA812)
                (_TESelView . #xA811)

                ;**************************************************
                ;Toolbox Utilities traps:

                (_BitAnd . #xA858)
                (_BitClr . #xA85F)
                (_BitNot . #xA85A)
                (_BitOr . #xA85B)
                (_BitSet . #xA85E)
                (_BitShift . #xA85C)
                (_BitTst . #xA85D)
                (_BitXOr . #xA859)
                (_FixMul . #xA868)
                (_FixRatio . #xA869)
                (_FixRound . #xA86C)
                (_GetCursor . #xA9B9)
                (_GetIcon . #xA9BB)
                (_GetPattern . #xA9B8)
                (_GetPicture . #xA9BC)
                (_GetString . #xA9BA)
                (_HiWord . #xA86A)
                (_LongMul . #xA867)
                (_LoWord . #xA86B)
                (_Munger . #xA9E0)
                (_NewString . #xA906)
                (_PlotIcon . #xA94B)
                (_SetString . #xA907)
                (_ShieldCursor . #xA855)
                (_FixAtan2 . #xA818)
                (_FixDiv . #xA84D)
                (_Fix2Frac . #xA841)
                (_Fix2Long . #xA840)
                (_Fix2X . #xA843)
                (_FracCos . #xA847)
                (_FracDiv . #xA84B)
                (_FracMul . #xA84A)
                (_FracSin . #xA848)
                (_FracSqrt . #xA849)
                (_Frac2Fix . #xA842)
                (_Frac2X . #xA845)
                (_Long2Fix . #xA83F)
                (_X2Fix . #xA844)
                (_X2Frac . #xA846)

                ;**************************************************
                ;Window Manager traps:

                (_BeginUpdate . #xA922)
                (_BringToFront . #xA920)
                (_CalcVBehind . #xA90A)
                (_CalcVis . #xA909)
                (_CheckUpdate . #xA911)
                (_ClipAbove . #xA90B)
                (_CloseWindow . #xA92D)
                (_DisposWindow . #xA914)
                (_DragGrayRgn . #xA905)
                (_DragTheRgn . #xA926)
                (_DragWindow . #xA925)
                (_DrawGrowIcon . #xA904)
                (_DrawNew . #xA90F)
                (_EndUpdate . #xA923)
                (_FindWindow . #xA92C)
                (_FrontWindow . #xA924)
                (_GetNewWindow . #xA9BD)
                (_GetWindowPic . #xA92F)
                (_GetWMgrPort . #xA910)
                (_GetWRefCon . #xA917)
                (_GetWTitle . #xA919)
                (_GrowWindow . #xA92B)
                (_HideWindow . #xA916)
                (_HiliteWindow . #xA91C)
                (_InitWindows . #xA912)
                (_InvalRect . #xA928)
                (_InvalRgn . #xA927)
                (_MoveWindow . #xA91B)
                (_NewWindow . #xA913)
                (_PaintBehind . #xA90D)
                (_PaintOne . #xA90C)
                (_PinRect . #xA94E)
                (_SaveOld . #xA90E)
                (_SelectWindow . #xA91F)
                (_SendBehind . #xA921)
                (_SetWindowPic . #xA92E)
                (_SetWRefCon . #xA918)
                (_SetWTitle . #xA91A)
                (_ShowHide . #xA908)
                (_ShowWindow . #xA915)
                (_SizeWindow . #xA91D)
                (_TrackGoAway . #xA91E)
                (_ValidRect . #xA92A)
                (_ValidRgn . #xA929)
                (_TrackBox . #xA83B)
                (_ZoomWindow . #xA83A)

                ;**************************************************
                ; Sound Manager Traps

                ;<according to Apple, these *don't* begin with underbars>
	
                (_SndDoCommand . #xA803)
                (_SndDoImmediate . #xA804)
                (_SndAddModifier . #xA802)
                (_SndNewChannel . #xA807)
                (_SndDisposeChannel . #xA801)
                (_SndPlay . #xA805)
                (_SndControl . #xA806)

                ;**************************************************
                ;Misc. traps:

                (_SCSIDispatch . #xA815)
                (_SysBeep . #xA9C8)
                ;the below added by cfry 10/22/87
                (_SCRNBITMAP . #xA833) 
                (_DELTAPOINT . #xA94F) 
                (_UPDTCONTROL . #xA953) 
                (_INITRESOURCES . #xA995) 
                (_SIZERSRC . #xA9A5) 
                (_CHANGEDRESOURCE . #xA9AA) 
                (_ADDREFERENCE . #xA9AC) 
                (_RMVEREFERENCE . #xA9AE) 
                (_METHODDISPATCH . #xA9F8) 
                (_DEBUGSTR . #xABFF)
                (_RGETRESOURCE . #xA80C) 
                (_INITPROCMENU . #xA808) 
                (_GETITEMCMD . #xA84E) 
                (_SETITEMCMD . #xA84F) 
                (_POPUPMENUSELECT . #xA80B) 
                (_KEYTRANS . #xA9C3) 
                (_TEGETOFFSET . #xA83C) 
                (_TEDISPATCH . #xA83D) 
                (_TESTYLENEW . #xA83E)
                (_PrintDispatch . #xA8FD)
 
               )))
  (dotimes (i (length traps))
    (let ((data (svref traps i)))
      (export (car data))
      (ccl::define-constant (car data) (cdr data))
      (setf (macro-function (car data)) macro-fn))))

;; ####################################################################
;; ######################## Register Traps ############################
;; ####################################################################
(let ((macro-fn #'register-trap-macro-function)
      (traps '#(
                ;**************************************************
                ;Event Manager traps:
                (_FlushEvents . #xA032)
                (_PostEvent . #xA02F)

                ;**************************************************
                ;Device Manager traps:

                (_Control . #xA004)
                (_AControl . #xA404)
                (_KillIO . #xA006)
                (_Status . #xA005)

                ;**************************************************
                ;File Manager traps:

                (_AddDrive . #xA04E)
                (_Allocate . #xA010)
                (_AllocContig . #xA210)
                (_Close . #xA001)
                (_Create . #xA008)
                (_Delete . #xA009)
                (_Eject . #xA017)
                (_FINITQUEUE . #xA016) 
                (_FlushFile . #xA045)
                (_FlushVol . #xA013)
                (_GetEOF . #xA011)
                (_GetFileInfo . #xA00C)
                (_GetFPos . #xA018)
                (_GetVol . #xA014)
                (_GetVolInfo . #xA007)
                (_HCreate . #xA208)
                (_HDelete . #xA209)
                (_HGetFileInfo . #xA20C)
                (_HGetVInfo . #xA207)
                (_HGetVol . #xA214)
                (_HOpen . #xA200)
                (_HOpenRF . #xA20A)
                (_HRename . #xA20B)
                (_HRstFLock . #xA242)
                (_HSetFileInfo . #xA20D)
                (_HSetFLock . #xA241)
                (_HSetVol . #xA215)
                (_InitQueue . #xA016)
                (_MountVol . #xA00F)
                (_Offline . #xA035)
                (_Open . #xA000)
                (_OpenRF . #xA00A)
                (_Read . #xA002)
                (_Rename . #xA00B)
                (_RstFilLock . #xA042)
                (_SetEOF . #xA012)
                (_SetFileInfo . #xA00D)
                (_SetFilLock . #xA041)
                (_SetFilType . #xA043)
                (_SetFPos . #xA044)
                (_SetPEOF . #xA212)
                (_SetVol . #xA015)
                (_UnmountVol . #xA00E)
                (_Write . #xA003)
                (_HFSDispatch . #xA260)

                ;**************************************************
                ;Memory Manager traps:

                (_BlockMove . #xA02E)
                (_CompactMem . #xA14C)
                (_DisposHandle . #xA023)
                (_DisposPtr . #xA01F)
                (_EmptyHandle . #xA02B)
                (_FreeMem . #xA01C)
                (_GetHandleSize . #xA025)
                (_GetPtrSize . #xA021)
                (_GetZone . #xA11A)
                (_HandleZone . #xA126)
                (_HLock . #xA029)
                (_HNoPurge . #xA04A)
                (_HPurge . #xA049)
                (_HUnlock . #xA02A)
                (_InitApplZone . #xA02C)
                (_InitZone . #xA019)
                (_MaxMem . #xA11D)
                (_MoreMasters . #xA036)
                (_NewHandle . #xA122)
                (_NewPtr . #xA11E)
                (_PtrZone . #xA148)
                (_PurgeMem . #xA14D)
                (_ReallocHandle . #xA027)
                (_RecoverHandle . #xA128)
                (_ReservMem . #xA140)
                (_SetAppBase . #xA857)
                (_SetApplLimit . #xA02D)
                (_SetGrowZone . #xA04B)
                (_SetHandleSize . #xA024)
                (_SetPtrSize . #xA020)
                (_SetZone . #xA01B)
                (_HClrRBit . #xA068)
                (_HGetState . #xA069)
                (_HSetRBit . #xA067)
                (_HSetState . #xA06A)
                (_MaxApplZone . #xA063)
                (_MaxBlock . #xA061)
                (_MoveHHi . #xA064)
                (_NewEmptyHandle . #xA166)
                (_PurgeSpace . #xA062)
                (_StackSpace . #xA065)

                ;**************************************************
                ;OS Event Manager traps:

                (_GetOSEvent . #xA031)
                (_OSEventAvail . #xA030)

                ;**************************************************
                ;OS Utilities traps:

                (_CmpString . #xA03C)
                (_Date2Secs . #xA9C7)
                (_Delay . #xA03B)
                (_Dequeue . #xA96E)
                (_DrvrInstall . #xA03D)
                (_DrvrRemove . #xA03E)
                (_Enqueue . #xA96F)
                (_GetTrapAddress . #xA146)
                (_HandAndHand . #xA9E4)
                (_HandToHand . #xA9E1)
                (_InitUtil . #xA03F)
                (_PtrAndHand . #xA9EF)
                (_PtrToHand . #xA9E3)
                (_PtrToXHand . #xA9E2)
                (_RDrvrInstall . #xA04F)
                (_ReadParm . #xA037)
                (_PutIcon . #xA9CA)
                (_ReadDateTime . #xA039)
                (_Secs2Date . #xA9C6)
                (_SetDateTime . #xA03A)
                (_SetTrapAddress . #xA047)
                (_SysError . #xA9C9)
                (_UprString . #xA854)
                (_VInstall . #xA033)
                (_VRemove . #xA034)
                (_WriteParam . #xA038)
                (_Chain . #xA9F3)
                (_Launch . #xA9F2)
                (_RelString . #xA050)

                ;**************************************************
                ;Miscellaneous new register traps:

                (_PPOSTEVENT . #xA12F) 
                (_RESRVMEM . #xA040) 
                (_SETAPPLBASE . #xA057) 
                (_READXPRAM . #xA051) 
                (_WRITEXPRAM . #xA052) 
                (_INSTIME . #xA058) 
                (_RMVTIME . #xA059) 
                (_PRIMETIME . #xA05A) 
                (_INITFS . #xA06C) 
                (_INITEVENTS . #xA06D) 
                (_STRIPADDRESS . #xA055) 
                (_SWAPMMUMODE . #xA05D) 
                (_SLOTVINSTALL . #xA06F) 
                (_SLOTVREMOVE . #xA070) 
                (_ATTACHVBL . #xA071) 
                (_DOVBLTASK . #xA072) 
                (_SINTINSTALL . #xA075) 
                (_SINTREMOVE . #xA076) 
                (_COUNTADBS . #xA077) 
                (_GETINDADB . #xA078) 
                (_GETADBINFO . #xA079) 
                (_SETADBINFO . #xA07A) 
                (_ADBREINIT . #xA07B) 
                (_ADBOP . #xA07C) 
                (_GETDEFAULTSTARTUP . #xA07D) 
                (_SETDEFAULTSTARTUP . #xA07E) 
                (_INTERNALWAIT . #xA07F)
                (_GETVIDEODEFAULT . #xA080) 
                (_SETVIDEODEFAULT . #xA081) 
                (_DTINSTALL . #xA082) 
                (_SETOSDEFAULT . #xA083) 
                (_GETOSDEFAULT . #xA084)
                (_SYSENVIRONS . #xA090)
                (_OSDispatch . #xA88F)
                (_Gestalt . #xa1ad)

                (_DEBUGGER . #xA9FF)

               )))
  (dotimes (i (length traps))
    (let ((data (svref traps i)))
      (export (car data))
      (ccl::define-constant (car data) (cdr data))
      (setf (macro-function (car data)) macro-fn))))
   ;end of REGISTER TRAPS


;; ####################################################################
;; ######################### Dispatching Traps ########################
;; ####################################################################
(let ((macro-fn #'dispatch-stack-trap-macro-function)
      (traps '#(
                (_DIBadMount 0 . _Pack2)
                (_DILoad 2 . _Pack2)
                (_DIUnload 4 . _Pack2)
                (_DIFormat 6 . _Pack2)
                (_DIVerify 8 . _Pack2)
                (_DIZero 10 . _Pack2)
                (_LActivate 0 . _Pack0)
                (_LAddColumn 4 . _Pack0)
                (_LAddRow 8 . _Pack0)
                (_LAddToCell 12 . _Pack0)
                (_LAutoScroll 16 . _Pack0)
                (_LCellSize 20 . _Pack0)
                (_LClick 24 . _Pack0)
                (_LClrCell 28 . _Pack0)
                (_LDelColumn 32 . _Pack0)
                (_LDelRow 36 . _Pack0)
                (_LDispose 40 . _Pack0)
                (_LDoDraw 44 . _Pack0)
                (_LDraw 48 . _Pack0)
                (_LFind 52 . _Pack0)
                (_LGetCell 56 . _Pack0)
                (_LGetSelect 60 . _Pack0)
                (_LLastClick 64 . _Pack0)
                (_LNew 68 . _Pack0)
                (_LNextCell 72 . _Pack0)
                (_LRect 76 . _Pack0)
                (_LScroll 80 . _Pack0)
                (_LSearch 84 . _Pack0)
                (_LSetCell 88 . _Pack0)
                (_LSetSelect 92 . _Pack0)
                (_LSize 96 . _Pack0)
                (_LUpdate 100 . _Pack0)

                (_SFPutFile 1 . _Pack3)
                (_SFGetFile 2 . _Pack3)
                (_SFPPutFile 3 . _Pack3)
                (_SFPGetFile 4 . _Pack3)

                (_iuDateString 0 . _Pack6)
                (_iuTimeString 2 . _Pack6)
                (_iuMetric 4 . _Pack6)
                (_iuGetIntl 6 . _Pack6)
                (_iuSetIntl 8 . _Pack6)
                (_iuMagString 10 . _Pack6)
                (_iuMagIDString 12 . _Pack6)
                (_iuDatePString 14 . _Pack6)
                (_iuTimePString 16 . _Pack6)

                ;Unfortunately the args are in registers but package selector is on stack.
                ;(_numtostring 0 . _Pack7)
                ;(_stringtonum 1 . _Pack7)
		
		; AppleEvents.  I have no idea about the leading underscore.
		(_AECreateDesc #x0825 . _Pack8)
		(_AECoercePtr #x0a02 . _Pack8)
		(_AECoerceDesc #x0603 . _Pack8)
		(_AEDisposeDesc #x0204 . _Pack8)
		(_AEDuplicateDesc #x0405 . _Pack8)
		
		(_AECreateList #x0706 . _Pack8)
		(_AECountItems #x0407 . _Pack8)
		(_AEPutPtr #x0a08 . _Pack8)
		(_AEPutDesc #x0609 . _Pack8)
		(_AEGetNthPtr #x100a . _Pack8)
		(_AEGetNthDesc #x0a0b . _Pack8)
		(_AESizeOfNthItem #x082a . _Pack8)
		(_AEGetArray #x0d0c . _Pack8)
		(_AEPutArray #x0b0d . _Pack8)
		(_AEDeleteIndex #x0403 . _Pack8)
		
		(_AEPutKeyPtr #x0a0f . _Pack8)
		(_AEPutParamPtr #x0a0f . _Pack8)
		(_AEPutKeyDesc #x0610 . _Pack8)
		(_AEPutParamDesc #x0610 . _Pack8)
		(_AEGetKeyPtr #x0e11 . _Pack8)
		(_AEGetParamPtr #x0e11 . _Pack8)
		(_AEGetKeyDesc #x0812 . _Pack8)
		(_AEGetParamDesc #x0812 . _Pack8)
		(_AESizeOfKeyDesc #x0829 . _Pack8)
		(_AESizeOfParam #x0829 . _Pack8)
		(_AEDeleteKeyDesc #x0413 . _Pack8)
		(_AEDeleteParam #x0413 . _Pack8)

		(_AEGetAttributePtr #x0e15 . _Pack8)
		(_AEGetAttributeDesc #x0826 . _Pack8)
		(_AESizeOfAttribute #x0828 . _Pack8)
		(_AEPutAttributePtr #x0a16 . _Pack8)
		(_AEPutAttributeDesc #x0627 . _Pack8)
		
		(_AECreateAppleEvent #x0b14 . _Pack8)
		(_AESend #x0d17 . _Pack8)
		(_AEProcessAppleEvent #x021b . _Pack8)
		(_AEResetTimer #x0219 . _Pack8)
		
		(_AESuspendTheCurrentEvent #x022b . _Pack8)
		(_AEResumeTheCurrentEvent #x0818 . _Pack8)
		(_AEGetTheCurrentEvent #x021a . _Pack8)
		(_AESetTheCurrentEvent #x022c . _Pack8)
		
		(_AEGetInteractionAllowed #x021d . _Pack8)
		(_AESetInteractionAllowed #x011e . _Pack8)
		(_AEInteractWithUser #x061c / _Pack8)
		
		(_AEInstallEventHandler #x091f . _Pack8)
		(_AERemoveEventHandler #x0720 . _Pack8)
		(_AEGetEventHandler #x0921 . _Pack8)

		(_AEInstallCoercionHandler #x0a22 . _Pack8)
		(_AERemoveCoercionHandler #x0723 . _Pack8)
		(_AEGetCoercionHandler #x0b24 . _Pack8)

		(_AEInstallSpecialHandler #x0500 . _Pack8)
		(_AERemoveSpecialHandler #x0500 . _Pack8)
		(_AEGetSpecialHandler #x052c . _Pack8)

                (_Fix2SmallFract 1 . _Pack12)
                (_SmallFract2Fix 2 . _Pack12)
                (_CMY2RGB 3 . _Pack12)
                (_RGB2CMY 4 . _Pack12)
                (_HSL2RGB 5 . _Pack12)
                (_RGB2HSL 6 . _Pack12)
                (_HSV2RGB 7 . _Pack12)
                (_RGB2HSV 8 . _Pack12)
                (_GetColor 9 . _Pack12)

                ;;The following are the new TextEdit traps
                (_TEStylPaste 0 . _TEDispatch)
                (_TESetStyle 1 . _TEDispatch)
                (_TEReplaceStyle 2 . _TEDispatch)
                (_TEGetStyle 3 . _TEDispatch)
                (_GetStylHandle 4 . _TEDispatch)
                (_SetStylHandle 5 . _TEDispatch)
                (_GetStylScrap 6 . _TEDispatch)
                (_TEStylInsert 7 . _TEDispatch)
                (_TEGetPoint 8 . _TEDispatch)
                (_TEGetHeight 9 . _TEDispatch)

                ;;Multifinder Traps
		(_MFMaxMem 21 . _OSDispatch)
                (_MFFreeMem 24 . _OSDispatch)
                (_MFTempNewHandle 29 . _OSDispatch)
		(_MFTempHLock 30 . _OSDispatch)
		(_MFTempHUnLock 31 . _OSDispatch)
		(_MFTempDisposHandle 32 . _OSDispatch)

                ;;Shutdown Manager
                (_SDPowerOff 1 . _ShutDown)
                (_SDRestart 2 . _ShutDown)
                (_SDInstall 3 . _ShutDown)
                (_SDRemove 4 . _ShutDown)

                ;;SCSI Manager
                (_SCSIReset 0 . _SCSIDispatch)
                (_SCSIGet 1 . _SCSIDispatch)
                (_SCSISelect 2 . _SCSIDispatch)
                (_SCSICmd 3 . _SCSIDispatch)
                (_SCSIComplete 4 . _SCSIDispatch)
                (_SCSIRead 5 . _SCSIDispatch)
                (_SCSIWrite 6 . _SCSIDispatch)
                (_SCSIRBlind 8 . _SCSIDispatch)
                (_SCSIWBlind 9 . _SCSIDispatch)
                (_SCSIStat 10 . _SCSIDispatch)
                (_SCSISelAtn 11 . _SCSIDispatch)
                (_SCSIMsgIn 12 . _SCSIDispatch)
                (_SCSIMsgOut 13 . _SCSIDispatch)

               )))
  (dotimes (i (length traps))
    (let ((data (svref traps i)))
      (export (car data))
      (ccl::define-constant (car data) (cadr data))
      (setf (gethash (car data) *dispatching-traps*) (cddr data))
      (setf (macro-function (car data)) macro-fn))))

;; ####################################################################
;; ##################### Long Dispatching Traps #######################
;; ####################################################################
(let ((macro-fn #'long-dispatch-stack-trap-macro-function)
      (traps '#(
                ;;Script Manager
                (_FontScript    #x82000000 . _ScriptUtil)
		(_IntlScript    #x82000002 . _ScriptUtil)
		(_KeyScript     #x80020004 . _ScriptUtil)
		(_Font2Script   #x82020006 . _ScriptUtil)
		(_GetEnvirons   #x84020008 . _ScriptUtil)
		(_SetEnvirons   #x8206000A . _ScriptUtil)
		(_GetScript     #x8404000C . _ScriptUtil)
		(_SetScript     #x8208000E . _ScriptUtil)
		(_CharByte      #x82060010 . _ScriptUtil)
		(_CharType      #x82060012 . _ScriptUtil)
		(_Pixel2Char    #x820E0014 . _ScriptUtil)
		(_Char2Pixel    #x820C0016 . _ScriptUtil)
		(_Transliterate #x820E0018 . _ScriptUtil)
		(_FindWord      #x8012001A . _ScriptUtil)
		(_HiliteText    #x800E001C . _ScriptUtil)
		(_DrawJust      #x8008001E . _ScriptUtil)
		(_MeasureJust   #x800C0020 . _ScriptUtil)
		(_ParseTable    #x82040022 . _ScriptUtil)

                ;;Print Manager Calls
                (_PrOpenDoc #x04000C00 . _PrintDispatch)
                (_PrClosDoc #x08000484 . _PrintDispatch)
                (_PrOpenPage #x10000808 . _PrintDispatch)
                (_PrClosPage #x1800040C . _PrintDispatch)
                (_PrintDefault #x20040480 . _PrintDispatch)
                (_PrStlDialog #x2A040484 . _PrintDispatch)
                (_PrJobDialog #x32040488 . _PrintDispatch)
                (_PrStlInit #x3C04040C . _PrintDispatch)
                (_PrJobInit #x44040410 . _PrintDispatch)
                (_PrDlgMain #x4A040894 . _PrintDispatch)
                (_PrValidate #x52040498 . _PrintDispatch)
                (_PrJobMerge #x5804089C . _PrintDispatch)
                (_PrPicFile #x60051480 . _PrintDispatch)
                (_PrHack #x6C070C80 . _PrintDispatch)
                (_PrGeneral #x70070480 . _PrintDispatch)
                (_PrDrvrOpen #x80000000 . _PrintDispatch)
                (_PrDrvrClose #x88000000 . _PrintDispatch)
                (_PrDrvrDCE #x94000000 . _PrintDispatch)
                (_PrDrvrVers #x9A000000 . _PrintDispatch)
                (_PrCtlCall #xA0000E00 . _PrintDispatch)
                (_PrPurge #xA8000000 . _PrintDispatch)
                (_PrNoPurge #xB0000000 . _PrintDispatch)
                (_PrErro #xBA000000 . _PrintDispatch)
                (_PrSetError #xC0000200 . _PrintDispatch)
                (_PrOpen #xC8000000 . _PrintDispatch)
                (_PrClose #xD0000000 . _PrintDispatch)

               )))
  (dotimes (i (length traps))
    (let ((data (svref traps i)))
      (export (car data))
      (ccl::define-constant (car data) (cadr data))
      (setf (gethash (car data) *dispatching-traps*) (cddr data))
      (setf (macro-function (car data)) macro-fn))))

;; ####################################################################
;; ########################### HFS Traps ##############################
;; ####################################################################
(let ((macro-fn #'hfs-trap-macro-function)
      (traps '#((_FSControl . 0)
                (_OpenWD . 1)
                (_CloseWD . 2)
                (_CatMove . 5)
                (_DirCreate . 6)
                (_GetWDInfo . 7)
                (_GetFCBInfo . 8)
                (_GetCatInfo . 9)
                (_SetCatInfo . 10)
                (_SetVolInfo . 11)
                (_SetPMSP . 12)
                (_LockRng . 16)
                (_UnlockRng . 17)
                (_GetVolParms .#x30)
                (_GetLogInInfo . #x31)
                (_GetDirAccess . #x32)
                (_SetDirAccess . #x33)
                (_MapID	. #x34)
                (_MapName . #x35)
                (_CopyFile . #x36)
                (_MoveRename . #x37)
                (_OpenDeny . #x38)
                (_OpenRFDeny . #x39)
               )))
  (dotimes (i (length traps))
    (let ((data (svref traps i)))
      (export (car data))
      (ccl::define-constant (car data) (cdr data))
      (setf (macro-function (car data)) macro-fn))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;almost all done

(provide 'traps)
