(in-package :user)

;;;____________________________________________________________________________________
;;;  collect-all-relations

(defun collect-all-relations ()
  (setf *all-relations* (copy-list *extensional-preds*)
        *all-relations* (nconc *all-relations* (copy-list *builtin-preds*))
        *all-relations* (nconc *all-relations* (copy-list *intensional-preds*))
        *all-relations* (stable-sort *all-relations* 
                                     #'(lambda (r1 r2) 
                                         (string< (p-name (cdr r1)) (p-name (cdr r2))))))
  *all-relations*)

;;;____________________________________________________________________________________
;;;  selector-dialog

(defclass selector-dialog (window)
  ((default-when-selected :initarg :default-when-selected :initform nil :accessor default-when-selected)
   (enable-only-when-selected :initarg :enable-only-when-selected :initform nil :accessor enable-only-when-selected)
   (default-when-unselected :initarg :default-when-unselected :initform nil :accessor default-when-unselected)))

;;;____________________________________________________________________________________
;;;  remove-keywords-and-values

(defun remove-keywords-and-values (list keywords)
  (dolist (keyword keywords)
    (setf list (remove-keyword-and-value list keyword)))
  list)

;;;____________________________________________________________________________________
;;;  remove-keyword-and-value

(defun remove-keyword-and-value (list keyword)
  (do* ((key t (not key))
        (prev nil rest)
        (rest list (cdr rest)))
       ((null rest) list)
    (when (and (eq (car rest) keyword) key)
      (cond (prev
             (rplacd prev (cddr rest))
             (setf rest prev))
            (t (setf list (cddr rest)
                     rest list))))))

;;;____________________________________________________________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window selector-dialog) &rest init-args &key &allow-other-keys)
  (let ((button-0-name (getf init-args :button-0-name))
        (button-0-enabled-p (getf init-args :button-0-enabled-p t))
        (button-0-default (getf init-args :button-0-default))         
        (button-0-action (getf init-args :button-0-action))
        (button-1-name (getf init-args :button-1-name))
        (button-1-enabled-p (getf init-args :button-1-enabled-p t))
        (button-1-default (getf init-args :button-1-default))         
        (button-1-action (getf init-args :button-1-action))
        (button-2-name (getf init-args :button-2-name))
        (button-2-enabled-p (getf init-args :button-2-enabled-p t))
        (button-2-default (getf init-args :button-2-default))         
        (button-2-action (getf init-args :button-2-action))
        (button-3-name (getf init-args :button-3-name))
        (button-3-enabled-p (getf init-args :button-3-enabled-p t))
        (button-3-default (getf init-args :button-3-default))         
        (button-3-action (getf init-args :button-3-action))
        (message (getf init-args :message "Select and item from list"))
        (table-sequence (getf init-args :table-sequence))
        (table-print-function (getf init-args :table-print-function #'princ))
        (selection-type (getf init-args :selection-type :single)))

    (setf init-args (remove-keywords-and-values init-args '(:button-0-name
                                                            :button-0-enabled-p
                                                            :button-0-default         
                                                            :button-0-action
                                                            :button-1-name
                                                            :button-1-enabled-p
                                                            :button-1-default       
                                                            :button-1-action
                                                            :button-2-name
                                                            :button-2-enabled-p
                                                            :button-2-default         
                                                            :button-2-action
                                                            :button-3-name
                                                            :button-3-enabled-p
                                                            :button-3-default         
                                                            :button-3-action
                                                            :message
                                                            :table-sequence
                                                            :table-print-function
                                                            :selection-type)))
    (apply #'call-next-method window init-args)

    (let* ((h (point-h (view-size window)))
           (v (point-v (view-size window)))
           (x 10)
           (y 5)
           (w (- h x x))
           (static-text-h 20)
           (selector-font '("Monaco" 9 :SRCOR :PLAIN))
           (button-0-pos (make-point (- h 90) (- v 30)))
           (button-1-pos (make-point (- h 160) (- v 30)))
           (button-2-pos (make-point (- h 230) (- v 30)))
           (button-3-pos (make-point (- h 300) (- v 30))))
 
      (add-subviews window
                    (make-dialog-item 'static-text-dialog-item
                                      (make-point x y) (make-point w static-text-h)
                                      message nil
                                      :view-nick-name :message)
                    (make-dialog-item 'sequence-dialog-item
                                      (make-point x (incf y 20)) (make-point w (- v y 41))
                                      "Untitled" nil
                                      :cell-size (make-point (- w 15) 14)
                                      :table-hscrollp nil
                                      :table-vscrollp t
                                      :table-sequence table-sequence
                                      :table-print-function table-print-function
                                      :selection-type selection-type
                                      :view-font selector-font
                                      :view-nick-name :item-selector))
      (when button-0-name
        (add-subviews window
                      (make-dialog-item 'button-dialog-item
                                        button-0-pos #@(60 20)
                                        button-0-name
                                        button-0-action
                                        :default-button button-0-default
                                        :dialog-item-enabled-p button-0-enabled-p
                                        :view-nick-name :button-0))
        (when button-1-name
         (add-subviews window
                       (make-dialog-item 'button-dialog-item
                                         button-1-pos #@(60 20)
                                         button-1-name
                                         button-1-action
                                         :default-button button-1-default
                                         :dialog-item-enabled-p button-1-enabled-p
                                         :view-nick-name :button-1))
         (when button-2-name
           (add-subviews window
                         (make-dialog-item 'button-dialog-item
                                           button-2-pos #@(60 20)
                                           button-2-name
                                           button-2-action
                                           :default-button button-2-default
                                           :dialog-item-enabled-p button-2-enabled-p
                                           :view-nick-name :button-2))
           (when button-3-name
             (add-subviews window
                           (make-dialog-item 'button-dialog-item
                                             button-3-pos #@(60 20)
                                             button-3-name
                                             button-3-action
                                             :default-button button-3-default
                                             :dialog-item-enabled-p button-3-enabled-p
                                             :view-nick-name :button-3)))))))))


;;;__________________________________________________________________________________
;;; select-corresponding-cell

(defmethod select-corresponding-cell ((dialog selector-dialog) contents)
  (let ((item-selector (view-named :item-selector dialog)))
    (cell-select item-selector
                 (index-to-cell item-selector (position contents (table-sequence item-selector))))))


;;;____________________________________________________________________________________
;;;  view-click-event-handler

(defmethod view-click-event-handler ((dialog selector-dialog) where)
  (declare (ignore where))
  (call-next-method)
  (let ((item-selector (view-named :item-selector dialog))
        (enable-disable-list (enable-only-when-selected dialog)))
    (cond ((selected-cells item-selector)
           (set-default-button dialog (view-named (default-when-selected dialog) dialog))
           (dolist (button-name enable-disable-list)
             (dialog-item-enable (view-named button-name dialog))))
          (t 
           (set-default-button dialog (view-named (default-when-unselected dialog) dialog))
           (dolist (button-name enable-disable-list)
             (dialog-item-disable (view-named button-name dialog)))))))


;;;____________________________________________________________________________________
;;;  set-view-size

(defmethod set-view-size ((window selector-dialog) h &optional v)
  (let ((old-size (view-size window)))
    (with-focused-view window 
      (apply #'call-next-method window h v)
      (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
        (_EraseRect :ptr view-rect)
        (_BeginUpdate :ptr (wptr window))
        (let* ((new-size (view-size window))
               (offset (subtract-points new-size old-size))
               (button-0 (view-named :button-0 window))
               (button-1 (view-named :button-1 window))
               (button-2 (view-named :button-2 window))
               (button-3 (view-named :button-3 window))
               (item-selector (view-named :item-selector window))
               (new-selector-size (add-points (view-size item-selector) offset)))
          
          (set-view-size item-selector new-selector-size)
          (set-cell-size item-selector (- (point-h new-selector-size) 15) 14)
          (when button-0
            (set-view-position button-0 (add-points (view-position button-0) offset))
            (when button-1
              (set-view-position button-1 (add-points (view-position button-1) offset))
              (when button-2
                (set-view-position button-2 (add-points (view-position button-2) offset))
                (when button-3
                  (set-view-position button-3 (add-points (view-position button-3) offset)))))))
        (_EndUpdate :ptr (wptr window))
        (_InvalRect :ptr view-rect)))))

;;;____________________________________________________________________________________
;;;  inspect-relations

(defun inspect-relations 
       (&key 
        (preds (collect-all-relations))
        (title "Relations")
        (message "Select a Relation to Inspect")
        (print-function #'(lambda (p stream)
                            (let ((pred (cdr p)))
                              (format stream "~(~A~) ~S"
                                      (pred-name pred)
                                      (pred-type pred)))))
        (action-function #'(lambda (item)
                             (let* ((item-selector (find-named-sibling item :item-selector))
                                    (cells-selected (selected-cells item-selector)))
                               (dolist (p 
                                        (nreverse 
                                         (mapcar #'(lambda (c) (cell-contents item-selector c)) cells-selected)))
                                 (inspect (cdr p))))))
        (view-position :centered)
        (view-size #@(400 200)))
  (when (window-p *RELATIONS-WINDOW*)
    (window-close *RELATIONS-WINDOW*))
  (when preds
    (setf *RELATIONS-WINDOW*
          (make-instance 
           'selector-dialog
           :window-title title
           :window-type :document-with-grow
           :close-box-p t
           :message message
           :view-position view-position
           :view-size view-size
           :table-sequence preds
           :table-print-function print-function
           :selection-type :disjoint
           :button-0-name " Inspect "
           :button-0-action action-function
           :button-0-enabled-p nil
           :default-when-selected :button-0
           :enable-only-when-selected '(:button-0)
           :default-when-unselected nil))))

;;;____________________________________________________________________________________
;;;  inspect-cliches

(defun inspect-cliches 
       (&key (view-position :centered)
             (view-size #@(400 200)))
  (when (window-p *CLICHES-WINDOW*)
    (window-close *CLICHES-WINDOW*))
  (when *available-relational-cliches*
    (setf *CLICHES-WINDOW*
          (make-instance 
           'selector-dialog
           :window-title "Clichs"
           :window-type :document-with-grow
           :close-box-p t
           :message "Select a Clich to Inspect"
           :view-position view-position
           :view-size view-size
           :table-sequence (sort (copy-list *available-relational-cliches*) #'string<)
           :table-print-function #'(lambda (cliche stream) (format stream "~S" cliche))
           :selection-type :disjoint
           :button-0-name " Inspect "
           :button-0-action #'(lambda (item)
                                (let* ((item-selector (find-named-sibling item :item-selector))
                                       (cells-selected (selected-cells item-selector)))
                                  (dolist (cliche-name
                                           (nreverse 
                                            (mapcar #'(lambda (c) (cell-contents item-selector c)) cells-selected)))
                                    (inspect (get-cliche-struct cliche-name)))))
           :button-0-enabled-p nil
           :default-when-selected :button-0
           :enable-only-when-selected '(:button-0)
           :default-when-unselected nil))))

;;____________________________________________________________________________________
;;  select-edit-method

(defun select-edit-method (operation pred-kind)
  (catch-cancel
   (if *expert-mode*
     (if (y-or-n-dialog  (format nil "~:(~A ~A~) using " operation pred-kind)
                         :yes-text "Graph" 
                         :no-text "Text"
                         :position :centered)
       :graph :text)
     :graph)))

;;____________________________________________________________________________________
;;  select-pred

(defun select-pred (preds pred-kind operation name)
  (if name
    (get name 'pred)
    (modal-dialog 
     (make-instance 
      'selector-dialog
      :window-type :document-with-grow
      :window-title (format nil "~:(~S ~S~)" operation pred-kind)
      :window-show nil
      :close-box-p nil
      :view-size #@(400 200)
      :view-position :centered
      :message (format nil "Select a ~:(~S~) to ~:(~S~)" pred-kind operation)
      :table-sequence preds
      :selection-type :single
      :table-print-function #'(lambda (p stream)
                                (let ((pred (cdr p)))
                                  (format stream "~(~A~) ~S" (pred-name pred) (pred-type pred))))
      :button-0-name " Cancel "
      :button-0-action #'(lambda (item) 
                           (declare (ignore item))
                           (return-from-modal-dialog :cancel))
      :button-1-name (format nil "~:(~A~)" operation)
      :button-1-action #'(lambda (item)
                           (let* ((item-selector (find-named-sibling item :item-selector))
                                  (selected-pred (cdr (cell-contents item-selector (car (selected-cells item-selector))))))
                             (when selected-pred
                               (return-from-modal-dialog selected-pred))))
      :button-1-enabled-p nil
      :default-when-selected :button-1
      :enable-only-when-selected '(:button-1)
      :default-when-unselected :button-0)
     t)))

;;____________________________________________________________________________________
;;  select-pred-and-edit-method

(defun select-pred-and-edit-method (preds pred-kind operation name)
  (let (pred edit-method)
    (catch-cancel
     (cond 
      ((and *expert-mode* name)
       (setf edit-method (select-edit-method operation 'fact)
             pred (get name 'pred)))
      (*expert-mode*
       (multiple-value-setq
         (edit-method pred)
         (values-list
          (modal-dialog 
           (make-instance 
            'selector-dialog
            :window-type :document-with-grow
            :window-title (format nil "~:(~A ~A~)" operation pred-kind)
            :close-box-p nil
            :view-position :centered
            :view-size #@(400 200)
            :message (format nil "Select a ~:(~A~) to ~:(~A~)" pred-kind operation)
            :table-sequence preds
            :selection-type :single
            :table-print-function #'(lambda (p stream)
                                      (let ((pred (cdr p)))
                                        (format stream "~(~A~) ~S" (pred-name pred) (pred-type pred))))
            :button-0-name " Cancel "
            :button-0-action #'(lambda (item) 
                                 (declare (ignore item))
                                 (return-from-modal-dialog :cancel))
            :button-1-name " Text "
            :button-1-action #'(lambda (item)
                                 (let* ((item-selector (find-named-sibling item :item-selector))
                                        (selected-pred (cdr (cell-contents item-selector (car (selected-cells item-selector))))))
                                   (when selected-pred
                                     (return-from-modal-dialog (list :text selected-pred)))))
            :button-1-enabled-p nil
            :button-2-name " Graph "
            :button-2-action #'(lambda (item)
                                 (let* ((item-selector (find-named-sibling item :item-selector))
                                        (selected-pred (cdr (cell-contents item-selector (car (selected-cells item-selector))))))
                                   (when selected-pred
                                     (return-from-modal-dialog (list :graph selected-pred)))))
            :button-2-enabled-p nil
            :default-when-selected :button-2
            :enable-only-when-selected '(:button-2 :button-1)
            :default-when-unselected :button-0)
           t))))
      (name
       (setf edit-method :graph
             pred (get name 'pred)))
      (t (setf edit-method :graph
               pred (select-pred preds pred-kind operation name))))
     (values edit-method pred))))


;;____________________________________________________________________________________
;;  select-item-or-type-in

(defun select-item-or-type-in (items title message type-in-message)
  (catch-cancel
   (modal-dialog 
    (make-instance 
     'selector-dialog
     :window-title title
     :window-type :document-with-grow
     :close-box-p nil
     :message message
     :view-position :centered
     :view-size #@(400 200)
     :table-sequence items
     :selection-type :single
     :button-0-name " Cancel "
     :button-0-action #'(lambda (item) 
                          (declare (ignore item))
                          (return-from-modal-dialog :cancel))
     :button-1-name " None "
     :button-1-action #'(lambda (item)
                          (declare (ignore item))
                          (return-from-modal-dialog :none))
     :button-2-name " Type In "
     :button-2-action #'(lambda (item)
                          (declare (ignore item))
                          (catch-cancel
                           (let ((pred (read-from-string (get-string-from-user type-in-message :position :centered) nil nil)))
                             (when pred
                               (return-from-modal-dialog pred)))))
     :button-2-default t
     :button-3-name " Accept "
     :button-3-action #'(lambda (item)
                          (let* ((item-selector (find-named-sibling item :item-selector))
                                 (selected-item (cell-contents item-selector (car (selected-cells item-selector)))))
                            (when selected-item
                              (return-from-modal-dialog selected-item))))
     :button-3-enabled-p nil
     :default-when-selected :button-3
     :enable-only-when-selected '(:button-3)
     :default-when-unselected :button-2)
    t)))

(provide :selectors)