;;;____________________________________________________________________________________
;;; The code in this file is designed to work in conjunction with grapher.lisp
;;; to animate opertionalization process of FOCL.
;;;
;;;  Created and designed by Clifford A. Brunk 05/20/91
;;;
;;;  Problems:
;;;    -  Doesn't handle back tracking
;;;    -  Will break if the node-frontier ever contains two identical conjunctions.
;;;    -  doesn't handle minus-1 operator very well
;;;    -  doesn't handle if-needed operator very well
;;;    -  probably won't handle recursive theories (????)
;;;
;;;   Idea behine this code:
;;;      - draw complete initial tree of the goal concept in an unoperationalized state
;;;      - highlight paths through this tree as they are operationalized
;;;      - however some paths aren't represented in the initial tree (recursion?, minus-1)..
;;;        [the code used to break when it couldn't find a node-conjunction coresponding to
;;;         the operationalized clause.  The code no longer insists on being able to
;;;         in such nodes.  The first time that no corresponding conjunction of node can
;;;         be found in the *node-frontier*, this code stops trying to graph the operationalization
;;;         along that path.
;;;____________________________________________________________________________________

(require 'grapher)

(in-package :user)

(defparameter *node-frontier* nil)

;;;____________________________________________________________________________________
;;;  reset-display-frontier

(defun reset-display-frontier (pos neg)
  (when (window-p *EBL-WINDOW*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (graph-root (graph-root view)))
      (clear-external-text view)
      (setf (node-external-text graph-root) (format nil "~A+ ~A- not covered" pos neg)
            (node-state graph-root) :ebl)
      (redraw-graph view :fast t)
      (setf *node-frontier* (list (list graph-root))))))


;;;____________________________________________________________________________________
;;;  graph-clause-operationalization
;;;
;;;  Given a prolog clause to be operationalized this function finds the node-conjunction
;;;  corresponding to the clause and changes the state of each node to :operational
;;;  The it changes the node-frontier to reflect the operationalization of this clause.
;;;
;;;  Operationalizing a conjunction means that no other conjunction with the same
;;;  consequent can be operationalized.  Additionally, the operationalized conjunction
;;;  itself can not be operationalized again, although its constituent literals can be.
;;;  Thus graph-clause-operationalization removes the operationalized node-conjunction
;;;  and all sibling node-conjunctions from the display-frontier.  Then it adds each node
;;;  which comprised the operationalized conjunction to the display-frontier as a seperate
;;;  conjunction.

(defun graph-clause-operationalization (clause)
  (when (and clause (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (conjunction (find-node-conjunction-corresponding-to-clause view
                                                                       clause
                                                                       *node-frontier*)))
      (cond (conjunction
             (dolist (node conjunction)
               (setf (node-state node) :ebl)
               (with-focused-view view (draw-node view node)))
             
             ;; Remove the operationalized node-conjunction and all sibling node-conjunctions
             (let ((parent (node-parent (car conjunction))))
               (if parent
                 (dolist (node-conjunction (node-children parent))
                   (setf *node-frontier*
                         (remove node-conjunction *node-frontier* :test #'equal)))
                 (setf *node-frontier* nil)))
             
             ;; Add each node which comprised the operationalized conjunction as its own conjunction
             (setf *node-frontier* (nconc *node-frontier* (mapcar #'list conjunction)))
             )
            (t (setf *node-frontier* nil)))
      )))

;;;____________________________________________________________________________________
;;;  replace-consequent-with-antecedents-in-frontier
;;;
;;;  Given a prolog literal (either negated or intensional which is about to be
;;;  replaced with one of its antecedents) this function finds the corresponding node,
;;;  removes it from and adds each of its child conjunctions to the operationalization-
;;;  frontier.

(defun replace-consequent-with-antecedents-in-frontier (literal)
  (when (and (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (conjunction (find-node-conjunction-corresponding-to-clause view
                                                                       (list literal) 
                                                                       *node-frontier*)))
      (if conjunction
        (setf *node-frontier* (remove conjunction *node-frontier* :test #'equal)
              *node-frontier* (nconc *node-frontier* (node-children (car conjunction))))
        (setf *node-frontier* nil))
      )))

;;;____________________________________________________________________________________
;;;  add-antecedents-to-frontier
;;;
;;;  Given a prolog literal (either negated???? or intensional which MAY be replaced with
;;;  a combination of its antecedents) this function finds the corresponding node,
;;;  and adds each of its child conjunctions to the display-frontier.

(defun add-antecedents-to-frontier (literal)
  (when (and (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (node (car (find-node-conjunction-corresponding-to-clause view
                                                                     (list literal) 
                                                                     *node-frontier*))))
      
        (if node 
          (setf *node-frontier* (nconc *node-frontier* (node-children node)))
          (setf *node-frontier* nil))
    )))


;;;____________________________________________________________________________________
;;;  remove-antecedents-from-frontier

(defun remove-antecedents-from-frontier (literal)
  (when (and (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (node (car (find-node-conjunction-corresponding-to-clause view
                                                                     (list literal) 
                                                                     *node-frontier*))))
      (if node
        (dolist (conjunction (node-children node))
          (setf *node-frontier*
                (remove conjunction *node-frontier* :test #'equal)))
        (setf *node-frontier* nil))
      )))


;;;____________________________________________________________________________________
;;;  remove-clause-from-frontier

(defun remove-clause-from-frontier (clause)
  (when (and (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (conjunction (find-node-conjunction-corresponding-to-clause view
                                                                       clause 
                                                                       *node-frontier*)))
      (if conjunction
        (setf *node-frontier* (remove conjunction *node-frontier* :test #'equal))
        (setf *node-frontier* nil))
      )))

;;;____________________________________________________________________________________
;;; display-ebl-gain
;;;

(defun display-ebl-gain (clause gain)
  (when (and (window-p *EBL-WINDOW*) *node-frontier*)
    (let* ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*)))
           (clause-nodes (find-node-conjunction-corresponding-to-clause view
                                                                        clause
                                                                        *node-frontier*)))
      (if clause-nodes
        (let* ((bottom-node (car clause-nodes))
               (max-v (node-bottom bottom-node)))
          (dolist (node clause-nodes)
            (if (> (node-bottom node) max-v)
              (setf bottom-node node
                    max-v (node-bottom bottom-node))))
          (setf (node-external-text bottom-node)
                (format nil "~A+ ~A- [~5F]" (gain-pp gain) (gain-nn gain) (gain-gain gain)))
          (with-focused-view view 
            (draw-node view bottom-node)))
        (setf *node-frontier* nil))
      )))

;;;____________________________________________________________________________________
;;;  setup-EBL-WINDOW

(defun setup-EBL-WINDOW (goal-concept)
  (cond
   (goal-concept
    (unless (window-p  *EBL-WINDOW*)
      (setf *EBL-WINDOW* (make-instance 'graph-window
                                        :window-show nil
                                        :window-title "Explanation Based Learning"
                                        :view-position #@(2 42)
                                        :kind :learning)))
    (let ((view (view-named :graph-view (view-named :graph-scroller *EBL-WINDOW*))))
      (setf (graph-expand view) :always
            (graph-orientation view) :horizontal)
      (generate-graph view (caar goal-concept))
      (resize-window *EBL-WINDOW*)
      (position-graph view :centered t)
      (hi-lite-subtree (graph-root view) :unoperationalized)))
   (t
    (if (window-p *EBL-WINDOW*)
      (window-close *EBL-WINDOW*)))))

(provide :animate-operationalization)
