
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

;;;____________________________________________________________________________________
;;;                               KR-FOCL-INTERFACE
;;;
;;; The code in this file is designed to work in conjunction with grapher.lisp and
;;; derivation-grapher.lisp to provide a interface which will facilitate the revision
;;; of a rule base.
;;;
;;;  Created and designed by Clifford A. Brunk 05/28/91
;;;
;;;  Problems:
;;;____________________________________________________________________________________

(require 'grapher)

(in-package :user)

(defvar *THEORY-WINDOW* nil)
(defvar *INDUCED-CONDITIONS* nil)
(defvar *INDUCED-CLAUSES* nil)
(defvar *KR-DIALOG* nil)

(defparameter *window-window-offset* 30)

;;;_______________________________________________________________________________
;;; *attach-induced-literal-dialog*

(defobject *attach-induced-literal-dialog* *dialog*)

(defobfun (exist *attach-induced-literal-dialog*) (init-list)

  (usual-exist (init-list-default init-list
                                  :window-title "Attach Induced Conditions"
                                  :window-size #@(400 200)
                                  :window-type :document
                                  :close-box-p nil
                                  :window-show t))

  (have 'done nil)

  (let ((window-h (point-h (window-size)))
        (window-v (point-v (window-size)))
        (h 10)
        (v 5)
        (indent 10)
        (induced-condition (getf init-list :induced-condition))
        (operationalized-clause (getf init-list :operationalized-clause)))

    (add-dialog-items 
     (oneof *static-text-dialog-item*
            :dialog-item-text "During learning the induced conditions:"
            :dialog-item-position (make-point h v)
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-font '("monaco" 9 :plain)
            :dialog-item-text (format nil "~A" induced-condition)
            :dialog-item-position (make-point (+ h indent) (incf v 20))
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-text "were added to the operationalized clause:"
            :dialog-item-position (make-point h (incf v 20))
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-font '("monaco" 9 :plain)
            :dialog-item-text (format nil "~A" operationalized-clause)
            :dialog-item-position (make-point (+ h indent) (incf v 20))
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-text
"This indicates that the induced conditions
should be added somewhere along the
operationalized path.

Select the attachment point and the induced
conditions to be added." 
            :dialog-item-position (make-point h (incf v 20))
            :dialog-item-size (make-point (- window-h 20) (- window-v v)))

     (oneof *button-dialog-item*
            :dialog-item-text " Attach "
            :dialog-item-nick-name :attach
            :default-button t
            :dialog-item-position (make-point (- window-h 70) (- window-v 60))
            :dialog-item-size #@(60 20)
            :dialog-item-enabled-p t
            :dialog-item-action #'(lambda ()
                                    (if (dialog-item-enabled-p)
                                      (do-literal-attachment))))

     (oneof *button-dialog-item*
            :dialog-item-text " Done "
            :dialog-item-nick-name :Done
            :dialog-item-position (make-point (- window-h 70) (- window-v 30))
            :dialog-item-enabled-p t
            :dialog-item-size #@(60 20)
            :dialog-item-action #'(lambda () 
                                    (ask my-dialog
                                      (window-hide)
                                      (setf done t))))

     )))



;;;_______________________________________________________________________________
;;; *attach-induced-clause-dialog*

(defobject *attach-induced-clause-dialog* *dialog*)

(defobfun (exist *attach-induced-clause-dialog*) (init-list)

  (usual-exist (init-list-default init-list
                                  :window-title "Attach Induced Clause"
                                  :window-size #@(400 200)
                                  :window-type :document
                                  :close-box-p nil
                                  :window-show t))

  (have 'done nil)

  (let ((window-h (point-h (window-size)))
        (window-v (point-v (window-size)))
        (h 10)
        (v 5)
        (indent 10)
        (induced-clause (getf init-list :induced-clause)))

    (add-dialog-items 
     (oneof *static-text-dialog-item*
            :dialog-item-text "During learning the clause:"
            :dialog-item-position (make-point h v)
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-font '("monaco" 9 :plain)
            :dialog-item-text (format nil "~A" induced-clause)
            :dialog-item-position (make-point (+ h indent) (incf v 20))
            :dialog-item-size (make-point (- window-h 20) 20))

     (oneof *static-text-dialog-item*
            :dialog-item-text
"was induced as a disjunct at the top level.

This clause can be attached as a 
disjunct anywhere in the rule base.

Select the attachment point and the
portion of the clause to be attached." 
            :dialog-item-position (make-point h (incf v 20))
            :dialog-item-size (make-point (- window-h 20) (- window-v v)))

     (oneof *button-dialog-item*
            :dialog-item-text " Attach "
            :dialog-item-nick-name :attach
            :default-button t
            :dialog-item-position (make-point (- window-h 70) (- window-v 60))
            :dialog-item-size #@(60 20)
            :dialog-item-enabled-p t
            :dialog-item-action #'(lambda ()
                                    (if (dialog-item-enabled-p)
                                      (do-clause-attachment))))
     (oneof *button-dialog-item*
            :dialog-item-text " Done "
            :dialog-item-nick-name :Done
            :default-button nil
            :dialog-item-position (make-point (- window-h 70) (- window-v 30))
            :dialog-item-enabled-p t
            :dialog-item-size #@(60 20)
            :dialog-item-action #'(lambda () 
                                    (ask my-dialog
                                      (window-hide)
                                      (setf done t))))
     )))

;;;_______________________________________________________________________________
;;; *delete-unoperationalized-dialog*

(defobject *delete-unoperationalized-dialog* *dialog*)

(defobfun (exist *delete-unoperationalized-dialog*) (init-list)

  (usual-exist (init-list-default init-list
                                  :window-title "Remove Erroneous Clauses and Conditions"
                                  :window-size #@(400 200)
                                  :window-type :document
                                  :close-box-p nil
                                  :window-show t))
  (have 'done nil)

  (let ((window-h (point-h (window-size)))
        (window-v (point-v (window-size))))

    (add-dialog-items 
     (oneof *static-text-dialog-item*
            :dialog-item-position #@ (10 5)
            :dialog-item-size (make-point (- window-h 15) 200)
            :dialog-item-text
"Deleting the marked clauses and conditions will
increase the accuracy of the top level concept.

 'Clear'  unmarks all selected nodes.
 'Mark'  marks all selected nodes for deletion.
 'Delete'  removes the marked nodes.")

     (oneof *button-dialog-item*
            :dialog-item-text " Clear "
            :dialog-item-position (make-point (- window-h 70) (- window-v 120))
            :dialog-item-size #@(60 20)
            :dialog-item-enabled-p t
            :dialog-item-action #'(lambda ()
                                    (ask *THEORY-WINDOW* 
                                        (ask gui-view (undelete-selected-nodes)))))
     (oneof *button-dialog-item*
            :dialog-item-text " Mark "
            :dialog-item-position (make-point (- window-h 70) (- window-v 90))
            :dialog-item-size #@(60 20)
            :dialog-item-enabled-p t
            :dialog-item-action #'(lambda ()
                                    (ask *THEORY-WINDOW* 
                                        (ask gui-view (delete-selected-nodes)))))
     (oneof *button-dialog-item*
            :dialog-item-text " Delete "
            :default-button t
            :dialog-item-position (make-point (- window-h 70) (- window-v 60))
            :dialog-item-size #@(60 20)
            :dialog-item-enabled-p t
            :dialog-item-action #'(lambda ()
                                    (ask *THEORY-WINDOW* 
                                        (ask gui-view (remove-all-deleted-nodes)))))
     (oneof *button-dialog-item*
            :dialog-item-text " Done "
            :dialog-item-position (make-point (- window-h 70) (- window-v 30))
            :dialog-item-enabled-p t
            :dialog-item-size #@(60 20)
            :dialog-item-action #'(lambda () 
                                    (ask my-dialog
                                      (window-hide)
                                      (setf done t))))
     )))


;;;_______________________________________________________________________________
;;;  learned-via
;;;
;;;  Returns :combination if the clause was learned by a combination of ebl and induction.
;;;          :induction   if the clause was learned by induction alone
;;;          :ebl         if the clause was learned by ebl alone

(defun learned-via (clause)
  (let ((induced nil)
        (operationalized nil))
    (do* ((literal clause (literal-next literal)))
         ((null literal))
      (unless (literal-deleted? literal)
        (case (derivation-type (literal-derivation literal))
          ((:induction :constructive-induction) (setf induced t))
          (:ebl (setf operationalized t))
          (otherwise nil))))
    (cond ((and induced operationalized) :combination)
          (induced :induction)
          (operationalized :ebl)
          (t nil))))


;;;_______________________________________________________________________________
;;;  duplicate-tree

(defun duplicate-tree (node &optional (parent nil))
  (when node
    (let ((copy (copy-node node)))
      (setf (node-parent copy) parent
            (node-selected? copy) nil
            (node-children copy) (mapcar 
                                  #'(lambda (children)
                                      (mapcar
                                       #'(lambda (child)
                                           (duplicate-tree child copy))
                                       children))
                                  (node-children copy)))
      copy)))


;;;_______________________________________________________________________________
;;;  nodes-in-tree

(defun nodes-in-tree (root)
  (let ((list-of-nodes nil))
    (labels 
      ((list-em (node)
                (cond ((null node) nil)
                      (t (setf list-of-nodes (push node list-of-nodes))
                         (dolist (children (node-children node))
                           (dolist (child children)
                             (list-em child)))))))
      (list-em root)
      list-of-nodes)))
  



;;;________________________________________________________________________________
;;; Knowledge Revision Heuristics
;;;________________________________________________________________________________

;;;________________________________________________________________________________
;;;  KR-Heuristic 1 - When a clause is learned exclusively by operationalizing a
;;;                   portion of the domain theory, it is a sign that none of the
;;;                   operationalized clauses used need to be modified.
;;;________________________________________________________________________________
;;;
;;;  KR-Heuristic-1 (learned-concept-description)
;;;
;;;  Returns a copy of learned-concept-description with the clauses learned by
;;;  operationalization alone removed.
;;;
;;;  This is accompished by determining what techniques were employed to learn each
;;;  clause in learned-concept-description.  If the clause was learned using :EBL
;;;  exclusively then that clause is removed.  As a consequnce when the theory
;;;  is a tree no rule used in learning that clause will be considered for
;;;  revision.
;;;
;;; [XXXX - Since the DT is not strictly a Tree it is possible that one of these 
;;;         rules could be flagged as needing revision by another heuristic. This
;;;         future modification must not invalidate the derivation used here.
;;;         Alternatively, this future modification could invalidate the proof
;;;         used here if it or some other modification covered the examples 
;;;         covered by the proof used here.
;;;         ie. the fact that this path leads to correct classification
;;;             "suggests" that it is correct, we can proceed under that assumption
;;;             until other information proves the assumption wrong.  At that point
;;;             the correctly classified examples become a constraint on the modification.
;;;             This will be implemented later.]
;;;
;;;________________________________________________________________________________

#|
(defun KR-Heuristic-1 (learned-concept-description)
  (mapcan #'(lambda (clause-body) 
              (if (eql (learned-via clause-body) :ebl)
                nil 
                (list clause-body)))
          learned-concept-description))
|#

;;;________________________________________________________________________________
;;;  KR-Heuristic 2 - When a clause is learned by operationalizing a portion of the
;;;                   domain theory and then inducing some predicates to further
;;;                   specialize the clause, it may be a sign that one or more of
;;;                   the operationalized clauses need to be modified by adding an
;;;                   extra condition from the induced predicates.
;;;________________________________________________________________________________
;;;
;;;  KR-Heuristic-2 (learned-concept-description)
;;;
;;;  Isolates the clauses of learned-concept-description learned by :EBL followed
;;;  by some inductive method. Then calls get-induced-literal-attachment which asks
;;;  the user where along the operationalized path to attach the induced literals.
;;;  
;;;________________________________________________________________________________

(defun KR-Heuristic-2 (LCD)
  (let ((combo-clauses nil)
        (clauses-with-same-ebl nil))
    (dolist (clause LCD)
      (when (eql (learned-via clause) :COMBINATION)
        (setf combo-clauses (nconc combo-clauses (list clause)))))
    (if combo-clauses
      (do* ((cs combo-clauses (cdr cs))
            (c (car cs) (car cs)))
           ((null cs) nil)
        (setf clauses-with-same-ebl (list c))
        (do* ((rcs cs (cdr rcs))
              (r (cadr rcs) (cadr rcs)))
             ((null (cdr rcs)) nil)
          (when (ebl-portion-equal c r)
            (setf clauses-with-same-ebl (nconc clauses-with-same-ebl (list r)))
            (rplacd rcs (cddr rcs))))
        (get-induced-literal-attachment clauses-with-same-ebl))
      (format t "~%Heuristic 1 for attaching induced conditions does not apply."))))


;;;________________________________________________________________________________
;;; ebl-portion-equal
;;; Returns true if the ebl portions of two clauses are equal.  The order of the
;;; literals in the cluases does not effect their equality.

(defun ebl-portion-equal (clause1 clause2)
  (let ((equal? t))
    (do ((l1 clause1 (literal-next l1)))
        ((null l1) equal?)
      (when (eql (derivation-type (literal-derivation l1)) :EBL)
        (do ((l2 clause2 (literal-next l2)))
            ((or (null l2)
                 (and (eql (derivation-type (literal-derivation l2)) :EBL)
                      (eql (literal-predicate-name l2) (literal-predicate-name l1))
                      (eql (literal-variablization l2) (literal-variablization l2))))
             (unless l2 (setf equal? nil))))))))



;;;________________________________________________________________________________
;;;  KR-Heuristic 3 - When a clause is learned exclusively by induction, it may be
;;;                   a sign that some unoperationalized clauses need to be changed.
;;;                   In this case KR-FOCL may suggest two types of revisons.
;;;                   KR-FOCL may suggest replacing a clause in the rule base which
;;;                   meets either of the following two conditions:
;;;
;;;                   a. The induced predicates subsume an unoperationalized clause.
;;;                      This may be a sign that the clause can be replaced by the
;;;                      induced predicates. In effect, this modification removes
;;;                      superfluous literals from a clause.
;;;                      [GENERALIZATION]
;;;
;;;                   b. The induced predicates are subsumed by an unoperationalized
;;;                      clause. This also can be a sign that the clause can be
;;;                      replaced by the induced predicates. In effect, this 
;;;                      modification adds extra literals to a clause.
;;;                      [SPECIALIZATION]
;;;
;;;                   Alternatively when a clause is learned exclusively by induction,
;;;                   it could be a sign that the induced predicates may be another
;;;                   clause for the top level predicate or for a predicate that is
;;;                   part of an unoperationalized clause.
;;;________________________________________________________________________________
;;;
;;;  KR-Heuristic-3 (learned-concept-description unoperationalized-clauses)
;;;
;;;  Isolates the clauses of learned-concept-description learned exclusively by
;;;  inductive methods. Then calls get-induced-clause-attachment which asks the
;;;  the user where to attach the induced clauses, following the suggestions
;;;  outlined above.
;;;  
;;;________________________________________________________________________________

(defun KR-Heuristic-3 (LCD)
  (let ((applied? nil))
    (dolist (induced-clause  (remove-if-not #'(lambda (C) (eql (learned-via C) :induction)) LCD))
      (setf applied? t)
      (get-induced-clause-attachment induced-clause))
    (unless applied?
      (format t "~%Heuristic 2 for attaching induced clauses does not apply."))))


;;;________________________________________________________________________________
;;;  KR-Heuristic 4 - When a clause is not operationalized during the entire 
;;;                   learning process, it is a sign ther the clause can be 
;;;                   eliminated.
;;;________________________________________________________________________________
;;;  KR-Heuristic-4 (unoperationalized-rules)
;;;
;;;  Calls delete-unoperationalized-clauses which alerts the user to any
;;;  modification of an unoperationalized-clause and then asks the users whether
;;;  or not to delete each unoperationalized-clause.
;;;
;;;________________________________________________________________________________

(defun KR-Heuristic-4 ()
  (delete-unoperationalized-clauses))



;;;________________________________________________________________________________
;;;  KR applies Heuristic 2 3 4

(defun KR (LCD TOP-LEVEL-LITERAL)

  (format t "~%~%Applying Knowledge Revision Heuristics...")

  ;; Create Window for Domain Theory
  (unless (ask *THEORY-WINDOW* (ownp 'wptr))
    (setf *THEORY-WINDOW*
          (oneof *gui-window*
                 :window-position #@(2 41)
                 :window-type :document-with-zoom
                 :window-title "DOMAIN THEORY"
                 :window-font '("Monaco" 9 :plain))))

  ;; Create Graph of Domain Theory indicating operationalized and unoperationalized portions
  ;; Specialize *THEORY-WINDOW* for displaying operationalized paths and attaching conditions
  (ask *THEORY-WINDOW*
    (have 'window-do-first-click t)
    (set-window-title "OPERATIONALIZED PATH")
    (ask (objvar gui-view)
      (setf (objvar graph-orientation) :diagonal
            (objvar node-selection-constraint) :single-not-leaf
            (objvar graph-expand) :always                
            (objvar graph-arguments) :from-substitution
            (objvar node-list) nil)
      (generate-graph TOP-LEVEL-LITERAL)
      (dolist (node (objvar node-list))
        (setf (node-state node) :unoperationalized))
      (operationalize-ebl-paths LCD)))

  (KR-Heuristic-2 LCD)

  ;; Specialize *THEORY-WINDOW* for displaying theory and attaching clauses
  (ask *THEORY-WINDOW*
    (ask gui-view
      (setf (objvar graph-orientation) :horizontal
            (objvar node-selection-constraint) :single-disjunctive-not-leaf)
      (dolist (node (objvar node-list))
        (setf (node-hidden? node) nil
              (node-selected? node) nil))
      (position-nodes))
    (set-window-title "DOMAIN THEORY")
    (set-window-size))

  (KR-Heuristic-3 LCD)



  (let ((applies? nil))
    (do* ((nodes (ask *THEORY-WINDOW* (ask (objvar gui-view) (objvar node-list))) (cdr nodes))
          (node (car nodes) (car nodes)))
         ((or applies? (null nodes)))
      (if (or (node-deleted? node)
              (eql (node-state node) :unoperationalized))
        (setf applies? t)))
    (when applies?
      ;; Specialize *THEORY-WINDOW* for revision (global editing)
      (ask *THEORY-WINDOW*
        (setf (objvar edit-kind) :edit)
        (ask gui-view
          (setf (objvar node-selection-constraint) nil)
          (dolist (node (objvar node-list))
            (setf (node-selected? node) nil)
            (if (eql (node-state node) :unoperationalized)   ;;  Should check to see if removing
              (setf (node-deleted? node) t))))               ;;  This would increase accuracy
        (window-select))
      (KR-Heuristic-4))
    (unless applies?
      (format t "~%Heuristic 3 for deleting erroneous conditions and clauses does not apply.~%")))

  nil)



;;;________________________________________________________________________________
;;; operationalize-ebl-paths
;;;
;;;  This code is specially ineffecient  (but hey it works)

(defobfun (operationalize-ebl-paths *gui-view*) (LCD)
  (let (derivation
        parent
        context
        clause-struct
        actual-parameters
        actual-clause)
    (dolist (clause LCD)
      (do ((literal clause (literal-next literal)))
          ((null literal))
        (setf parent (objvar graph-root)
              context (list parent)
              derivation (literal-derivation literal))
        (when (eql (derivation-type derivation) :ebl)
          (setf (node-state parent) :operationalized)
          (dolist (step (butlast (derivation-path derivation)))
            (setf parent (find-if #'(lambda (node) (equal (node-literal node) (cdr step))) context)
                  clause-struct (car step)
                  actual-parameters (rest (node-literal parent))
                  actual-clause (substitute1 (clause-body clause-struct)
                                             (unify-list (clause-parameters clause-struct) 
                                                         actual-parameters))
                  context (find-node-conjunction-corresponding-to-clause actual-clause
                                                                         (node-children parent)))

              (dolist (node context)
                (setf (node-state node) :operationalized)))

          (when (literal-deleted? literal)                                    ;;;  This is a hack
            (let ((deleted-literal (car (last (derivation-path derivation)))))
              (dolist (node context)
                (when (literals-correspond (node-literal node) deleted-literal)
                  (setf (node-deleted? node) t)))))
          )))))

;;;________________________________________________________________________________
;;; User Interface Routines
;;;________________________________________________________________________________


;;;______________________________________________________________________________
;;;______________________________________________________________________________
;;;  get-induced-literal-attachment (prolog-root-literal clause) 
;;;
;;;  For each clause in learned-clauses get the correct attachment point along the
;;;  operationalized path for the literals learned using induction.
;;;
;;;  Display entire path learned by :EBL (this will be a connected graph [probably an and-tree])
;;;  Display the literals learned by :INDUCTION (these will be isolated nodes)
;;;  Display the literals learned by :CONSTRUCTIVE-INDUCTION (these will be disjoint connected 
;;;                                                            graphs [probably and-trees])
;;;
;;;  Allow the user to draw and connectors from any :EBL node to any induced 
;;;  literal and vice-versa.  Allow the user to stop at any point, but notify
;;;  the user that correct classification is not ensured if a fronteer of the
;;;  indeuced literals have not been connected.
;;;
;;;  5/29/91  Well that was the ideal this is what I have working so far.
;;;
;;;  DO THIS -->  show the default attachment point and selection

(defun get-induced-literal-attachment (clauses)
  (let ((v 42))
    (ask *THEORY-WINDOW*
      (ask (objvar gui-view)
        (show-only-derivation-ebl-portion (car clauses)))
      (set-window-size)
      (ask gui-view (center-graph))
      (set-window-position (round (- *screen-width* (point-h (window-size))) 2) v)
      (window-select)
      (incf v (+ *window-window-offset* (point-v (window-size)))))
        
    (unless (ask *INDUCED-CONDITIONS* (ownp 'wptr))
      (setf *INDUCED-CONDITIONS*
            (oneof *gui-window*
                   :window-type :document-with-zoom
                   :window-title "INDUCED CONDITIONS"
                   :window-font '("Monaco" 9 :plain))))
    (ask *INDUCED-CONDITIONS* 
      (have 'window-do-first-click t)
      (ask (objvar gui-view)
        (setf (objvar graph-orientation) :horizontal
              (objvar graph-expand) :always
              (objvar graph-arguments) :from-substitution
              (objvar node-list) nil)
        (graph-induced-portion-of-clauses clauses))
      (set-window-position (round (- *screen-width* (point-h (window-size))) 2)
                           (min v (- *screen-width* (point-v (window-size)) 2)))
      (window-select)
      (setf v (+ (point-v (window-position)) (point-v (window-size)) *window-window-offset*)))
    
    (setf *KR-DIALOG* (oneof *attach-induced-literal-dialog*
                             :window-position (list ':top  (min v (- *screen-height* 222)))
                             :induced-condition 
                             (if (cdr clauses)
                               (let ((cs (clause-string (car clauses) :induction? t)))
                                 (dolist (c (cdr clauses) cs)
                                   (setf cs (concatenate 'string cs"; " (clause-string c :induction? t)))))
                               (clause-string (car clauses) :induction? t))
                             :operationalized-clause (clause-string (car clauses) :ebl? t)))
    (ask *KR-DIALOG* (have 'window-do-first-click t))
    (do ()
        ((ask *KR-DIALOG* (objvar done))
         (ask *INDUCED-CONDITIONS* (window-close))
         (ask *THEORY-WINDOW* (window-hide))
         (ask *KR-DIALOG* (window-close))
         nil)
      (event-dispatch))))

;;;_______________________________________________________________________________
;;;  clause-string

(defun clause-string (clause &key (ebl? nil) (induction? nil) (both? nil))
  (let ((c-string ""))
  (do* ((literal clause (literal-next literal)))
       ((null literal))
    (cond ((literal-deleted? literal))
          ((and (literal-negated? literal)
                (or both?
                    (and ebl? (eql (derivation-type (literal-derivation literal)) :EBL))
                    (and induction? (not (eql (derivation-type (literal-derivation literal)) :EBL)))))
          
           (setf c-string (concatenate 'string
                                       c-string
                                       "(NOT "
                                       (clause-string (literal-negated-literals literal)
                                                      :ebl? ebl?
                                                      :induction? induction?
                                                      :both? both?)
                                       ") "
                                       )))
          ((or both?
               (and ebl? (eql (derivation-type (literal-derivation literal)) :EBL))
               (and induction? (not (eql (derivation-type (literal-derivation literal)) :EBL))))
           (setf c-string
                 (if (string-equal c-string "")
                   (format nil "~A" (cons (literal-predicate-name literal) 
                                          (literal-variablization literal)))
                   (concatenate 'string c-string ", " 
                                (format nil "~A" 
                                        (cons (literal-predicate-name literal) 
                                              (literal-variablization literal)))))))
          (t )))
    (if induction?
      c-string
      (concatenate 'string c-string "."))))


;;;________________________________________________________________________________
;;; show-only-derivation-ebl-portion
;;;
;;;  This code is specially ineffecient  (but hey it works)

(defobfun (show-only-derivation-ebl-portion *gui-view*) (clause)
  (setf last-node-selected nil)
  (dolist (node (objvar node-list))
    (setf (node-hidden? node) t
          (node-selected? node) nil))
  (let (derivation
        parent
        context
        clause-struct
        actual-parameters
        actual-clause)
    (do ((literal clause (literal-next literal)))
        ((null literal))
      (setf parent (objvar graph-root)
            context (list parent)
            derivation (literal-derivation literal))
      (when (eql (derivation-type derivation) :ebl)
        (setf (node-hidden? parent) nil)
        (dolist (step (butlast (derivation-path derivation)))
          (setf parent (find-if #'(lambda (node) (equal (node-literal node) (cdr step))) context)
                clause-struct (car step)
                actual-parameters (rest (node-literal parent))
                actual-clause (substitute1 (clause-body clause-struct)
                                           (unify-list (clause-parameters clause-struct) 
                                                       actual-parameters))
                context (find-node-conjunction-corresponding-to-clause actual-clause
                                                                       (node-children parent)))

          (dolist (node context)
            (setf (node-hidden? node) nil)))

          (when (literal-deleted? literal)                                    ;;;  This is a hack
            (let ((deleted-literal (car (last (derivation-path derivation)))))
              (dolist (node context)
                (when (literals-correspond (node-literal node) deleted-literal)
                  (setf (node-deleted? node) t)))))
          )))
  (position-nodes))




;;;_______________________________________________________________________________
;;;  do-literal-attachment

(defun do-literal-attachment ()
  (let ((head-node nil)
        (added-conditions-nodes nil))
    (ask *THEORY-WINDOW*
      (ask (objvar gui-view)
        (setf head-node (objvar last-node-selected))))
    (ask *INDUCED-CONDITIONS*
      (ask (objvar gui-view)
        (dolist (node (objvar node-list))
          (if (node-selected? node)
            (setf added-conditions-nodes (push node added-conditions-nodes))))))

    (when (and head-node
               added-conditions-nodes)
      (let ((copy-of-added-conditions-nodes nil)
            (added-nodes nil)
            (duplicate nil)
            (new-v 0)
            (node-conjunction nil))
        (dolist (children (node-children head-node))
          (unless (node-hidden? (car children))
            (setf node-conjunction children)))
        (dolist (node added-conditions-nodes)
          (setf duplicate (duplicate-tree node)
                (node-parent duplicate) head-node
                copy-of-added-conditions-nodes (push duplicate copy-of-added-conditions-nodes)
                added-nodes (nconc added-nodes (nodes-in-tree duplicate))))
        (ask *THEORY-WINDOW*
          (ask gui-view
            (setf node-list (nconc node-list added-nodes))
            (nconc node-conjunction copy-of-added-conditions-nodes)
            (position-nodes))
          (set-window-size)
          (setf new-v (+ *window-window-offset* (point-v (window-position)) (point-v (window-size))))
          (ask gui-view
            (center-graph)
            (view-draw-contents)))
        (ask *INDUCED-CONDITIONS*
          (set-window-position (point-h (window-position)) 
                               (min new-v (- *screen-height* (point-v (window-size)) 20)))
          (incf new-v (+ *window-window-offset* (point-v (window-size)))))
        (ask *KR-DIALOG*
          (set-window-position (point-h (window-position)) (min new-v (- *screen-height* 220)))))
      )))


;;;________________________________________________________________________________
;;;  get-induced-clause-attachment (learned-clauses unoperationalized-clauses)
;;;
;;;  For each clause in learned-clauses get the correct attachment point for the 
;;;  clause. Search all unoperationalized-clause for a clause which is either
;;;  more general or more specific than the learned-clause. Suggest replacing the
;;;  unoperationalized-clause with the learned clause. If no such suggestion is 
;;;  accepted (or exists) suggest attaching the learned-clause as a disjunct for
;;;  the target-concept or for a rule that is part of an unoperationalized-clause.
;;;
;;;  Yah, right!

(defun get-induced-clause-attachment (induced-clause)
  (let ((v 42))
    (ask *THEORY-WINDOW*
      (set-window-position (round (- *screen-width* (point-h (window-size))) 2) v)
      (window-select)
      (incf v (+ *window-window-offset* (point-v (window-size)))))

  (unless (ask *INDUCED-CLAUSES* (ownp 'wptr))
    (setf *INDUCED-CLAUSES*
          (oneof *gui-window*
                 :window-type :document-with-zoom
                 :window-title "INDUCED CLAUSES"
                 :window-font '("Monaco" 9 :plain))))
  (ask *INDUCED-CLAUSES* 
    (have 'window-do-first-click t)
    (ask (objvar gui-view)
      (setf (objvar graph-orientation) :horizontal
            (objvar graph-expand) :always                
            (objvar graph-arguments) :from-substitution
            (objvar node-list) nil)
      (graph-induced-portion-of-clauses (list induced-clause)))
    (set-window-position (round (- *screen-width* (point-h (window-size))) 2)
                         (min v (- *screen-height* (point-v (window-size)) 2)))
    (window-select)
    (ask (objvar gui-view) (view-draw-contents))
    (setf v (+ (point-v (window-position)) (point-v (window-size)) *window-window-offset*)))
 
  (setf *KR-DIALOG* (oneof *attach-induced-clause-dialog*
                           :induced-clause (clause-string induced-clause :both? t)
                           :window-position  (list ':top  (min v (- *screen-height* 220)))))
  (ask *KR-DIALOG* (have 'window-do-first-click t))

  (do ()
      ((ask *KR-DIALOG* (objvar done))
       (ask *INDUCED-CLAUSES* (window-close))
       (ask *THEORY-WINDOW* (window-hide))
       (ask *KR-DIALOG* (window-close))
       nil)
    (event-dispatch))))
  

;;;________________________________________________________________________________
;;;  do-clause-attachment

(defun do-clause-attachment ()
  (let ((head-node nil)
        (added-conditions-nodes nil))

    (ask *THEORY-WINDOW*
      (ask (objvar gui-view)
        (setf head-node (objvar last-node-selected))))

    (ask *INDUCED-CLAUSES*
      (ask (objvar gui-view)
        (dolist (node (objvar node-list))
          (if (node-selected? node)
            (setf added-conditions-nodes (push node added-conditions-nodes))))))

    (when (and head-node
               added-conditions-nodes)

      ;;;  Add a copy of the added-nodes as a disjunction of the head node.

      (let ((copy-of-added-conditions-nodes nil)
            (added-nodes nil)
            (duplicate nil)
            (new-v 0))
        (dolist (node added-conditions-nodes)
          (setf duplicate (duplicate-tree node)
                (node-parent duplicate) head-node
                copy-of-added-conditions-nodes (push duplicate copy-of-added-conditions-nodes)
                added-nodes (nconc added-nodes (nodes-in-tree duplicate))))
      
        (ask *THEORY-WINDOW*
          (ask (objvar gui-view)
            (setf node-list (nconc node-list added-nodes)
                  (node-children head-node) (nconc (node-children head-node) 
                                                   (list copy-of-added-conditions-nodes)))
            (position-nodes))
          (set-window-size)
          (setf new-v (+ 30 (point-v (window-position)) (point-v (window-size))))
          (ask gui-view
            (view-draw-contents)))
                
        (ask *INDUCED-CLAUSES*
          (set-window-position (point-h (window-position)) 
                               (min new-v (- *screen-height* (point-v (window-size)) 20)))
          (incf new-v (+ *window-window-offset* (point-v (window-size)))))
        (ask *KR-DIALOG*
          (set-window-position (point-h (window-position)) (min new-v (- *screen-height* 220))))))
    ))



;;;________________________________________________________________________________
;;;  delete-unoperationalized-clauses
;;;

(defun delete-unoperationalized-clauses ()


  (setf *KR-DIALOG* (oneof *delete-unoperationalized-dialog*
                           :window-position (list ':top (min (ask *THEORY-WINDOW* 
                                                               (+ (point-v (window-position))
                                                                  (point-v (window-size))
                                                                  40))
                                                             (- *screen-height* 220)))))
  (ask *KR-DIALOG* (have 'window-do-first-click t))
  (do ()
      ((ask *KR-DIALOG* (objvar done))
       (ask *KR-DIALOG* (window-close))
       nil)
    (event-dispatch)))





;;;_______________________________________________________________________________
;;;  Real calls to derivation grapher
;;;_______________________________________________________________________________

;;;_______________________________________________________________________________
;;;  connect-clause-derivation-induced-portion

(defobfun (connect-clause-derivation-induced-portion *gui-view*) (parent-node clause)
  (let ((use-existing-paths t))
    (do* ((literal clause (literal-next literal)))
         ((null literal))
      (if (not (eql (derivation-type (literal-derivation literal)) :EBL))
        (connect-literal-derivation parent-node literal use-existing-paths)))))

;;;_______________________________________________________________________________
;;;  graph-induced-portion-of-clauses
;;;
;;;  given a linked list of literal stuctures this fuction returns a graph
;;;  of the induced conditions.  The top-level node and conjunction are
;;;  hidden (maybe they should be made explicit and given new selection
;;;  semantics.
;;;
;;;  (connect-literal parent-node literal &optional (expand t))

(defobfun (graph-induced-portion-of-clauses *gui-view*) (clauses)
  (setf node-list nil
        graph-root (create-node :kind :intensional :state :induced :hidden? t)
        (node-text graph-root) " TOP ")
  (size-node graph-root)

  (let ((derivation nil)
        (conjunction nil)
        (disjunction nil)
        (next-literal nil)
        (new-node nil))

    (dolist (clause clauses)
      (setf conjunction nil)
    
      (do ((literal clause (literal-next literal)))
          ((null literal) nil)
        (setf derivation (literal-derivation literal))

        (case (derivation-type derivation)
          (:EBL nil)

          (:INDUCTION 
           (setf next-literal (literal-next literal)
                 (literal-next literal) nil
                 new-node (connect-literal graph-root (car (convert-to-prolog literal)) t)
                 conjunction (nconc conjunction (list new-node))
                 (literal-next literal) next-literal)
           (hi-lite-subtree-as-induced new-node))

          (:CONSTRUCTIVE-INDUCTION
           (let ((prolog-literal (cdar (derivation-path derivation)))
                 (subtree nil))
             (dolist (node conjunction)
               (if (eql prolog-literal (node-literal node))
                 (setf subtree node)))
             (unless subtree
               (setf subtree (connect-literal graph-root prolog-literal t)
                     conjunction (nconc conjunction (list subtree)))
               (unoperationalize-subtree subtree))
             
             (hi-lite-operationalize-path-as-induced subtree clause)))
          ))
      (setf disjunction (nconc disjunction (list conjunction))))
    (setf (node-children graph-root) disjunction))

  (when (cdr clauses)
    (setf (node-hidden? graph-root) nil
          (node-literal graph-root) (cons (gen-rel-name graph-root) (gen-rel-vars graph-root))
          (node-text graph-root)  (format nil "~A" (node-literal graph-root) ))
    (size-node graph-root))

  (position-nodes)
  (ask gui-window (set-window-size))
  (center-graph)
  (view-draw-contents))

;;___________________________________________________________________________
;; gen-rels-name
;; this function is designed to create a name for a newly created rule.
 
(defun gen-rel-name (node)
  (let ((name ""))
    (dolist (children (butlast (node-children node)))
      (dolist (child (butlast children))
        (setf name (format nil "~A~A-AND-" name (node-n child))))
      (setf name (format nil "~A~A" name (node-n (car (last children)))))
      (setf name (format nil "~A-OR-" name)))

    (let ((children (car (last (node-children node)))))
      (dolist (child (butlast children))
        (setf name (format nil "~A~A-AND-" name (node-n child))))
      (setf name (format nil "~A~A" name (node-n (car (last children)))))
    name)))

(defun node-n (node)
  (if (eql (node-kind node) :NOT)
    (format nil "NOT-~A" (gen-rel-name node))
    (car (node-literal node))))
  



;;___________________________________________________________________________
;; gen-rel-vars
;; this function is designed to create an agrument list for an newly
;; created rule.  Currently all the variables in the children are added
;; to the parent.  This should be modified to check mode ...
 
(defun gen-rel-vars (node)
  (let ((vars nil))
    (dolist (children (node-children node))
      (dolist (child children)
        (dolist (v (node-vars child))
          (setf vars (nconc vars (list v))))))
    (sort vars #'string<)))

(defun node-vars (node)
  (if (eql (node-kind node) :NOT)
    (let ((vars nil))
      (dolist (children (node-children node))
        (dolist (child children)
          (dolist (v (node-vars child))
            (setf vars (nconc vars (list v))))))
      vars)
    (remove-duplicates (cdr (node-literal node)))))



(defobfun (unoperationalize-subtree *gui-view*) (node)
  (when node
    (setf (node-state node) :unoperationalized)
    (dolist (children (node-children node))
      (dolist (child children)
        (unoperationalize-subtree child)))))

(defobfun (hi-lite-subtree-as-induced *gui-view*) (node)
  (when node
    (setf (node-state node) :induced)
    (dolist (children (node-children node))
      (dolist (child children)
        (hi-lite-subtree-as-induced child)))))

;;;  Code very similar to the following is used two other places in KR-FOCL
;;;  perhaps it would be better to abstract it slightly and consolidate it.
;;;
;;;  needed parameters:
;;;     clause
;;;     start-node
;;;     derivation-type
;;;     operationalized-node-state

(defobfun (hi-lite-operationalize-path-as-induced *gui-view*) (subtree clause)
  (let (derivation
        parent
        context
        clause-struct
        actual-parameters
        actual-clause)
    (do ((literal clause (literal-next literal)))
        ((null literal))
      (setf parent subtree
            context (list parent)
            derivation (literal-derivation literal))
      (when (eql (derivation-type derivation) :CONSTRUCTIVE-INDUCTION)
        (setf (node-state parent) :INDUCED)
        (dolist (step (butlast (derivation-path derivation)))
          (setf parent (find-if #'(lambda (node) (equal (node-literal node) (cdr step))) context)
                clause-struct (car step)
                actual-parameters (rest (node-literal parent))
                actual-clause (substitute1 (clause-body clause-struct)
                                           (unify-list (clause-parameters clause-struct) 
                                                       actual-parameters))
                context (find-node-conjunction-corresponding-to-clause actual-clause
                                                                       (node-children parent)))
          
          (dolist (node context)
            (setf (node-state node) :INDUCED)))
        
        (when (literal-deleted? literal)                                    ;;;  This is a hack
          (let ((deleted-literal (car (last (derivation-path derivation)))))
            (dolist (node context)
              (when (literals-correspond (node-literal node) deleted-literal)
                (setf (node-deleted? node) t)))))
        ))))



;;;_______________________________________________________________________________
;;;  Delete (d D delete)  is   delete-selected-nodes
;;;  Undelete (clear)     is   undelete-selected-nodes
;;;  Really-Delete (control-delete) is  remove-all-deleted-nodes
;;;  Undelete-all (control-clear)  is  undelete-all-deleted-nodes
;;;_______________________________________________________________________________

;;____________________________________________________________________________________
;; remove-child-from-parent

(defobfun (remove-child-from-parent *gui-view*) (node)
  (let ((parent (node-parent node)))
    (when parent
      (do* ((conjunctions (node-children parent) (cdr conjunctions))
            (conjunction (car conjunctions) (car conjunctions)))
           ((member node conjunction)
            (rplaca conjunctions (delete node conjunction))))
      (setf (node-children parent) (delete nil (node-children parent)))
      (if (null (node-children parent))
        (setf (node-kind parent) :undefined)))))

;;;_______________________________________________________________________________
;;; delete-selected-nodes

(defobfun (delete-selected-nodes *gui-view*) ()
  (if (node-selected? graph-root)
    (case (catch-cancel 
            (y-or-n-dialog (format nil "~%Do you want to delete the rule ~A from the rule base?"
                                   (car (node-literal graph-root)))))
      ((nil) nil)
      (:cancel nil)
      (t (format t "~%The rule ~A should be deleted." (car (node-literal graph-root)))
         (ask gui-window (window-close t))))
    (dolist (node node-list)
      (when (node-selected? node)
        (setf (node-deleted? node) t)
        (with-focused-view (self)
          (draw-node node))))))

;;;_______________________________________________________________________________
;;; remove-all-deleted-nodes

(defobfun (remove-all-deleted-nodes *gui-view*) ()
  (dolist (node node-list)
    (when (node-deleted? node)
      (hide-all-descendents node)
      (setf (node-hidden? node) t)
      (remove-child-from-parent node)))
  (setf node-list (delete-if #'node-hidden? node-list))
  (position-nodes)
  (force-graph-redraw))

;;;_______________________________________________________________________________
;;; undelete-selected-nodes

(defobfun (undelete-selected-nodes *gui-view*) ()
  (dolist (node node-list)
    (when (node-selected? node)
      (setf (node-deleted? node) nil)
      (rlet ((rect :rect
                   :left (node-left node)
                   :top (node-top node)
                   :right (node-right node)
                   :bottom (node-bottom node)))
        (with-focused-view (self)
          (_EraseRect :ptr rect)
          (draw-node node))))))

;;;_______________________________________________________________________________
;;; undelete-all-deleted-nodes

(defobfun (undelete-all-deleted-nodes *gui-view*) ()
   (dolist (node node-list)
    (if (node-deleted? node)
      (setf (node-deleted? node) nil)))
  (force-graph-redraw))








;;;____________________________________________________________________________________
;;;  Revise Menu


(defparameter *apply-heuristics*
  (oneof *menu-item*
         :menu-item-title "Apply Heuristics"
         :menu-item-action 
         #'(lambda () (eval-enqueue '(apply-heuristics)))))

(ask *apply-heuristics*
  (fhave 'menu-item-update #'(lambda () (if (and (boundp '*LCD*) *LCD*
                                                 (boundp '*GOAL-CONCEPT*) *GOAL-CONCEPT*)
                                          (menu-item-enable)
                                          (menu-item-disable)))))

(defparameter *edit-theory*
  (oneof *menu-item*
         :menu-item-title "Edit Theory"
         :menu-item-action 
         #'(lambda () (eval-enqueue '(edit-theory *top-level-call*)))))

(ask *edit-theory*
  (fhave 'menu-item-update #'(lambda () (if (and (boundp '*top-level-call*) *top-level-call*)
                                          (menu-item-enable)
                                          (menu-item-disable)))))

(defparameter *revise-menu*
  (oneof *menu* 
         :menu-title "Revise"
         :menu-items (list *apply-heuristics*
                           (oneof *menu-item* :menu-item-title "-")
                           *edit-theory*)))



;;;__________________________________________________________________________________
;;; apply-heuristics

(defun apply-heuristics ()
  (when (and (boundp '*LCD*) *LCD*
             (boundp '*GOAL-CONCEPT*) *GOAL-CONCEPT*)
    (hide-listener)
    (catch-cancel 
      (KR *LCD* *GOAL-CONCEPT*))
    (unhide-listener)))



;;;__________________________________________________________________________________
;;; edit-theory

(defun edit-theory (root-literal)
  (declare (ignore root-literal))
  (message-dialog "
  Theory editor is not complete.
" :position :centered))

(provide :KR-FOCL)
