;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

(defparameter *true-indicator* 't)
(defparameter *pos-class-indicator* '+)
(defparameter *neg-class-indicator* '-)

(defvar *display-tuple-kind* nil)

;;;=======================================
;;;  array-dialog-item

(defclass array-dialog-item (table-dialog-item)
  ((table-array :initarg :table-array :accessor table-array)))

(defmethod initialize-instance ((item array-dialog-item) &rest rest &key table-array)
  (let* ((array-dimensions (array-dimensions table-array)))
    (apply #'call-next-method item
           :table-dimensions (make-point (elt array-dimensions 1) (elt array-dimensions 0))  rest)))

(defmethod cell-contents ((item array-dialog-item) h &optional v)
  (if v
    (aref (table-array item) v h)
    (aref (table-array item) (point-v h) (point-h h))))

(defmethod set-cell-contents ((item array-dialog-item) cell value)
  (setf (aref (table-array item) (point-v cell) (point-h cell)) value))

(defmethod set-cell-contents ((item sequence-dialog-item) cell value)
  (case (slot-value item 'ccl::sequence-order)
    (:horizontal (setf (nth (point-h cell) (table-sequence item)) value))
    (:vertical (setf (nth (point-v cell) (table-sequence item)) value))))

(defun table-print (object stream)
  (cond ((null object) (format stream ""))
        (t (format stream "~S" object))))

;;;=======================================
;;;  examples-window

(defclass examples-window (window) 
  ((example-template :initarg :example-template :accessor example-template)
   (english :initarg :english :accessor english)
   (no-edit :initarg :no-edit :accessor no-edit)
   (changed :initarg :changed :accessor changed)))

(defmethod initialize-instance ((window examples-window) &rest rest &key relation-list var-list type-list mode-list class-list element-array
                                template english induction commutative constraint no-edit compute-neg)
  (apply #'call-next-method window rest)
  (setf (example-template window) template
        (english window) english
        (no-edit window) no-edit
        (changed window) nil)
  (let ((size (view-size window)))
    (add-subviews
     window
     
     (make-dialog-item
      'sequence-dialog-item #@(16 2) (make-point (- (point-h size) 31) 11) ""
      #'(lambda (item) (select-relations-cell item (first (selected-cells item))))
      :table-sequence relation-list
      :view-nick-name :relations
      :sequence-order :horizontal
      :table-vscrollp nil
      :table-hscrollp nil
      :table-print-function #'table-print)
     
     (make-dialog-item
      'sequence-dialog-item #@(16 16) (make-point (- (point-h size) 31) 11) ""
      #'(lambda (item) (select-vars-types-modes-cell item (first (selected-cells item))))
      :table-sequence var-list
      :view-nick-name :vars
      :sequence-order :horizontal
      :table-vscrollp nil
      :table-hscrollp nil
      :table-print-function #'table-print)
     
     (make-dialog-item
      'sequence-dialog-item #@(16 30) (make-point (- (point-h size) 31) 11) ""
      #'(lambda (item) (select-vars-types-modes-cell item (first (selected-cells item))))
      :table-sequence type-list
      :view-nick-name :types
      :sequence-order :horizontal
      :table-vscrollp nil
      :table-hscrollp nil
      :table-print-function #'table-print)
     
     (make-dialog-item
      'sequence-dialog-item #@(16 44) (make-point (- (point-h size) 31) 11) ""
      #'(lambda (item) (select-vars-types-modes-cell item (first (selected-cells item))))
      :table-sequence mode-list
      :view-nick-name :modes
      :sequence-order :horizontal
      :table-vscrollp nil
      :table-hscrollp nil
      :table-print-function #'table-print)
     
     (make-dialog-item
      'sequence-dialog-item #@(2 58) (make-point 11 (- (point-v size) 120)) "" 
      #'(lambda (item) (select-class-cell item (first (selected-cells item))))
      :table-sequence class-list
      :sequence-order :vertical
      :table-vscrollp nil
      :table-hscrollp nil
      :table-print-function #'table-print
      :view-nick-name :class)
     
     (make-dialog-item
      'array-dialog-item #@(16 58) (make-point (- (point-h size) 16) (- (point-v size) 105)) ""
      #'(lambda (item) (select-data-cell item (first (selected-cells item))))
      :table-array element-array
      :selection-type :single
      :table-hscrollp t
      :table-vscrollp t
      :table-print-function #'table-print
      :view-nick-name :data)
     
     (make-dialog-item 
      'editable-text-dialog-item
      (make-point 16 (- (point-v size) 20)) #@(200 11) "" nil
      :view-font '("monaco" 9)
      :dialog-item-enabled-p (not no-edit)
      :view-nick-name :edit)
     
     (make-dialog-item
      'check-box-dialog-item (make-point 20 (- (point-v size) 44))  nil "compute-neg" nil
      :dialog-item-enabled-p (not (or no-edit template))
      :check-box-checked-p compute-neg :view-nick-name :compute-neg)

     (make-dialog-item
      'check-box-dialog-item (make-point 195 (- (point-v size) 44))  nil "induction" nil
      :dialog-item-enabled-p (not (or no-edit template))
      :check-box-checked-p induction :view-nick-name :induction)
     
     (make-dialog-item
      'check-box-dialog-item (make-point 285 (- (point-v size) 44)) nil "commutative" nil
      :dialog-item-enabled-p (not (or no-edit template))
      :check-box-checked-p commutative :view-nick-name :commutative)
     
     (make-dialog-item
      'check-box-dialog-item (make-point 400 (- (point-v size) 44)) nil "constraint" nil
      :dialog-item-enabled-p (not (or no-edit template))
      :check-box-checked-p constraint :view-nick-name :constraint)
     
     (make-dialog-item
      'button-dialog-item (make-point 235 (- (point-v size) 25)) #@(60 20) " New "
      #'(lambda (item) (new-value item))
      :dialog-item-enabled-p (not no-edit)
      :view-nick-name :new)
     
     (make-dialog-item
      'button-dialog-item (make-point 300 (- (point-v size) 25)) #@(60 20) " Delete "
      #'(lambda (item) (delete-value item))
      :dialog-item-enabled-p (not no-edit)
      :view-nick-name :delete)
     
     (make-dialog-item
      'button-dialog-item (make-point 365 (- (point-v size) 25)) #@(60 20) " English "
      #'(lambda (item) (setf (english window) (edit-translation (first (table-sequence (find-named-sibling item :relations)))
                                                                (table-sequence (find-named-sibling item :vars))
                                                                :extensional
                                                                (english (view-container item)))))
      :dialog-item-enabled-p (not (or no-edit template))
      :view-nick-name :english)
     
     (make-dialog-item
      'button-dialog-item (make-point 430 (- (point-v size) 25)) #@(60 20) " Define "
      #'(lambda (item) (re-def-pred-from-example-window (view-container item)))
      :dialog-item-enabled-p (not no-edit)
      :view-nick-name :define)))
  (set-cell-size window 90 11)
  )

(defun select-relations-cell (item cell)
  (let ((window (view-container item)))
    (if (or (example-template window)
            (no-edit window))
      (update-examples-window-table-cells item nil nil)
      (when cell
        (let ((h (point-h cell))
              (data (find-named-sibling item :data)))
          (cell-select item cell)
          (update-examples-window-table-cells item (first (selected-cells item)) t)
          (cell-select (view-named :vars window) cell)
          (cell-select (view-named :types window) cell)
          (cell-select (view-named :modes window) cell)
          (dotimes (v (point-v (table-dimensions data)))
            (cell-select data h v)))))))

(defun select-vars-types-modes-cell (item cell)
  (let ((window (view-container item)))
    (if (or (example-template window)
            (no-edit window))
      (update-examples-window-table-cells item nil nil)
      (when cell
        (update-examples-window-table-cells item cell nil)
        (cell-select item cell)))))

(defun select-class-cell (item cell)
  (when cell
    (let ((v (point-v cell))
          (data (find-named-sibling item :data)))
      (update-examples-window-table-cells item cell nil)
      (cell-select item cell)
      (dotimes (h (point-h (table-dimensions data)))
        (cell-select data h v)))))

(defun select-data-cell (item cell)
  (when cell
    (cell-select item cell)
    (update-examples-window-table-cells item cell t)))

(defmethod size-items ((window examples-window))
  (without-interrupts
   (with-focused-view window
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
       (#_eraserect view-rect)
       (#_beginupdate (wptr window))
       (let* ((size (view-size window))
              (v (point-v size))
              (h (point-h size)))
         (set-view-size (view-named :relations window) (- h 31) 11)
         (set-view-size (view-named :vars window) (- h 31) 11)
         (set-view-size (view-named :types window) (- h 31) 11)
         (set-view-size (view-named :modes window) (- h 31) 11)
         (set-view-size (view-named :class window) 11 (- v 120))
         (set-view-size (view-named :data window) (- h 16) (- v 105))
         (set-view-position (view-named :compute-neg window) 20 (- v 44))
         (set-view-position (view-named :induction window) 195 (- v 44))
         (set-view-position (view-named :commutative window) 285 (- v 44))
         (set-view-position (view-named :constraint window) 400 (- v 44))
         (set-view-position (view-named :edit window) 16 (- v 20))
         (set-view-position (view-named :new window) 235 (- v 25))
         (set-view-position (view-named :delete window) 300 (- v 25))
         (set-view-position (view-named :english window) 365 (- v 25))
         (set-view-position (view-named :define window) 430 (- v 25)))
       (#_endupdate (wptr window))
       (#_invalrect view-rect)))))

(defmethod view-click-event-handler ((window examples-window) where)
  (update-selected-cell window)
  (let ((old-sp (scroll-position (view-named :data window))))
    (call-next-method window where)
    (let ((new-sp (scroll-position (view-named :data window))))
      (unless (= old-sp new-sp)
        (scroll-examples-window-sequences window (point-h new-sp) (point-v new-sp))))))

(defmethod scroll-examples-window-data ((window examples-window) h v)
  (scroll-to-cell (view-named :data window) h v)
  (scroll-examples-window-sequences window h v))

(defmethod scroll-examples-window-sequences ((window examples-window) h v)
  (scroll-to-cell (view-named :relations window) h 0)
  (scroll-to-cell (view-named :vars window) h 0)
  (scroll-to-cell (view-named :types window) h 0)
  (scroll-to-cell (view-named :modes window) h 0)
  (scroll-to-cell (view-named :class window ) 0 v))

(defun display-in-edit (edit value)
  (unless (no-edit (view-container edit))
    (select-all edit)
    (clear edit)
    (buffer-insert (fred-display-start-mark edit) value)
    (fred-update edit)
    (select-all edit)))

(defmethod update-selected-cell ((window examples-window))
  (let (item name cell i)
    (dolist (n '(:vars :types :modes :data :class :relations))  ;; this order is important!
      (setf i (view-named  n window))
      (when (selected-cells i)
        (setf item i
              name n
              cell (first (selected-cells item)))))
    (when cell
      (unless (no-edit window)
        (let ((old-contents (cell-contents item cell))
              (new-contents (value-from-dialog-item-text (view-named :edit window))))
          (unless (equal old-contents new-contents)
            (setf (changed window) t)
            (cond ((eq name :relations)
                   (unless (example-template window)
                     (set-table-sequence item (make-list (length (table-sequence item)) :initial-element new-contents))
                     (invalidate-view item)))
                  (t
                   (set-cell-contents item cell new-contents)
                   (redraw-cell item cell)))))))
    (values item cell)))

(defmethod view-key-event-handler ((window examples-window) char)
  (case char
    ((#\Enter #\Tab #\Newline #\BackArrow #\ForwardArrow #\UpArrow #\DownArrow)     
     (multiple-value-bind (item cell) (update-selected-cell window)
       (when cell
         (case char
           (#\BackArrow (move-cell-selection item cell -1 0 :wrap))
           (#\Tab (move-cell-selection item cell 1 0 :create-new-lines))
           (#\ForwardArrow (move-cell-selection item cell 1 0 :wrap))
           (#\UpArrow (move-cell-selection item cell 0 -1 :wrap))
           (#\DownArrow (move-cell-selection item cell 0 1 :wrap))))))
    (otherwise
     (call-next-method window char))))

(defmethod window-close ((window examples-window))
  (if (changed window)
    (if (y-or-n-dialog "The contents of the window have been changed.  Do you want to close the window without defining the relation?"
                       :size #@(300 150)
                       :position :centered
                       :yes-text " No "
                       :no-text " Yes"
                       :cancel-text nil)
      (throw-cancel nil)
      (call-next-method window))
    (call-next-method window)))

(defun move-cell-selection (item cell delta-h delta-v wrap-around)
  (let* ((max-dims (table-dimensions item))
         (old-h (point-h cell))
         (old-v (point-v cell))
         (new-h (min (max 0 (+ old-h delta-h)) (- (point-h max-dims) 1)))
         (new-v (min (max 0 (+ old-v delta-v)) (- (point-v max-dims) 1))))
    (if (and (= old-h new-h) (= old-v new-v))
      (move-cell-between-items item cell delta-h delta-v wrap-around)
      (let* ((new-cell (make-point new-h new-v))
             (vd (visible-dimensions item))
             (sp (scroll-position item))
             (sph (point-h sp))
             (spv (point-v sp)))
        (case (view-nick-name item)
          ((:vars :types :modes :data) 
           (cell-deselect item cell)
           (cell-select item new-cell))
          (:relations
           (cell-deselect item cell)
           (select-relations-cell item new-cell))
          (:class
           (cell-deselect item cell)
           (select-class-cell item new-cell)))
        (unless (and (<= sph new-h) (< new-h (+ sph (point-h vd)))
                     (<= spv new-v) (< new-v (+ spv (point-v vd))))
          (scroll-examples-window-data (view-container item) (+ sph delta-h) (+ spv delta-v)))
        
        (display-in-edit (find-named-sibling item :edit) (if (cell-contents item new-cell) (format nil "~S" (cell-contents item new-cell)) ""))
        ))))

(defun move-cell-between-items (item cell delta-h delta-v wrap-around)
  (let* ((window (view-container item))
         (limited (or (no-edit window) (example-template window)))
         (relations (find-named-sibling item :relations))
         (vars (find-named-sibling item :vars))
         (types (find-named-sibling item :types))
         (modes (find-named-sibling item :modes))
         (data (find-named-sibling item :data))
         (class (find-named-sibling item :class)))
    (case (view-nick-name item)
      (:data
       (cond ((= delta-h -1) (select-class-cell class cell))
             ((= delta-h 1) (case wrap-around
                              (:create-new-lines
                               (let ((v (point-v cell)))
                                 (cell-deselect data cell)
                                 (new-instance-line class data (make-point 0 v))
                                 (select-class-cell class (make-point 0 (+ v 1)))))
                              (:wrap (let ((max-v (- (point-v (table-dimensions data)) 1))
                                           (v (point-v cell)))
                                       (if (= v max-v)
                                         (select-data-cell data cell)
                                         (select-class-cell class (make-point 0 (+ v 1))))))
                              (otherwise nil)))
             ((= delta-v -1) (unless limited (select-vars-types-modes-cell modes cell)))
             ((= delta-v 1) nil)  ))
      (:class
       (cond ((= delta-h -1) (case wrap-around
                               ((:create-new-lines :wrap)
                                (let ((v (point-v cell)))
                                  (cond ((= v 0)
                                         (select-class-cell class cell))
                                        (t
                                         (update-examples-window-table-cells data cell nil)
                                         (select-data-cell data (make-point (- (point-h (table-dimensions data)) 1) (- v 1)))
                                         ))))
                               (otherwise nil)))
             ((= delta-h 1) (update-examples-window-table-cells data cell nil) (cell-select data cell))
             ((= delta-v -1) nil)
             ((= delta-v 1) nil)  ))
      (:modes
       (cond ((= delta-h 1) nil)
             ((= delta-h -1) nil)
             ((= delta-v -1) (select-vars-types-modes-cell types cell))
             ((= delta-v 1) (select-data-cell data cell))
             ))
      (:types
       (cond ((= delta-h 1) nil)
             ((= delta-h -1) nil)
             ((= delta-v -1) (select-vars-types-modes-cell vars cell))
             ((= delta-v 1) (select-vars-types-modes-cell modes cell))
             ))
      (:vars
       (cond ((= delta-h 1) nil)
             ((= delta-h -1) nil)
             ((= delta-v -1) (select-relations-cell relations cell))
             ((= delta-v 1) (select-vars-types-modes-cell types cell))
             ))
      (:relations
       (cond ((= delta-h 1) nil)
             ((= delta-h -1) nil)
             ((= delta-v -1) nil)
             ((= delta-v 1) (select-vars-types-modes-cell vars cell))
             ))
      )))

(defmethod set-view-size ((window examples-window) h &optional v)
  (without-interrupts
   (apply #'call-next-method window h v)
   (size-items window)))

(defmethod window-zoom-event-handler ((window examples-window) message)
  (declare (ignore message))
  (without-interrupts
   (call-next-method)
   (size-items window)))

(defmethod set-cell-size ((window examples-window) h &optional v)
  (unless v (setf v (point-v h)
                  h (point-h h)))
  (set-cell-size (view-named :relations window) h (point-v (cell-size (view-named :relations window))))
  (set-cell-size (view-named :vars window) h (point-v (cell-size (view-named :vars window))))
  (set-cell-size (view-named :types window) h (point-v (cell-size (view-named :types window))))
  (set-cell-size (view-named :modes window) h (point-v (cell-size (view-named :modes window))))
  (set-cell-size (view-named :class window) (point-h (cell-size (view-named :class window))) v)
  (set-cell-size (view-named :data window) h v)
  (invalidate-view window t))

(defun update-examples-window-table-cells (item cell exclude-item-cells?)
  (let ((edit (find-named-sibling item :edit)) i)
    (display-in-edit edit (if (and cell (cell-contents item cell)) (format nil "~S" (cell-contents item cell)) ""))
    (dolist (n '(:relations :vars :types :modes :class :data))
      (setf i (find-named-sibling item n))
      (unless (and exclude-item-cells? (eq i item))
        (dolist (cell (selected-cells i))
          (cell-deselect i cell))))))

(defun delete-value (button)
  (let* ((window (view-container button))
         (edit (view-named :edit window))
         (relations (view-named :relations window))
         (vars (view-named :vars window))
         (types (view-named :types window))
         (modes (view-named :modes window))
         (class (view-named :class window))
         (data (view-named :data window)))
    (display-in-edit edit "")
    (cond ((selected-cells relations)
           (when (rest (table-sequence relations))
             (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-example-window-delete-column *user-monitor*)))
             (let* ((cell (first (selected-cells relations)))
                    (index (point-h cell)))
               (update-examples-window-table-cells relations nil nil)
               (let* ((dims (array-dimensions (table-array data)))
                      (v (first dims))
                      (h (- (second dims) 1)))
                 (setf (table-sequence relations) (delete (cell-contents relations cell) (table-sequence relations) :start index :end (+ index 1))
                       (table-sequence vars) (delete (cell-contents vars cell) (table-sequence vars) :start index :end (+ index 1))
                       (table-sequence types) (delete (cell-contents types cell) (table-sequence types) :start index :end (+ index 1))
                       (table-sequence modes) (delete (cell-contents modes cell) (table-sequence modes) :start index :end (+ index 1))
                       (table-array data) (delete-column (table-array data) index))
                 (set-table-dimensions data  h v))
               (invalidate-view relations t)
               (invalidate-view vars t)
               (invalidate-view types t)
               (invalidate-view modes t)
               (invalidate-view data))))
          
          ((selected-cells class)
           (when (rest (table-sequence class))   ;; do not allow deletion of last example
             (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-example-window-delete-row *user-monitor*)))
             (let* ((cell (first (selected-cells class)))
                    (index (point-v cell))
                    (dims (array-dimensions (table-array data)))
                    (v (- (first dims) 1))
                    (h (second dims)))
               (update-examples-window-table-cells class nil nil)
               (setf (table-sequence class) (delete (cell-contents class cell) (table-sequence class) :start index :end (+ index 1))
                     (table-array data) (delete-row (table-array data) index))
               (set-table-dimensions data  h v)
               (invalidate-view class t)
               (invalidate-view data))))
          
          ((selected-cells vars)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-delete-var *user-monitor*)))
           (let* ((cell (first (selected-cells vars)))
                  (index (point-h cell)))
             (update-examples-window-table-cells vars nil nil)
             (setf (table-sequence vars) (delete (cell-contents vars cell) (table-sequence vars) :start index :end (+ index 1))))
           (invalidate-view vars t))
          
          ((selected-cells types)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-delete-type *user-monitor*)))
           (let* ((cell (first (selected-cells types)))
                  (index (point-h cell)))
             (update-examples-window-table-cells types nil nil)
             (setf (table-sequence types) (delete (cell-contents types cell) (table-sequence types) :start index :end (+ index 1))))
           (invalidate-view types t))
          
          ((selected-cells modes)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-delete-mode *user-monitor*)))
           (let* ((cell (first (selected-cells modes)))
                  (index (point-h cell)))
             (update-examples-window-table-cells modes nil nil)
             (setf (table-sequence modes) (delete (cell-contents modes cell) (table-sequence modes) :start index :end (+ index 1))))
           (invalidate-view modes t))
          
          ((selected-cells data)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-delete-datum *user-monitor*)))
           (let ((cell (first (selected-cells data))))
             (set-cell-contents data cell nil)
             (redraw-cell data cell)))) ))
                 

(defun new-value (button)
  (let* ((window (view-container button))
         (edit (view-named :edit window))
         (relations (view-named :relations window))
         (vars (view-named :vars window))
         (types (view-named :types window))
         (modes (view-named :modes window))
         (class (view-named :class window))
         (data (view-named :data window)))
    (display-in-edit edit "")
    (cond ((selected-cells relations)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-column *user-monitor*)))
           (let* ((cell (first (selected-cells relations)))
                  (dims (array-dimensions (table-array data)))
                  (v (first dims))
                  (h (second dims))
                  (contents (cell-contents relations cell)))
             (display-in-edit edit contents)
             (setf (table-sequence relations) (list-insert contents  (table-sequence relations) h)
                   (table-sequence vars) (list-insert '?arg (table-sequence vars) h)
                   (table-sequence types) (list-insert ':anything (table-sequence types) h)
                   (table-sequence modes) (list-insert ':? (table-sequence modes) h)
                   (table-array data) (insert-column (table-array data) h))
             (set-table-dimensions data (+ h 1) v))
           (invalidate-view relations)
           (invalidate-view vars)
           (invalidate-view types)
           (invalidate-view modes)
           (invalidate-view data))

          ((selected-cells class)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-row *user-monitor*)))
           (new-instance-line class data (first (selected-cells class))))

          ((selected-cells vars)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-var *user-monitor*)))
           (let ((cell (first (selected-cells vars))))
             (setf (table-sequence vars) (list-insert nil (table-sequence vars) (+ (point-h cell) 1))))
           (invalidate-view vars))
           
          ((selected-cells types)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-type *user-monitor*)))
           (let ((cell (first (selected-cells types))))
             (setf (table-sequence types) (list-insert nil (table-sequence types) (+ (point-h cell) 1))))
           (invalidate-view types))

          ((selected-cells modes)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-mode *user-monitor*)))
           (let ((cell (first (selected-cells modes))))
             (setf (table-sequence modes) (list-insert nil (table-sequence modes) (+ (point-h cell) 1))))
           (invalidate-view modes))
          
          ((selected-cells data)
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-example-window-new-datum *user-monitor*)))
           (let* ((cell (first (selected-cells data)))
                  (dims (array-dimensions (table-array data)))
                  (v (first dims))
                  (h (second dims))
                  (index (+ (point-v cell) 1)))
             (setf (table-sequence class) (list-insert nil (table-sequence class) index)
                   (table-array data) (insert-row (table-array data) index))
             (set-table-dimensions data h (+ v 1))
             (invalidate-view class)
             (invalidate-view data)))
    )))

;;;_______________________________________
;;;

(defun new-instance-line (class data cell)
  (let* ((dims (array-dimensions (table-array data)))
         (v (first dims))
         (h (second dims))
         (index (+ (point-v cell) 1)))
    (setf (table-sequence class) (list-insert nil (table-sequence class) index)
          (table-array data) (insert-row (table-array data) index))
    (set-table-dimensions data h (+ v 1))
    (invalidate-view class)
    (invalidate-view data)))



;;;_______________________________________
;;;  display-examples-or-tuples

(defun display-examples-or-tuples (vars pos neg title)
  (with-cursor *watch-cursor*
    (let* ((name (predicate-being-learned))
           (pred (get-pred name)))
      (cond ((and pred *map-tuples-back-to-examples*)
             (setf pos (return-originals-extended (r-pos pred) pos)
                   neg (return-originals-extended (r-neg pred) neg))
             (if *use-templates-when-possible*
               (display-examples name :pos pos :neg neg :title title :no-edit t)
               (display-examples nil :pos pos :neg neg :title title :display-tuples-only t :vars vars :no-edit t)))
            (t (display-examples nil :pos pos :neg neg :title title :display-tuples-only t :vars vars :no-edit t))))))

;;;_______________________________________
;;; display-top-level-examples

(defun display-top-level-examples ()
  (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-display-top-level-examples *user-monitor*)))
  (display-examples (predicate-being-learned)))


;;;_______________________________________
;;; valid-template

(defun valid-template (template)
  (let ((invalid nil)
        (error-message "")
        (undefined-relations nil)
        (wrong-args nil)
        (type-clash nil)
        (no-args nil))
    (if (not (example-template-p template))
      (setf error-message "Example template is not valid."
            invalid t)
      (let* ((name (example-template-name template))
             (r-struct (get-pred name))
             (vars (example-template-vars template))
             (facts (example-template-facts template)))
        (cond ((not (r-p r-struct))
               (setf error-message (format nil "Example template is not valid.~%The primary fact is not defined.")
                     invalid t))
              ((not (= (length vars) (r-arity r-struct)))
               (setf error-message (format nil "Example template is not valid.~%The fact ~(~S~) should have ~A arguments." name (r-arity r-struct))
                     invalid t))
              (t
               (let ((var-type-alist (mapcar #'cons vars (r-type r-struct))))
                 (dolist (fact facts)
                   (let* ((fact-name (first fact))
                          (fact-vars (rest fact))
                          (fact-r (get-pred fact-name)))
                     (cond ((not (r-p fact-r))
                            (push fact-name undefined-relations)
                            (setf invalid t))
                           ((not (= (length (r-type fact-r)) (length fact-vars)))
                            (push fact wrong-args)
                            (setf invalid t))
                           ((not (some #'(lambda (v) (member v vars :test #'var-eq)) fact-vars))
                            (push fact no-args)
                            (setf invalid t))
                           (t
                            (let ((fact-types (r-type fact-r)))
                              (multiple-value-bind (new-vars new-types) (compute-new-vars-and-types fact-vars fact-types vars nil nil)
                                (when new-vars
                                  (setq vars (append vars new-vars)
                                        var-type-alist (nconc var-type-alist (mapcar #'cons new-vars new-types))))
                                (do* ((args fact-vars (rest args))
                                      (arg (first args) (first args))
                                      (arg-types fact-types (rest arg-types))
                                      (type (first arg-types) (first arg-types)))
                                     ((null args))
                                  (when (member arg vars :test #'var-eq)
                                    (unless (type-eq type (rest (assoc arg var-type-alist)))
                                      (pushnew fact type-clash)
                                      (setf invalid t))))))))))
                 (when invalid
                   (setf error-message (with-output-to-string (out)
                                         (format out "Example template ~S is not valid." name)
                                         (when undefined-relations
                                           (format out "~%  Relations not defined extensionally:~%     ~{~(~s~)~^, ~}"  undefined-relations))
                                         (when wrong-args
                                           (format out "~%  Facts with wrong number of arguments:~%     ~{~(~s~)~^, ~}" wrong-args))
                                         (when no-args
                                           (format out "~%  Facts that don't reference template vars:~%     ~{~(~s~)~^, ~}" no-args))
                                         (when type-clash
                                           (format out "~%  Facts containing type clashes:~%     ~{~(~s~)~^, ~}" type-clash)))))
                 )))))
    (cond (invalid
           (if *user-interface-available*
             (message-dialog error-message :position :centered :size #@(500 200))
             (format t "~%~%~A~%" error-message))
           nil)
          (t template))))


;;;_______________________________________
;;;  display-examples

(defun display-examples (relation-name &key 
                                       (pos :default)
                                       (neg :default)
                                       (title (format nil "~(~s~) examples" relation-name))
                                       (use-template-when-possible *use-templates-when-possible*)
                                       (no-edit nil)
                                       (display-tuples-only nil)
                                       (vars nil)
                                       (pred (get-pred relation-name)))
  (with-cursor *watch-cursor*
    (let (relation-list var-list type-list mode-list class-list element-array template)
      (when pred
        (when (eq pos :default) (setf pos (r-pos pred)))
        (when (eq neg :default) (setf neg (if (r-compute-neg pred) nil (r-neg pred)))))
      (cond
       (display-tuples-only
        (multiple-value-setq (relation-list var-list type-list mode-list class-list element-array)
          (create-example-arrays-for-tuples vars pos neg)))
       ((and use-template-when-possible (setf template (valid-template (find-example-template relation-name))))
        (multiple-value-setq (relation-list var-list type-list mode-list class-list element-array)
          (create-example-arrays-using-template template pos neg)))
       (t
        (multiple-value-setq (relation-list var-list type-list mode-list class-list element-array)
          (create-example-arrays-without-template pred pos neg))) )
      
      (window-select (make-instance 'examples-window
                       :window-title title
                       :window-show nil
                       :view-position :centered
                       :view-size #@(520 300)
                       :relation-list relation-list
                       :var-list var-list
                       :type-list type-list
                       :mode-list mode-list
                       :class-list class-list
                       :element-array element-array
                       :template template
                       :english (unless (or display-tuples-only template) (copy-list (r-questions pred)))
                       :induction (unless (or display-tuples-only template) (r-induction pred))
                       :commutative (unless (or display-tuples-only template) (r-commutative pred))
                       :constraint (unless (or display-tuples-only template) (r-constraint pred))
                       :no-edit no-edit
                       :compute-neg (unless (or display-tuples-only template) (r-compute-neg pred))
                       )))))


(defun re-def-pred-from-example-window (window)
  (let ((relation-list (table-sequence (view-named :relations window)))
        (vars (table-sequence (view-named :vars window)))
        (type (table-sequence (view-named :types window)))
        (mode (table-sequence (view-named :modes window)))
        (class-list (table-sequence (view-named :class window)))
        (element-array (table-array (view-named :data window)))
        (template (example-template window))
        (english (english window))
        (induction (check-box-checked-p (view-named :induction window)))
        (commutative (check-box-checked-p (view-named :commutative window)))
        (constraint (check-box-checked-p (view-named :constraint window)))
        (compute-neg (check-box-checked-p (view-named :compute-neg window))))
    (if template
      (define-relations-from-example-window class-list element-array template)
      (define-relation-from-table (first relation-list) vars type mode class-list element-array english
        induction commutative constraint compute-neg))
    (when (user-monitor-p *user-monitor*)
      (if template
        (incf (user-monitor-example-window-define-from-template *user-monitor*))
        (incf (user-monitor-example-window-define *user-monitor*)))))
  (setf (changed window) nil))

(defun create-example-arrays-for-tuples (vars pos neg)
  (let* ((number-fields (length (or vars (first pos) (first neg))))
         (var-list (or vars (make-old-vars number-fields)))
         (number-pos (length pos))
         (number-neg (length neg))
         (class-list (or (nconc (make-list number-pos :initial-element *pos-class-indicator*)
                                (make-list number-neg :initial-element *neg-class-indicator*))
                         (list nil)))
         (element-array (make-array (list (max 1  (+ number-pos number-neg)) (max 1 number-fields)))))
    (setf element-array (insert-info-into-arrays pos 0 number-fields element-array)
          element-array (insert-info-into-arrays neg number-pos number-fields element-array))
    (values nil nil nil var-list class-list element-array)))

(defun create-example-arrays-without-template (r-struct pos neg)
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-example-window-display-without-template *user-monitor*)))
  (let* ((number-fields (r-arity r-struct))
         (var-list (copy-list (r-vars r-struct)))
         (type-list (copy-list (r-type r-struct)))
         (mode-list (if (r-mode r-struct) (copy-list (r-mode r-struct)) (make-list number-fields :initial-element :?)))
         (relation-list (make-list number-fields :initial-element (r-name r-struct)))
         (number-pos (length pos))
         (number-neg (length neg))
         (class-list (or (nconc (make-list number-pos :initial-element *pos-class-indicator*)
                                (make-list number-neg :initial-element *neg-class-indicator*))
                         (list nil)))
         (element-array (make-array (list (max 1  (+ number-pos number-neg)) (max 1 number-fields)) :adjustable t)))
    (setf element-array (insert-info-into-arrays pos 0 number-fields element-array)
          element-array (insert-info-into-arrays neg number-pos number-fields element-array))
    (values relation-list var-list type-list mode-list class-list element-array)))
    
(defun insert-info-into-arrays (tuples tuple-index number-fields element-array)
  (dolist (tuple tuples)
    (dotimes (field-index number-fields)
      (setf (aref element-array tuple-index field-index) (nth field-index tuple)))
    (incf tuple-index))
  element-array)

(defun create-example-arrays-using-template (template pos neg)
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-example-window-display-using-template *user-monitor*)))
  (let* ((t-name (example-template-name template))
         (t-vars (example-template-vars template))
         (t-facts (example-template-facts template))
         (r-struct (get-r-struct t-name))
         (relation-list (make-list (r-arity r-struct) :initial-element t-name))
         (var-list (reverse (r-vars r-struct)))
         (type-list (reverse (r-type r-struct)))
         (mode-list (if (r-mode r-struct) (reverse (r-mode r-struct)) (make-list (r-arity r-struct) :initial-element :?)))
         (element-array (make-array (list (+ (length pos) (length neg)) (length relation-list)) :adjustable t))
         (class-list nil)
         (v 0)
         (bound-vars t-vars)
         name r one-var?)
    (dolist (fact t-facts)
      (setq name (first fact)
            r (get-r-struct name)
            one-var? nil)
      (when (r-p r)
        (do* ((lvars (rest fact) (rest lvars))
              (lvar (first lvars) (first lvars))
              (vars (r-vars r) (rest vars))
              (types (r-type r) (rest types))
              (modes (r-mode r) (rest modes)))
             ((null lvars)
              (unless one-var? 
                (setq relation-list (cons name relation-list)
                      var-list (cons nil var-list)
                      type-list (cons nil type-list)
                      mode-list (cons nil mode-list))))
          (unless (member lvar bound-vars :test #'var-eq)
            (unless (equal lvar *filtered-indicator*)
              (setq bound-vars (cons lvar bound-vars)))
            (setq one-var? t
                  relation-list (cons name relation-list)
                  var-list (cons (first vars) var-list)
                  type-list (cons (first types) type-list)
                  mode-list (cons (or (first modes) :?) mode-list))))))
    (multiple-value-setq (element-array class-list v) 
      (insert-example-info-into-arrays pos *pos-class-indicator* v t-vars t-facts element-array class-list))
    (multiple-value-setq (element-array class-list v)
      (insert-example-info-into-arrays neg *neg-class-indicator* v t-vars t-facts element-array class-list))
    (values (nreverse relation-list) (nreverse var-list) (nreverse type-list) (nreverse mode-list) class-list element-array)))

(defun return-object-var-defining-facts (t-facts defined-vars)
  (let (var)
    (mapcan #'(lambda (fact)
                (when (setq var (find-if-not #'(lambda (v) (or (member v defined-vars :test #'var-eq)
                                                               (equal v *filtered-indicator*)))
                                             (rest fact)))
                  (setq defined-vars (push var defined-vars))
                  (list (list fact var))))
            t-facts)))

(defun insert-example-info-into-arrays (tuples indicator v t-vars t-facts array class-list)
  (let* ((defining-facts (return-object-var-defining-facts t-facts t-vars)))
    (dolist (tuple tuples)
      (let ((h -1) old-v)
        (dolist (d tuple)
          (array-insert d array v (incf h)))
        (setq old-v v
              v (insert-fact-info-into-array (direct-mapping t-vars tuple) t-facts defining-facts array h v v)
              class-list (nconc class-list (cons indicator (make-list (- v old-v 1) :initial-element nil))))
        ))
    (values array class-list v)))

(defun insert-fact-info-into-array (mapping t-facts defining-facts array h v max-v)
  (if (null t-facts)
    (values (max max-v (incf v)) v)
    (let* ((fact (first t-facts))
           (pred (get-pred (first fact)))
           (vars (rest fact))
           (pattern (direct-substitute vars mapping t))
           (datas (cond ((every #'(lambda (p) (eq p *filtered-indicator*)) pattern)
                         nil)
                        ((some #'filtered? pattern)
                         (return-filtered-example-data (r-pos pred) pattern))
                        ((some #'(lambda (e) (example-matches-pattern e pattern)) (r-pos pred))
                         (list (list *true-indicator*)))))
           (count (max 1 (count-if #'filtered? pattern)))
           )
      (cond ((eq fact (first (first defining-facts)))
             (let* ((var (second (first defining-facts)))
                    (position (do* ((p 0)
                                    (es vars (rest es))
                                    (e (first es) (first es)))
                                   ((equal e var) p)
                                (unless (member e mapping :test #'equal :key #'first) (incf p))))
                    (mapping (cons (list nil nil) mapping)))
               (dolist (data datas)
                 (insert-data-into-array data array v h)
                 (rplaca (first mapping) var)
                 (rplaca (rest (first mapping)) (nth position data))
                 (setq v (insert-fact-info-into-array mapping (rest t-facts) (rest defining-facts) array (+ h count) v v)
                       max-v (max max-v v))))
             (values max-v v))
            
            (t
             (let ((current-v v)
                   (current-max-v max-v))
               (dolist (data datas)
                 (insert-data-into-array data array v h)
                 (setq max-v (max max-v (incf v))))
               (insert-fact-info-into-array mapping (rest t-facts) defining-facts array (+ h count) current-v (max current-max-v max-v))))
            ))))

(defun insert-data-into-array (data array v h)
  (let ((nh h))
    (dolist (datum data)
      (array-insert datum array v (incf nh)))))

;;;_______________________________________
;;;  array-insert

(defun array-insert (value array &rest subscripts)
  (unless (apply #'array-in-bounds-p array subscripts)
    (let ((dims (array-dimensions array)))
      (adjust-array array (mapcar #'+ (mapcar #'(lambda (x y) (max (+ (- x y) 1) 0)) subscripts dims) dims))))
  (setf (apply #'aref array subscripts) value))


;;;_______________________________________
;;; array manipulations

(defun delete-column (array index-to-be-removed)
  (let* ((dims (array-dimensions array))
         (v (first dims))
         (h (- (second dims) 1)))
    (dotimes (j v)
      (do ((i index-to-be-removed (incf i)))
          ((= i h))
        (setf (aref array j i) (aref array j (+ 1 i))))
      (setf (aref array j h) nil))
     (adjust-array array (list v h))))

(defun insert-column (array index-to-be-added)
  (let* ((dims (array-dimensions array))
         (v (first dims))
         (h (second dims)))
    (adjust-array array (list v (+ h 1)))
    (dotimes (j v)
      (do ((i h (decf i)))
          ((= i index-to-be-added))
        (setf (aref array j i) (aref array j (- i 1))))
      (setf (aref array j index-to-be-added) nil)))
  array)


(defun delete-row (array index-to-be-removed)
  (let* ((dims (array-dimensions array))
         (v (- (first dims) 1))
         (h (second dims)))
    (dotimes (i h)
      (do ((j index-to-be-removed (incf j)))
          ((= j v))
        (setf (aref array j i) (aref array (+ j 1) i)))
      (setf (aref array v i) nil))
    (adjust-array array (list v h))))

(defun insert-row (array index-to-be-added)
  (let* ((dims (array-dimensions array))
         (v (first dims))
         (h (second dims)))
    (adjust-array array (list (+ v 1) h))
    (dotimes (i h)
      (do ((j v (decf j)))
          ((= j index-to-be-added))
        (setf (aref array j i) (aref array (- j 1) i)))
      (setf (aref array index-to-be-added i) nil)))
  array)

(defun list-insert (atom list position)
  (if (= position 0)
    (if list
      (cons atom list)
      (list atom))
    (do ((p nil r)
         (r list (rest r))
         (i 0 (incf i)))
        ((or (= i position) (null r))
             (rplacd p (cons atom r))
             list))))


(defun define-relation-from-table (name vars type mode class-list element-array english induction commutative constraint compute-neg)
  (with-cursor *watch-cursor*
    (when (relation-table-valid-p vars type mode class-list element-array english induction commutative constraint compute-neg)
      (let* ((arity (length vars))
             (pos-tuples nil)
             (neg-tuples nil)
             (example-index -1)
             tuple)
        (dolist (class class-list)
          (incf example-index)
          (setf tuple nil)
          (do ((field-index arity))
              ((= 0 field-index))
            (setf tuple (push (aref element-array example-index (decf field-index)) tuple)))
          (when (every #'null tuple)
            (format t "~%Warning - ~A is probably not a valid example of relation ~A." tuple name))
          (cond
           ((eq class *neg-class-indicator*) (setf neg-tuples (push tuple neg-tuples)))
           ((eq class *pos-class-indicator*) (setf pos-tuples (push tuple pos-tuples)))
           (t (unless (every #'null tuple)
                (format t "~%Warning - ~A in the relation ~A was not assigned a class and is being added as a positive example." tuple name)
                (setf pos-tuples (push tuple pos-tuples))))))
        (eval `(def-pred ,name
                 :vars ,vars
                 :type ,type
                 :mode ,(if (every #'(lambda (m) (eq m :?)) mode) nil mode)
                 :questions ,english
                 :pos ,(nreverse pos-tuples)
                 :neg ,(if compute-neg :computed (nreverse neg-tuples))
                 :induction ,induction
                 :commutative ,commutative
                 :constraint ,constraint)))
      (setf *FACTS-CHANGED* t))))

(defun relation-table-valid-p (vars type mode class-list element-array english induction commutative constraint compute-neg)
  english induction commutative constraint
  (let ((valid t))

    (unless (= (length vars) (length type) (length mode) (array-dimension element-array 1))
      (message-dialog "Error - Vars, Type, Mode, and Tuples do not have same arity.")
      (setf valid nil))

    (unless (= (length class-list) (array-dimension element-array 0))
      (message-dialog "Error - Number of class indicators and Tuples do not agree.")
      (setf valid nil))

    (when (some #'(lambda (c) (not (or (eq c *pos-class-indicator*) (eq c *neg-class-indicator*)))) class-list)
      (message-dialog "Warning - Some class indicator is not valid."))

    (when (and (some #'(lambda (c) (eq c *neg-class-indicator*)) class-list) compute-neg)
      (message-dialog "Warning - explictly defined negative tuples will be removed when compute-neg is set."))

    ;;;  vars and english correspond
    ;;;  data elements are well formed no variables in positive examples
    valid))



;;;_______________________________________
;;;  return-filtered-example-data

(defun return-filtered-example-data (examples pattern)
  (delete-duplicates (mapcan #'(lambda (example) (when (example-matches-pattern example pattern)
                                                   (list (filtered-example-info example pattern))))
                             examples) :test #'equal))

(defun display-tuple-dialog ()
  (let*
    ((window-h 200)
     (window-v 120)
     (dialog
      (make-instance
        'dialog
        :window-show nil
        :window-type :double-edge-box
        :view-position :centered
        :view-size (make-point window-h window-v)
        :close-box-p nil
        :view-font '("Chicago" 12 :srcor :plain)
        :view-subviews
        (list 
         (make-dialog-item
          'static-text-dialog-item (make-point 10 5) nil "Display Tuples..." nil)
         
         (make-dialog-item
          'radio-button-dialog-item (make-point 15 30) nil "Input to Literal"
          #'(lambda (item) item (setf *display-tuple-kind* :input))
          :radio-button-cluster 0 :view-nick-name :input :radio-button-pushed-p (eql *display-tuple-kind* :input))
         (make-dialog-item
          'radio-button-dialog-item (make-point 15 45) nil "Covered by Literal"
          #'(lambda (item) item (setf *display-tuple-kind* :output))
          :radio-button-cluster 0 :view-nick-name :output :radio-button-pushed-p (not (or (eql *display-tuple-kind* :input)
                                                                                          (eql *display-tuple-kind* :excluded))))
         (make-dialog-item
          'radio-button-dialog-item (make-point 15 60) nil "Excluded by Literal" 
          #'(lambda (item) item (setf *display-tuple-kind* :excluded))
          :radio-button-cluster 0 :view-nick-name :excluded :radio-button-pushed-p (eql *display-tuple-kind* :excluded))
         
         ;;; Exit Buttons ___________________
         (make-dialog-item
          'button-dialog-item
          (make-point (- window-h 165) (- window-v 27)) #@(70 20) " Display "
          #'(lambda (item) (return-from-modal-dialog (view-nick-name (pushed-radio-button (view-container item) 0))))
          :default-button t)
         (make-dialog-item
          'button-dialog-item
          (make-point (- window-h 80) (- window-v 27)) #@(70 20) " Cancel "
          #'(lambda (item) item (return-from-modal-dialog :cancel))
          :default-button nil)))))
    (modal-dialog dialog)))


;;;_______________________________________
;;;  show-tuples-last-node-selected

(defmethod show-tuples-last-node-selected ((view graph-view))
  (catch-cancel
    (let ((node (last-node-selected view)))
      (when (node-p node)
        (let* ((coverages (remove-if-not #'coverage-p (node-coverage node)))
               (aux (node-aux node))
               (literal (when (and (literal-p aux) (or (literal-pos aux) (literal-new-pos aux))) aux)))
          (if (or coverages literal)
            (let* ((kind (display-tuple-dialog))
                   (title (format nil "~A ~(~A~) tuples" (node-string node) kind)))
              (if coverages
                (dolist (coverage coverages)
                  (case kind
                    (:input
                     (display-examples-or-tuples (coverage-input-vars coverage) (coverage-input-pos coverage) (coverage-input-neg coverage) title))
                    (:excluded
                     (display-examples-or-tuples (coverage-input-vars coverage)
                                                 (return-originals-not-extended (coverage-input-pos coverage) (coverage-output-pos coverage))
                                                 (return-originals-not-extended (coverage-input-neg coverage) (coverage-output-neg coverage)) title))
                    (otherwise
                     (display-examples-or-tuples (coverage-output-vars coverage) (coverage-output-pos coverage) (coverage-output-neg coverage) title))
                    ))
                (case kind
                  (:input
                   (display-examples-or-tuples nil (literal-pos literal) (literal-neg literal) title))
                  (:excluded
                   (display-examples-or-tuples nil
                                               (return-originals-not-extended (literal-pos literal) (literal-new-pos literal))
                                               (return-originals-not-extended (literal-neg literal) (literal-new-neg literal)) title))
                  (:output
                   (display-examples-or-tuples nil (literal-new-pos literal) (literal-new-neg literal) title)))))
            (message-dialog (format nil "No coverage information was found!") :position :centered)))))))

(defun define-relations-from-example-window (class-list element-array template)
  (with-cursor *watch-cursor*
    (let* ((t-vars (example-template-vars template))
           (t-facts (example-template-facts template))
           (number-relations (length t-facts))
           (assert-lists (make-list number-relations))
           (retract-lists (make-list number-relations))
           (number-fields-per-relation (mapcar #'(lambda (f) (count *filtered-indicator* f)) t-facts))
           (number-fields-in-example (length t-vars))
           (mapping (mapcar #'list t-vars))
           (example nil)
           tuple-to-assert tuple-to-retract)
      (define-primary-fact-from-example-window class-list element-array template)
      (dotimes (example-index (first (array-dimensions element-array)))
        (setf example nil)
        (dotimes (i number-fields-in-example)
          (push (aref element-array example-index i) example))
        (when (and (some #'(lambda (x) x) example)
                   (nth example-index class-list))
          (do ((vs (nreverse example) (rest vs))
               (ss mapping (rest ss)))
              ((null vs))
            (rplacd (first ss) (first vs))))
        (do* ((asserts assert-lists (rest asserts))
              (retracts retract-lists (rest retracts))
              (facts t-facts (rest facts))
              (fact (first facts) (first facts))
              (field-index number-fields-in-example (incf field-index (max 1 (or (first number-fields) 0))))
              (number-fields number-fields-per-relation (rest number-fields)))
             ((null facts))
          (multiple-value-setq
            (tuple-to-assert tuple-to-retract)
            (assemble-tuple (rest fact) mapping element-array example-index field-index))
          (when tuple-to-assert (rplaca asserts (push tuple-to-assert (first asserts))))
          (when tuple-to-retract (rplaca retracts (push tuple-to-retract (first retracts))))))
      
      (do* ((facts t-facts (rest facts))
            (name (caar facts) (caar facts))
            (asserts assert-lists (rest asserts))
            (retracts retract-lists (rest retracts)))
           ((null facts))
        (let* ((pred (get-pred name))
               (assertion-to-retract (delete-if-not #'(lambda (e) (some #'(lambda (p) (example-matches-pattern e p)) (first retracts))) 
                                                    (remove-if #'(lambda (x) (member x (first asserts) :test #'equalp)) (r-pos pred)) ))
               (assertion-to-assert (delete-if #'(lambda (x) (member x (r-pos pred) :test #'equal)) (first asserts))))
          
          (dolist (tuple assertion-to-retract)
            (retract-fact nil nil name tuple))
          (dolist (tuple assertion-to-assert)
            (assert-fact nil nil name tuple))
          
          (setf *new-facts-neg* nil
                *new-facts-pos* nil)))
      (setf *FACTS-CHANGED* t))))

(defun define-primary-fact-from-example-window (class-list element-array template)
    (with-cursor *watch-cursor*
      (let ((name (example-template-name template))
            (example-length (length (example-template-vars template)))
            (example nil)
            (pos nil)
            (neg nil))
        (do* ((classes class-list (rest classes))
              (class (first classes) (first classes))
              (example-index 0 (incf example-index)))
             ((null classes))
          (when (or class (aref element-array example-index 0))
            (setf example nil)
            (dotimes (i example-length)
              (push (aref element-array example-index i) example))
            (setf example (nreverse example))
            (when (some #'null example)
              (format t "~%Warning - ~A is probably not a valid example of relation ~S." example name))
            (cond
             ((eq class *pos-class-indicator*) (push example pos))
             ((eq class *neg-class-indicator*) (push example neg))
             (t (format t "~%Warning - ~A in the relation ~A was not assigned a class and~%          being added as a positive example." example name))
             (push example pos))))
        (let* ((r-struct (get-r-struct name))
               (pos-to-be-retracted (set-difference (r-pos r-struct) pos :test #'equal))
               (neg-to-be-retracted (set-difference (r-neg r-struct) neg :test #'equal))
               (pos-to-be-asserted (set-difference pos (r-pos r-struct) :test #'equal))
               (neg-to-be-asserted (set-difference neg (r-neg r-struct) :test #'equal)))
          
          (dolist (tuple pos-to-be-retracted)
            (when (null (rest tuple))
              (unless (member tuple neg-to-be-asserted :test #'equalp)
                (delete-example-facts (first tuple)))))
          
          (dolist (tuple neg-to-be-retracted)
            (when (null (rest tuple))
              (unless (member tuple pos-to-be-asserted :test #'equalp)
                (delete-example-facts (first tuple)))))
          
          (dolist (tuple pos-to-be-retracted)
            (retract-fact nil nil name tuple))
          (dolist (tuple neg-to-be-retracted)
            (retract-fact nil t name tuple))
          (dolist (tuple pos-to-be-asserted)
            (assert-fact nil nil name tuple))
          (dolist (tuple neg-to-be-asserted)
            (assert-fact nil t name tuple))
          
          (setf *new-facts-neg* nil
                *new-facts-pos* nil
                *FACTS-CHANGED* t)
          ))))


(defun assemble-tuple (tuple-template mapping array example-index field-index)
  (let* ((array-refed? nil)
         (array-ref-non-nil? nil)
         (index (- field-index 1))
         (tuple-to-assert nil)
         (tuple-to-retract nil)
         value)
    (dolist (term tuple-template)
      (cond ((setf value (rest (assoc term mapping)))
             (push value tuple-to-assert)
             (push value tuple-to-retract))
            (t
             (setf value (aref array example-index (incf index))
                   array-refed? t)
             (when value (setf array-ref-non-nil? t))
             (push value tuple-to-assert)
             (push *filtered-indicator* tuple-to-retract))))
    (if (= (count *filtered-indicator* tuple-template) 0)
      (if (aref array example-index (incf index))
        (values (nreverse tuple-to-assert) nil)
        (values nil (nreverse tuple-to-retract)))
      (if (and array-refed? (not array-ref-non-nil?))
        (values nil (nreverse tuple-to-retract))
        (values (nreverse tuple-to-assert) (nreverse tuple-to-retract))))))