
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

(defvar *deanalyzed-state* :cliche)


;;;=======================================
;;;  ANALYZE-WINDOW

(defclass analyze-window (learning-window)
  ((only-count-first-cover :initarg only-count-first-cover :initform nil :accessor analyze-window-only-count-first-cover)
   (pred :initarg pred :initform nil :accessor analyze-window-pred)
   (rule :initarg pred :initform nil :accessor analyze-window-rule)))

;;;_______________________________________
;;;  INITIALIZE-INSTANCE

(defmethod initialize-instance ((window analyze-window) &rest initargs)
  (apply #'call-next-method window initargs)
  (add-subviews 
   window
   (make-dialog-item 'static-text-dialog-item #@(10 2) #@(500 16) "" nil
                     :view-nick-name :examples)
   (make-dialog-item 'button-dialog-item (make-point (- (point-h (view-size window)) 100) 1) #@(85 17) "Re-Analyze"
                     #'(lambda (item) (re-analyze-coverage (view-container item)))
                     :view-nick-name :re-analyze)))

;;;_______________________________________
;;;   SET-VIEW-SIZE

(defmethod set-view-size ((window analyze-window) h &optional (v nil))
  (call-next-method window h v)
  (if (null v)
    (setf v (point-v h)
          h (point-h h)))
  (set-view-position (view-named :re-analyze window) (make-point (- h 100) 1)))

;;;_______________________________________
;;;  NODE-CHANGED?

(defun node-changed? (node) (eq (node-aux node) :changed))
(defun set-node-changed (node) (setf (node-aux node) :changed))
(defun reset-node-changed (node) (setf (node-aux node) nil))

;;;_______________________________________
;;;  ANALYZE-CREATE-PRED-POP-UP-MENU

(defun analyze-create-pred-pop-up-menu (x y title)
  (let ((pred-menu
         (make-dialog-item 
          'pop-up-menu (make-point x y) #@(380 22) title nil :view-nick-name :pred-name
          :menu-items
          (mapcar #'(lambda (name) (make-instance 'menu-item :menu-item-title (format nil "~(~A~)" name)))
                  (mapcan #'(lambda (r) (if (pred-p r) (list (r-name r)))) *r-structs*)))))
    (dolist (item (menu-items pred-menu))
      (set-menu-item-action-function item #'(lambda () (disable-invalid-goal-concepts (view-container pred-menu)))))
    pred-menu))

;;;_______________________________________
;;;  ANALYZE-CREATE-RULE-POP-UP-MENU

(defun analyze-create-rule-pop-up-menu (x y title)
  (let ((rule-menu
         (make-dialog-item 
          'pop-up-menu (make-point x y) #@(380 22) title nil :view-nick-name :rule-name
          :menu-items (append (list (make-instance 'menu-item :menu-item-title "none")
                                    (make-instance 'menu-item :menu-item-title "-"))
                              (mapcar #'(lambda (name) (make-instance 'menu-item :menu-item-title (format nil "~(~A~)" name)))
                                      (mapcan #'(lambda (r) (if (rule-p r) (list (r-name r)))) *r-structs*)))
          )))
    rule-menu))

;;;_______________________________________
;;;  ANALYZE-COVERAGE-SETTING-DIALOG

(defun analyze-coverage-setting-dialog (pred rule depth only-count-first-cover)
  (let* ((window-h 400) (window-v 140)
         (dialog
          (make-instance
            'dialog
            :window-show nil
            :window-type :double-edge-box
            :view-position :centered
            :view-size (make-point window-h window-v)
            :close-box-p nil
            :view-font '("Chicago" 12 :srcor :plain)
            :view-subviews
            (list 
             (make-dialog-item
              'static-text-dialog-item (make-point 10 5) #@(250 16) "Analyze Coverage..." nil)
             (analyze-create-pred-pop-up-menu 10 30 "Examples from Fact:")
             (analyze-create-rule-pop-up-menu 10 55 "Coverage of Rule:")
             (make-dialog-item 'editable-text-dialog-item (make-point 12 85) #@(18 16) (format nil "~A" depth) nil :allow-returns nil :view-nick-name :depth)
             (make-dialog-item 'static-text-dialog-item (make-point 35 85) nil "Analysis Depth" nil)
             
             (make-dialog-item 'check-box-dialog-item (make-point 12 110) nil "Only count first cover" nil
                               :check-box-checked-p only-count-first-cover :view-nick-name :only-count-first-cover)
             (make-dialog-item 'button-dialog-item (make-point (- window-h 165) (- window-v 27)) #@(70 20) " Analyze "
                               #'(lambda (item) (multiple-value-bind (p r) (pred-name-and-rule-name-from-dialog (view-container item))
                                                  (return-from-modal-dialog
                                                   (values p r
                                                           (number-from-dialog-item-text (find-named-sibling item :depth))
                                                           (check-box-checked-p (find-named-sibling item :only-count-first-cover))))))
                               :default-button t)
             (make-dialog-item
              'button-dialog-item
              (make-point (- window-h 80) (- window-v 27)) #@(70 20) " Cancel "
              #'(lambda (item) item (return-from-modal-dialog :cancel))
              :default-button nil)))))
    (when pred
      (select-pop-up-menu-item (view-named :pred-name dialog) pred))
    (when rule
      (select-pop-up-menu-item (view-named :rule-name dialog) rule))
    (modal-dialog dialog)))

;;;_______________________________________
;;;  ANALYZE-COVERAGE

(defvar *analyze-depth* 1)
(defvar *analyze-only-count-first-cover* t)
(defvar *analyze-batch* t)

(defun analyze-coverage (&optional pred-name rule-name (depth *analyze-depth*) (only-count-first-cover *analyze-only-count-first-cover*))
  (catch-cancel
    (unless (and pred-name rule-name)
      (multiple-value-setq (pred-name rule-name depth only-count-first-cover)
        (analyze-coverage-setting-dialog (or *analyze-pred* (predicate-being-learned)) (or *analyze-rule* (goal-concept-name)) depth only-count-first-cover))
      (setq *analyze-depth* depth
            *analyze-only-count-first-cover* only-count-first-cover
            *analyze-pred* pred-name
            *analyze-rule* rule-name)
      (unless (and pred-name rule-name)
        (message-dialog "Both a fact and a rule must be selected to analyze coverage." :position :centered)
        (analyze-coverage)))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-analyze-coverage *user-monitor*)))
    (when (uses-cut? rule-name)
      (message-dialog (format nil "The definition of the rule ~(~A~)~%contains a cut therefore the analysis~%will probably not be accurate." rule-name) :position :centered))
    (let* ((*batch* *analyze-batch*)
           (*maintain-prolog-rule-trace* nil)
           (pred (get-pred pred-name))
           (rule (get-rule rule-name))
           (neg (r-neg pred))
           (pos (r-pos pred))
           (graph (generate-graph (list rule) nil :every-use depth))
           (base (graph-a-base graph))
           (window (make-instance 'analyze-window :window-show nil :window-title (format nil "Analyze Coverage of ~(~S~)" rule-name)))
           (view (graph-view window)))
      (setf (expand view) :always
            (analyze-window-only-count-first-cover window) only-count-first-cover
            (analyze-window-pred window) pred
            (analyze-window-rule window) rule)
      (set-dialog-item-text (view-named :examples window) (format nil "Examples:  ~(~S~)   ~A+  ~A-" (r-name pred) (length pos) (length neg)))
      (display-in-window graph window)
      (with-focused-view view
        (set-view-font (graph-window view) (view-font view))
        (insert-node-tuples (list base) pos neg (node-vars base) (r-type rule) :analyze view t t only-count-first-cover))
      (invalidate-view window t))))

;;;_______________________________________
;;;  HILIGHT-NODE-COVERAGE

(defun hilight-node-coverage (node extended-pos-tuples extended-neg-tuples &optional (negated? nil))
  (declare (ignore negated?))
  (cond ((eq extended-pos-tuples :uses-undefined-relation) (setf (node-state node) :undefined))
        ((and extended-neg-tuples (null extended-pos-tuples)) (setf (node-state node) :covers-neg))
        ((and extended-neg-tuples extended-pos-tuples) (setf (node-state node) :covers-pos-and-neg))
        ((and (null extended-neg-tuples) extended-pos-tuples) (setf (node-state node) :covers-pos))
        ((and (null extended-neg-tuples) (null extended-pos-tuples)) (setf (node-state node) :covers-none))))

;;;_______________________________________
;;;  NODES-OR-ANTECEDENTS-CHANGED?

(defun nodes-or-antecedents-changed? (nodes)
  (cond ((null nodes) nil)
        ((node-p nodes) (or (node-changed? nodes) (nodes-or-antecedents-changed? (node-antecedents nodes))))
        (t (or (nodes-or-antecedents-changed? (first nodes)) (nodes-or-antecedents-changed? (rest nodes))))))

;;;_______________________________________
;;;  RE-ANALYZE-COVERAGE

(defun re-analyze-coverage (window)
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-analyze-coverage *user-monitor*)))
  (let* ((*batch* *analyze-batch*)
         (view (graph-view window))
         (graph (graph view))
         (base (graph-a-base graph))
         (coverage (find :analyze (node-coverage base) :key #'coverage-from)))
    (deselect-node (graph-root graph) t)
    (with-focused-view view
      (set-view-font (graph-window view) (view-font view))
      (if (some #'node-changed? (graph-used-nodes graph))
        (insert-node-tuples (list base) (coverage-input-pos coverage) (coverage-input-neg coverage) (coverage-input-vars coverage) (coverage-input-type coverage)
                            :analyze view t t (analyze-window-only-count-first-cover window))
        (analyze-new-nodes-below base view (analyze-window-only-count-first-cover window))))
    (mapc #'(lambda (node) (if (node-changed? node) (reset-node-changed node))) (graph-used-nodes graph)))
  (invalidate-view window t))

;;;_______________________________________
;;;  ANALYZE-NEW-NODES-BELOW

(defun analyze-new-nodes-below (node view only-count-first-cover)
  (when (node-p node)
    (let ((antecedents (node-antecedents node))
          (coverage (find :analyze (node-coverage node) :key #'coverage-from)))
      (if (some #'(lambda (c) (some #'(lambda (n) (not (find :analyze (node-coverage n) :key #'coverage-from))) c)) antecedents)
        (insert-node-tuples antecedents (coverage-input-pos coverage) (coverage-input-neg coverage) (coverage-input-vars coverage) (coverage-input-type coverage)
                            :analyze view t t only-count-first-cover)
        (dolist (c antecedents)
          (dolist (n c)
            (analyze-new-nodes-below n view only-count-first-cover)))))))

;;;_______________________________________
;;;  DEANALYZE-NODE

(defun deanalyze-node (node view)
  (setf (node-state node) *deanalyzed-state*
        (cell-external-text (node-cell view node)) nil))

;;;_______________________________________
;;;  DEANALYZE-UP

(defun deanalyze-up (node view)
  (when (and (node-p node) (not (node-root? node)))
    (deanalyze-node node view)
    (deanalyze-down (rest (member node (conjunction-containing-node node))) view)
    (deanalyze-up (node-consequent node) view)))

;;;_______________________________________
;;;  DEANALYZE-DOWN

(defun deanalyze-down (nodes view)
  (cond ((null nodes) nil)
        ((node-p nodes)
         (deanalyze-node nodes view)
         (deanalyze-down (node-antecedents nodes) view))
        (t
         (dolist (n nodes)
           (deanalyze-down n view)))))

;;;_______________________________________
;;;  DEANALYZE-PATH-CONTAINING-NODE

(defun deanalyze-path-containing-node (node view)
  (set-node-changed node)
  (deanalyze-down (member node (conjunction-containing-node node)) view)
  (deanalyze-up node view))



