;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

(defparameter *simplify-even-if-overly-general* nil)
(defparameter *check-for-shared-constants* nil)

;;;___________________________________________
;;;  SET-DELETION-HASH-FROM-CLAUSE

(defun set-deletion-hash-from-clause (rule-name clause)
  (do ((literal clause (literal-next literal))
       (literal-index 0 (incf literal-index)))
      ((null literal))
    (if (literal-deleted? literal)
      (delete-literal rule-name 0 literal-index)
      (undelete-literal rule-name 0 literal-index))))

;;;___________________________________________
;;;  SOME-LITERAL-NOT-DERIVED-VIA-EBL

(defun some-literal-not-derived-via-ebl (clause)
  (do ((non-ebl-literal? nil)
       (literal clause (next-literal literal)))
      ((or (null literal) non-ebl-literal?) non-ebl-literal?)
    (unless (eql (derivation-type (literal-derivation literal)) :ebl)
      (setq non-ebl-literal? t))))

;;;___________________________________________
;;;  SIMPLIFY-CLAUSE

(defun simplify-clause (clause     ;; a linked list of literals
                        pos-tuples ;; all remaining positive tuples
                        neg-tuples ;; all negative tuples
                        old-vars   ;; bound variables
                        old-modes  ;; modes of bound variables
                        )
  (when (and *simplify-clauses*
             (or (not *simplify-operationalizations*)
                 (some-literal-not-derived-via-ebl clause)))
    (simplify clause pos-tuples neg-tuples old-vars old-modes :simplify-c))
  (multiple-value-bind (extended-pos-tuples extended-neg-tuples extended-variables)
                       (insert-literal-tuples clause pos-tuples neg-tuples old-vars)
    (let ((pos-tuples-not-covered (return-originals-not-extended pos-tuples extended-pos-tuples))
          (neg-tuples-covered (return-originals-extended neg-tuples extended-neg-tuples)))
      (when *display-learning?* (replace-last-clause-in-lcd-graph clause (length extended-pos-tuples) (length extended-neg-tuples)))
      (values clause pos-tuples-not-covered neg-tuples-covered extended-variables))))

;;;___________________________________________
;;;  SIMPLIFY-OPERATIONAL-CLAUSE

(defun simplify-operational-clause (winners
                                    winner
                                    pos-tuples ;; all positive tuples not covered by previous clauses
                                    neg-tuples ;; all negative tuples covered by current clause
                                    old-vars   ;; bound variables
                                    current-state-value
                                    use-negative-hash)
  (declare (ignore use-negative-hash))
  (when *simplify-operationalizations*
    (let ((clause (copy-literals (winner-literal winner))))
      (multiple-value-bind (modified? p-proved p-unproved n-proved n-unproved) (simplify clause pos-tuples neg-tuples old-vars nil :simplify-o)
        (declare (ignore p-unproved n-unproved))
        (when modified?
          (multiple-value-bind (new-vars new-types) (new-vars-still-bound (winner-vars winner) (winner-types winner) clause)
            (multiple-value-setq (clause new-vars new-types) (fix-new-vars clause old-vars new-vars new-types))
            (multiple-value-bind (new-pos-tuples new-neg-tuples) (insert-tuples clause pos-tuples neg-tuples old-vars)
              (update-winner winners *delete-better-function* t
                             (gain-metric current-state-value p-proved p-proved n-proved n-proved) clause :simplify-o
                             :vars new-vars
                             :types new-types
                             :pos new-pos-tuples
                             :neg new-neg-tuples)))))))
  winners)


;;;___________________________________________
;;;  SIMPLIFY
;;;
;;;  A clause generalization routine that attempts to delete each literal
;;;  in the clause, in an attempt to increase the coverage of positive examples
;;;  while not increasing the coverage of negative examples.
;;;  Basis of simplify-clause and simplify-operational-clause.
;;;
;;;  delete each literal in turn from clause if when deleted the clause if "better" (ebl)
;;;  or "at least as good" (induction ) without the literal.
;;;  "better" covers more positive tuples and not more negative tuples
;;;  "at least as good" covers at least as many positive tuples and not more negative tuples

(defun simplify (clause     ;; a linked list of literals
                 pos-tuples ;; all remaining positive tuples
                 neg-tuples ;; all negative tuples
                 old-vars   ;; bound variables
                 old-modes  ;; modes of bound variables
                 source)
  (push-status source)
  (let* ((some-literal-deleted? nil)
         (delete-literal? t)
         (ip (length pos-tuples))
         (in (length neg-tuples))
         extended-pos-tuples extended-neg-tuples
         p-proved n-proved p-unproved n-unproved
         new-extended-pos-tuples new-extended-neg-tuples
         new-p-proved new-p-unproved new-n-proved new-n-unproved
         new-literal-vars bound-vars)
    (multiple-value-setq (extended-pos-tuples extended-neg-tuples) (insert-literal-tuples clause pos-tuples neg-tuples old-vars nil))
    (when (or (null extended-neg-tuples) *simplify-even-if-overly-general*)
      (setq p-proved (count-originals-extended pos-tuples extended-pos-tuples)
            n-proved (count-originals-extended neg-tuples extended-neg-tuples)
            p-unproved (- ip p-proved)
            n-unproved (- in n-proved))
      (when *display-learning?*
        (display-winner-gain *CURRENT-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved)
        (display-winner-gain *BEST-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved))
      (do* ((literal clause (if delete-literal? clause (literal-next literal))))
           ((null literal))
        (if delete-literal?
          (setq new-literal-vars (compute-new-vars literal old-vars)
                bound-vars (append old-vars new-literal-vars)
                delete-literal? nil)
          (setq new-literal-vars (compute-new-vars literal bound-vars)
                bound-vars (append bound-vars new-literal-vars)))
        (cond ((literal-deleted? literal))
              ((eq (derivation-type (literal-derivation literal)) :object)                                ;; Delete uneeded object identity literals
               (unless (every #'(lambda (var) (or (member var old-vars :test #'var-eq)                    ;; Assumption a literal of with derivation-type
                                                  (variable-occurs-in-a-previous-literal var literal)))   ;; :object is of the form (not (= ?X ?Y))
                              (literal-variablization (literal-negated-literals literal)))
                 (setq delete-literal? t)))
              ((eq (derivation-type (literal-derivation literal)) :determinate)                           ;; Delete uneeded determinate literals
               (unless (some #'(lambda (var) (variable-occurs-in-a-subsequent-literal var literal t))
                             new-literal-vars)
                 (setq delete-literal? t)))
              ((and *check-for-shared-constants* (literal-causes-unbound-vars literal old-vars clause)))  ;; for compatibility with glenn's checks
              (t
               (setf (literal-deleted? literal) t)
               (when (or *check-for-shared-constants*
                         (all-variables-in-clause-bound-p clause old-vars old-modes))
                 (setq delete-literal? :test))
               (setf (literal-deleted? literal) nil)))
        (when delete-literal?
          (setf (literal-deleted? literal) t)
          (multiple-value-setq (new-extended-pos-tuples new-extended-neg-tuples) (insert-literal-tuples clause pos-tuples neg-tuples old-vars nil))
          (setq new-p-proved (count-originals-extended pos-tuples new-extended-pos-tuples)
                new-n-proved (count-originals-extended neg-tuples new-extended-neg-tuples)
                new-p-unproved (- ip new-p-proved)
                new-n-unproved (- in new-n-proved))
          (when *display-learning?*
            (display-winner-gain *CURRENT-GAIN-WINDOW* clause :source source :gain nil :pos new-p-proved :neg new-n-proved))
          (cond ((or (eql delete-literal? t)
                     (and (eql delete-literal? :test)
                          (<= new-n-proved n-proved)                                     ;; Criteria for deleting a literal
                          (if (eql (derivation-type (literal-derivation literal)) :ebl)  ;;
                            (< new-p-unproved p-unproved)                                ;;
                            (<= new-p-unproved p-unproved))))                            ;;
                 (setq some-literal-deleted? t
                       delete-literal? t
                       p-proved new-p-proved 
                       p-unproved new-p-unproved
                       n-proved new-n-proved
                       n-unproved new-n-unproved)
                 (when *display-learning?*
                   (display-winner-gain *BEST-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved)))
                (t
                 (setf delete-literal? nil
                       (literal-deleted? literal) nil))))))
    (pop-status)
    (values some-literal-deleted? p-proved p-unproved n-proved n-unproved clause)))

;;;___________________________________________
;;;  fix-new-vars

(defun fix-new-vars (clause old-vars new-vars new-types)
  (when new-vars
    (let* ((renamed-new-vars (make-old-vars (length new-vars) (length old-vars)))
           (mapping (direct-mapping new-vars renamed-new-vars)))
      (do ((literal clause (literal-next literal)))
          ((null literal))
        (if (literal-negated? literal)
          (fix-new-vars (literal-negated-literals literal) old-vars new-vars new-types)
          (setf (literal-variablization literal) (direct-substitute (literal-variablization literal) mapping))))
      (setq new-vars renamed-new-vars)))
  (values clause new-vars new-types))

;;;___________________________________________
;;;  new-vars-still-bound
;;;
;;;  added to determine which new-vars are still bound in clause
;;;  (after marked literals have been deleted)

(defun new-vars-still-bound (new-vars new-types clause)
  (do ((vars (reverse new-vars) (cdr vars))
       (types (reverse new-types) (cdr types))
       (bound-new-vars)
       (bound-new-types))
      ((null vars) (values bound-new-vars bound-new-types))
    (cond 
     ((do ((literal clause (literal-next literal)))
          ((null literal) nil)
        (if (and (not (literal-deleted? literal))
                 (member (car vars) (literal-variablization literal) :test #'var-eq))
          (return t)))
      (setf bound-new-vars (cons (car vars) bound-new-vars)
            bound-new-types (cons (car types) bound-new-types)))
     (t nil))))


;;;___________________________________________
;;;  literal-causes-unbound-vars

(defun literal-causes-unbound-vars (literal old-vars clause)
  (let* ((bound-vars
          (do ((vars old-vars)
               (l clause (literal-next l)))
              ((eql l literal) vars)
            (setq vars (append (literal-variablization l) vars))))
         (literal-new-vars 
          (all-images 
           #'(lambda (var)
               (if (not (member var bound-vars)) var))
           (literal-variablization literal))))
    (do ((l (literal-next literal) (literal-next l)))
        ((null l) nil)
      (if 
        (some 
         #'(lambda (var) 
             (member var literal-new-vars :test #'equalp)) 
         (literal-variablization l))
        (return t)))))


;;;___________________________________________
;;;  all-variables-in-clause-bound-p

(defun all-variables-in-clause-bound-p (clause old-vars old-modes)
  (when (literal-p clause)
    (let ((all-bound t))
      (multiple-value-bind (bound-vars unbound-vars) (bound-and-unbound-vars old-vars old-modes)
        (do ((literal clause (literal-next literal)))
            ((or (null literal) (null all-bound)) (and all-bound (null unbound-vars)))
          (unless (literal-deleted? literal)
            (let ((args (literal-args literal)))
              (when unbound-vars
                (dolist (arg args)
                  (if (member arg unbound-vars :test #'var-eq)
                    (setf unbound-vars (delete arg unbound-vars :test #'var-eq)))))
              (unless (and (some #'(lambda (var) (or (member var bound-vars :test #'var-eq)
                                                     (variable-occurs-in-a-previous-literal var literal)))
                                 args)
                           (every #'(lambda (var) (or (member var bound-vars :test #'var-eq)
                                                      (variable-occurs-in-a-previous-literal var literal)))
                                  (literal-vars-that-need-binding literal)))
                (setf all-bound nil)))))))))


;;;___________________________________________
;;;  bound-and-unbound-vars

(defun bound-and-unbound-vars (vars modes)
  (let ((bound-vars nil)
        (unbound-vars nil))
    (cond ((null modes) (values (remove-if-not #'pcvar-p vars) nil))
          (t (mapc #'(lambda (var mode)
                       (when (pcvar-p var)
                         (if (eq mode :-)
                           (push var unbound-vars)
                           (push var bound-vars))))
                   vars modes)
             (values bound-vars unbound-vars)))))

;;;___________________________________________
;;;  variable-occurs-in-a-previous-literal

(defun variable-occurs-in-a-previous-literal (variable literal)
  (let ((occurs nil))
    (do ((L (literal-prev literal) (literal-prev L)))
        ((or occurs (null L)))
      (cond ((literal-negated? L) nil)
            ((literal-deleted? L) nil)
            ((member variable (literal-args L) :test #'equalp) (setf occurs t))))
    occurs))

;;;___________________________________________
;;;  variable-occurs-in-a-subsequent-literal

(defun variable-occurs-in-a-subsequent-literal (variable literal &optional (exclude-object-not-= nil))
  (let ((occurs nil))
    (do ((L (literal-next literal) (literal-next L)))
        ((or occurs (null L)))
      (unless (and exclude-object-not-= (eq (derivation-type (literal-derivation L)) :object))
        (cond ((literal-deleted? L) nil)
              ((member variable (literal-args L) :test #'equalp) (setf occurs t)))) )
    occurs))

;;;___________________________________________
;;;  flatten-completely

(defun flatten-completely (expression)
  (cond ((null expression) nil)
        ((consp expression) (nconc (flatten-completely (first expression))
                                   (flatten-completely (rest expression))))
        (t (list expression))))

;;;___________________________________________
;;;  expression-vars

(defun expression-vars (expression)
  (delete-duplicates (delete-if-not #'pcvar-p (flatten-completely expression))))

;;;___________________________________________
;;;  literal-args

(defun literal-args (literal)
  (if (literal-negated? literal)
    (clause-args (literal-negated-literals literal))
    (let ((relation (literal-predicate-name literal)))
      (if (or (eql relation 'bagof)
              (eql relation 'call)
              (eql relation 'find-proofs)
              (eql relation 'setof)
              (eql relation 'is))
        (multiple-value-bind (ov nv) (variables-used-in (literal-variablization literal) nil)
          (nconc ov nv))
        (literal-variablization literal)))))

;;;___________________________________________
;;;  variables-used-in

(defun variables-used-in (form old-vars &optional (ov nil) (nv nil) (add-to-new t))
  (cond ((pcvar-p form) (if (or (member form ov :test #'var-eq)
                                (member form nv :test #'var-eq))
                          (values ov nv)
                          (if (member form old-vars)
                            (values (cons form ov) nv)
                            (if add-to-new
                              (values ov (cons form nv))
                              (values ov nv)))))
        ((consp form)
         (let ((relation (first form)))
           (cond ((eql relation 'not) (variables-used-in (rest form) old-vars ov nv nil))
                 ((eql relation 'or)  (variables-used-in (rest form) old-vars ov nv nil))
                 ((eql relation 'and) (variables-used-in (rest form) old-vars ov nv add-to-new))
                 (t (multiple-value-setq (ov nv) (variables-used-in relation old-vars ov nv add-to-new))
                    (variables-used-in (rest form) old-vars ov nv add-to-new)))))
        (t (values ov nv))))

;;;___________________________________________
;;;  clause-args

(defun clause-args (clause)
  (let ((vars nil))
    (do ((literal clause (literal-next literal)))
        ((null literal) vars)
      (setf vars (append vars (literal-args literal))))))

;;;___________________________________________
;;;  literal-vars-that-need-binding

(defun literal-vars-that-need-binding (literal)
  (cond ((literal-negated? literal) (clause-vars-that-need-binding (literal-negated-literals literal)))
        ((eq (literal-predicate-name literal) 'is) (expression-vars (rest (literal-variablization literal))))
        (t (mapcan #'(lambda (var mode) (when (and (eq mode :+) (pcvar-p var)) (list var)))
                   (literal-variablization literal)
                   (r-mode (get-r-struct (literal-predicate-name literal)))))))

;;;___________________________________________
;;;  clause-vars-that-need-binding

(defun clause-vars-that-need-binding (clause)
  (let ((vars nil))
    (do ((literal clause (literal-next literal)))
        ((null literal) vars)
      (setf vars (append vars (literal-vars-that-need-binding literal))))))



#|
;;;___________________________________________
;;;  literal-args

(defun literal-args (literal)
  (cond ((literal-negated? literal) (clause-args (literal-negated-literals literal)))
        ((eq (literal-predicate-name literal) 'is) (flatten-completely (literal-variablization literal)))
        (t (literal-variablization literal))))

;;;___________________________________________
;;;  all-variables-in-clause-bound-p

(defun all-variables-in-clause-bound-p (clause old-vars old-modes)
  (when clause
    (let ((all-bound t))
      (multiple-value-bind (bound-vars unbound-vars)
                           (bound-and-unbound-vars old-vars old-modes)
        (do ((literal clause (literal-next literal)))
            ((or (null literal) (null all-bound)) (and all-bound (null unbound-vars)))
          (unless (literal-deleted? literal)
            (when unbound-vars
              (dolist (arg (literal-args literal))
                (setf unbound-vars (delete arg unbound-vars :test #'equalp))))
            (unless (every #'(lambda (var) (or (member var bound-vars :test #'equalp)
                                               (variable-occurs-in-a-previous-literal var literal)))
                           (literal-vars-that-need-binding literal))
              (setf all-bound nil))))))))

;;;___________________________________________
;;;  count-proved-and-unproved-tuples

(defun count-proved-and-unproved-tuples (function tuples)
  (let ((proved 0)
        (unproved 0)
        (cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (if (catch 'reg-prove (apply function cont nil tuple))
        (incf proved)
        (incf unproved)))
    (values proved unproved)))

;;;___________________________________________
;;;  compile-and-simplify
;;;
;;;  compiles a function for the clause and uses the prolog deletion code to perform simplification.

(defun compile-and-simplify (clause     ;; a linked list of literals
                             pos-tuples ;; all remaining positive tuples
                             neg-tuples ;; all negative tuples
                             old-vars   ;; bound variables
                             old-modes  ;; modes of bound variables
                             source)
  (push-status source)
  (clear-clause-and-literal-deletions)
  (let* ((*compile-allowing-deletions* t)
         (rule-name (unique-r-name 'clause))
         (prolog-function (eval (focl-create-prolog-function rule-name (length old-vars) (list (cons (cons rule-name old-vars) (convert-literals-to-prolog clause t))))))
         (some-literal-deleted? nil)
         (delete-literal? t)
         literal-index new-literal-vars bound-vars new-p-proved new-p-unproved new-n-proved new-n-unproved)
    (set-deletion-hash-from-clause rule-name clause)
    (multiple-value-bind (p-proved p-unproved) (count-proved-and-unproved-tuples prolog-function pos-tuples)
      (multiple-value-bind (n-proved n-unproved) (count-proved-and-unproved-tuples prolog-function neg-tuples)
        (when (or (= n-proved 0) *simplify-even-if-overly-general*)
          (when *display-learning?*
            (display-winner-gain *CURRENT-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved)
            (display-winner-gain *BEST-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved))
          (do* ((literal clause (if delete-literal? clause (literal-next literal))))
               ((null literal))
            (if delete-literal?
              (setq literal-index 0
                    new-literal-vars (compute-new-vars literal old-vars)
                    bound-vars (append old-vars new-literal-vars)
                    delete-literal? nil)
              (setq literal-index (incf literal-index)
                    new-literal-vars (compute-new-vars literal bound-vars)
                    bound-vars (append bound-vars new-literal-vars)))
            (cond ((literal-deleted? literal))
                  ((eq (derivation-type (literal-derivation literal)) :object)                                ;; Delete uneeded object identity literals
                   (unless (every #'(lambda (var) (or (member var old-vars :test #'var-eq)                    ;; Assumption a literal of with derivation-type
                                                      (variable-occurs-in-a-previous-literal var literal)))   ;; :object is of the form (not (= ?X ?Y))
                                  (literal-variablization (literal-negated-literals literal)))
                     (setq delete-literal? t)))
                  ((eq (derivation-type (literal-derivation literal)) :determinate)                           ;; Delete uneeded determinate literals
                   (unless (some #'(lambda (var) (variable-occurs-in-a-subsequent-literal var literal t))
                                 new-literal-vars)
                     (setq delete-literal? t)))
                  ((and *check-for-shared-constants* (literal-causes-unbound-vars literal old-vars clause)))  ;; for compatibility with glenn's checks
                  (t
                   (setf (literal-deleted? literal) t)
                   (when (or *check-for-shared-constants*
                             (all-variables-in-clause-bound-p clause old-vars old-modes))
                     (setq delete-literal? :test))
                   (setf (literal-deleted? literal) nil)))
            (when delete-literal?
              (setf (literal-deleted? literal) t)
              (delete-literal rule-name 0 literal-index)
              (multiple-value-setq (new-p-proved new-p-unproved) (count-proved-and-unproved-tuples prolog-function pos-tuples))
              (multiple-value-setq (new-n-proved new-n-unproved) (count-proved-and-unproved-tuples prolog-function neg-tuples))
              (when *display-learning?*
                (display-winner-gain *CURRENT-GAIN-WINDOW* clause :source source :gain nil :pos new-p-proved :neg new-n-proved))
              (cond ((or (eql delete-literal? t)
                         (and (eql delete-literal? :test)
                              (<= new-n-proved n-proved)                                     ;; Criteria for deleting a literal
                              (if (eql (derivation-type (literal-derivation literal)) :ebl)  ;;
                                (< new-p-unproved p-unproved)                                ;;
                                (<= new-p-unproved p-unproved))))                            ;;
                     (setq some-literal-deleted? t
                           delete-literal? t
                           p-proved new-p-proved 
                           p-unproved new-p-unproved
                           n-proved new-n-proved
                           n-unproved new-n-unproved)
                     (when *display-learning?*
                       (display-winner-gain *BEST-GAIN-WINDOW* clause :source source :gain nil :pos p-proved :neg n-proved)))
                    (t
                     (setf delete-literal? nil
                           (literal-deleted? literal) nil)
                     (undelete-literal rule-name 0 literal-index))))))
        (clear-clause-and-literal-deletions)
        (pop-status)
        (values some-literal-deleted? p-proved p-unproved n-proved n-unproved clause)))))

|#