;;;; copyright (c) 1990, 1991 by the university of california, irvine. 
;;;; this program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  this program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the regents of the university of california.
;;;; this program was written by michael pazzani, cliff brunk, glenn silverstein
;;;; and kamal ali.  

(in-package :user)

(proclaim '(inline return-reduced-extended))

;;;___________________________________________
;;;  return-reduced-extended

(defun return-reduced-extended (original-tuples extended-tuples)
  (let ((excess (- (length (first extended-tuples)) (length (first original-tuples)))))
    (if (zerop excess)
      extended-tuples
      (mapcar #'(lambda (tuple) (butlast tuple excess)) extended-tuples))))

;;;___________________________________________
;;;  return-originals-not-extended

(defun return-originals-not-extended (original-tuples extended-tuples)
  (let ((reduced-tuples (return-reduced-extended original-tuples extended-tuples)))
    (all-images #'(lambda (tuple) (unless (member tuple reduced-tuples :test #'equal) tuple)) original-tuples)))

;;;___________________________________________
;;;  return-originals-extended

(defun return-originals-extended (original-tuples extended-tuples)
  (let ((reduced-tuples (return-reduced-extended original-tuples extended-tuples)))
    (all-images #'(lambda (tuple) (when (member tuple reduced-tuples :test #'equal) tuple)) original-tuples)))

;;;___________________________________________
;;;  count-originals-extended

(defun count-originals-extended (original-tuples extended-tuples)
  (let ((reduced-tuples (return-reduced-extended original-tuples extended-tuples)))
    (count-if #'(lambda (tuple) (member tuple reduced-tuples :test #'equal)) original-tuples)))


;;;_____________________________________________________
;;;  FIND-LITERAL-FROM-GOAL-CONCEPT

(defun find-literal-from-goal-concept (current-state-value goal-concept-and-rule variables variables-type pos-tuples neg-tuples use-hash-tables winners)
  (if goal-concept-and-rule
    (op-literal (first (first goal-concept-and-rule)) variables variables-type pos-tuples neg-tuples current-state-value :ebl use-hash-tables winners)
    :fail))

;;;___________________________________________
;;;  compute-gain

(defun compute-gain (thing current-state-value pos new-pos new-neg neg)
  (let* ((number-new-pos (length new-pos))
         (number-new-neg (length new-neg))
         (number-original-pos-extended (count-originals-extended pos new-pos))
         (number-original-neg-extended (count-originals-extended neg new-neg))
         (gain (gain-metric current-state-value number-original-pos-extended number-new-pos number-new-neg number-original-neg-extended)))
    (when (and *trace-learning?* (member :o *focl-trace-level*))
      (format t "~% ~a~% ~40Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              thing
              number-new-pos number-original-pos-extended (length pos)
              number-new-neg number-original-neg-extended (length neg) (gain-gain gain)))
    gain))

;;;___________________________________________
;;;  compute-negated-gain

(defun compute-negated-gain (thing current-state-value pos not-pos not-neg neg)
  (let* ((ip (length pos))
         (in (length neg))
         (p (- ip (count-originals-extended neg not-neg)))
         (n (- in (count-originals-extended pos not-pos)))
         (gain (gain-metric current-state-value ip p n in)))
    (when (and *trace-learning?* (member :o *focl-trace-level*))
      (format t "~% ~a~% ~40Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              thing p p ip n n in (gain-gain gain)))
    gain))

;;;__________________________________
;;; compute-conjunction-gain

(defun compute-conjunction-gain (conjunction variables pos-tuples neg-tuples source current-state-value)
  (declare (ignore variables))
  (select-node conjunction)
  (let ((node (first conjunction)) gain)
    (if (or (rest conjunction)
            (node-second-order? node))
      (let ((*compile-allowing-deletions* nil)
            function consequent vars types )
        (setq consequent (node-real-consequent node)
              vars (node-vars consequent)
              types (r-type (node-r-struct consequent))
              function (or (gethash conjunction *conjunction-function-hash*)
                           (setf (gethash conjunction *conjunction-function-hash*) (convert-tree-to-prolog-function conjunction vars)))
              gain (info-gain-prove conjunction function current-state-value pos-tuples neg-tuples vars types)))
      (setq gain (info-gain (node-r-struct node) (node-vars node) nil pos-tuples neg-tuples current-state-value)))
    (when (and *display-learning?* (eq source :ebl))
      (display-conjunction-gain *EBL-WINDOW* conjunction gain))
    (deselect-node conjunction t)
    gain))

;;;___________________________________________
;;;  new-vars-and-types

(defun new-vars-and-types (original-vars extended-vars extended-types)
  (do ((o-vars original-vars (rest o-vars))
       (e-vars extended-vars (rest e-vars))
       (e-types extended-types (rest e-types)))
      ((null o-vars) (values e-vars e-types))))

;;;___________________________________________
;;;  op-literal

(defun op-literal (prolog-literal
                   variables
                   variable-types
                   pos-tuples            
                   neg-tuples
                   current-state-value   ;; (current-metric (length pos) (length neg))
                   source                ;; :ebl or :intensional
                   use-hash-tables
                   winners)
  (if *fast-operationalization*
    (fast-op-literal prolog-literal variables variable-types pos-tuples neg-tuples current-state-value source use-hash-tables winners)
    (when (and pos-tuples prolog-literal)
      (let (graph base winner literal extended-vars extended-variable-types new-pos-tuples new-neg-tuples gain)
        (cond ((eq source :ebl)
               (setf graph *ebl-graph*
                     base (graph-base *ebl-graph*))
               (undelete-node base t)
               (deselect-node base t)
               (select-node base)
               (when *display-learning?*
                 (re-init-ebl-window *EBL-WINDOW* base pos-tuples neg-tuples)))
              (t
               (setf graph (create-graph :permanent? t)
                     base (connect-literal graph (graph-root graph) prolog-literal nil source :never 0))
               (set-graph-base graph base)
               (select-node base)))
        (case *refinement*
          (:leaves
           (reset-last-new-var-id)
           (let ((nodes))
             (multiple-value-setq
               (nodes extended-vars extended-variable-types new-pos-tuples new-neg-tuples gain)
               (op-node base variables variable-types pos-tuples neg-tuples source current-state-value graph))
             (setf nodes (reduce-nodes nodes)
                   literal (cond ((node-p nodes) (convert-nodes-to-literals nodes graph))
                                 ((conjunction-p nodes) (convert-nodes-to-literals nodes graph))
                                 ((disjunction-p nodes) (break "The leaf operationalize is not supposed to return a disjunction"))))
             (insert-tuples literal pos-tuples neg-tuples variables)))
          (:frontier
           (multiple-value-setq
             (literal extended-vars extended-variable-types new-pos-tuples new-neg-tuples gain)
             (op-frontier graph variables variable-types pos-tuples neg-tuples source current-state-value))))
        (cond (literal
               (multiple-value-bind (new-vars new-variable-types) (new-vars-and-types variables extended-vars extended-variable-types)
                 (multiple-value-setq (winners winner)
                   (update-winner winners *literal-better-function* nil
                                  gain ;; (compute-gain literal current-state-value pos-tuples new-pos-tuples new-neg-tuples neg-tuples)
                                  literal source
                                  :vars new-vars
                                  :types new-variable-types
                                  :pos new-pos-tuples
                                  :neg new-neg-tuples
                                  :negated? nil))
                 (simplify-operational-clause winners winner pos-tuples neg-tuples variables current-state-value use-hash-tables)
                 (mapc #'(lambda (winner) (if (<= (winner-gain winner) 0) (remove-winner winner winners))) (winners-new-winners winners))
                 (setf (winners-new-winners winners) nil)
                 winners))
              (t :fail))))))

;;;__________________________________
;;;  unique-r-name

(defun unique-r-name (&optional (old-name nil))
  (let* ((base (cond ((null old-name) 'new_relation)
                     ((numberp old-name) (intern (format nil "~a~a" 'new_relation old-name)))
                     (t old-name)))
         (new-name base))
    (do ((count 1 (incf count)))
        ((null (get-r-struct new-name)))
      (setf new-name (intern (format nil "~a_~a" base count))))
    new-name))

;;;_______________________________________
;;; fixup-learning-window

(defun fixup-learning-window (window &optional node)
  (when (window-open? window)
    (without-interrupts
     (let ((view (graph-view window)))
       (unless node (setf node (root view)))
       (display-tree-cells view node)
       (position-cells view)
       (reset-graph-size view 80 0)
       (grow-window-if-needed window)
       (re-position-graph view :centered t)
       (reset-scroll-bars (graph-scroller window))
       (invalidate-view window t)))))

;;;_______________________________________
;;; get-antecedents

(defun get-antecedents (node source graph &optional (display t) (state :unoperationalized))
  (case (node-kind node)
    ((:or :and :not) (node-antecedents node))
    ((:intensional)
     (let* ((r-struct (node-r-struct node))
            (parameters (node-vars node))
            (antecedents (node-antecedents node))
            (clauses (all-antecedents r-struct parameters t)))
       (cond (antecedents (update-vars antecedents clauses source))
             (t (setf antecedents (connect-clauses graph node clauses state :never 0)
                      (node-recursive? node) nil
                      (node-antecedents node) antecedents)
                (when (and display (eq source :ebl) *display-learning?*)
                  (fixup-learning-window *EBL-WINDOW* node))))
       antecedents))
    (otherwise nil)))

;;;_______________________________________
;;; update-vars

(defun update-vars (nodes literals source)
  (cond ((consp nodes)
         (update-vars (first nodes) (first literals) source)
         (update-vars (rest nodes) (rest literals) source))
        ((node-p nodes)
         (case (node-kind nodes)
           ((:or :and :not) (update-vars (first (node-antecedents nodes)) (rest literals) source))
           (t (setf (node-vars nodes) (rest literals))
              (when (and *display-learning?* (eq source :ebl))
                (when (window-open? *EBL-WINDOW*)
                  (without-interrupts 
                   (let* ((view (graph-view  *EBL-WINDOW*))
                          (cell (node-cell view nodes)))
                     (with-focused-view view
                       (setf (cell-text cell) (node-string nodes))
                       (draw-cell cell)))))))))))

;;;__________________________________
;;; graphically-operationalize-node

(defun graphically-operationalize-node (node source)
  (setf (node-state node) source)
  (when (node-intensional? node)
    (deselect-node node))
  (when (and *display-learning?* (eq source :ebl))
    (when (window-open? *EBL-WINDOW*)
      (without-interrupts 
       (let ((cell (node-cell (graph-view *EBL-WINDOW*) node)))
         (when cell
           (setf (cell-text cell) (node-string node))
           (focus-and-draw-cell cell)))))))

;;;__________________________________
;;; display-conjunction-gain

(defun display-conjunction-gain (window conjunction gain)
  (without-interrupts 
   (when (window-open? window)
     (update-external-text (node-cell (graph-view window) (first (last conjunction)))
                           (format nil "~A+ ~A- [~4F] "
                                   (gain-pp gain) (gain-nn gain) (gain-gain gain))))))

;;;__________________________________
;;; min-new-id

(defun min-new-id (thing &optional (min-id 0))
  (cond ((pcvar-p thing) (min min-id (pcvar-id thing)))
        ((node-p thing) (min-new-id (node-vars thing) min-id))
        ((consp thing) (min (min-new-id (first thing) min-id) (min-new-id (rest thing) min-id)))
        (t min-id)))

;;;_______________________________________
;;; bind-parameters

(defun bind-parameters (parameters parameter-types variables 
                                   &optional (existing-alist nil) (next-id (length variables)))
  (let* ((new nil)
         (new-variables nil)
         (new-types nil)
         (alist nil)
         (new-parameters (mapcar 
                          #'(lambda (parameter type)
                              (cond
                               ((not (pcvar-p parameter)) parameter)
                               ((member parameter variables :test #'equalp) parameter)
                               ((rest (assoc parameter alist :test #'equalp)))
                               ((rest (assoc parameter existing-alist :test #'equalp)))
                               (t (setf new (make-pcvar :id next-id))
                                  (incf next-id)
                                  (push (cons parameter new) alist)
                                  (push new new-variables)
                                  (push type new-types)
                                  new)))
                          parameters parameter-types)))
    (values new-parameters (nreverse new-variables) (nreverse new-types) alist)))

;;;_______________________________________
;;; bind-is-parameters

(defun bind-is-parameters (vars.expression variables &optional (existing-alist nil) (next-id (length variables)))
  (let ((vars (first vars.expression))
        (expression (second vars.expression)))
    (unless (consp vars) (setf vars (list vars)))
    (let* ((tranfered-expression (transfer-parameter-bindings expression existing-alist))
           (tranfered-vars (transfer-parameter-bindings vars existing-alist))
           (new-vars (delete-duplicates (remove-if #'(lambda (v) (member v variables)) tranfered-vars)))
           (new-new-vars (make-old-vars (length new-vars) next-id))
           (new-types (make-list (length new-vars) :initial-element :anything))
           (alist (mapcar #'cons new-vars new-new-vars))
           (bound-vars (transfer-parameter-bindings tranfered-vars alist))
           (new-parameters (list (if (rest bound-vars) bound-vars (first bound-vars)) tranfered-expression)))
      (values new-parameters new-new-vars new-types alist))))

;;;_______________________________________
;;; transfer-parameter-bindings

(defun transfer-parameter-bindings (parameters alist)
  (cond ((consp parameters)
         (mapcar 
          #'(lambda (parameter)
              (cond ((consp parameter) (mapcar #'(lambda (e) (transfer-parameter-bindings e alist)) parameter))
                    ((rest (assoc parameter alist :test #'equalp)))
                    (t parameter)))
          parameters))
        ((rest (assoc parameters alist :test #'equalp)))
        (t parameters)))


;;;========================================================================================
;;; LEAF OPERATIONALIZER

(defvar *op-negated* :leaf)    ;;  :leaf :frontier :self

;;;_______________________________________
;;; op-node

(defun op-node (node variables variable-types
                     pos neg source current-state-value derivation
                     &optional (unbound-bound-alist nil))
  (unless (> (node-depth node) *max-op-depth*)
    (case (node-kind node)
      (:extensional (op-extensional-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      (:builtin (op-builtin-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      (:is (op-is-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      (:= (op-=-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      (:intensional (op-intensional-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      ((:and :or) (op-and-or-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      (:not (op-not-node node variables variable-types pos neg source current-state-value derivation unbound-bound-alist))
      )))

;;;_______________________________________
;;; op-extensional-node

(defun op-extensional-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist &optional (negated? nil))
  (declare (ignore derivation))
  (let* ((r-struct (node-r-struct node))
         (parameters (node-vars node))
         (parameter-types (r-type r-struct)))
    (multiple-value-bind
      (new-parameters new-variables new-variable-types new-alist)
      (bind-parameters parameters parameter-types variables unbound-bound-alist)
      (let* ((new-pos (extend-tuples-extensional r-struct pos new-parameters new-variables))
             (new-neg (extend-tuples-extensional r-struct neg new-parameters new-variables))
             (gain (if negated?
                     (compute-negated-gain node current-state-value pos new-pos new-neg neg)
                     (compute-gain node current-state-value pos new-pos new-neg neg))))
        (setf (node-vars node) new-parameters)
        (graphically-operationalize-node node source)
        (values (list (list node))
                (append variables new-variables)
                (append variable-types new-variable-types)
                new-pos new-neg gain
                (append unbound-bound-alist new-alist))))))

;;;_______________________________________
;;; op-builtin-node

(defun op-builtin-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist &optional (negated? nil))
  (declare (ignore derivation))
  (let* ((r-struct (node-r-struct node))
         (parameters (node-vars node))
         (new-parameters (transfer-parameter-bindings parameters unbound-bound-alist))
         (new-pos (extend-tuples-builtin r-struct pos new-parameters))
         (new-neg (extend-tuples-builtin r-struct neg new-parameters))
         (gain (if negated?
                 (compute-negated-gain node current-state-value pos new-pos new-neg neg)
                 (compute-gain node current-state-value pos new-pos new-neg neg))))
    (setf (node-vars node) new-parameters)
    (graphically-operationalize-node node source)
    (values (list (list node))
            variables variable-types
            new-pos new-neg gain
            unbound-bound-alist)))

;;;_______________________________________
;;; op-is-node

(defun op-is-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist &optional (negated? nil))
  (declare (ignore derivation))
  (multiple-value-bind
    (new-parameters new-variables new-variable-types new-alist)
    (bind-is-parameters (node-vars node) variables unbound-bound-alist)
    (let* ((new-pos (extend-tuples-is pos variables new-parameters new-variables))
           (new-neg (extend-tuples-is neg variables new-parameters new-variables))
           (gain (if negated?
                   (compute-negated-gain node current-state-value pos new-pos new-neg neg)
                   (compute-gain node current-state-value pos new-pos new-neg neg))))
      (setf (node-vars node) new-parameters)
      (graphically-operationalize-node node source)
      (values (list (list node))
              (append variables new-variables)
              (append variable-types new-variable-types)
              new-pos new-neg gain
              (append unbound-bound-alist new-alist)))))

;;;_______________________________________
;;; op-=-node

(defun op-=-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist &optional (negated? nil))
  (declare (ignore derivation))
  (multiple-value-bind
    (new-parameters new-variables new-variable-types new-alist)
    (bind-is-parameters (node-vars node) variables unbound-bound-alist)
    (let* ((new-pos (extend-tuples-= pos variables new-parameters new-variables))
           (new-neg (extend-tuples-= neg variables new-parameters new-variables))
           (gain (if negated?
                   (compute-negated-gain node current-state-value pos new-pos new-neg neg)
                   (compute-gain node current-state-value pos new-pos new-neg neg))))
      (setf (node-vars node) new-parameters)
      (graphically-operationalize-node node source)
      (values (list (list node))
              (append variables new-variables)
              (append variable-types new-variable-types)
              new-pos new-neg gain
              (append unbound-bound-alist new-alist)))))

;;;_______________________________________
;;; op-intensional-node

(defun op-intensional-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (setf (node-vars node) (transfer-parameter-bindings (node-vars node) unbound-bound-alist))
  (let ((antecedents (get-antecedents node source derivation)))
    (graphically-operationalize-node node source)
    (multiple-value-bind
      (definition new-variables new-variable-types new-pos new-neg new-gain new-alist)
      (op-disjunction antecedents variables variable-types
                      pos neg source current-state-value derivation
                      unbound-bound-alist)
      (setf (node-vars node) (transfer-parameter-bindings (node-vars node) new-alist))
      (graphically-operationalize-node node source)
      
      (values definition new-variables new-variable-types
              new-pos new-neg new-gain new-alist))))

;;;_______________________________________
;;; op-and-or-node

(defun op-and-or-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (graphically-operationalize-node node source)
  (op-disjunction (node-antecedents node) variables variable-types
                  pos neg source current-state-value derivation
                  unbound-bound-alist))

;;;_______________________________________
;;; op-not-node

(defun op-not-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (let ((antecedents (node-antecedents node))
        definition function new-pos new-neg)
    (graphically-operationalize-node node source)
    (case *op-negated*
      (:leaf
       (multiple-value-bind (not-definition not-variables not-variable-types not-neg not-pos not-gain not-alist)
                            (op-disjunction antecedents variables variable-types neg pos source current-state-value derivation unbound-bound-alist)
         not-variables not-variable-types not-gain not-alist
         (setf definition (list (list (get-node derivation :r-struct (get-r-struct 'not) :kind :not :antecedents not-definition :state source)))
               new-pos (return-originals-not-extended pos not-pos)
               new-neg (return-originals-not-extended neg not-neg))))
      (:frontier
       (let* ((graph (node-graph node))
              (selected-nodes (all-images #'(lambda (n) (when (node-selected? n) n)) (graph-used-nodes graph)))
              frontier-graph base coverage)
         (dolist (n selected-nodes) (setf (node-selected? n) nil))
         (select-node antecedents)
         (setq *initially-bound-frontier-vars* variables)
         (return-best-frontier graph variables variable-types pos neg source current-state-value)
         (clear-clause-and-literal-deletions)
         (setq frontier-graph (extract-frontier graph nil)
               base (graph-base frontier-graph))
         (dolist (n selected-nodes) (setf (node-selected? n) t))
         (insert-node-tuples base pos neg variables variable-types source nil t t)
         (setf coverage (find source (node-coverage base) :key #'coverage-from)
               definition (get-node derivation :r-struct (get-r-struct 'not) :kind :not :state source)
               (node-antecedents definition) (list (list (duplicate-node base definition graph)))
               definition (list (list definition))
               new-pos (coverage-output-pos coverage)
               new-neg (coverage-output-neg coverage))))
      (:self
       (select-node definition)
       (setf function (convert-tree-to-prolog-function definition variables)
             new-pos (filter-proved-tuples function pos)
             new-neg (filter-proved-tuples function neg))
       (deselect-node definition t)))
    (values definition variables variable-types new-pos new-neg (compute-gain node current-state-value pos new-pos new-neg neg) unbound-bound-alist)))

;;;__________________________________
;;; op-conjunction

(defun op-conjunction (conjunction variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (let ((conjunction-definition nil)
        (new-pos pos)
        (new-neg neg)
        (extended-variables variables)
        (extended-types variable-types)
        (extended-alist unbound-bound-alist)
        node-definition
        gain)
    (do ((nodes conjunction (rest nodes)))
        ((null nodes))
      (multiple-value-setq
        (node-definition extended-variables extended-types new-pos new-neg gain extended-alist)
        (op-node (first nodes) extended-variables extended-types new-pos new-neg source current-state-value derivation extended-alist))
      (setf conjunction-definition (nconc conjunction-definition (first node-definition))))
    (values (list conjunction-definition)
            extended-variables extended-types
            new-pos new-neg gain
            extended-alist)))

;;;__________________________________
;;; op-disjunction

(defun op-disjunction (disjunction variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (let ((best-disjunct (best-disjunct disjunction variables variable-types pos neg source current-state-value derivation unbound-bound-alist)))
    (when best-disjunct
      (op-conjunction best-disjunct variables variable-types pos neg source current-state-value derivation unbound-bound-alist))))

;;;__________________________________
;;; best-disjunct

(defun best-disjunct (disjunction variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (declare (ignore variable-types derivation unbound-bound-alist))
  (let ((gain nil)
        (best-disjunct (first disjunction))
        (best-gain nil))
    (when (rest disjunction)
      (dolist (disjunct disjunction)
        (select-node disjunct)
        (setf gain (compute-conjunction-gain disjunct variables pos neg source current-state-value))
        (when (or (null best-gain)
                  (> (gain-gain gain) (gain-gain best-gain)))
          (setf best-gain gain
                best-disjunct disjunct))
        (deselect-node disjunct t)))
    (select-node best-disjunct)
    best-disjunct))




#|
;;;===================================================================================
;;;===================================================================================
;;; Retain this junk for a while


(:all-defs
 (setf definition (all-defs-node node variables variable-types source derivation unbound-bound-alist))
 (select-node definition)
 (setf function (convert-tree-to-prolog-function definition variables)
       new-pos (filter-unproved-tuples function pos)
       new-neg (filter-unproved-tuples function neg))
 (deselect-node definition t))

;;;===================================================================================
;;; ALL EXTENSIONAL DEFINITIONS
;;;
;;; Each of the functions all-def-xxx return all disjuctive extensional definitions of
;;; collection of nodes input to it regardless of gain.

;;;__________________________________
;;; all-combinations

(defun all-combinations (list-a list-b)
  (cond ((and list-a list-b)
         (let ((combination-list nil))
           (dolist (element-a list-a)
             (dolist (element-b list-b)
               (setf combination-list (nconc combination-list (list (append element-a element-b))))))
           combination-list))
        (list-a list-a)
        (list-b list-b)))

;;;__________________________________
;;; all-defs-node

(defun all-defs-node (node variables variable-types source graph unbound-bound-alist)
  (let* ((r-struct (node-r-struct node))
         (parameters (node-vars node))
         (parameter-types (when r-struct (r-type r-struct))))

    (unless (> (node-depth node) *max-op-depth*)

      (case (node-kind node)
        
        (:not
         (graphically-operationalize-node node source)
         (values (list (mapcar #'(lambda (disjunct)
                                   (get-node graph :kind :not :state source :antecedents (list disjunct)))
                               (all-defs-disjunction (get-antecedents node source graph)
                                                     variables variable-types source graph unbound-bound-alist)))
                 variables
                 variable-types 
                 unbound-bound-alist
                 ))

        ((:or :and)
         (graphically-operationalize-node node source)
         (values (all-defs-disjunction (get-antecedents node source graph)
                                       variables variable-types source graph unbound-bound-alist)
                 variables
                 variable-types 
                 unbound-bound-alist))
        
        ((:is :=)
         (multiple-value-bind
           (new-parameters new-variables new-variable-types new-alist)
           (bind-is-parameters (node-vars node) variables unbound-bound-alist)
           (setf (node-vars node) new-parameters)
           (graphically-operationalize-node node source)
           (values (list (list node))
                   (append variables new-variables)
                   (append variable-types new-variable-types)
                   (append unbound-bound-alist new-alist)
                  )))
        
        (:intensional
         (multiple-value-bind
           (new-parameters new-variables new-variable-types new-alist)
           (bind-parameters parameters parameter-types variables unbound-bound-alist)
           (setf (node-vars node) new-parameters)
           (graphically-operationalize-node node source)
           (let ((extended-variables (append variables new-variables))
                 (extended-variable-types (append variable-types new-variable-types))
                 (extended-unbound-bound-alist (append unbound-bound-alist new-alist)))
             (values (all-defs-disjunction (get-antecedents node source graph)
                                           extended-variables extended-variable-types
                                           source graph extended-unbound-bound-alist)
                     extended-variables
                     extended-variable-types
                     extended-unbound-bound-alist
                    ))))
        
        (:extensional
         (multiple-value-bind
           (new-parameters new-variables new-variable-types new-alist)
           (bind-parameters parameters parameter-types variables unbound-bound-alist)
           (setf (node-vars node) new-parameters)
           (graphically-operationalize-node node source)
           (values (list (list node))
                   (append variables new-variables)
                   (append variable-types new-variable-types)
                   (append unbound-bound-alist new-alist)
                  )))
        
        (:builtin
         (setf (node-vars node) (transfer-parameter-bindings (node-vars node) unbound-bound-alist))
         (graphically-operationalize-node node source)
         (values (list (list node))
                 variables
                 variable-types
                 unbound-bound-alist
                ))
        ))))

;;;__________________________________
;;; all-defs-conjunction

(defun all-defs-conjunction (conjunction variables variable-types source graph unbound-bound-alist)
  (when conjunction
    (multiple-value-bind (node-def new-variables new-variable-types new-alist)
                         (all-defs-node (first conjunction) variables variable-types source graph unbound-bound-alist)
      (all-combinations node-def (all-defs-conjunction (rest conjunction) new-variables new-variable-types source graph new-alist)))))

;;;__________________________________
;;; all-defs-disjunction

(defun all-defs-disjunction (disjunction variables variable-types source graph unbound-bound-alist)
  (mapcan #'(lambda (conjunction) (all-defs-conjunction conjunction variables variable-types source graph unbound-bound-alist)) disjunction))



(defun compute-conjunction-gain (conjunction variables pos neg source current-state-value)
  (select-node conjunction)
  (let* ((function (convert-tree-to-prolog-function conjunction variables))
         (p (count-prove function variables pos))
         (n (count-prove function variables neg))
         (gain (gain-metric current-state-value p p n n)))
    (when (and *display-learning?* (eq source :ebl))
      (display-conjunction-gain *EBL-WINDOW* conjunction gain))
    (when (and *trace-learning?* (member :o *focl-trace-level*))
      (format t "~% ~a~% ~40Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              conjunction
              p p (length pos)
              n n (length neg) (gain-gain gain)))
    (deselect-node conjunction t)
    gain))


(defun op-not-node (node variables variable-types pos neg source current-state-value derivation unbound-bound-alist)
  (graphically-operationalize-node node source)
  (let* ((antecedents (get-antecedents node source derivation))
         (negated-node (first (first antecedents)))
         not-pos not-neg new-pos new-neg
         function not-definition not-variables not-variable-types not-gain)
    (cond ((or (rest antecedents)
               (rest (first antecedents))
               (eq (node-kind negated-node) :intensional)
               (eq (node-kind negated-node) :or)
               (eq (node-kind negated-node) :and))
           (setf not-definition (all-defs-node node variables variable-types source derivation unbound-bound-alist))
           (select-node not-definition)
           (setf function (convert-tree-to-prolog-function not-definition variables)
                 new-pos (filter-unproved-tuples function pos)
                 new-neg (filter-unproved-tuples function neg))
           (deselect-node not-definition t))
          (t
           (case (node-kind negated-node)
             (:extensional
              (multiple-value-setq
                (not-definition not-variables not-variable-types not-neg not-pos not-gain)
                (op-extensional-node  negated-node variables variable-types
                                      neg pos source current-state-value derivation
                                      unbound-bound-alist t))
              (setf not-definition (list (list (get-node derivation :r-struct (get-r-struct 'not) :kind :not :antecedents not-definition :state source)))
                    new-pos (return-originals-not-extended pos not-pos)
                    new-neg (return-originals-not-extended neg not-neg)))
          
             (:builtin
              (multiple-value-setq
                (not-definition not-variables not-variable-types not-neg not-pos not-gain)
                (op-builtin-node negated-node variables variable-types neg pos source current-state-value derivation unbound-bound-alist t))
              (setf not-definition (list (list (get-node derivation :r-struct (get-r-struct 'not) :kind :not :antecedents not-definition :state source)))
                    new-pos (return-originals-not-extended pos not-pos)
                    new-neg (return-originals-not-extended neg not-neg)))
          
             (:is
              (multiple-value-setq
                (not-definition not-variables not-variable-types not-neg not-pos not-gain)
                (op-is-node negated-node variables variable-types neg pos source current-state-value derivation unbound-bound-alist t))
              (setf not-definition (list (list (get-node derivation :r-struct (get-r-struct 'not) :kind :not :antecedents not-definition :state source)))
                    new-pos (return-originals-not-extended pos not-pos)
                    new-neg (return-originals-not-extended neg not-neg)))

             (:not
              (multiple-value-setq
                (not-definition not-variables not-variable-types not-pos not-neg not-gain)
                (op-disjunction (get-antecedents negated-node source derivation) variables variable-types
                                pos neg source current-state-value derivation unbound-bound-alist))
              (setf new-pos (return-originals-extended pos not-pos)
                    new-neg (return-originals-extended neg not-neg))) )))

    not-variables not-variable-types not-gain
    
    (values not-definition
            variables variable-types
            new-pos new-neg
            (compute-gain node current-state-value pos new-pos new-neg neg)
            unbound-bound-alist)))
|#