;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)


#|

;; Standard Set of Frontier Operators

(setf *ACTIVE-FRONTIER-OPERATORS*
      '(FRONTIER-OP_replace_node_by_one_antecedent_disjunct
        FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct
        FRONTIER-OP_delete_one_conjunct
        FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct
        FRONTIER-OP_delete_one_disjunct
        FRONTIER-OP_add_one_existing_disjunct))

;; Multiple-Step-Look-Ahead Frontier Operator

(setf *ACTIVE-FRONTIER-OPERATORS* '(FRONTIER-OP_multiple_step_look_ahead))

;; Pseudo-Leaf Operationalization Frontier Operator

(setf *ACTIVE-FRONTIER-OPERATORS* '(FRONTIER-OP_leaf_operationalization))

|#


;;;__________________________________
;;;  REAL-OPERATOR

(defun real-operator (operator modification)
  (cond ((eql operator 'FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct)
         (first modification))
        (t operator)))

;;;__________________________________
;;; OPERATOR-PREFERED?

(defun operator-prefered? (method)
  (or (and *prefer-children* 
           (or (eql method 'FRONTIER-OP_replace_node_by_one_antecedent_disjunct)
               (eql method 'FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct)
               (eql method 'FAST-FRONTIER-OP_delete_one_literal_from_rule)
               (eql method 'FAST-FRONTIER-OP_replace_rule_by_one_clause)
               (eql method 'FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause)))
      (and *prefer-deletions* 
           (or (eql method 'FRONTIER-OP_delete_one_conjunct)
               (eql method 'FRONTIER-OP_delete_one_disjunct)
               (eql method 'FRONTIER-OP_replace_node_by_one_antecedent_disjunct)
               (eql method 'FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct)
               (eql method 'FAST-FRONTIER-OP_delete_one_literal_from_clause)
               (eql method 'FAST-FRONTIER-OP_delete_one_clause)
               (eql method 'FAST-FRONTIER-OP_delete_one_literal_from_rule)
               (eql method 'FAST-FRONTIER-OP_replace_rule_by_one_clause)
               (eql method 'FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause)))))

;;;_______________________________________
;;; NEXT-ELEMENT

(defun next-element (element list)
  (first (rest (member element list :test #'equal))))

;;;_______________________________________
;;; SUBSEQUENT-ELEMENTS

(defun subsequent-elements (element list)
  (rest (member element list :test #'equal)))

;;;_______________________________________
;;; GET-ANTECEDENTS-QUICK

(defun get-antecedents-quick (node graph)
  (case (node-kind node)
    ((:or :and :not) (node-antecedents node))
    ((:intensional)
     (let ((antecedents (node-antecedents node)))
       (unless antecedents 
         (setf antecedents (connect-clauses graph node (all-antecedents (node-r-struct node) (node-vars node) t) (node-state node) :never 0)
               (node-antecedents node) antecedents
               (node-recursive? node) nil)
         (when *display-learning?*
           (dolist (cell (node-cells node))
             (fixup-learning-window (graph-window (cell-view cell))))))
       antecedents))
    (otherwise nil)))

;;;============================================================================
;;; FRONTIER-OPs

;;;____________________________________________________________________________
;;; FRONTIER-OP_replace_node_by_one_antecedent_disjunct

(def-frontier-operator FRONTIER-OP_replace_node_by_one_antecedent_disjunct (graph node modification action)
  (case action
    (:apply
     (deselect-node node)
     (select-node modification))

    (:next
     (let ((antecedents (get-antecedents-quick node graph)))
       (cond (modification
              (deselect-node modification)
              (select-node (setf modification (next-element modification antecedents)))
              (unless modification
                (select-node node)))
             ((rest (rest antecedents))
              (deselect-node node)
              (select-node (setf modification (first antecedents)))))))
    )
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct

(def-frontier-operator FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct (graph node modification action)
  (case action
    (:apply
     (deselect-node node)
     (select-node (node-antecedents node))
     (deselect-node modification)
     (when *frontier-prevent-adding-deleted-disjuncts*
       (delete-node modification)))

    (:next
     (let ((antecedents (get-antecedents-quick node graph)))
       (cond (modification
              (select-node modification)
              (deselect-node (setf modification (next-element modification antecedents)))
              (unless modification
                (deselect-node antecedents)
                (select-node node)))
             ((rest antecedents)
              (deselect-node node)
              (select-node antecedents)
              (deselect-node (setf modification (first antecedents)))))))
    )
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_delete_one_conjunct

(def-frontier-operator FRONTIER-OP_delete_one_conjunct (graph node modification action)
  graph
  (case action
    (:apply
     (deselect-node modification t))
    
    (:next
     (unless *expanding-best-frontier*
       (cond (modification
              (select-node modification)
              (setf modification nil))
             (t
              (let ((base (graph-base graph)))
                (deselect-node node)
                (if (and (some #'next-selected-node-below (conjunction-containing-node node))
                         (or (not *frontier-check-for-unbound-variables*) 
                             (all-frontier-vars-bound base *initially-bound-frontier-vars*)))    ;; perhaps this should be passed in                                                                  
                  (setf modification node)
                  (select-node node)))))))
    )
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_delete_one_disjunct

(def-frontier-operator FRONTIER-OP_delete_one_disjunct (graph node modification action)
  (case action
    (:apply
     (deselect-node modification t)
     (when *frontier-prevent-adding-deleted-disjuncts*
       (delete-node modification)))
    
    (:next
     (unless *expanding-best-frontier*
       (let ((disjunct (conjunction-containing-node node))
             (base (graph-base graph)))
         (cond (modification
                (undelete-node modification)
                (setf modification nil))
               ((eq node (find-if #'node-selected? disjunct))
                (delete-node (setf modification disjunct))
                (multiple-value-bind (bound vars selected) (all-frontier-vars-bound base *initially-bound-frontier-vars*)    ;; perhaps this should be passed in  
                  (declare (ignore vars))
                  (unless (and bound selected)
                    (undelete-node modification)
                    (setf modification nil))))))))
    )
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_add_one_existing_disjunct

(def-frontier-operator FRONTIER-OP_add_one_existing_disjunct (graph node modification action)
  graph
  (labels ((find-first-selected-node (node-or-collection)
             (cond ((node-p node-or-collection) (when (node-selected? node-or-collection) node-or-collection))
                   ((consp node-or-collection) (or (find-first-selected-node (first node-or-collection))
                                                    (find-first-selected-node (rest node-or-collection)))))))
    
    (case action
      (:apply
       (select-node modification))
      
      (:next
       (let ((sibling-disjuncts (node-antecedents (node-consequent node))))
         (cond (modification
                (deselect-node modification)
                (select-node (setf modification (find-if-not #'(lambda (disjunct) (or (some #'next-selected-node-below disjunct)
                                                                                      (every #'node-deleted? disjunct)))
                                                             (subsequent-elements modification sibling-disjuncts)))))
               ((eq node (find-first-selected-node sibling-disjuncts))
                (select-node (setf modification (find-if-not #'(lambda (disjunct) (or (some #'next-selected-node-below disjunct)
                                                                                      (every #'node-deleted? disjunct)))
                                                             sibling-disjuncts)))))))
      ))
  modification)

;;;____________________________________________________________________________
;;; FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct

(defparameter *FRONTIER-OPERATORS_expand_conjunctively_defined_single_node_disjunct*
  '(FRONTIER-OP_replace_node_by_one_antecedent_disjunct
    FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct
    FRONTIER-OP_delete_one_conjunct))

(def-frontier-operator FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct (graph node modification action)
  (labels ((expand-conjunctively-defined-single-node-disjunct (node graph)
             (case (node-kind node)
               ((:intensional :not :and :or)
                (let ((antecedents (get-antecedents-quick node graph)))
                  (when (and antecedents (null (rest antecedents)))
                    (if (null (rest (first antecedents)))
                      (expand-conjunctively-defined-single-node-disjunct (first (first antecedents)) graph)
                      (first antecedents)))))
               (otherwise nil))))
    
    (let ((m-operator (first modification))
          (m-node (second modification))
          (m-modification (third modification))
          (conjunction nil))
      (case action
        (:apply
         (select-node (conjunction-containing-node m-node))
         (deselect-node node)
         (funcall (symbol-function m-operator) graph m-node m-modification action))
        
        (:next
         (cond (modification
                (setf modification
                      (cond
                       ((setf m-modification (funcall (symbol-function m-operator) graph m-node m-modification action))
                        (list m-operator m-node m-modification))
                       ((setf m-operator (next-element m-operator *FRONTIER-OPERATORS_expand_conjunctively_defined_single_node_disjunct*))
                        (FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct graph node (list m-operator m-node m-modification) action))
                       ((setf m-node (next-element m-node (conjunction-containing-node m-node)))
                        (setf m-operator (first *FRONTIER-OPERATORS_expand_conjunctively_defined_single_node_disjunct*))
                        (FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct graph node (list m-operator m-node m-modification) action))
                       (t
                        (deselect-node node t)
                        (select-node node)
                        (setf modification nil)))))
               ((setf conjunction (expand-conjunctively-defined-single-node-disjunct node graph))
                (select-node conjunction)
                (deselect-node node)
                (setf modification (list (first *FRONTIER-OPERATORS_expand_conjunctively_defined_single_node_disjunct*) (first conjunction) nil)
                      modification (FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct graph node modification action)))))))
    )
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_multiple_step_look_ahead
;;;
;;; [XXXX] this will generate numerous redudent frontiers, introduce an indexing scheme and store gain???

(defparameter *FRONTIER-OPERATORS_multiple_step_look_ahead*
  '(FRONTIER-OP_replace_node_by_one_antecedent_disjunct
    FRONTIER-OP_replace_node_by_all_but_one_antecedent_disjunct
    RONTIER-OP_delete_one_conjunct
    FRONTIER-OP_expand_conjunctively_defined_single_node_disjunct
    FRONTIER-OP_delete_one_disjunct
    FRONTIER-OP_add_one_existing_disjunct))


(def-frontier-operator FRONTIER-OP_multiple_step_look_ahead (graph node stack action)
  (labels ((next-modification-operator-node (graph operators stack)
             (when stack
               (let* ((triple (first stack))
                      (operator (first triple))
                      (node (second triple))
                      (modification (third triple)))
                 (cond
                  ((setq modification (funcall (symbol-function operator) graph node modification :next))
                   (cons (list operator node modification) (rest stack)))
                  ((setq operator (next-element operator operators))
                   (next-modification-operator-node graph operators (cons (list operator node modification) (rest stack))))
                  ((setq node (next-selected-node node :upper-bound node))
                   (next-modification-operator-node graph operators (cons (list (first operators) node modification) (rest stack))))
                  (t
                   (next-modification-operator-node graph operators (rest stack))))))))
    
    (when (and (numberp *frontier-look-ahead-level*) (> *frontier-look-ahead-level* 0))
      (case action
        (:apply
         (let ((triple (first (last stack))))
           (funcall (symbol-function (first triple)) graph (second triple) (third triple) action)))
        (:next
         (setf stack
               (if (= (length stack) *frontier-look-ahead-level*)
                 (next-modification-operator-node graph *FRONTIER-OPERATORS_multiple_step_look_ahead* stack)
                 (let ((temp-stack (list (list (first *FRONTIER-OPERATORS_multiple_step_look_ahead*)
                                               (next-selected-node node :include-self t :upper-bound node)
                                               nil))))
                   (if (setf temp-stack (next-modification-operator-node graph *FRONTIER-OPERATORS_multiple_step_look_ahead* temp-stack))
                     (nconc temp-stack stack)
                     (next-modification-operator-node graph *FRONTIER-OPERATORS_multiple_step_look_ahead* stack)))))))
      stack)))


;;;____________________________________________________________________________
;;; FRONTIER-OP_complete_operationalization
;;;
;;; [XXXX] this only returns one step in the complete operationalization

(def-frontier-operator FRONTIER-OP_complete_operationalization (graph node modification action)
  (cond ((and modification (eql action :next))
         (deselect-node node t)
         (select-node node)
         (setf modification nil))
        (t
         (do ((n (next-selected-node node :include-self t :upper-bound node) (next-selected-node n :upper-bound node)))
             ((null n))
           (when (or (and (node-intensional? n) (not (self-ancestor-p n)))
                     (node-not? n)
                     (node-and? n)
                     (node-or? n))
             (deselect-node n)
             (let ((antecedents (get-antecedents-quick n graph)))
               (select-node antecedents))))
         (setf modification :all-ops)))
  modification)


;;;____________________________________________________________________________
;;; FRONTIER-OP_leaf_operationalization
;;;
;;;  An attempt to duplicate the leaf operationalization process in
;;;  the frontier framework
;;;
;;;  Requires two hacks
;;;    - slight modification to   EVALUATION-OPERATOR>  
;;;    - The frontier framework applies operators to each selected node, while
;;;      the leaf operationalization operator should be applied to only the
;;;      first intensional node.  To achieve this the FRONTIER-OP_leaf_operationalization
;;;      operationalization determines if the input node is the first expandable node
;;;      in the frontier.
;;;
;;; [XXXX] this returns only one step in should return upto the next choice point!!!

(def-frontier-operator FRONTIER-OP_leaf_operationalization (graph node modification action)
  (labels ((first-selected-expandable-node (n)
             (cond ((node-p n)
                    (unless (node-deleted? n)
                      (if (and (node-selected? n)
                               (or (node-intensional? n)
                                   (node-not? n)
                                   (node-and? n)
                                   (node-or? n)))
                        n
                        (first-selected-expandable-node (node-antecedents n)))))
                   ((consp n)
                    (or (first-selected-expandable-node (first n))
                        (first-selected-expandable-node (rest n)))))))

  (case action
    (:apply
     (deselect-node node)
     (select-node modification) )

    (:next
     (cond (modification
            (deselect-node modification)
            (select-node (setf modification (next-element modification (get-antecedents-quick node graph))))
            (unless modification (select-node node)))
           ((eq node (first-selected-expandable-node (graph-base graph)))
            (deselect-node node)
            (select-node (setf modification (first (get-antecedents-quick node graph))))
            ))))
    )
  modification)

;;;____________________________________________________________________________
;;; FRONTIER-OP_best_first_leaf_operationalization

(def-frontier-operator FRONTIER-OP_best_first_leaf_operationalization (graph node modification action)
  (case action
    (:apply
     (deselect-node node)
     (select-node modification))

    (:next
     (let ((antecedents (get-antecedents-quick node graph)))
       (cond (modification
              (deselect-node modification)
              (select-node (setf modification (next-element modification antecedents)))
              (unless modification (select-node node)))
             (antecedents
              (deselect-node node)
              (select-node (setf modification (first antecedents)))))))
    )
  modification)


;;;=================================================================
;;; CODE TO DETERMINE IF CONJUNCT CAN BE DELETED
            
;;;____________________________________________________________________________
;;; all-node-vars-bound

(defun all-node-vars-bound (node bound-vars)
  (let ((mode (r-mode (node-r-struct node)))
        (all-bound t))
    (when mode
      (mapcar #'(lambda (v m) 
                  (when (and (eql m :+) (pcvar-p v))
                    (unless (member v bound-vars) 
                      (setq all-bound nil))))
              (node-vars node) mode))
    all-bound))

;;;____________________________________________________________________________
;;; all-frontier-vars-bound

(defun all-frontier-vars-bound (nodes bound-vars)
  (cond ((null nodes) (values t bound-vars nil))
        
        ((node-p nodes)
         (if (node-deleted? nodes)
           (values t bound-vars nil)
           (let ((new-vars (compute-node-new-vars-and-types nodes bound-vars))
                 (antecedents (node-antecedents nodes))
                 (extended-bound-vars nil)
                 (all-bound t)
                 (some-selected nil))
             (case (node-kind nodes)
               (:intensional
                (cond ((node-selected? nodes)
                       (setq all-bound (all-node-vars-bound nodes bound-vars)
                             extended-bound-vars (if new-vars (append bound-vars new-vars) bound-vars)
                             some-selected t))
                      (antecedents
                       (setq all-bound (let (bound vars selected)
                                         (every #'(lambda (conjunction) 
                                                    (multiple-value-setq (bound vars selected) (all-frontier-vars-bound conjunction bound-vars))
                                                    (when selected (setq some-selected t))
                                                    (or (null selected)
                                                        (and bound (every #'(lambda (v) (member v vars)) new-vars))))
                                                antecedents)))
                       (if (and some-selected all-bound)
                         (setq extended-bound-vars (if new-vars (append bound-vars new-vars) bound-vars))
                         (setq extended-bound-vars bound-vars)))))
               ((:not :or)
                (multiple-value-setq (all-bound extended-bound-vars some-selected) (all-frontier-vars-bound antecedents bound-vars))
                (setq extended-bound-vars bound-vars))
               (:and
                (multiple-value-setq (all-bound extended-bound-vars some-selected) 
                  (all-frontier-vars-bound (first antecedents) bound-vars)))
               (:cut
                (setq extended-bound-vars bound-vars))
               (otherwise
                (if (node-selected? nodes)
                  (setq some-selected t
                        all-bound (all-node-vars-bound nodes bound-vars)
                        extended-bound-vars (if (and all-bound new-vars) (append bound-vars new-vars) bound-vars))
                  (setq extended-bound-vars bound-vars))))
             (values all-bound extended-bound-vars some-selected))))
        
        ((conjunction-p nodes)
         (let (all-vars-bound all-bound selected (some-selected nil))
           (setq all-bound (every #'(lambda (node) 
                                      (multiple-value-setq (all-vars-bound bound-vars selected) (all-frontier-vars-bound node bound-vars))
                                      (when selected (setq some-selected t))
                                      all-vars-bound)
                                  nodes))
           (values all-bound bound-vars some-selected)))
        
        ((disjunction-p nodes)
         (let (all-vars-bound all-bound vars selected (some-selected nil))
           (setq all-bound (every #'(lambda (conjunction) 
                                      (multiple-value-setq (all-vars-bound vars selected) (all-frontier-vars-bound conjunction bound-vars))
                                      (when selected (setq some-selected t))
                                      all-vars-bound)
                                  nodes))
           (values all-bound bound-vars some-selected)))))