;;;; copyright (c) 1990, 1991 by the university of california, irvine. 
;;;; this program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  this program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the regents of the university of california.
;;;; this program was written by michael pazzani, cliff brunk, glenn silverstein
;;;; and kamal ali.  

(in-package :user)

;;; This is a hack for something that should be in utilities!!!!

(defun convert-prolog-to-literals (prolog source)
  (let* ((graph (make-graph))
         (literals (convert-nodes-to-literals (convert-prolog-to-nodes prolog source graph) nil nil nil)))
    (dispose-graph graph)
    literals))

(defun fast-op-literal (prolog-literal
                        variables
                        variable-types
                        pos-tuples            
                        neg-tuples
                        current-state-value   ;; (current-metric (length pos) (length neg))
                        source                ;; :ebl or :intensional
                        use-hash-tables
                        winners)
  
  (when (and pos-tuples prolog-literal)
    (let (op-prolog-literals p n winner literal extended-vars extended-variable-types new-pos-tuples new-neg-tuples gain)
      (case *refinement*
        (:leaves
         (reset-last-new-var-id)
         (multiple-value-setq (op-prolog-literals new-pos-tuples new-neg-tuples extended-vars extended-variable-types)
             (fast-op-l  prolog-literal pos-tuples neg-tuples variables variable-types nil current-state-value source winners)))
        (:frontier
         (multiple-value-setq (op-prolog-literals new-pos-tuples new-neg-tuples extended-vars extended-variable-types)
           (fast-op-frontier prolog-literal pos-tuples neg-tuples variables variable-types nil current-state-value source winners))
         ))
      (cond (op-prolog-literals
             (setq p (length new-pos-tuples)
                   n (length new-neg-tuples)
                   gain (gain-metric current-state-value p p n n)
                   literal (convert-prolog-to-literals op-prolog-literals source))
             (insert-tuples literal pos-tuples neg-tuples variables)
             (multiple-value-bind (new-vars new-variable-types) (new-vars-and-types variables extended-vars extended-variable-types)
               (multiple-value-setq (winners winner)
                 (update-winner winners *literal-better-function* nil
                                gain ;; (compute-gain literal current-state-value pos-tuples new-pos-tuples new-neg-tuples neg-tuples)
                                literal source
                                :vars new-vars
                                :types new-variable-types
                                :pos new-pos-tuples
                                :neg new-neg-tuples
                                :negated? nil))
               (simplify-operational-clause winners winner pos-tuples neg-tuples variables current-state-value use-hash-tables)
               (mapc #'(lambda (winner) (if (<= (winner-gain winner) 0) (remove-winner winner winners))) (winners-new-winners winners))
               (setf (winners-new-winners winners) nil)
               winners))
            (t :fail)))))

;;;===================================================================================
;;; FAST VERSION OF LEAF OPERATIONALIZATION
;;;
;;; 
;;;

(defun fast-op-l (literal pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners)
  (if (consp literal)
    (let* ((name (first literal))
           (variablization (rest literal))
           (r-struct (get-r-struct name)))
      (case (r-kind r-struct)
        (:intensional
         (fast-op-d (all-antecedents r-struct variablization t) pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
        (:not
         (fast-op-not-c variablization pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
        (:and
         (fast-op-c variablization pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
        (:or
         (fast-op-d (mapcar #'list variablization) pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
        (otherwise
         (setq variablization (direct-substitute variablization mapping))
         (multiple-value-bind (new-vars new-types) (compute-r-struct-new-vars-and-types r-struct variablization old-vars)
           (when new-vars
             (setq mapping (nconc (direct-mapping new-vars (make-old-vars (length new-vars) (length old-vars))) mapping)
                   new-vars (direct-substitute new-vars mapping)
                   variablization (direct-substitute variablization mapping)))
           (values (list (cons name variablization))
                   (generalized-extend-tuples r-struct pos-tuples variablization nil new-vars old-vars)
                   (generalized-extend-tuples r-struct neg-tuples variablization nil new-vars old-vars)
                   (if new-vars (append old-vars new-vars) old-vars)
                   (if new-vars (append old-types new-types) old-types)
                   mapping)))))
    (values literal pos-tuples neg-tuples old-vars old-types mapping)))

(defun fast-op-c (conjunction pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners)
  (let (op)
    (values (mapcan #'(lambda (literal)
                        (multiple-value-setq (op pos-tuples neg-tuples old-vars old-types mapping)
                          (fast-op-l literal pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
                        op)
                    conjunction) pos-tuples neg-tuples old-vars old-types mapping)))

(defun fast-op-d (disjunction pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners)
  (let (best-op best-gain best-pos-tuples best-neg-tuples best-old-vars best-old-types best-mapping
                op gain p n new-pos-tuples new-neg-tuples new-old-vars new-old-types new-mapping)
    (dolist (conjunction disjunction)
      (multiple-value-setq (op new-pos-tuples new-neg-tuples new-old-vars new-old-types new-mapping)
        (fast-op-c conjunction pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners))
      (setq p (length new-pos-tuples)
            n (length new-neg-tuples)
            gain (gain-metric current-state-value p p n n))
      (when (or (null best-gain)
                (> (gain-gain gain) (gain-gain best-gain)))
        (setq best-op op
              best-gain gain
              best-pos-tuples new-pos-tuples
              best-neg-tuples new-neg-tuples
              best-old-vars new-old-vars
              best-old-types new-old-types
              best-mapping new-mapping)))
    (values best-op best-pos-tuples best-neg-tuples best-old-vars best-old-types best-mapping)))


;;; We want to be able to do different things when operationalizing a negation
;;; The problems is that operationalization returns an over specialization  this is fine except in the
;;; case where this specialization is negated becoming overly general and requiring induction even when
;;; the initial theory is correct.  Really want to do frontier op and return a disjunction.

(defun fast-op-not-c (conjunction pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners)
  (let (definition new-pos new-neg)
    (case *op-negated*
      (:leaf
       (multiple-value-bind (op extended-neg extended-pos)
                            (fast-op-c conjunction neg-tuples pos-tuples old-vars old-types mapping current-state-value source winners)
         (setq definition (list (cons 'not op))
               new-pos (return-originals-not-extended pos-tuples extended-pos)
               new-neg (return-originals-not-extended neg-tuples extended-neg))))
      (:frontier
       (multiple-value-bind (op extended-pos extended-neg)
                            (fast-op-frontier conjunction pos-tuples neg-tuples old-vars old-types mapping current-state-value source winners)
         (setq definition (list (cons 'not op))
               new-pos (return-originals-not-extended pos-tuples extended-pos)
               new-neg (return-originals-not-extended neg-tuples extended-neg))))
      (:self
       (setq conjunction (direct-mapping conjunction mapping)
             definition (cons 'not conjunction))
       (if (or (rest conjunction)
               (null (r-prolog-function (get-r-struct (first (first conjunction))))))
         (let ((function (focl-compile-clause-function (cons (cons 'dummy old-vars) conjunction) (length old-vars))))
           (setq new-pos (filter-proved-tuples function pos-tuples)
                 new-neg (filter-proved-tuples function neg-tuples)))
         (let* ((literal (first conjunction))
                (r-struct (get-r-struct (first literal)))
                (function (r-prolog-function r-struct))
                (variablization (rest literal))
                (new-vars (compute-r-struct-new-vars-and-types r-struct variablization old-vars)))
           (setq new-pos (filter-proved-tuples-with-variablization variablization function pos-tuples new-vars)
                 new-neg (filter-proved-tuples-with-variablization variablization function neg-tuples new-vars)))))
      )
    (values definition new-pos new-neg old-vars old-types mapping)))

;;;===================================================================================
;;; FAST FRONTIER DAG

(defvar *expand-trivial-conjunctive-rules* nil)
(defvar *expand-all-conjunctive-rules* nil)
(defvar *expanding-best-frontier* nil)
(defvar *prolog-frontier* nil)

(defvar *used-dags* nil)
(defvar *free-dags* nil)

(defstruct dag
  (root nil)
  (cycle-node nil)
  (used-nodes nil)
  (free-nodes nil)
  (used-conjunctions nil)
  (free-conjunctions nil)
  (used-so-nodes nil)
  (free-so-nodes nil)
  )

(defstruct (dag-node (:print-function (lambda (dag-node stream depth)
                                        (declare (ignore depth))
                                        (format stream "{DAG-NODE ~A }" (dag-node-name dag-node)))))
  (name nil)
  (conjunctions nil)
  (number-of-conjunctions nil)
  (used-in nil))


(defstruct (dag-conjunction (:print-function (lambda (dag-conjunction stream depth)
                                               (declare (ignore depth))
                                               (format stream "{DAG-CONJUNCTION ~A }" (dag-conjunction-literals dag-conjunction)))))
  (used-in nil)
  (name nil)
  (number 0)
  (literals nil)
  (length 0))

(defstruct (dag-so-node (:print-function (lambda (dag-so-node stream depth)
                                           (declare (ignore depth))
                                           (format stream "{DAG-S0-NODE-~A ~A}" (dag-so-node-name dag-so-node) (dag-so-node-literals dag-so-node)))))
  
  (name nil)
  (literals nil)
  (used-in nil))

(defun get-dag ()
  (let ((dag nil))
    (if *free-dags*
      (setq dag (first *free-dags*)
            *free-dags* (rest *free-dags*))
      (setq dag (make-dag)))
    (setq *used-dags* (push dag *used-dags*))
    dag))

(defun dispose-dag (dag)
  (setq *used-dags* (delete dag *used-dags*)
        *free-dags* (push dag *free-dags*))
  (dolist (node (dag-used-nodes dag))
    (clear-dag-node node))
  (dolist (conjunction (dag-used-conjunctions dag))
    (clear-dag-conjunction conjunction))
  (dolist (so-node (dag-used-so-nodes dag))
    (clear-dag-so-node so-node))
  (setf (dag-free-nodes dag) (nconc (dag-free-nodes dag) (dag-used-nodes dag))
        (dag-used-nodes dag) nil
        (dag-free-conjunctions dag) (nconc (dag-free-conjunctions dag) (dag-used-conjunctions dag))
        (dag-used-conjunctions dag) nil
        (dag-free-so-nodes dag) (nconc (dag-free-so-nodes dag) (dag-used-so-nodes dag))
        (dag-used-so-nodes dag) nil)
  dag)

(defun dispose-node-and-below (dag node)
  (dolist (conjunction (dag-node-conjunctions node))
    (dispose-conjunction-and-below dag conjunction))
  (clear-dag-node node)
  (setf (dag-used-nodes dag) (delete node (dag-used-nodes dag))
        (dag-free-nodes dag) (push node (dag-free-nodes dag)))
  node)

(defun dispose-conjunction-and-below (dag conjunction)
  (dolist (node (dag-conjunction-literals conjunction))
    (cond ((dag-node-p node)
           (unless (rest (dag-node-used-in node))
             (dispose-node-and-below dag node)))
          ((dag-so-node-p node)
           (dispose-so-node-and-below dag node))))
  (clear-dag-conjunction conjunction)
  (setf (dag-used-conjunctions dag) (delete conjunction (dag-used-conjunctions dag))
        (dag-free-conjunctions dag) (push conjunction (dag-free-conjunctions dag)))
  conjunction)

(defun dispose-so-node-and-below (dag so-node)
  (dolist (node (dag-so-node-literals so-node))
    (cond ((dag-node-p node)
           (unless (rest (dag-node-used-in node))
             (dispose-node-and-below dag node)))
          ((dag-so-node-p node)
           (dispose-so-node-and-below dag node))))
  (clear-dag-so-node so-node)
  (setf (dag-used-so-nodes dag) (delete so-node (dag-used-so-nodes dag))
        (dag-free-so-nodes dag) (push so-node (dag-free-so-nodes dag)))
  so-node)

(defun clear-dag-node (node)
  (setf (dag-node-name node) nil
        (dag-node-conjunctions node) nil
        (dag-node-number-of-conjunctions node) nil
        (dag-node-used-in node) nil)
  node)

(defun clear-dag-conjunction (conjunction)
  (setf (dag-conjunction-used-in conjunction) nil
        (dag-conjunction-name conjunction) nil
        (dag-conjunction-number conjunction) 0
        (dag-conjunction-literals conjunction) nil
        (dag-conjunction-length conjunction) 0)
  conjunction)

(defun clear-dag-so-node (so-node)
  (setf (dag-so-node-name so-node) nil
        (dag-so-node-literals so-node) nil
        (dag-so-node-used-in so-node) nil)
  so-node)

(defun get-dag-node (dag)
  (let ((node nil)
        (free-nodes (dag-free-nodes dag)))
    (if free-nodes
      (setf node (first free-nodes)
            (dag-free-nodes dag) (rest free-nodes))
      (setq node (make-dag-node)))
    (setf (dag-used-nodes dag) (push node (dag-used-nodes dag)))
    node))

(defun get-dag-so-node (dag)
  (let ((so-node nil)
        (free-so-nodes (dag-free-so-nodes dag)))
    (if free-so-nodes
      (setf so-node (first free-so-nodes)
            (dag-free-so-nodes dag) (rest free-so-nodes))
      (setq so-node (make-dag-so-node)))
    (setf (dag-used-so-nodes dag) (push so-node (dag-used-so-nodes dag)))
    so-node))

(defun get-dag-conjunction (dag)
  (let ((conjunction nil)
        (free-conjunctions (dag-free-conjunctions dag)))
    (if free-conjunctions
      (setf conjunction (first free-conjunctions)
            (dag-free-conjunctions dag) (rest free-conjunctions))
      (setq conjunction (make-dag-conjunction)))
    (setf (dag-used-conjunctions dag) (push conjunction (dag-used-conjunctions dag)))
    conjunction))


(defun find-dag-node (dag name)
  (find-if #'(lambda (node) (equal name (dag-node-name node))) (dag-used-nodes dag)))

(defun used-below (upper-dag-element lower-dag-element)
  (cond ((null upper-dag-element) nil)
        ((null lower-dag-element) nil)
        ((eq upper-dag-element lower-dag-element) t)
        ((dag-node-p upper-dag-element)
         (some #'(lambda (conjunction) (used-below conjunction lower-dag-element)) (dag-node-conjunctions upper-dag-element)))
        ((dag-so-node-p upper-dag-element)
         (some #'(lambda (literal) (used-below literal lower-dag-element)) (dag-so-node-literals upper-dag-element)))
        ((dag-conjunction-p upper-dag-element)
         (some #'(lambda (literal) (used-below literal lower-dag-element)) (dag-conjunction-literals upper-dag-element)))))


(defun get-dag-node-for-relation (dag name used-in)
  (let ((node (find-dag-node dag name)))
    (cond ((dag-node-p node)
           (when (used-below node used-in)
             (setq node (dag-cycle-node dag)))
           (unless (member used-in (dag-node-used-in node))
             (setf (dag-node-used-in node) (push used-in (dag-node-used-in node)))))
          (t
           (let ((clauses (get-clauses name)))
             (setf node (get-dag-node dag)
                   (dag-node-name node) name
                   (dag-node-conjunctions node) (if (or (and *expand-trivial-conjunctive-rules*
                                                             clauses
                                                             (null (rest clauses))
                                                             (null (rest (rest (first clauses)))))
                                                        (and *expand-all-conjunctive-rules*
                                                             clauses
                                                             (null (rest clauses))))
                                                  (get-dag-conjunctions-for-clauses dag clauses name node)
                                                  nil)
                   (dag-node-number-of-conjunctions node) (length clauses)
                   (dag-node-used-in node) (list used-in)))))
    node))

(defun get-dag-so-node-for-second-order-literal (dag name body used-in)
  (let ((so-node (get-dag-so-node dag)))
    (setf (dag-so-node-name so-node) name
          (dag-so-node-literals so-node) (mapcar #'(lambda (literal) (get-dag-node-for-literal dag literal so-node)) body)
          (dag-so-node-used-in so-node) used-in)
    so-node))

(defun get-dag-node-for-literal (dag literal used-in)
  (if (consp literal)
    (let ((name (first literal)))
      (cond ((eql name 'not)         (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'and)         (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'or)          (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'bagof)       (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'call)        (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'find-proofs) (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            ((eql name 'setof)       (get-dag-so-node-for-second-order-literal dag name (rest literal) used-in))
            (t                       (get-dag-node-for-relation dag name used-in))))
    (get-dag-node-for-relation dag literal used-in)))

(defun get-dag-conjunction-for-clause-body (dag clause-body rule-name used-in clause-number)
  (let ((conjunction (get-dag-conjunction dag)))
    (setf (dag-conjunction-used-in conjunction) used-in
          (dag-conjunction-name conjunction) rule-name
          (dag-conjunction-number conjunction) clause-number
          (dag-conjunction-length conjunction) (length clause-body)
          (dag-conjunction-literals conjunction) (mapcar #'(lambda (literal) (get-dag-node-for-literal dag literal conjunction)) clause-body))
    conjunction))

(defun get-dag-conjunctions-for-clauses (dag clauses rule-name used-in)
  (let ((clause-number -1))
    (mapcar #'(lambda (clause) (get-dag-conjunction-for-clause-body dag (rest clause) rule-name used-in (incf clause-number))) clauses)))

(defun create-dag-conjunctions-for-node (dag node)
  (let ((name (dag-node-name node)))
    (when (get-rule name)
      (get-dag-conjunctions-for-clauses dag (get-clauses name) name node))))

(defun expand-dag-node (dag name)
  (let ((node (find-dag-node dag name)))
    (when (dag-node-p node)
      (or (dag-node-conjunctions node)
          (setf (dag-node-conjunctions node) (create-dag-conjunctions-for-node dag node))))))

(defun active-path-from-dag-element-to-root (dag-element root)
  (cond ((eq dag-element root) t)
        ((dag-node-p dag-element)
         (some #'(lambda (element)
                   (if (dag-conjunction-p element)
                     (let ((rule-name (dag-conjunction-name element))
                           (clause-index (dag-conjunction-number element))
                           (literal-index -1))
                       (and (some #'(lambda (node)
                                      (and (not (literal-deleted rule-name clause-index (incf literal-index)))
                                           (eq node dag-element)))
                                  (dag-conjunction-literals element))
                            (active-path-from-dag-element-to-root element root)))
                     (active-path-from-dag-element-to-root element root)))
               (dag-node-used-in dag-element)))
        ((dag-so-node-p dag-element)
         (let ((element (dag-so-node-used-in dag-element)))
           (if (dag-conjunction-p element)
             (let ((rule-name (dag-conjunction-name element))
                   (clause-index (dag-conjunction-number element))
                   (literal-index -1))
               (and (some #'(lambda (node)
                              (and (not (literal-deleted rule-name clause-index (incf literal-index)))
                                   (eq node dag-element)))
                          (dag-conjunction-literals element))
                    (active-path-from-dag-element-to-root element root)))
             (active-path-from-dag-element-to-root element root))))
        ((dag-conjunction-p dag-element)
         (and (not (clause-deleted (dag-conjunction-name dag-element) (dag-conjunction-number dag-element)))
              (active-path-from-dag-element-to-root (dag-conjunction-used-in dag-element) root)))
        ((consp dag-element)
         (some #'(lambda (element) (active-path-from-dag-element-to-root element root)) dag-element))))

(defun setup-initial-dag (prolog)
  (unless (consp (first prolog))
    (setq prolog (list prolog)))
  (let* ((dag (get-dag))
         (root (get-dag-node dag))
         (cycle (get-dag-node dag)))
    (setf (dag-root dag) root
          (dag-node-name root) :root
          (dag-node-conjunctions root) (list (get-dag-conjunction-for-clause-body dag prolog :root root 0))
          (dag-node-number-of-conjunctions root) 1
          (dag-node-used-in root) nil
          (dag-cycle-node dag) cycle
          (dag-node-name cycle) :cycle
          (dag-node-conjunctions cycle) nil
          (dag-node-number-of-conjunctions cycle) 0
          (dag-node-used-in cycle) nil)
    dag))

;;;===================================================================================
;;; FAST FRONTIER OPERATIONALIZATION

(defun variables-types-used-in (prolog old-vars)
  (multiple-value-bind (old-vars old-types new-vars new-types) (vtui prolog nil old-vars nil nil nil nil t)
    (values (nreverse old-vars) (nreverse old-types) (nreverse new-vars) (nreverse new-types))))

(defun vtui (prolog type old-vars ov ot nv nt add-to-new)
  (cond ((pcvar-p prolog) (if (or (member prolog ov :test #'var-eq)
                                (member prolog nv :test #'var-eq))
                          (values ov ot nv nt)
                          (if (member prolog old-vars)
                            (values (cons prolog ov) (cons (or type :anything) ot) nv nt)
                            (if add-to-new
                              (values ov ot (cons prolog nv) (cons (or type :anything) nt))
                              (values ov ot nv nt)))))
        ((consp prolog)
         (let* ((first-prolog (first prolog))
                (r-struct (get-r-struct first-prolog)))
           (cond ((r-p r-struct)
                  (case (r-kind r-struct)
                    (:not (vtui (rest prolog) nil old-vars ov ot nv nt nil))
                    (:or (vtui (rest prolog) nil old-vars ov ot nv nt nil))
                    (:and (vtui (rest prolog) nil old-vars ov ot nv nt add-to-new))
                    (otherwise (vtui (rest prolog) (r-type r-struct) old-vars ov ot nv nt add-to-new))))
                 (t
                  (multiple-value-setq (ov ot nv nt) (vtui (first prolog) (first type) old-vars ov ot nv nt add-to-new))
                  (vtui (rest prolog) (rest type) old-vars ov ot nv nt add-to-new)))))
        (t (values ov ot nv nt))))

(defun number-of-deletable-clauses (rule-name number-of-clauses)
  (let ((number-of-deletable-clauses 0))
    (dotimes (clause-index number-of-clauses)
      (unless (or (clause-deleted rule-name clause-index)
                  (clause-permanently-added rule-name clause-index))
        (incf number-of-deletable-clauses)))
    number-of-deletable-clauses))

(defun number-of-addable-clauses (rule-name number-of-clauses)
  (let ((number-of-addable-clauses 0))
    (dotimes (clause-index number-of-clauses)
      when (clause-deleted-but-not-permanently rule-name clause-index)
      (incf number-of-addable-clauses))
    number-of-addable-clauses))

(defun number-of-deletable-literals (rule-name clause-index number-of-literals)
  (let ((number-of-deletable-literals 0)
        (number-of-undeleted-literals 0))
    (dotimes (literal-index number-of-literals)
      (cond ((literal-deleted rule-name clause-index literal-index))
            ((literal-permanently-added rule-name clause-index literal-index) (incf number-of-undeleted-literals))
            (t (incf number-of-undeleted-literals)
               (incf number-of-deletable-literals))))
    (values number-of-deletable-literals number-of-undeleted-literals)))

;;;============================================================================
;;; FAST-FRONTIER-OPs

;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_replace_rule_by_one_clause
;;;
;;;  Complete
;;;  Definition      Try           Try             Try
;;;  A :- B,C.       A :- B,C.
;;;  A :- D,E,F.                   A :- D,E,F.
;;;  A :- G,H.                                     A :- G,H.
;;;
;;;  Apply to all disjunctive rules with heads that appear in the current frontier,
;;;  only apply to rules that have more than 2 clauses (avoid duplicating work)

(def-fast-frontier-operator FAST-FRONTIER-OP_replace_rule_by_one_clause
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and  *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_replace_rule_by_one_clause"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name clause-index number-of-clauses) (values-list best-modification)
       (expand-dag-node dag rule-name)
       (dotimes (index number-of-clauses)
         (delete-clause rule-name index))
       (permanently-undelete-clause rule-name clause-index)))
    (:test
     (let ((root (dag-root dag))
           (cycle (dag-cycle-node dag))
           rule-name evaluation number-of-clauses)
       (dolist (node (dag-used-nodes dag))
         (when (and (not (eq node root))
                    (not (eq node cycle))
                    (null (dag-node-conjunctions node))
                    (> (setq number-of-clauses (dag-node-number-of-conjunctions node)) 2)
                    (active-path-from-dag-element-to-root node root))
           (setq rule-name (dag-node-name node))
           (dotimes (clause-index number-of-clauses)
             (delete-clause rule-name clause-index))
           (dotimes (clause-index number-of-clauses)
             (undelete-clause rule-name clause-index)
             (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
             (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_replace_rule_by_one_clause)
               (setf best-operator 'FAST-FRONTIER-OP_replace_rule_by_one_clause
                     best-modification (list rule-name clause-index number-of-clauses)
                     best-evaluation evaluation))
             (delete-clause rule-name clause-index))
           (dotimes (clause-index number-of-clauses)
             (undelete-clause rule-name clause-index)))))))
  (values best-evaluation best-operator best-modification))

;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause
;;;
;;;  Complete
;;;  Definition      Try           Try             Try
;;;  A :- B,C.       A :- B,C.                     A :- B,C.
;;;  A :- D,E,F.     A :- D,E,F.   A :- D,E,F.
;;;  A :- G,H.                     A :- G,H.       A :- G,H.
;;;
;;;  Apply to all disjunctive rules with heads that appear in the current frontier,
;;;  only apply to rules that have more than 1 clause.

(def-fast-frontier-operator FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause 
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and  *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name clause-index) (values-list best-modification)
       (expand-dag-node dag rule-name)
       (permanently-delete-clause rule-name clause-index)))
    (:test
     (let ((root (dag-root dag))
           (cycle (dag-cycle-node dag))
           rule-name evaluation number-of-clauses)
       (dolist (node (dag-used-nodes dag))
         (when (and (not (eq node root))
                    (not (eq node cycle))
                    (null (dag-node-conjunctions node))
                    (> (setq number-of-clauses (dag-node-number-of-conjunctions node)) 1)
                    (active-path-from-dag-element-to-root node root))
           (setq rule-name (dag-node-name node))
           (dotimes (clause-index number-of-clauses)
             (delete-clause rule-name clause-index)
             (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
             (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause)
               (setf best-operator 'FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause
                     best-modification (list rule-name clause-index)
                     best-evaluation evaluation))
             (undelete-clause rule-name clause-index)))))))
  (values best-evaluation best-operator best-modification))

;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_delete_one_clause
;;;
;;;  Complete           Current
;;;  Definition         Definition    Try             Try
;;;  A :- B,C.          A :- B,C.                     A :- B,C.
;;;  A :- D,E,F.        A :- D,E,F.   A :- D,E,F.
;;;  A :- G,H.
;;;
;;;  Apply to all rules with heads that have previously appeared the best frontier
;;;  and which have more than 1 undeleted clause.
;;;
;;;  Note: If a rules head previously appeared in the best frontier and no longer
;;;        does then some operator has modified the rules definition and the rule
;;;        will appear as modified in the deletion hash table.
;;;
;;;  Note: This operator isn't applied when *expanding-best-frontier* because
;;;        all frontiers produced by applying this operator to the expanded frontier
;;;        would have been produced in the unexpanded frontier using the operator
;;;        FAST-FRONTIER-OP_replace_rule_by_all_but_one_clause.

(def-fast-frontier-operator FAST-FRONTIER-OP_delete_one_clause
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and  *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_delete_one_clause"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name clause-index) (values-list best-modification)
       (permanently-delete-clause rule-name clause-index)))
    (:test
     (unless *expanding-best-frontier*
       (let ((root (dag-root dag))
             (cycle (dag-cycle-node dag))
             rule-name evaluation number-of-clauses number-of-deletable-clauses)
         (dolist (node (dag-used-nodes dag))
           (setq rule-name (dag-node-name node)
                 number-of-clauses (dag-node-number-of-conjunctions node))
           (when (and (not (eq node root))
                      (not (eq node cycle))
                      (dag-node-conjunctions node)
                      (>= (setq number-of-deletable-clauses (number-of-deletable-clauses rule-name number-of-clauses)) 1)
                      (active-path-from-dag-element-to-root node root))
             (cond ((> number-of-deletable-clauses 1)
                    (dotimes (clause-index number-of-clauses)
                      (unless (or (clause-deleted rule-name clause-index)
                                  (clause-permanently-added rule-name clause-index))
                        (delete-clause rule-name clause-index)
                        (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
                        (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_delete_one_clause)
                          (setf best-operator 'FAST-FRONTIER-OP_delete_one_clause
                                best-modification (list rule-name clause-index)
                                best-evaluation evaluation))
                        (undelete-clause rule-name clause-index))))
                   ((= number-of-deletable-clauses 1)
                    (dotimes (clause-index number-of-clauses)
                      (unless (or (clause-deleted rule-name clause-index)
                                  (clause-permanently-added rule-name clause-index))
                        (permanently-undelete-clause rule-name clause-index)))))))))))
  (values best-evaluation best-operator best-modification))

;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_add_one_clause
;;;
;;;  Complete           Current
;;;  Definition         Definition    Try             Try
;;;  A :- B,C.          A :- B,C.     A :- B,C.       A :- B,C.
;;;  A :- D,E,F.                      A :- D,E,F.
;;;  A :- G,H.                                        A :- G,H.
;;;
;;;  Apply to all rules with heads that have previously appeared the best frontier
;;;  and which have more than 1 deleted clause.
;;;
;;;  Note: If a rules head previously appeared in the best frontier and no longer
;;;        does then some operator has modified the rules definition and the rule
;;;        will appear as modified in the deletion hash table.
;;;
;;;  Note: This operator isn't applied when *expanding-best-frontier* because
;;;        all frontiers produced by applying this operator to the expanded frontier
;;;        would also have been produced in the unexpanded frontier.


(def-fast-frontier-operator FAST-FRONTIER-OP_add_one_clause
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_add_one_clause"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name clause-index) (values-list best-modification)
       (permanently-undelete-clause rule-name clause-index)))
    (:test
     (unless *expanding-best-frontier*
       (let ((root (dag-root dag))
             (cycle (dag-cycle-node dag))
             rule-name evaluation number-of-clauses number-of-addable-clauses)
         (dolist (node (dag-used-nodes dag))
           (setq rule-name (dag-node-name node)
                 number-of-clauses (dag-node-number-of-conjunctions node))
           (when (and (not (eq node root))
                      (not (eq node cycle))
                      (dag-node-conjunctions node)
                      (>= (setq number-of-addable-clauses (number-of-addable-clauses rule-name number-of-clauses)) 1)
                      (active-path-from-dag-element-to-root node root))
             (cond ((> number-of-addable-clauses 1)
                    (dotimes (clause-index number-of-clauses)
                      (when (clause-deleted-but-not-permanently rule-name clause-index)
                        (undelete-clause rule-name clause-index)
                        (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
                        (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_add_one_clause)
                          (setf best-operator 'FAST-FRONTIER-OP_add_one_clause
                                best-modification (list rule-name clause-index)
                                best-evaluation evaluation))
                        (delete-clause rule-name clause-index))))
                   ((= number-of-addable-clauses 1)
                    (dotimes (clause-index number-of-clauses)
                      (when (clause-deleted-but-not-permanently rule-name clause-index)
                        (permanently-delete-clause rule-name clause-index)))))))))))
  (values best-evaluation best-operator best-modification))


;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_delete_one_literal_from_clause
;;;
;;;  Complete           Current
;;;  Definition         Definition    Try             Try
;;;  A :- B,C.          A :- B,C.     A :- B.       A :- C.
;;;  A :- D,E,F.                      
;;;  A :- G,H.


(def-frontier-operator FAST-FRONTIER-OP_delete_one_literal_from_clause
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_delete_one_literal_from_clause"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name clause-index literal-index) (values-list best-modification)
       (permanently-delete-literal rule-name clause-index literal-index)))
    (:test
     (let ((root (dag-root dag))
           (cycle (dag-cycle-node dag))
           rule-name clause-index evaluation number-of-literals number-of-deletable-literals number-of-undeleted-literals)
       (dolist (node (dag-used-nodes dag))
         (when (and (not (eq node root))
                    (not (eq node cycle))
                    (dag-node-conjunctions node)
                    (active-path-from-dag-element-to-root node root))
           (setq rule-name (dag-node-name node))
           (dolist (conjunction (dag-node-conjunctions node))
             (setq clause-index (dag-conjunction-number conjunction))
             (unless (clause-deleted rule-name clause-index)
               (setq number-of-literals (dag-conjunction-length conjunction))
               (multiple-value-setq (number-of-deletable-literals number-of-undeleted-literals) (number-of-deletable-literals rule-name clause-index number-of-literals))
               (when (and (> number-of-undeleted-literals 1)
                          (>= number-of-deletable-literals 1))
                 (dotimes (literal-index number-of-literals)
                   (unless  (or (literal-deleted rule-name clause-index literal-index)
                                (literal-permanently-added rule-name clause-index literal-index))
                     (delete-literal rule-name clause-index literal-index)
                     (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
                     (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_delete_one_literal_from_clause)
                       (setf best-operator 'FAST-FRONTIER-OP_delete_one_literal_from_clause
                             best-modification (list rule-name clause-index literal-index)
                             best-evaluation evaluation))
                     (undelete-literal rule-name clause-index literal-index)))))))))))
  (values best-evaluation best-operator best-modification))

;;;____________________________________________________________________________
;;; FAST-FRONTIER-OP_delete_one_literal_from_rule
;;;
;;;  Complete           
;;;  Definition         Try           Try           Try
;;;  A :- B,C,D.        A :- C,D.     A :- B,D.     A :- B,C.
;;;
;;;  Apply to all conjunctive rules with heads that appear in the current frontier,
;;;  only apply to rules that have exactly 1 clause.

(def-frontier-operator FAST-FRONTIER-OP_delete_one_literal_from_rule
  (function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification action)
  (when (and *trace-learning?* (member :o *focl-trace-level*))
    (format t "~%~%FAST-FRONTIER-OP_delete_one_literal_from_rule"))
  (ccase action
    (:apply
     (multiple-value-bind (rule-name literal-index) (values-list best-modification)
       (expand-dag-node dag rule-name)
       (permanently-delete-literal rule-name 0 literal-index)))
    (:test
     (let ((root (dag-root dag))
           (cycle (dag-cycle-node dag))
           rule-name evaluation number-of-undeleted-literals)
       (dolist (node (dag-used-nodes dag))
         (setq rule-name (dag-node-name node))
         (when (and (not (eq node root))
                    (not (eq node cycle))
                    (null (dag-node-conjunctions node))
                    (= (dag-node-number-of-conjunctions node) 1)
                    (> (setq number-of-undeleted-literals (- (length (first (get-clauses rule-name))) 1)) 1)
                    (active-path-from-dag-element-to-root node root))
           (dotimes (literal-index number-of-undeleted-literals )
             (delete-literal rule-name 0 literal-index)
             (setq evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples))
             (when (evaluation-operator< best-evaluation best-operator evaluation 'FAST-FRONTIER-OP_delete_one_literal_from_rule)
               (setf best-operator 'FAST-FRONTIER-OP_delete_one_literal_from_rule
                     best-modification (list rule-name literal-index)
                     best-evaluation evaluation))
             (undelete-literal rule-name 0 literal-index)))))))
  (values best-evaluation best-operator best-modification))

(defun get-function-variablization-type-from-prolog (prolog old-variables)
  (let (function variablization types)
    (when (consp prolog)
      (let ((first-prolog (first prolog)))
        (if (or (consp first-prolog)
                (eql first-prolog 'not)
                (eql first-prolog 'and)
                (eql first-prolog 'or))
          (let ((f.v.t (gethash prolog *conjunction-function-hash*)))
            (if f.v.t
              (setq function (first f.v.t)
                    variablization (second f.v.t)
                    types (third f.v.t))
              (multiple-value-bind (old-vars old-types new-vars new-types) (variables-types-used-in prolog old-variables)
                (setq variablization (append old-vars new-vars)
                      types (append old-types new-types)
                      function (focl-compile-clause-function (if (consp first-prolog)
                                                               (cons (cons 'dummy variablization) prolog)
                                                               (list (cons 'dummy variablization) prolog))
                                                             (length variablization)))
                (setf (gethash prolog *conjunction-function-hash*) (list function variablization types)))))
          (let ((r-struct (get-r-struct first-prolog)))
            (setq function (r-prolog-function r-struct)
                  variablization (rest prolog)
                  types (r-type r-struct))))))
    (values function variablization types)))

(defun fast-evaluate-frontier (function variablization types current-state-value pos-tuples neg-tuples)
  (when (and *trace-learning?* (member :o *focl-trace-level*))
    (display-clause-and-literal-deletions))
  (info-gain-prove *prolog-frontier* function current-state-value pos-tuples neg-tuples variablization types))

(defun fast-op-frontier (prolog pos-tuples neg-tuples variables variable-types mapping current-state-value source winners)
  (declare (ignore source winners))
  (setq prolog (direct-substitute prolog mapping)
        *prolog-frontier* prolog)
  (clear-clause-and-literal-deletions)
  (multiple-value-bind (function variablization types) (get-function-variablization-type-from-prolog prolog variables)
    (let ((dag (setup-initial-dag prolog))
          (best-evaluation (fast-evaluate-frontier function variablization types current-state-value pos-tuples neg-tuples)))
      (unless (and *operationalize-only-when-inital-frontier-has-positive-gain*
                   (not (evaluation-good-enough-to-continue best-evaluation)))
        (fast-find-best-frontier dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation)
        (multiple-value-bind (new-vars new-types) (compute-new-vars-and-types variablization types variables nil nil)
          (let ((op-prolog-literals (create-rule-for-frontier prolog))
                (new-pos-tuples (extend-tuples-prove function pos-tuples variablization types nil new-vars))
                (new-neg-tuples (extend-tuples-prove function neg-tuples variablization types nil new-vars))
                (extended-vars (if new-vars (append variables new-vars) variables))
                (extended-types (if new-vars (append variable-types new-types) variable-types)))
            (clear-clause-and-literal-deletions)
            (values op-prolog-literals new-pos-tuples new-neg-tuples extended-vars extended-types)))))))

(defun fast-find-best-frontier (dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation)
  (let ((modified? nil))
    (multiple-value-setq (dag best-evaluation modified?)
      (real-fast-find-best-frontier dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation))
    (when *expand-best-frontier*
      (setq *expanding-best-frontier* t)
      (when (and *trace-learning?* (member :o *focl-trace-level*))
        (format t "~%~%Expanding Best Frontier"))
      (fast-expand-frontier dag)
      (multiple-value-setq (dag best-evaluation modified?)
        (real-fast-find-best-frontier dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation))
      (when (and *trace-learning?* (member :o *focl-trace-level*))
        (format t "~%~%Undoing Expansion Best Frontier"))
      (fast-unexpand-frontier dag)
      (setq *expanding-best-frontier* nil)
      (when modified?
        (multiple-value-setq (dag best-evaluation modified?)
          (fast-find-best-frontier dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation))))
    (values dag best-evaluation modified?)))

(defun real-fast-find-best-frontier (dag function variablization types pos-tuples neg-tuples current-state-value best-evaluation)
  (let ((modified? nil))
    (do ((terminate nil)
         (best-operator nil nil)
         (best-modification nil nil))
        (terminate)
      (dolist (operator *ACTIVE-FAST-FRONTIER-OPERATORS*)
        (multiple-value-setq (best-evaluation best-operator best-modification)
          (funcall (symbol-function operator) function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification :test)))
      (cond ((and best-operator (evaluation-good-enough-to-continue best-evaluation))
             (funcall (symbol-function best-operator) function variablization types current-state-value pos-tuples neg-tuples dag best-evaluation best-operator best-modification :apply)
             (setq modified? t))
            (t
             (setq terminate t)))
      (when (and *trace-learning?* (member :o *focl-trace-level*))
        (format t "~%~%Selecting...")
        (format t "~%Operator :     ~S" best-operator)
        (format t "~%Modification : ~S" best-modification)
        (format t "~%Evaluation :   ~S" best-evaluation)
        (display-clause-and-literal-deletions)))
    (values dag best-evaluation modified?)))

(defvar *expanded-dag-nodes* nil)

(defun fast-expand-frontier (dag)
  (setq *expanded-dag-nodes* nil)
  (dolist (node (dag-used-nodes dag))
    (unless (dag-node-conjunctions node)
      (when (setf (dag-node-conjunctions node) (create-dag-conjunctions-for-node dag node))
        (setq *expanded-dag-nodes* (push node *expanded-dag-nodes*))))))

(defun fast-unexpand-frontier (dag)
  (let (modified? rule-name clause-index)
    (dolist (node *expanded-dag-nodes*)
      (setq modified? nil
            rule-name (dag-node-name node)
            clause-index -1)
      (do ((cs (dag-node-conjunctions node) (rest cs)))
          ((or modified? (null cs)))
        (if (clause-deleted rule-name (incf clause-index))
          (setq modified? t)
          (dotimes (literal-index (dag-conjunction-length (first cs)))
            (if (literal-deleted rule-name clause-index literal-index)
              (setq modified? t)))))
      (unless modified?
        (dolist (conjunction (dag-node-conjunctions node))
          (dispose-conjunction-and-below dag conjunction))))))

(defun rules-that-need-to-be-redefined ()
  (let* ((rules-using-modified-rules nil)
         (goal-name (first *goal-concept*))
         (relations-called-by-goal-concept (cons goal-name (relations-called-by-relation goal-name)))
         name)
    (maphash #'(lambda (key value)
                 (when (or (eql value t)
                           (eql value :deleted))
                   (setq name (first key))
                   (unless (member name rules-using-modified-rules :test #'equal)
                     (push name rules-using-modified-rules)
                     (dolist (r (rules-using-relation name))
                       (pushnew (r-name r) rules-using-modified-rules :test #'equal)))))
             *deleted-clauses-and-literals*)
    (intersection rules-using-modified-rules relations-called-by-goal-concept)))

(defun modified-clause-body (rule clause clause-index)
  (unless (clause-deleted rule clause-index)
    (let ((literal-index -1))
      (all-images #'(lambda (literal)
                      (unless (literal-deleted rule clause-index (incf literal-index))
                        literal))
                  (rest clause)))))

(defun modified-clause-bodies (rule)
  (let ((clause-index -1))
    (all-images #'(lambda (clause) (modified-clause-body rule clause (incf clause-index))) (get-clauses rule))))

(defun modified-clauses (rule)
  (let ((clause-index -1)
        (modified-body nil))
    (all-images #'(lambda (clause)
                    (when (setq modified-body (modified-clause-body rule clause (incf clause-index)))
                      (cons (first clause) modified-body)))
                (get-clauses rule))))

(defun create-rule-for-frontier (prolog)
  (let* ((rules-to-redefine (rules-that-need-to-be-redefined))
         (mapping (mapcar #'(lambda (name) (list name (unique-r-name name))) rules-to-redefine))
         name)
    (do* ((names rules-to-redefine remaining-names)
          (remaining-names (rest names) (rest names)))
         ((null names))
      (setq name (first names))
      (unless (intersection (relations-called-by-relation name) remaining-names :test #'equal)
        (setq rules-to-redefine (delete name rules-to-redefine :test #'equal)
              remaining-names rules-to-redefine
              mapping (find-or-defined-desired-rule name (get-rule name) (modified-clauses name) mapping))))
    (dolist (name rules-to-redefine)
      (setq mapping (find-or-defined-desired-rule name (get-rule name) (modified-clauses name) mapping)))
    (direct-substitute prolog mapping)))

(defun find-or-defined-desired-rule (old-name old-rule clauses mapping)
  (setq clauses (direct-substitute clauses (remove old-name mapping :key #'first :test #'equal)))
  (if (and (null (rest clauses))
           (null (rest (rest (first clauses))))
           (equal (rest (first (first clauses))) (rest (second (first clauses)))))
    (setf (second (assoc old-name mapping)) (first (second (first clauses))))
    (let ((found-rule (find-if #'(lambda (r) 
                                   (and (rule-p r)
                                        (= (r-arity old-rule) (r-arity r))
                                        (equal (r-type old-rule) (r-type r))
                                        (let ((r-clauses (get-clauses (r-name r))))
                                          (and (= (length r-clauses) (length clauses))
                                               (clauses-equal r-clauses clauses)))))
                               *r-structs*)))
      (if found-rule
        (setf (second (assoc old-name mapping)) (r-name found-rule))
        (let ((sub (assoc old-name mapping)))
          (re-def-rule old-rule :name (second sub) :clauses (direct-substitute clauses (list sub)) :show nil)))))
  mapping)

(defun clauses-equal (G-clauses S-clauses)
  (cond ((and (null G-clauses) (null S-clauses)) t)
        ((null G-clauses) nil)
        ((null S-clauses) nil)
        (t (let* ((G-clause (first G-clauses))
                  (G-body (rest G-clause))
                  (G-args (rest (first G-clause)))
                  (match nil))
             (do* ((untried S-clauses (rest untried))
                   (S-clause (first untried) (first untried)))
                  ((or (setq match (and (= (length G-clause) (length S-clause))
                                        (conjunctions-match G-body (rest S-clause) (create-var-mapping-from-a-to-b G-args (rest (first S-clause))))
                                        (clauses-equal (rest G-clauses) (remove S-clause S-clauses))))
                       (null untried))))
             match))))








    
  

