;;;; copyright (c) 1990, 1991 by the university of california, irvine. 
;;;; this program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  this program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the regents of the university of california.
;;;; this program was written by michael pazzani, cliff brunk, glenn silverstein
;;;; and kamal ali.  

(in-package :user)


;;;_____________________________________________________
;;; info-gain
;;;
;;; all-purpose general infomation gain function for r-structs
;;; used by induction, constructive induction, and ebl respecitively
;;; variabilization is the variabilization to be tested
;;; negative? tests negation of clause ; shouldn't be used by ebl
;;; it's less expensive to compute original info once, outside of this
;;; since it's not a function of the variabilization
;;; use-negative-hash-tables is t if hash tables should be used
;;; they store the # of matches of negative tuples
;;; it should only be used for the first literal of a clause (since the neg-tuples are the same)
;;; and avoid recomputation of this number for subsequent clauses
;;;
;;; returns 4 values ;ka 07
;;;
;;; 1. info gain
;;; 2. max possible gain for any specialized version
;;; 3. stop- a flag to determine if this is the best you can do
;;;     ( 4. a structure (an instance of conj-info) that is non-nil for extensional preds whose
;;;          variabilization's are appropriate for computing conjunctions )
;;; 4. num of pos tuples that had 1 or more extensions  ;ka 07
;;;
;;;  note: to save space we could pass in and recycle a conj-info structure
;;;        (from find-max-literal, etc.)

(defun info-gain (r-struct variablization negative? pos-tuples neg-tuples
                           &optional (current-state-value nil) (use-negative-hash-table nil)
                           &key (instantiated-cliche nil) (position-in-cliche nil) (lpos (length pos-tuples)) (lneg (length neg-tuples)))
  (let (gain max-possible-gain pos-matches neg-matches orig-pos-matches (orig-neg-matches ':-)) ;; formally ':didnt-compute
    (setq *max-number-of-observed-extensions* 0)
    (unless current-state-value
      (setq current-state-value (current-metric lpos lneg)))
    
    (multiple-value-setq (pos-matches orig-pos-matches)
      (generalized-count-matches r-struct pos-tuples variablization negative? :instantiated-cliche instantiated-cliche))
    
    ;even when pos-matches == 0, we want the right gain calculated
    ;want varzns with -ve gain being set to 0 ;;ka 07
    
    (cond
     ((and use-negative-hash-table
           (setq neg-matches (gethash variablization (neg-hash-table r-struct negative?)))))
     (t (multiple-value-setq (neg-matches orig-neg-matches)
          (generalized-count-matches r-struct neg-tuples variablization negative? :instantiated-cliche instantiated-cliche))
        (when use-negative-hash-table
          (setf (gethash variablization (neg-hash-table r-struct negative?)) neg-matches))))
    
    (setq max-possible-gain (compute-max-possible-gain pos-matches current-state-value)
          gain (GAIN-METRIC current-state-value orig-pos-matches pos-matches neg-matches orig-neg-matches))
    
    ;;; ges 5/2 - added printing of cliche - note if conjunction is too long this will mess up the indenting
    (when (and *trace-learning?* (member :i *focl-trace-level*))
      (if instantiated-cliche (print-cliche instantiated-cliche position-in-cliche))
      (format t "~%~a~a~a ~40Tpos: ~4@a: ~4@a/~4a neg: ~4@a: ~4@a/~4a gain: ~6,2f"
              (if negative? "~" " ")
              (r-name r-struct)
              variablization
              pos-matches orig-pos-matches lpos
              neg-matches orig-neg-matches lneg 
              (gain-gain gain)))
    
    (values gain
            max-possible-gain
            (and (= orig-pos-matches lpos)
                 (or (null neg-matches)
                     (= 0 neg-matches)))
            )))

;;;_____________________________________________________
;;; INFO-GAIN-PROVE
;;;
;;;  Compute info gain of a prolog-function

(defun info-gain-prove (thing function current-state-value pos-tuples neg-tuples variablization types)
  (setq *max-number-of-observed-extensions* 0)
  (multiple-value-bind (pos-matches orig-pos-matches) (count-prove function variablization types pos-tuples)
    (multiple-value-bind (neg-matches orig-neg-matches) (count-prove function variablization types neg-tuples)
      (let ((gain (GAIN-METRIC current-state-value pos-matches pos-matches neg-matches neg-matches)))
        (when (and *trace-learning?* (member :i *focl-trace-level*))
          (cond ((graph-p thing) (format t "~% ~a" (convert-tree-to-prolog (graph-base thing))))
                ((conjunction-p thing) (format t "~% ~a" (convert-tree-to-prolog thing)))
                ((node-p thing) (format t "~% ~a" (node-string thing)))
                ((consp thing) (format t "~% ~a" thing))
                (t (format t "~% ~a ~a" thing variablization)))
          (format t "~% ~40Tpos: ~4@a: ~4@a/~4a neg: ~4@a: ~4@a/~4a gain: ~6,2f"
                  pos-matches orig-pos-matches (length pos-tuples)
                  neg-matches orig-neg-matches (length neg-tuples) (gain-gain gain)))
        gain))))

#|
;;;_____________________________________________________
;;; INFO-GAIN-PROVE-IMMEDIATE

(defun info-gain-prove-immediate (name function current-state-value pos-tuples neg-tuples args negative-hash-table)
  (let (pos-matches orig-pos-matches neg-matches orig-neg-matches gain)
    (multiple-value-setq (pos-matches orig-pos-matches) (count-prove-immediate function pos-tuples))
    (cond ((and negative-hash-table
                (setq neg-matches (gethash args negative-hash-table)
                      orig-neg-matches neg-matches)))
          (t (multiple-value-setq (neg-matches orig-neg-matches) (count-prove-immediate function neg-tuples))
             (when negative-hash-table
               (setf (gethash args negative-hash-table) neg-matches))))
    (setq gain (GAIN-METRIC current-state-value orig-pos-matches pos-matches neg-matches orig-neg-matches))
    (when (and *trace-learning?* (member :i *focl-trace-level*))
      (format t "~% ~a~a~% ~40Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              name args
              pos-matches orig-pos-matches (length pos-tuples)
              neg-matches orig-neg-matches (length neg-tuples) (gain-gain gain)))
    gain))
|#

;;;_____________________________________________________
;;; COUNT-PROVE-IMMEDIATE
;;;
;;;  like count-prove, but tuples is in order of argument of function

(defun count-prove-immediate (function tuples)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (matches 0)
        (original-matches 0))
    (clear-last-tuple)
    (cond (*count-examples-as-original-matches*
           (dolist (tuple tuples)
             (setq *var-counter* 0)
             (setf (fill-pointer *trail*) 0)
             (when (catch 'reg-prove (apply function cont nil tuple))
               (incf matches)
               (when (tuple-derived-from-different-example-than-last-tuple tuple)
                 (incf original-matches)))))
          (t (dolist (tuple tuples)
               (setq *var-counter* 0)
               (setf (fill-pointer *trail*) 0)
               (when (catch 'reg-prove (apply function cont nil tuple))
                 (incf matches)))
             (setq original-matches matches)))
    (unless (zerop matches)
      (setq *max-number-of-observed-extensions* (max *max-number-of-observed-extensions* 1)))
    (values matches original-matches)))

;;;_____________________________________________________
;;;  Gain and Content Functions
;;;
;;;   c is the current state value
;;;   p is the number of positive tuples matched
;;;   n is the number of negative tuples matched
;;;   t++ is the number of previous positive tuples matched

(defun r-gain (c p n)
  (- (if (= p 0) 0 (/ p (+ p n))) c))
 
(defun r-content (p n)
  (if (= p 0) 0 (/ p (+ p n))))


(defun i-content (p n)
  (if (= p 0) 0 (log (/ (+ p n) p)  2)))

(defun i-gain (c t++ p n)
  (if (= p 0) 0 (* t++ (if (= c 0) 1  ;;<- HACK
                           (- c (log (/ (+ p n) p) 2))))))

(defun gain-metric (c t++ p n n++)
  (incf *variablizations-checked*)
  (increment-work *literal-work* (current-status))
  (case *gain-function*
    (:information (make.gain (i-gain c t++ p n) t++ p n n++))
    (:ratio       (make.gain (r-gain c p n)     c   p n n++))
    (:prob        (make.gain (p-gain p n)                    ;;for field gain
                             (p-gain (car c) (cdr c))        ;;for field t++
                             p n n++))
    (otherwise    (make.gain (i-gain c t++ p n) t++ p n n++))))

(defun current-metric (p n)
  (case *gain-function*
    (:information (i-content p n))
    (:ratio       (r-content p n))
    (:prob        (cons p n))
    (otherwise    (i-content p n))))

;;; the 4 args below only allow calculation of the posterior
;;; p-content. 
;;; returns ls and ln gains over the original (at beginning of lcd) 
;;; set of tuples

;;; now p-gain is def to be ls(wrt orig-set)*coverage
(defun p-gain (p1 n1)
  (let ((e-given-h (float (/ (+ 1 p1) (+ 1 *pos-examples-num*)))) ;laplace ratio for 1 class
        (e-given-~h (float (/ (+ 1 n1) (+ 1 *neg-examples-num*))))
        (~e-given-h (float (/ (+ 1 (- *pos-examples-num* p1)) (+ 1 *pos-examples-num*))))
        (~e-given-~h (float (/ (+ 1 (- *neg-examples-num* n1)) (+ 1 *neg-examples-num*)))))
    (values (* (/ e-given-h e-given-~h) (+ p1 n1))
            (* (/ ~e-given-h ~e-given-~h) (+ p1 n1)))))



;;;_____________________________________________________
;;; UPDATE-ARGUMENTS-FROM-TUPLE

(defun update-arguments-from-tuple (variablization tuple new-vars arguments &aux v)
  (if (null new-vars)
    (do ((vs variablization (rest vs))
         (as arguments (rest as)))
        ((null vs))
      (setq v (first vs))
      (rplaca as (if (pcvar-p v) (nth (pcvar-id v) tuple) v)))
    (do ((vs variablization (rest vs))
         (as arguments (rest as)))
        ((null vs))
      (setq v (first vs))
      (rplaca as (if (pcvar-p v) (if (member v new-vars :test #'var-eq) v (nth (pcvar-id v) tuple)) v)))))

;;;_____________________________________________________
;;; MEETS-OBJECT-IDENTITY-CRITERION

(defun meets-object-identity-criterion (arguments types new-tuple tuple)
  (every #'(lambda (arg type value)
             (or (not (object-variarble-type? type))
                 (not (new-var? arg))
                 (not (member value tuple :test #'equal))))
         arguments types new-tuple))

;;;_____________________________________________________
;;; RETURN-NEW-OBJECT-VARIABLES

(defun return-new-object-variables (variablization types new-vars)
  (mapcan #'(lambda (var type)
              (when (and (member var new-vars)
                         (object-variarble-type? type))
                (list var)))
          variablization types))

;;;_____________________________________________________
;;; QUICK-MATCH
;;;
;;; like unify except
;;; 1. pattern can contain variables but tuple can not
;;; 2. mapping has been allocated memory else where and is
;;;    initialized to contain a cons cell for each variable
;;;    in pattern

(defun quick-match (pattern tuple mapping)
  (let ((match t) pat tup val pair)
    (dolist (p mapping) (rplacd p nil))
    (do ((ps pattern (rest ps))
         (ts tuple (rest ts)))
        ((or (null ps) (null match)))
      (setq pat (first ps)
            tup (first ts))
      (if (setq pair (assoc pat mapping))
        (if (setq val (rest pair))
          (unless (eql val tup) (setq match nil))
          (rplacd pair tup))
        (unless (eql pat tup) (setq match nil))))
    match))

;;;_____________________________________________________
;;; QUICK-SUBSTITUTE

(defun quick-substitute (form mapping)
  (cond ((consp form) (cons (quick-substitute (first form) mapping)
                            (quick-substitute (rest form) mapping)))
        ((rest (assoc form mapping)))
        (t form)))

;;;_____________________________________________________
;;; GENERALIZED-COUNT-MATCHES

(defun generalized-count-matches (r-struct tuples variablization negative? &key instantiated-cliche)
  (cond ((null tuples)
         (values 0 0))
        ((pred-p r-struct)                                            
         (if negative?
           (count-negation-matches r-struct tuples variablization :instantiated-cliche instantiated-cliche)
           (count-matches r-struct tuples variablization :instantiated-cliche instantiated-cliche)))
        ((builtin-p r-struct)                                         
         (if negative?
           (count-builtin-negation-matches (r-function r-struct) tuples variablization :instantiated-cliche instantiated-cliche)
           (count-builtin-matches (r-function r-struct) tuples variablization :instantiated-cliche instantiated-cliche)))
        (t
         (count-prove (r-prolog-function r-struct) variablization (r-type r-struct) tuples negative?))
        ))

;;;_____________________________________________________
;;; COUNT-MATCHES
;;;
;;;  returns 2 values
;;;   1 number of extended tuples matched
;;;   2 number of original tuples matched
;;;
;;;  pred is an r structure 
;;;  tuples is a list of tuples (assumes that first var is ?0, second ?1 etc
;;;  variabilization is a list of variables where ?-n indicates new, ?n indicates
;;;  nth constant of tuple

(defun count-matches (pred tuples variablization &key instantiated-cliche)
  (let* ((matches 0)
         (original-matches 0)
         (key (classify-variabilization-for-induction variablization))
         (types (r-type pred))
         (test-for-object-identity? (some #'object-variarble-type? types))
         (arguments (copy-list variablization))
         (new-vars (compute-new variablization))
         (mapping (unless (eql key :all-bound) (mapcar #'(lambda (v) (cons v nil)) new-vars)))
         (matched-tuples nil)
         new-matches)
    (clear-last-tuple)
    (dolist (tuple tuples)
      (update-arguments-from-tuple variablization tuple new-vars arguments)
      (setq new-matches 0)
      (dolist (new-tuple (retrieve-superset-of-matching-tuples pred arguments key))
        (when (if (eql key :all-bound)
                (equal arguments new-tuple)
                (and (quick-match arguments new-tuple mapping)
                     (if test-for-object-identity? (meets-object-identity-criterion arguments types new-tuple tuple) t)))
          (incf new-matches)))
      (unless (zerop new-matches)
        (incf matches new-matches)        ;; increment n if n extended tuples match
        (setq *max-number-of-observed-extensions* (max *max-number-of-observed-extensions* new-matches))
        (if instantiated-cliche
          (push tuple matched-tuples)
          (if *count-examples-as-original-matches*
            (when (tuple-derived-from-different-example-than-last-tuple tuple)
              (incf original-matches))    ;; increment 1 if any extended tuple matches and tuple is not derived from the same example the last tuple was
            (incf original-matches))      ;; increment 1 if any extended tuple matches
        )))
    (values matches (if instantiated-cliche (count-orig-matches matched-tuples instantiated-cliche) original-matches))))

;;;_____________________________________________________
;;; COUNT-NEGATION-MATCHES
;;;
;;;  a negation never binds variables, instead it's true for each tuple is there
;;;  is no fact that unifies with that tuple
;;;
;;;  returns 2 values:  
;;;  1   number of extended tuples matched
;;;  2   number of original tuples matched (will equal 1st val. returned for this function)

(defun count-negation-matches (pred tuples variablization &key instantiated-cliche)
  (let* ((key (classify-variabilization-for-induction variablization))
         (arguments (copy-list variablization))
         (new-vars (compute-new variablization))
         (mapping (unless (eql key :all-bound) (mapcar #'(lambda (v) (cons v nil)) new-vars)))
         (matched-tuples nil)
         (matches 0)
         (original-matches 0)
         match?)
    (clear-last-tuple)
    (dolist (tuple tuples)
      (update-arguments-from-tuple variablization tuple new-vars arguments)
      (setq match? nil)
      (do ((new-tuples (retrieve-superset-of-matching-tuples pred arguments key) (rest new-tuples)))
          ((or (null new-tuples) match?))
        (when (if (eql key :all-bound)
                (equal arguments (first new-tuples))
                (quick-match arguments (first new-tuples) mapping))
          (setq match? t)))
      (unless match?
        (incf matches)                    ;; increment 1 if no extended tuple matches
        (if instantiated-cliche
          (push tuple matched-tuples)
          (if *count-examples-as-original-matches*
            (when (tuple-derived-from-different-example-than-last-tuple tuple)
              (incf original-matches))    ;; increment 1 if no extended tuple matches and tuple is not derived from the same example the last tuple was
            (incf original-matches))      ;; increment 1 if no extended tuple matches
          )))
    (values matches (if instantiated-cliche (count-orig-matches matched-tuples instantiated-cliche) original-matches))))

;;;_____________________________________________________
;;; EXTEND-TUPLES-EXTENSIONAL
;;;
;;;  like count tuples, but instantiates new vars and append to end of teach tuple
;;;  note that new-vars are now indicated by a list of new-variables
;;;  all-unique-images dousn't add duplicates

(defun extend-tuples-extensional (pred tuples variablization &optional (new-vars (compute-new variablization)))
  (if new-vars
    (let ((key (classify-variabilization-for-induction variablization new-vars))
          (arguments (copy-list variablization))
          (mapping (mapcar #'(lambda (v) (cons v nil)) new-vars))
          (object-vars (return-new-object-variables variablization (r-type pred) new-vars)))
      (mapcan #'(lambda (tuple)
                  (update-arguments-from-tuple variablization tuple new-vars arguments)
                  (all-unique-images #'(lambda (new-tuple)
                                         (when (and (quick-match arguments new-tuple mapping)
                                                    (null (intersection (quick-substitute object-vars mapping) tuple :test #'equal)))
                                           (append tuple (quick-substitute new-vars mapping))))
                                     (retrieve-superset-of-matching-tuples pred arguments key)))
              tuples))
    (filter-unproved-tuples-extensional pred tuples variablization)))

;;;_____________________________________________________
;;; FILTER-UNPROVED-TUPLES-EXTENSIONAL

(defun filter-unproved-tuples-extensional (pred tuples variablization)
  (let ((arguments (copy-list variablization))
        (hash-table (r-pos-hash pred))
        (proved-tuples nil))
    (dolist (tuple tuples)
      (update-arguments-from-tuple variablization tuple nil arguments)
      (when (gethash arguments hash-table nil)
        (push tuple proved-tuples)))
    (nreverse proved-tuples)))

;;;_____________________________________________________
;;; EXTEND-NEGATION-TUPLES
;;;
;;;  negations dont extend, they just filter (using remove if not))

(defun extend-negation-tuples (pred tuples variablization &optional (new-vars (compute-new variablization)))
  (if new-vars
    (let ((key (classify-variabilization-for-induction variablization new-vars))
          (arguments (copy-list variablization))
          (mapping (mapcar #'(lambda (v) (cons v nil)) new-vars))
          (unproved-tuples nil))
      (dolist (tuple tuples)
        (update-arguments-from-tuple variablization tuple new-vars arguments)
        (unless (some #'(lambda (new-tuple)
                          (quick-match arguments new-tuple mapping))
                      (retrieve-superset-of-matching-tuples pred arguments key))
          (push tuple unproved-tuples)))
      (nreverse unproved-tuples))
    (filter-proved-tuples-extensional pred tuples variablization)))

;;;_____________________________________________________
;;; FILTER-PROVED-TUPLES-EXTENSIONAL

(defun filter-proved-tuples-extensional (pred tuples variablization)
  (let ((arguments (copy-list variablization))
        (hash-table (r-pos-hash pred))
        (unproved-tuples nil))
    (dolist (tuple tuples)
      (update-arguments-from-tuple variablization tuple nil arguments)
      (unless (gethash arguments hash-table nil)
        (push tuple unproved-tuples)))
    (nreverse unproved-tuples)))

;;;______________________________________
;;; EXTEND-TUPLES
;;;
;;; node-or-literal is a single intensional, extensional, or builtin
;;; node or literal structure.  Second order stuctures are not handled

(defun extend-tuples (node-or-literal pos-tuples neg-tuples old-vars old-types new-vars)
  (declare (ignore old-types))
  (let* ((all-vars (append old-vars new-vars))
         (mapping (direct-mapping all-vars (make-old-vars (length all-vars))))
         (new-new-vars (direct-substitute new-vars mapping))
         (new-old-vars (direct-substitute old-vars mapping))
         (r-struct (cond ((node-p node-or-literal) (node-r-struct node-or-literal))
                         ((literal-p node-or-literal) (get-r-struct (literal-predicate-name node-or-literal)))))
         (variablization (direct-substitute (cond ((node-p node-or-literal) (node-vars node-or-literal))
                                                  ((literal-p node-or-literal) (literal-variablization node-or-literal)))
                                            mapping)))
    (if (uses-undefined-relation? (cons (r-name r-struct) variablization))
      :uses-undefined-relation
      (values (generalized-extend-tuples r-struct pos-tuples variablization nil new-new-vars new-old-vars)
              (generalized-extend-tuples r-struct neg-tuples variablization nil new-new-vars new-old-vars)))))

;;;_____________________________________________________
;;; COUNT-PROVE
;;;
;;;  counts the number of tuples that are true for a variablization of a predicate
;;;  return two values pos-matches and original-pos-matches
;;;  these are equal since we are not expanding tuples

#|
(defun count-prove (function variablization tuples &optional (negative nil))
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (arguments (copy-list variablization))
        (tuple-length (length (first tuples)))
        (matches 0))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (update-arguments-from-tuple variablization tuple tuple-length arguments)
      (if (catch 'reg-prove (apply function cont nil arguments))
        (unless negative (incf matches))
        (when negative (incf matches))))
    (values matches matches)))
|#

(defun count-prove (function variablization types tuples &optional (negation nil) (new-vars (compute-new variablization)))
  (let (arguments
        (bindings nil)
        (new-prolog-vars nil))
    (clear-last-tuple)
    (cond (negation
           (simple-count-prove function variablization tuples negation new-vars))
          
          ((and new-vars *count-all-matches*)
           (let ((object-vars (return-new-object-variables variablization types new-vars))
                 (matches 0)
                 (original-matches 0)
                 (new-matches 0))
             (dolist (tuple tuples)
               (multiple-value-setq (arguments bindings)
                 (create-parameter-bindings-from-tuple variablization tuple bindings new-vars))
               (setq new-prolog-vars (mapcar #'(lambda (x) (rest (assoc x bindings))) new-vars)
                     new-matches 0)
               (dolist (new (setof-function function arguments new-prolog-vars))
                 (when (or (null object-vars)
                           (every #'(lambda (object-var)
                                      (not (member (rest (assoc (rest (assoc object-var bindings)) new)) tuple :test #'equal)))
                                  object-vars))
                   (incf new-matches)))
               (unless (zerop new-matches)
                 (incf matches new-matches)
                 (setq *max-number-of-observed-extensions* (max *max-number-of-observed-extensions* new-matches))
                 (if *count-examples-as-original-matches*
                   (when (tuple-derived-from-different-example-than-last-tuple tuple)
                     (incf original-matches))
                   (incf original-matches))))
             (values matches original-matches)))
          
          (new-vars
           (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
                 (object-vars (return-new-object-variables variablization types new-vars))
                 (matches 0)
                 (original-matches 0)
                 new)
             (dolist (tuple tuples)
               (multiple-value-setq (arguments bindings)
                 (create-parameter-bindings-from-tuple variablization tuple bindings new-vars))
               (setq new-prolog-vars (mapcar #'(lambda (x) (rest (assoc x bindings))) new-vars))
               (when (and (setq new (catch 'reg-prove (apply function cont nil arguments)))
                          (or (null object-vars)
                              (every #'(lambda (object-var)
                                         (not (member (rest (assoc (rest (assoc object-var bindings)) new)) tuple :test #'equal)))
                                     object-vars)))
                 (incf matches)
                 (if *count-examples-as-original-matches*
                   (when (tuple-derived-from-different-example-than-last-tuple tuple)
                     (incf original-matches))
                   (incf original-matches))))
             (unless (zerop matches)
               (setq *max-number-of-observed-extensions* (max *max-number-of-observed-extensions* 1)))
             (values matches original-matches)))
          
          (t (simple-count-prove function variablization tuples negation new-vars)))))

;;;_____________________________________________________
;;; SIMPLE-COUNT-PROVE

(defun simple-count-prove (function variablization tuples negative new-vars)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (arguments (copy-list variablization))
        (matches 0)
        (original-matches 0))
    (clear-last-tuple)
    (cond (*count-examples-as-original-matches*
           (dolist (tuple tuples)
             (setq *var-counter* 0)
             (setf (fill-pointer *trail*) 0)
             (update-arguments-from-tuple variablization tuple new-vars arguments)
             (if (catch 'reg-prove (apply function cont nil arguments))
               (unless negative
                 (incf matches)
                 (when (tuple-derived-from-different-example-than-last-tuple tuple)
                   (incf original-matches)))
               (when negative
                 (incf matches)
                 (when (tuple-derived-from-different-example-than-last-tuple tuple)
                   (incf original-matches))))))
          (t 
           (dolist (tuple tuples)
             (setq *var-counter* 0)
             (setf (fill-pointer *trail*) 0)
             (update-arguments-from-tuple variablization tuple new-vars arguments)
             (if (catch 'reg-prove (apply function cont nil arguments))
               (unless negative (incf matches))
               (when negative (incf matches) )))
           (setq original-matches matches)))
    (unless (zerop matches)
      (setq *max-number-of-observed-extensions* (max *max-number-of-observed-extensions* 1)))
    (values matches original-matches)))

;;;_____________________________________________________
;;; FILTER-PROVED-TUPLES

(defun filter-proved-tuples (function tuples)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (unproved-tuples nil))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (unless (catch 'reg-prove (apply function cont nil tuple))
        (push tuple unproved-tuples)))
    (nreverse unproved-tuples)))

;;;_____________________________________________________
;;; FILTER-UNPROVED-TUPLES

(defun filter-unproved-tuples (function tuples)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (proved-tuples nil))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (when (catch 'reg-prove (apply function cont nil tuple))
        (push tuple proved-tuples)))
    (nreverse proved-tuples)))

;;;_____________________________________________________
;;; FILTER-PROVED-TUPLES-WITH-VARIABLIZATION

(defun filter-proved-tuples-with-variablization (variablization function tuples new-vars)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (arguments (copy-list variablization))
        (unproved-tuples nil))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (update-arguments-from-tuple variablization tuple new-vars arguments)
      (unless (catch 'reg-prove (apply function cont nil arguments))
        (push tuple unproved-tuples)))
    (nreverse unproved-tuples)))

;;;_____________________________________________________
;;; FILTER-UNPROVED-TUPLES-WITH-VARIABLIZATION

(defun filter-unproved-tuples-with-variablization (variablization function tuples new-vars)
  (let ((cont #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t)))
        (arguments (copy-list variablization))
        (proved-tuples nil))
    (dolist (tuple tuples)
      (setq *var-counter* 0)
      (setf (fill-pointer *trail*) 0)
      (update-arguments-from-tuple variablization tuple new-vars arguments)
      (when (catch 'reg-prove (apply function cont nil arguments))
        (push tuple proved-tuples)))
    (nreverse proved-tuples)))


;;;_____________________________________________________
;;; EXTEND-TUPLES-INTENSIONAL
;;;
;;;  like extend-tuples

#|
(defun extend-tuples-intensional (rule tuples variablization &optional (negation nil) (new-vars (compute-new variablization)))
  (let (argument
        (bindings nil)
        (new-prolog-vars nil)
        (rule-function (r-prolog-function rule)))
    (cond ((and new-vars (not negation))
           (mapcan #'(lambda(tuple)
                       (multiple-value-setq (argument bindings)
                         (create-parameter-bindings-from-tuple variablization tuple bindings new-vars))
                       (when (null new-prolog-vars)
                         (setq new-prolog-vars (mapcar #'(lambda(x)(cdr (assoc x bindings))) new-vars)))
                       (mapcar #'(lambda(new) (append tuple new))
                               (setof-function rule-function argument new-prolog-vars)))
                   tuples))
          (negation (filter-proved-tuples-with-variablization variablization rule-function tuples))
          (t (filter-unproved-tuples-with-variablization variablization rule-function tuples)))))
|#

;;;_____________________________________________________
;;; EXTEND-TUPLES-PROVE

(defun extend-tuples-prove (function tuples variablization types &optional (negation nil) (new-vars (compute-new variablization)))
  (let (argument
        (bindings nil)
        (new-prolog-vars nil))
    (cond (negation (filter-proved-tuples-with-variablization variablization function tuples new-vars))
          (new-vars (let ((object-vars (return-new-object-variables variablization types new-vars)))
                      (mapcan #'(lambda (tuple)
                                  (multiple-value-setq (argument bindings)
                                    (create-parameter-bindings-from-tuple variablization tuple bindings new-vars))
                                  (when (null new-prolog-vars)
                                    (setq new-prolog-vars (mapcar #'(lambda(x)(cdr (assoc x bindings))) new-vars)))
                                  (all-images #'(lambda (new) (unless (and object-vars
                                                                           (intersection (subst new new-vars object-vars :test #'equalp) tuple :test #'equalp)))
                                                 (append tuple new))
                                              (setof-function function argument new-prolog-vars)))
                              tuples)))
          (t (filter-unproved-tuples-with-variablization variablization function tuples new-vars)))))

;;;_____________________________________________________
;;; EXTEND-TUPLES-INTENSIONAL

(defun extend-tuples-intensional (rule tuples variablization &optional (negation nil) (new-vars (compute-new variablization)))
  (extend-tuples-prove (r-prolog-function rule) tuples variablization (r-type rule) negation new-vars))


;;;_____________________________________________________
;;; EXTEND-TUPLES-=

(defun extend-tuples-= (tuples variables parameters &optional (new-vars (compute-new-vars (first parameters) variables)))
  (let ((substitution (mapcar #'(lambda (var) (list var nil)) variables))
        (binding-cons (cons t nil))
        (assigned (first parameters))
        (expression (second parameters))
        expression-value binding)
    (mapcan
     #'(lambda (tuple)
         (mapc #'(lambda (pair new-value) (setf (second pair) new-value)) substitution tuple)
         (setf binding (rplacd binding-cons substitution)
               expression-value (substitute-vars expression binding)
               binding (unify assigned expression-value binding))
         (when binding
           (if new-vars
             (list (append tuple (substitute-vars new-vars binding)))
             (list tuple))))
     tuples)))

;;;_____________________________________________________
;;; EXTEND-NEGATION-TUPLES-=

(defun extend-negation-tuples-= (tuples variables parameters)
  (let ((substitution (mapcar #'(lambda (var) (list var nil)) variables))
        (binding-cons (cons t nil))
        (assigned (first parameters))
        (expression (second parameters))
        expression-value binding)
    (mapcan
     #'(lambda (tuple)
         (mapc #'(lambda (pair new-value) (setf (second pair) new-value)) substitution tuple)
         (setq binding (rplacd binding-cons substitution)
               expression-value (substitute-vars expression binding)
               binding (unify assigned expression-value binding))
         (unless binding
           (list tuple)))
     tuples)))

;;;_____________________________________________________
;;; EXTEND-TUPLES-IS

(defun extend-tuples-is (tuples variables parameters &optional (new-vars (compute-new-vars (first parameters) variables)))
  (let ((substitution (mapcar #'(lambda (var) (list var nil)) variables))
        (binding-cons (cons t nil))
        (assigned (first parameters))
        (expression (second parameters))
        expression-value binding)
    (mapcan
     #'(lambda (tuple)
         (mapc #'(lambda (pair new-value) (setf (second pair) new-value)) substitution tuple)
         (setf binding (rplacd binding-cons substitution)
               expression-value (eval (substitute-vars expression binding))
               binding (unify assigned expression-value binding))
         (when binding
           (if new-vars
             (list (append tuple (substitute-vars new-vars binding)))
             (list tuple))))
     tuples)))

;;;_____________________________________________________
;;; EXTEND-NEGATION-TUPLES-IS

(defun extend-negation-tuples-is (tuples variables parameters)
  (let ((substitution (mapcar #'(lambda (var) (list var nil)) variables))
        (binding-cons (cons t nil))
        (assigned (first parameters))
        (expression (second parameters))
        expression-value binding)
    (mapcan
     #'(lambda (tuple)
         (mapc #'(lambda (pair new-value) (setf (second pair) new-value)) substitution tuple)
         (setq binding (rplacd binding-cons substitution)
               expression-value (eval (substitute-vars expression binding))
               binding (unify assigned expression-value binding))
         (unless binding
           (list tuple)))
     tuples)))

;;;_____________________________________________________
;;; EXTEND-TUPLES-IS-OP

(defun extend-tuples-is-op (r-struct tuples variablization old-vars)
  (extend-tuples-is tuples old-vars (cons (first variablization) (list (cons (r-function r-struct) (rest variablization))))))

;;;_____________________________________________________
;;; GENERALIZED-EXTEND-TUPLES
;;;
;;;  extends both positive and negative tuples and returns the result note pred
;;;  is a structure here might want to use this in find-literal

(defun generalized-extend-tuples (r-struct tuples variablization negated? new-vars &optional tuple-vars)
  (cond ((pred-p r-struct)
         (if negated?
           (extend-negation-tuples r-struct tuples variablization new-vars)
           (extend-tuples-extensional r-struct tuples variablization new-vars)))
        ((builtin-p r-struct)
         (if negated?
           (extend-negation-tuples-builtin r-struct tuples variablization)
           (extend-tuples-builtin r-struct tuples variablization)))
        ((is-p r-struct)
         (if negated?
           (extend-negation-tuples-is tuples tuple-vars variablization)
           (extend-tuples-is tuples tuple-vars variablization new-vars)))
        ((=-p r-struct)
         (if negated?
           (extend-negation-tuples-= tuples tuple-vars variablization)
           (extend-tuples-= tuples tuple-vars variablization new-vars)))
        ((is-op-p r-struct)
         (if negated?
           nil
           (extend-tuples-is-op r-struct tuples variablization tuple-vars)))
        (t
         (extend-tuples-intensional r-struct tuples variablization negated? new-vars))))


;;;_____________________________________________________
;;; This code is designed to determine if a tuple was
;;; derived from the same example as the previous tuple.
;;; It is called by the various counting function the
;;; when *count-examples-as-original-matches* is set to t.
;;; The assumption underlying this code is that extended
;;; tuples derived from the same example will be contiquous
;;; in the list of tuples input to the count functions.

(defvar *last-tuple* nil)

(defun clear-last-tuple () (setq *last-tuple* nil))

(defun tuple-derived-from-different-example-than-last-tuple (tuple)
  (prog1 
    (not (first-n-elements-identical? tuple *last-tuple* *example-arity*))
    (setq *last-tuple* tuple)))

(defun first-n-elements-identical? (list1 list2 n)
  (cond ((zerop n) t)
        ((equal (first list1) (first list2)) (first-n-elements-identical? (rest list1) (rest list2) (- n 1)))))