
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)
;;; variabilization restriction code

;;  00  glenn  02/09/91   added to implement variabilization-restrictions for cliches
;;  01  glenn  05/05/91   updated to reflect changes in cliche implementation

(defun apply-variabilization-restrictions (pred restrictions variabilizations old-vars old-types
                                           original-vars original-types; for recursive predicate
                                           instantiated-cliche position-in-cliche)
  (if (null restrictions)  
    variabilizations
    (remove-if-not
     #'(lambda (var-struct)
         (every #'(lambda (restriction) 
                    (variabilization-satisfies-restriction pred restriction 
                                                           (variabilization-struct-variabilization var-struct)
                                                           old-vars old-types original-vars original-types
                                                           instantiated-cliche position-in-cliche))
                restrictions))
     variabilizations)))

;;  00  glenn  05/05/91  hack to deal with restrictions for threshold builtins 
;; should tap into variabilization-satisfies-restriction but I need to figure out what to do 
;;  with the missing information as the pred is not yet know etc.

(defun apply-thresh-var-restrictions (restrictions variables instantiated-cliche)
  (let ((include-new-var-restr
         (car (membcar 'include-new-var restrictions))))
    (cond (include-new-var-restr
           (intersection (cliche-pred-new-vars-of-type instantiated-cliche 
                                                       (second include-new-var-restr)
                                                       (third include-new-var-restr))
                         variables :test #'var-eq))
          (t variables))))



;;  00  glenn  02/09/91   auxiliary function to apply-variabilization-restrictions, checks a 
;;     single restriction for a single variabilization

;;;  need to fix up so that it uses the cliche and can refer to variables in prior elements of
;;;  the cliche

;;;  functions supplied
;;;  (pred-new-vars cliche position) - retrieves new-vars of pred in position of cliche
;;;  (vars-satisfy-type type vars) - returns those vars that are of type type

(defun cliche-pred-new-vars (cliche position)
  (literal-info-new-vars (nth position cliche)))

(defun cliche-pred-new-types (cliche position)
  (literal-info-new-types (nth position cliche)))

(defun cliche-pred-vars (cliche position)
  (literal-info-vars (nth position cliche)))

;;;  restrictions supported
;;;  (include-new-var &opt type) - variabilizations must have at least one new var that
;;;    satisfies type (which is optional)

;;; accepts a cliche, a position, and optionally a type and returns those new vars that satisfy
;;; type or all new-vars if type is nil

(defun cliche-pred-new-vars-of-type (cliche position &optional type)
  (let ((new-vars (cliche-pred-new-vars cliche position))
        (new-types (cliche-pred-new-types cliche position)))
    (retrieve-vars-that-satisfy-type new-vars new-types type)))

(defun retrieve-vars-that-satisfy-type (vars &optional types target-type)
  (cond ((or (null target-type) (eql target-type :anything)) vars)
        ((eql target-type 'numeric-type)
         (remove-if #'null
                    (mapcar #'(lambda (var type) (if (number-type type) var)) vars types)))
        ((eql target-type 'non-numeric-type)
         (remove-if #'null
                    (mapcar #'(lambda (var type) (if (non-numeric-type type) var)) vars types)))
        (t
         (remove-if #'null
                    (mapcar #'(lambda (var type) (if (eql type target-type) var)) vars types)))))

;;;  returns true if type satisfie target-type
;;;  note this is used elsewhere (e.g. in builtins) and should maintain the same input and 
;;;  output behavior

(defun satisfies-type (type target-type)
  (cond ((or (null target-type) (eql target-type :anything)) t)
        ((eql target-type 'numeric-type) (number-type type))
        ((eql target-type 'non-numeric-type) (not (number-type type)))
        (t (eql type target-type))))

(defun var-types-satisfies-restriction (types target-type &optional (one? t))
  (if one?
    (some #'(lambda (type) (satisfies-type type target-type)) types)
    (every #'(lambda (type) (satisfies-type type target-type)) types)))

(defun compute-types (vars all-vars all-types)
  (mapcar #'(lambda (var) (nth (position var all-vars :test #'var-eq) all-types)) vars))



;;; minor changes adding mapping field here - ges 11/14

(defstruct var-restriction
  name
  source-pos
  var-att
  var-pos
  type
  include-all-vars?
  unordered?
  num-subst
  subst-opt?
  mapping) ; ges 11/14

(defvar *all-var-restrictions* nil)

(defmacro def-var-restriction (name &key source-pos var-att var-pos type 
                                    include-all-vars? (unordered? t) num-subst 
                                    (subst-opt? t)
                                    mapping) ; ges 11/14
  `(let ((name ',name)
         (source-pos ',source-pos)
         (var-att ',var-att)
         (var-pos ',var-pos)
         (type ',type)
         (include-all-vars? ',include-all-vars?)
         (unordered? ',unordered?)
         (num-subst ',num-subst)
         (subst-opt? ',subst-opt?)
         (mapping ',mapping))
     (let* ((bucket (assoc name *all-var-restrictions*))
            (var-restriction 
             (make-var-restriction :name name :source-pos source-pos :var-att var-att :var-pos
                                   var-pos :type type :include-all-vars? include-all-vars?
                                   :unordered? unordered? :num-subst num-subst :subst-opt? 
                                   subst-opt? :mapping mapping)))
       (setf (get name 'var-restriction) var-restriction)
       (if bucket 
         (setf (cdr bucket) var-restriction)
         (push (cons name var-restriction) *all-var-restrictions*)))))


;;; definition of variabilization restrictions 


;;; (include-new-var pred-pos &opt type var-pos)

(def-var-restriction include-new-var 
  :source-pos pred-pos
  :var-att new
  :var-pos var-num
  :type var-type
  :mapping ((source-pos . 1) (type . 2) (var-pos . 3)))


;;; (introduces-new-var &opt type var-pos)

(def-var-restriction introduces-new-var 
  :source-pos current-pos 
  :var-att new
  :var-pos var-num
  :type var-type
  :mapping ((type . 1) (var-pos . 2)))



;;; note this one is inconsistent on the ordering (with respect to the others)

;;; (include-old-var pred-pos &opt var-pos var-type)

(def-var-restriction include-old-var 
  :source-pos (head old) 
  :var-att all
  :var-pos var-num
  :type var-type
  :mapping ((source-pos . 1) (var-pos . 2) (type . 3)))


;;; (use-same-vars pred-pos &opt num-subst (unordered? t) (subst-opt? t))

(def-var-restriction use-same-vars
  :source-pos pred-pos
  :var-att all
  :var-pos nil
  :type nil
  :include-all-vars? t
  :unordered? t-or-nil
  :num-subst var-num
  :subst-opt? t-or-nil
  :mapping ((source-pos . 1) (num-subst . 2) (unordered? . 3) (subst-opt? . 4)))

(defun variabilization-satisfies-restriction (pred restriction variabilization old-vars old-types
                                              original-vars original-types instantiated-cliche 
                                              position-in-cliche)
  (if (eql (car restriction) 'reduction-variabilization)
    (every #'(lambda (restriction) 
               (variabilization-satisfies-restriction pred
                                                      restriction 
                                                      variabilization
                                                      old-vars
                                                      old-types
                                                      original-vars 
                                                      original-types
                                                      instantiated-cliche 
                                                      position-in-cliche))
           (reduction-pred-reduction-variabilization pred))
    (let* ((restriction-template (get (car restriction) 'var-restriction))
           (mapping (var-restriction-mapping restriction-template))
           (source-vars 
            (retrieve-source-vars restriction restriction-template pred variabilization old-vars 
                                  old-types original-vars original-types instantiated-cliche 
                                  position-in-cliche mapping))
           (include-all-vars?
            (get-restriction-value restriction 
                                   (var-restriction-include-all-vars? restriction-template)
                                   'include-all-vars?
                                   mapping)))
      (when source-vars
        (if include-all-vars? 
          (process-include-all-vars-restrictions restriction restriction-template variabilization
                                                 source-vars mapping)
          (let ((var-pos 
                 (get-restriction-value restriction 
                                        (var-restriction-var-pos restriction-template)
                                        'var-pos
                                        mapping)))
            (if var-pos
              (member (nth var-pos variabilization) source-vars :test #'var-eq)
              (some #'(lambda (sv) (member sv variabilization :test #'var-eq))
                    source-vars))))))))

;;; note if num-subst is nil then assumes that the list is ordered (num-subst should be 0 
;;; if the intent was an unordered list with 0 substitutions)

(defun process-include-all-vars-restrictions (restriction restriction-template variabilization
                                              source-vars mapping)
  (let ((unordered? (get-restriction-value restriction 
                                           (var-restriction-unordered? restriction-template)
                                           'unordered?
                                           mapping))
        (num-subst (get-restriction-value restriction 
                                          (var-restriction-num-subst restriction-template)
                                          'num-subst
                                          mapping))
        (subst-opt? (get-restriction-value restriction 
                                           (var-restriction-subst-opt? restriction-template)
                                           'subst-opt?
                                           mapping)))
    (if (null num-subst)
      (equalp source-vars variabilization)
      (let ((missmatches (var-missmatches source-vars variabilization unordered?)))
        (if subst-opt?
          (<= (length missmatches) num-subst)
          (= (length missmatches) num-subst))))))


;;; accepts a restriction and a restriction template and filters through the first three 
;;; arguments of the template, i.e., 
;;; (1) retrieves the variables associated with the pred in pred-pos
;;;     a). pred-pos = a number - the number is an index into the cliche
;;;     b). pred-pos = head - use head vars
;;;     c). pred-pos = old - old variables before beginning of cliche
;;;     d). pred-pos = all-vars - all variables used up to and including the current position
;;;         in the cliche
;;; (2). new or all variables of the source pred (from pred-pos) are used depending on the 
;;;      value of var-att in the template
;;; (3). if type restrictions filter the remaining variables

(defun retrieve-source-vars (restriction restriction-template pred variabilization old-vars 
                             old-types original-vars original-types instantiated-cliche 
                             position-in-cliche mapping)
  (let* ((source-pos (get-restriction-value restriction 
                                            (var-restriction-source-pos restriction-template)
                                            'source-pos
                                            mapping))
         (source-pred (if (numberp source-pos) (nth source-pos instantiated-cliche)))
         (var-att (var-restriction-var-att restriction-template)))
    (multiple-value-bind
       (source-vars source-types)
       (if source-pred
         (if (eql var-att 'new)
           (values (literal-info-new-vars source-pred)
                   (literal-info-new-types source-pred))    
           (values (literal-info-variabilization source-pred)
                   (r-type (literal-info-pred source-pred))))
         (case source-pos
           (current-pos
            (if (eql var-att 'all)
              (values variabilization (r-type pred))
              (let ((all-old-vars
                     (if (zerop position-in-cliche) 
                       old-vars
                       (literal-info-vars (nth (1- position-in-cliche) instantiated-cliche)))))
                (dual-set-difference variabilization (r-type pred) all-old-vars))))
           (head (values original-vars original-types))
           (old (values old-vars old-types))
           (all-vars 
            (progn (setq source-pred (nth position-in-cliche instantiated-cliche))
                   (values (literal-info-vars source-pred) 
                           (literal-info-types source-pred))))
           (t (error "got unimplemented source pos: ~a  source pred is ~a" source-pos source-pred))))
      (retrieve-vars-that-satisfy-type source-vars source-types 
                                       (get-restriction-value restriction 
                                                              (var-restriction-type  
                                                               restriction-template)
                                                              'type
                                                              mapping)))))


;;; might want to make this an in-line procedure or something for efficiency
(defun get-restriction-value (restriction val-spec val-type mapping)
  (let ((index (cdr (assoc val-type mapping))))
    (if index
      (nth index restriction)
      val-spec)))

;;; accepts a set and its companion set (set1 and companion1 - e.g., like vars and types)
;;; and a second set (set2) and returns those elements of set1 and their companions that
;;; do not appear in set2.

(defun dual-set-difference (set1 companion1 set2 &key (test #'eql))
  (let ((set-diff nil)
        (companion-diff nil))
    (mapc 
     #'(lambda (e1 c1)
         (unless (or (member e1 set2 :test test) (member e1 set-diff))
           (push e1 set-diff)
           (push c1 companion-diff)))
     set1 companion1)
    (values (reverse set-diff) (reverse companion-diff))))



;;;  accepts two lists of vars and resturns the number of missmatches - if unordered? is true
;;;  then the lists are treated as sets

(defun var-missmatches (source-vars target-vars unordered?)
  (if unordered? 
    (set-difference target-vars source-vars :test #'var-eq)
    (let ((missmatches nil))
      (mapc #'(lambda (s-var t-var) (if (not (var-eq s-var t-var)) (push t-var missmatches)))
            source-vars target-vars)
      missmatches)))


;;; predicate restrictions

;;; possibly useful in alternate partof cliche?  - though not currently used
(defun pred-type-restriction-part-type (pred)
  (member (r-name pred) '(handle bottom body support concavity)))


(defun get-pred-type-restr (restr)
  (some #'(lambda (r) (if (symbolp r) r)) restr))

(defun retrieve-consistent-reduction-preds ()
  (let* ((recursive-var-types (r-type (get-r-struct *predicate-being-learned*)))
        reduction-type)
    (remove-if-not 
     #'(lambda (pred) 
         (and (reduction-pred-p (cdr pred))
              (setq reduction-type (reduction-pred-reduction-type (cdr pred)))
              (some 
               #'(lambda (type)
                   (satisfies-type type reduction-type))
               recursive-var-types)))
     *extensional-preds*)))

;; note won't quite work for unconstrained cliche - first cut

(defun filter-pred-restrictions (restrictions max-new-vars)
  (let* ((pred-type (get-pred-type-restr restrictions))
         (preds 
          (case pred-type
            (pred (append *extensional-preds* *builtin-preds*))
            (ext-pred *extensional-preds*)
            (comp *builtin-preds*)
            (arith-op *is-ops*)
            (t nil))))
    (apply-pred-restrictions preds restrictions max-new-vars)))


;;; note pred is in alist form ((pred . struct) ...) like *extensional-preds* etc.

(defun apply-pred-restrictions (preds restrictions &optional max-new-vars)
  (remove-if-not
            #'(lambda (pred)
                (every #'(lambda (restriction) 
                           (pred-satisfies-restriction (r-info-struct pred)
                                                       restriction max-new-vars))
                       restrictions))
            (top-level-pred-restrictions preds restrictions)))

;;; get's r-struct from element of list like *extensional-preds* - done here for generality
;;; i.e., we can change the form of *extensional-preds* etc.
(defun r-info-struct (pair) (rest pair))

(defun make-pred-info (name) (list (cons name (get-r-struct name))))

;;; may want to make these non-exclusive
(defun top-level-pred-restrictions (preds pred-restr &aux include-restr class-restr)
  (cond ((setq include-restr (car (membcar 'include-pred pred-restr)))
         (let ((include-pred nil))
           (cond ((get-r-struct (setq include-pred (second include-restr))) ; predicate is defined
                  (make-pred-info include-pred))
                 (t nil))))
        ((eql (setq class-restr (second (car (membcar 'pred-class pred-restr)))) 'reduction-pred)
         (retrieve-consistent-reduction-preds))
        ((eql class-restr 'recursive-pred)
         ;(break "got recursive pred restr")
         (make-pred-info *predicate-being-learned*))
        (t preds)))

;;;  retrictions so far:

;;; (1) exclude-pred - removes pred from consideration
;;; (2) supports-new-var (optional type) - mode restrictions allow for at least one new variable
;;; (3) include-pred - opposite of 
;;; (4) pred-class - for reduction and recursive preds this portion of the code doesn't filter
;;;;    them
(defun pred-satisfies-restriction (pred restriction max-new-vars)
  (if (listp restriction)
    (case (car restriction)
      (supports-new-var 
       (and max-new-vars ; builtins won't supply max-new-vars but should support new vars
            (> max-new-vars 0) 
            (if (second restriction)
              (var-types-satisfies-restriction (new-var-types pred) (second restriction))
              (new-var-types pred))))
      (exclude-pred (not (eql (r-name pred) (second restriction)))) ; ges added 11/5
      (t t))
      (case restriction
        (pred t)
        (ext-pred (pred-p pred))
        (comp (builtin-p pred))
        (var-comp (and (builtin-p pred) (not (r-try-constants pred))))
        (arith-op (is-op-p pred))
        (t t))))

(defun new-var-types (pred)
  (let ((types (r-type pred))
        (mode (r-mode pred)))
    (if (null mode)
      types ; all variables can be new
      (remove-if #'null
                 (mapcar #'(lambda (m ty) (if (or (eql m :-) (eql m :?)) ty)) mode types)))))
