;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein 
;;;; and Kamal Ali.  

(in-package :user)

;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins
;;  01  glenn  10/04/91   updated to count orig-tuples correctly
;;  02  cliff  08/28/93   revised to deal with multiple arity variabilizations
;;
;;;  additions on 10/04/91
;;;    instantiated-cliche was added to provide the cliche structure being processed
;;;    match-fn is either #'count-if (for normal predicates) and #'remove-if-not - for
;;;      the case when comp is part of a cliche - for cliches we need to collect the
;;;      tuples to allow the computation of orig-matches (T++)

(defun count-builtin-matches (function tuples variabilization &key instantiated-cliche)
  (let ((matches 0)
        (original-matches 0)
        (extended-tuples nil))
    (clear-last-tuple)
    (if instantiated-cliche
      (setq extended-tuples (real-extend-tuples-builtin function tuples variabilization)
            matches (length extended-tuples))
      (if (= (length variabilization) 2)
        (let* ((var1 (first variabilization))
               (var2 (second variabilization))
               (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
               (var-pos2 (if (pcvar-p var2) (pcvar-id var2))))
          (if *count-examples-as-original-matches*
            (cond ((and var-pos1 var-pos2)
                   (dolist (tuple tuples)
                     (when (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (var-pos1
                   (dolist (tuple tuples)
                     (when (funcall function (nth var-pos1 tuple) var2)
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (var-pos2
                   (dolist (tuple tuples)
                     (when (funcall function var1 (nth var-pos2 tuple))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (t 
                   (cond ((funcall function var1 var2)
                          (setq matches (length tuples))
                          (dolist (tuple tuples)
                            (when (tuple-derived-from-different-example-than-last-tuple tuple)
                              (incf original-matches))))
                         (t (setq matches 0
                                  original-matches 0)))))
            (progn
              (cond ((and var-pos1 var-pos2)
                     (dolist (tuple tuples)
                       (when (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple)) (incf matches))))
                    (var-pos1
                     (dolist (tuple tuples)
                       (when (funcall function (nth var-pos1 tuple) var2) (incf matches))))
                    (var-pos2
                     (dolist (tuple tuples)
                       (when (funcall function var1 (nth var-pos2 tuple)) (incf matches))))
                    (t 
                     (setq matches (if (funcall function var1 var2) (length tuples) 0))))
              (setq original-matches matches))))
        (if (some #'pcvar-p variabilization)
          (let ((arguments (copy-list variabilization)))
            (cond (*count-examples-as-original-matches*
                   (dolist (tuple tuples)
                     (update-arguments-from-tuple variabilization tuple nil arguments)
                     (when (apply function (first arguments) (rest arguments))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (t
                   (dolist (tuple tuples)
                     (update-arguments-from-tuple variabilization tuple nil arguments)
                     (when (apply function (first arguments) (rest arguments))
                       (incf matches)))
                   (setq original-matches matches))))
          (cond ((apply function (first variabilization) (rest variabilization))
                 (setq matches (length tuples))
                 (if *count-examples-as-original-matches*
                   (dolist (tuple tuples)
                     (when (tuple-derived-from-different-example-than-last-tuple tuple)
                       (incf original-matches)))
                   (setq original-matches matches)))
                (t (setq matches 0
                         original-matches 0))))))
      (values matches (if instantiated-cliche (count-orig-matches extended-tuples instantiated-cliche) original-matches))))

;;  rv  who    date       reason
;;  00  glenn  10/04/91   updated as in count-builtin-matches to count orig-tuples properly
;;  01  cliff  08/28/93   revised to deal with multiple arity variabilizations

(defun count-builtin-negation-matches (function tuples variabilization &key instantiated-cliche)
  (let ((matches 0)
        (original-matches 0)
        (extended-tuples nil))
    (clear-last-tuple)
    (if instantiated-cliche
      (setq extended-tuples (real-extend-negation-tuples-builtin function tuples variabilization)
            matches (length extended-tuples))
      (if (= (length variabilization) 2)
        (let* ((var1 (first variabilization))
               (var2 (second variabilization))
               (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
               (var-pos2 (if (pcvar-p var2) (pcvar-id var2))))
          (if *count-examples-as-original-matches*
            (cond ((and var-pos1 var-pos2)
                   (dolist (tuple tuples)
                     (unless (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (var-pos1
                   (dolist (tuple tuples)
                     (unless (funcall function (nth var-pos1 tuple) var2)
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (var-pos2
                   (dolist (tuple tuples)
                     (unless (funcall function var1 (nth var-pos2 tuple))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (t 
                   (cond ((funcall function var1 var2)
                          (setq matches (length tuples))
                          (dolist (tuple tuples)
                            (when (tuple-derived-from-different-example-than-last-tuple tuple)
                              (incf original-matches))))
                         (t
                          (setq matches 0
                                original-matches 0)))))
            (progn
              (cond ((and var-pos1 var-pos2)
                     (dolist (tuple tuples)
                       (unless (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple)) (incf matches))))
                    (var-pos1
                     (dolist (tuple tuples)
                       (unless (funcall function (nth var-pos1 tuple) var2) (incf matches))))
                    (var-pos2
                     (dolist (tuple tuples)
                       (unless (funcall function var1 (nth var-pos2 tuple)) (incf matches))))
                    (t 
                     (setq matches (if (funcall function var1 var2) (length tuples) 0))))
              (setq original-matches matches))))
        (if (some #'pcvar-p variabilization)
          (let ((arguments (copy-list variabilization)))
            (cond (*count-examples-as-original-matches*
                   (dolist (tuple tuples)
                     (update-arguments-from-tuple variabilization tuple nil arguments)
                     (unless (apply function (first variabilization) (rest variabilization))
                       (incf matches)
                       (when (tuple-derived-from-different-example-than-last-tuple tuple)
                         (incf original-matches)))))
                  (t
                   (dolist (tuple tuples)
                     (update-arguments-from-tuple variabilization tuple nil arguments)
                     (unless (apply function (first variabilization) (rest variabilization)) (incf matches)))
                   (setq original-matches matches))))
          (cond ((apply function (first variabilization) (rest variabilization))
                 (setq matches 0
                       original-matches 0))
                (t
                 (setq matches (length tuples))
                 (if *count-examples-as-original-matches*
                   (dolist (tuple tuples)
                     (when (tuple-derived-from-different-example-than-last-tuple tuple)
                       (incf original-matches)))
                   (setq original-matches matches)))))))
    (values matches (if instantiated-cliche (count-orig-matches extended-tuples instantiated-cliche) matches))))

(defun count-orig-matches (extended-tuples instantiated-cliche)
  (count-originals-extended (literal-info-pos (first instantiated-cliche)) extended-tuples))

(defun return-orig-matches (extended-tuples instantiated-cliche)
  (return-originals-extended (literal-info-pos (first instantiated-cliche)) extended-tuples))

(defun compute-midpt (a b) (/ (+ a b) 2.0))


;;  rv  who    date       reason
;;  00  glenn  02/20/91   revised to deal with nil for pos-points or neg-points
;;  00  glenn  05/13/91   generalized to deal with non-numerics

(defun compute-boundary-points (pos-points neg-points &optional (sort-fn #'<))
  (do ((pos pos-points)
       (neg neg-points)
       (pos-is-less 
        (and pos-points neg-points) (funcall sort-fn (car pos-points) (car neg-points)))
       (boundary-pts))
      ((or (null pos) (null neg)) (nreverse boundary-pts))
    (cond (pos-is-less
           (cond ((or (null (cadr pos))
                      (not (funcall sort-fn (cadr pos) (car neg)))) ; not still <
                  (setq pos-is-less nil) 
                  (pushnew (car pos) boundary-pts)
                  (pushnew (car neg) boundary-pts))
                 (t nil))
           (setq pos (cdr pos)))
          (t
           (cond ((or (null (cadr neg))
                      (not (funcall sort-fn (cadr neg) (car pos)))) ; not still <
                  (setq pos-is-less t)
                  (pushnew (car pos) boundary-pts)
                  (pushnew (car neg) boundary-pts))
                 (t nil))
           (setq neg (cdr neg))))))


;;  rv  who    date       reason
;;  00  glenn  05/13/91   added misc. functions to compute subsets of builtins that
;;    satisfy various properties


;; the following are functions useful for retrieving subsets of builtins that satisfy
;; various properties (i.e., equality, non-numeric, numeric, relational (i.e., the 
;; non-equality builtins))


;;; determine if type is a numeric type

(defun number-type (type)
  (or (equal type :integer) (equal type :real) (equal type :numeric)(equal type :number)))


;;; for builtins typically concerned with type of first arg (should be same for all
;;; comparator builtins in general)

(defun get-builtin-type (builtin)
  (car (r-type builtin)))

;;; accepts 
(defun overall-builtin-type (builtins)
  (let ((numeric-type nil)
        (non-numeric-type nil))
    (dolist (b builtins)
      (if (number-type (get-builtin-type (cdr b)))
        (setq numeric-type t)
        (setq non-numeric-type t)))
    (if numeric-type
      (if non-numeric-type 
        :anything
        'numeric-type)
      (if non-numeric-type
        'non-numeric-type))))

;;; retrieve those builtins that support a threshold

(defun thresh-builtins (builtins )
  (all-images #'(lambda (b) (if (r-try-constants (cdr b)) b)) builtins))

;;;  not of numeric type

(defun non-numeric-type (type)
  (not (number-type type)))

;;; note this distinction (numeric vs. non-numeric) may not be useful at the moment
;;; but may be in the future (there may be some merit in separating the processing of
;;; numeric and non-numeric builtins)

(defun numeric-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (number-type (car (r-type (cdr b))))
                          (r-try-constants (cdr b))))
                 builtins))

(defun non-numeric-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (non-numeric-type (car (r-type (cdr b))))
                          (r-try-constants (cdr b))))
                 builtins))

;;;  returns those non-equality builtins that support a threshold
(defun relational-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (r-try-constants (cdr b))
                          (not (r-equality? (cdr b)))))
                 builtins))

;;;  returns those equality builtins that support the use of a constant
(defun equality-constant-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (r-try-constants (cdr b))
                          (r-equality? (cdr b))))
                 builtins))

;;; this is used in cliches and probably should be moved there

(defun pred-type-restriction-supports-threshold (pred-type-restriction)
  (or (eql pred-type-restriction 'pred)
      (eql pred-type-restriction 'comp)
      (eql pred-type-restriction 'thresh)))


; note: bring back code that checks the negated literal  - also figure out what to do about
; variabilizations


;;__________________________________________________________________________________________
;; FIND-LITERAL-BUILTIN
;;
;;  finds builtin literal with maximum gain-  see find-max for details of params
;;
;;  returns 6 values
;;  1 new-literal             -literal structure with maximum gain
;;                            -suitble for adding to end of clause
;;  2 new-vars                -names of newvariables (renamed to be old variables)
;;  3 new-types               -types of new variables
;;  4 new-pos-tuples          -next set of positive tuples
;;  5 new-neg-tuples          -next set of negative tuples
;;  6 max-gain                -maximum information gain 
;;                            (to compare aginst intensional if necessary)
;;
;;  returns nil if no literal has positive gain 
;;
;;  rv  who    date       reason
;;  00  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits
;;                        [????]   What is the 0 in values for?
;;  01  glenn  05/02/91   got rid of conjunction baggage
 
(defun find-literal-builtin (current-state-value ;passed to info-gain
                             predicate-being-learned	; used to detect recursion
                             variables
                             variable-types
                             pos-tuples
                             neg-tuples
                             original-vars
                             use-hash-tables
                              winners)	; previous high for minimum-gain
  
     
    ;;; try variable pairs first
  (unless *builtin-threshold-only*			; ges 3/15

      (find-maximum-literal current-state-value
					 predicate-being-learned
					 variables
					 variable-types
					 0
					 pos-tuples
					 neg-tuples 
					 original-vars
					 use-hash-tables
					 (inductive-builtins *builtin-preds*)
					 winners :builtin))
  (unless *covered-all-pos-tuples*
    (find-literal-builtin-thresh current-state-value
				     variables
				     variable-types
				     pos-tuples
				     neg-tuples 
				     (inductive-builtins *builtin-preds*) 
				     winners :builtin))
  ;;;convert "new winners"- Note- the same structure appears on new and all-winners
  (mapc #'(lambda (winner) (create-winner-literal variables pos-tuples neg-tuples  winner :builtin))
	(winners-new-winners winners))
  (setf (winners-new-winners winners) nil)
      
  winners)

;;  this is a generally useful function and should be moved to a utilities file
;;  01  glenn  05/05/91  used when given a subset of two lists in correspondence and you want
;;    the corresponding set of the second list

(defun retrieve-corresponding-elements (new-list1 old-list1 old-list2)
  (if (equal new-list1 old-list1)
      old-list2
    (mapcar #'(lambda (e1) (nth (position e1 old-list1) old-list2)) new-list1)))

;;__________________________________________________________________________________________
;; find-literal-builtin-thresh
;;
;; returns the predicate variabilization of pred with the maximum information gain
;; works for intension and extensional pred
;;
;; returns 4 values 
;; 1. predicate-with-maximum-gain       - builtin predicate (not a structure like other routines)
;; 2. variabilization-with-maximum-gain -variabilization ?0 thru ?n are old ?-n indicates new
;; 3. max-negated?                      -t if negation of predicate has maximum gain
;; 4. max-gain                          -the amount of information gain
;;
;; returns nil if no variabilization has gain greater than 0
;;
;; (1) apply-variabilization-restrictions wants variabilizations - since we don't 
;;     know what the thresholds are yet - the variabilizations are simply the 
;;     variables themselves each in an individual list - taking the car of the
;;     varzns returned gets us back to variables


(defun find-literal-builtin-thresh (current-state-value  ;passed to info-gain
                                    variables	         ;list of old variables
                                    variable-types       ;types of old variables
                                    pos-tuples
                                    neg-tuples
                                    builtins
                                    winner
                                    source
                                    &key
                                    var-restrictions
                                    pred-restrictions
                                    instantiated-cliche 
                                    position-in-cliche)
  (let (covered-all-pos-tuples
	negative?
        gain
        (builtins-that-satisfy-restrictions
	  (apply-pred-restrictions builtins pred-restrictions))
        relational-builtins
        equality-builtins
        variabilization
        max-possible-gain)
    covered-all-pos-tuples max-possible-gain
    (setq relational-builtins 
          (relational-thresh-builtins builtins-that-satisfy-restrictions))
    (setq equality-builtins
          (equality-constant-builtins builtins-that-satisfy-restrictions))
    
    (process-threshold-variabilizations 
      relational-builtins
      var-restrictions
      variables
      variable-types
      pos-tuples
      neg-tuples
      variabilization
      instantiated-cliche
						; exit condition for loop over-variables
      nil
						; body of inner loop
       ;;;  try built-ins with a single variable and a threshold value first

      (multiple-value-setq 
        (gain max-possible-gain covered-all-pos-tuples)
        (info-gain comp
                   variabilization 
                   nil 
                   pos-tuples 
                   neg-tuples 
                   current-state-value
                   nil 
                   :instantiated-cliche instantiated-cliche 
                   :position-in-cliche position-in-cliche))

      (when (update-winner? winner *literal-better-function* nil gain comp source
                            :vars variabilization :negated? nil
                            :instantiated-cliche instantiated-cliche)
	(update-winner winner *literal-better-function* nil gain comp source
                       :vars variabilization :negated? nil
                       :instantiated-cliche instantiated-cliche))

      (discard-gain gain)
     
      (when (>= 0 (gain-gain gain))

	(multiple-value-setq 
          (gain max-possible-gain covered-all-pos-tuples)
          (info-gain comp
                     variabilization 
                     t 
                     pos-tuples 
                     neg-tuples 
                     current-state-value
                     nil 
                     :instantiated-cliche instantiated-cliche 
                     :position-in-cliche position-in-cliche))

        (when (update-winner? winner *literal-better-function* t gain comp source
                              :vars variabilization :negated? t
                              :instantiated-cliche instantiated-cliche)
          (update-winner winner *literal-better-function* t gain comp source
                         :vars variabilization :negated? t
                         :instantiated-cliche instantiated-cliche))

	(discard-gain gain)
	))

						; handle non-numeric builtins - i.e., generating constants
    (process-equality-constant-varzns 
      equality-builtins
      var-restrictions
      variables
      variable-types
      pos-tuples
      neg-tuples
      variabilization
      negative?
      instantiated-cliche
						; exit condition for loop over-variables
      nil
						; body of inner loop
       ;;;  try built-ins with a single variable and a threshold value first

      (multiple-value-setq 
        (gain max-possible-gain covered-all-pos-tuples)
        (info-gain comp
                   variabilization 
                   negative? 
                   pos-tuples 
                   neg-tuples 
                   current-state-value
                   nil 
                   :instantiated-cliche instantiated-cliche 
                   :position-in-cliche position-in-cliche))

      (when (update-winner? winner *literal-better-function* nil gain comp source
                            :vars variabilization :negated? negative?
                            :instantiated-cliche instantiated-cliche)
        (update-winner winner *literal-better-function* negative? gain comp source
                       :vars variabilization :negated? negative?
                       :instantiated-cliche instantiated-cliche))

      (discard-gain gain)
      )
    winner))


;;;  for builtin predicates computes the information gain of comp using a single variable 
;;;  (var) and a threshold value for the comparison.  The threshold with the best info-gain is
;;;  returned provided it beats current-best.  A triple is returned consisting of the gain
;;;  of the best threshold, the threshold value (nil signifies that we couldn't beat current-best),
;;;  and the max-negated? flag. 

;;__________________________________________________________________________________________
;; compute-threshold-gain
;;
;;  for builtin predicates computes the information gain of all builtins (note this should be
;;  the subset of builtins that have their try-constants flag set) using a single variable 
;;  (var) and a threshold value for the comparison.  The threshold with the best info-gain is
;;  returned provided it beats current-best.  4 values are returned consisting of the gain of 
;;  the best threshold and builtin, the best builtin (comp-with-max-gain), the threshold value 
;;  (nil signifies that we couldn't beat current-best), and the max-negated? flag.
;;
;;  rv  who    date       reason
;;  00  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits
;;  01  glenn  05/02/91   got rid of conjunction baggage
;;  02  glenn  05/05/91   moved loop over builtin predicates into compute-threshold gain to
;;                        avoid resorting the pts etc. for each predicate
;;  03  cliff  05/08/91   replaced "real-info-gain" with "info-gain"

(defun compute-threshold-gain (builtins var pos-tuples neg-tuples current-best current-state-value
                                        &key instantiated-cliche position-in-cliche)
  (let ((max-gain current-best)
        (pos (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) pos-tuples) #'<))
        (neg (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) neg-tuples) #'<))
        (threshold nil)
        (comp-with-max-gain nil)
        (gain 0)
        (comp nil)
        (max-negated? nil)
        (max-possible-gain nil)
        (covered-all-pos-tuples nil)
        (boundary-points nil))
    max-possible-gain
    (setq boundary-points (compute-boundary-points pos neg))
    (dolist (pair builtins)
      (setq comp (rest pair))
      (dolist (boundary-point boundary-points)
        ; fix so it computes boundary points
        (multiple-value-setq 
          (gain max-possible-gain covered-all-pos-tuples)
          (info-gain comp (list var boundary-point) nil pos-tuples neg-tuples current-state-value nil
                     :instantiated-cliche instantiated-cliche :position-in-cliche position-in-cliche))
        (when (<= max-gain gain)
          (setf max-gain gain
                threshold boundary-point
                comp-with-max-gain comp
                max-negated? nil))
        (check-pruning-III covered-all-pos-tuples compute-threshold-gain 
                           (values max-gain comp-with-max-gain threshold max-negated? 
                                   covered-all-pos-tuples))
        (when (>= 0 gain)
          (multiple-value-setq 
            (gain max-possible-gain covered-all-pos-tuples)
            (info-gain comp (list var boundary-point) t pos-tuples neg-tuples current-state-value nil 
                       :instantiated-cliche instantiated-cliche :position-in-cliche position-in-cliche))
          (when (< max-gain gain)
            (setf max-gain gain
                  threshold boundary-point
                  comp-with-max-gain comp
                  max-negated? t))
          (check-pruning-III covered-all-pos-tuples compute-threshold-gain 
                             (values max-gain comp-with-max-gain threshold max-negated? 
                                     covered-all-pos-tuples)))))
    (values max-gain comp-with-max-gain threshold max-negated? covered-all-pos-tuples)))

(defun bound-var? (var)
  (or (numberp var) (and (pcvar-p var) (> (pcvar-id var) -1))))

;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins
;;  01  cliff  08/28/93   revised to deal with multiple arity variabilizations

;;; like count-builtin matches, but returns those tuples that match
;;; note variablization currently should be a list with only two elements - first element 
;;; is the variable undergoing the comparison and the second var is the comp val

(defun extend-tuples-builtin (builtin tuples variabilization)
  (real-extend-tuples-builtin (r-function builtin) tuples variabilization))

(defun real-extend-tuples-builtin (function tuples variabilization)
  (if (= (length variabilization) 2)
    (let* ((var1 (first variabilization))
           (var2 (second variabilization))
           (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
           (var-pos2 (if (pcvar-p var2) (pcvar-id var2))))
      (cond ((and var-pos1 var-pos2)
             (all-images #'(lambda (tuple) (when (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple)) tuple)) tuples))
            (var-pos1
             (all-images #'(lambda (tuple) (when (funcall function (nth var-pos1 tuple) var2) tuple)) tuples))
            (var-pos2
             (all-images #'(lambda (tuple) (when (funcall function var1 (nth var-pos2 tuple)) tuple)) tuples))
            (t 
             (if (funcall function var1 var2) tuples nil))))
    (if (some #'pcvar-p variabilization)
      (let ((arguments (copy-list variabilization)))
        (all-images #'(lambda (tuple)
                        (update-arguments-from-tuple variabilization tuple nil arguments)
                        (when (apply function (first arguments) (rest arguments)) tuple)) tuples))
      (if (apply function (first variabilization) (rest variabilization)) tuples nil))))

;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins
;;  01  cliff  08/28/93   revised to deal with multiple arity variabilizations

;;; filters those tuples that satisfy built-in predicate pred
;;; note variablization currently should be a list with only two elements - first element 
;;; is the variable undergoing the comparison and the second var is the comp val

(defun extend-negation-tuples-builtin (builtin tuples variabilization)
  (real-extend-negation-tuples-builtin (r-function builtin) tuples variabilization))

(defun real-extend-negation-tuples-builtin (function tuples variabilization)
  (if (= (length variabilization) 2)
    (let* ((var1 (first variabilization))
           (var2 (second variabilization))
           (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
           (var-pos2 (if (pcvar-p var2) (pcvar-id var2))))
      (cond ((and var-pos1 var-pos2)
             (all-images #'(lambda (tuple) (unless (funcall function (nth var-pos1 tuple) (nth var-pos2 tuple)) tuple)) tuples))
            (var-pos1
             (all-images #'(lambda (tuple) (unless (funcall function (nth var-pos1 tuple) var2) tuple)) tuples))
            (var-pos2
             (all-images #'(lambda (tuple) (unless (funcall function var1 (nth var-pos2 tuple)) tuple)) tuples))
            (t 
             (if (funcall function var1 var2) nil tuples))))
    (if (some #'pcvar-p variabilization)
      (let ((arguments (copy-list variabilization)))
        (all-images #'(lambda (tuple) 
                        (update-arguments-from-tuple variabilization tuple nil arguments)
                        (unless (apply function (first arguments) (rest arguments)) tuple)) tuples))
      (if (apply function (first variabilization) (rest variabilization)) tuples nil))))
