
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

(defun number-correct (function pos-tuples neg-tuples)
  (multiple-value-bind (p o c n) (evaluate-prolog-function function pos-tuples neg-tuples)
      (declare (ignore o c))
      (+ p n)))

;;;______________________________________________________________________________
;;; REDUCED-ERROR-PRUNING
;;;
;;;   Destructively prunes concept-description to increase its accuracy
;;;   on the set { pos-tuples neg-tuples }.  During one pass the pruning algorithm
;;;   independently applies each operator in all posisble ways and retains
;;;   the single modification which leads to the greatest improvement in
;;;   accuracy.  Mulitple passes are made over the concept-description until
;;;   all operators, if applied, would result in a decrease in accuracy on
;;;   the pruning set.  At this point pruning terminates and the pruned 
;;;   concept-description is returned.

(defun Reduced-Error-Pruning (&key (concept-description-head *learned-description-head*)   ;;; r-struct
                                   (concept-description *learned-description*)             ;;; list of literal structures
                                   pos-tuples
                                   neg-tuples
                                   (operators (list #'delete-last-literal #'drop-any-clause)))
  (let* ((*compile-allowing-deletions* t)
         (*batch* t)
         (*maintain-prolog-rule-trace* nil)
         (name (r-name concept-description-head))
         (vars (r-vars concept-description-head))
         (arity (r-arity concept-description-head)))
    (setf (r-prolog-function concept-description-head) (eval (focl-create-prolog-function name arity (mapcar #'(lambda (clause) (cons (cons 'name vars) (convert-literals-to-prolog clause t))) concept-description))))
    (do ((clauses concept-description (rest clauses))
         (clause-index 0 (incf clause-index)))
        ((null clauses))
      (do ((literal (first clauses) (literal-next literal))
           (literal-index 0 (incf literal-index)))
          ((null literal))
        (if (literal-deleted? literal)
          (delete-literal name clause-index literal-index)
          (undelete-literal name clause-index literal-index))))
    (let ((number-to-beat (number-correct (r-prolog-function concept-description-head) pos-tuples neg-tuples))
          most-correct)
      (do ((best-operator nil nil)
           (terminate nil))
          (terminate)
        (decf number-to-beat)
        (setq terminate t)
        (dolist (operator operators)
          (setq most-correct (funcall operator concept-description-head concept-description pos-tuples neg-tuples :test))
          (if (> most-correct number-to-beat)
            (setq number-to-beat most-correct
                  best-operator operator
                  terminate nil)))
        (unless terminate
          (setq concept-description (funcall best-operator concept-description-head concept-description pos-tuples neg-tuples :apply))))))
  (delete nil concept-description))



;;;==============================================================================
;;;  REDUCED-ERROR-PRUNING OPERATORS
;;;==============================================================================

;;;______________________________________________________________________________
;;; delete-last-literal
;;;
;;;  :test Independently marks the last undeleted literal of each clause in
;;;        concept-description as deleted and determines the number of examples
;;;        correctly classified.  Records the deletion which leads to the most
;;;        correctly classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Deletes the literal from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.

(let ((best-literal-to-delete nil)
      (best-literal-to-delete-clause-index nil)
      (best-literal-to-delete-literal-index nil))
  (defun delete-last-literal (concept-description-head concept-description pos-tuples neg-tuples op)
    (let ((most-correct 0)
          (current-correct 0)
          (last-literal nil)
          (name (r-name concept-description-head))
          (vars (r-vars concept-description-head))
          (mode (r-mode concept-description-head))
          (function (r-prolog-function concept-description-head))
          (clause-index -1)
          (last-literal-index 0))
      (case op
        (:test (dolist (clause-body concept-description)
                 (incf clause-index)
                 (when clause-body
                   (setq last-literal (last-undeleted-literal clause-body))
                   (when last-literal
                     (setf last-literal-index (do ((literal clause-body (literal-next literal))
                                                   (literal-index 0 (incf literal-index)))
                                                  ((eq literal last-literal) literal-index))
                           (literal-deleted? last-literal) t)
                     (delete-literal name clause-index last-literal-index)
                     (when (all-variables-in-clause-bound-p clause-body vars mode)
                       (setq current-correct (number-correct function pos-tuples neg-tuples))
                       (when (> current-correct most-correct)
                         (setq most-correct current-correct
                               best-literal-to-delete last-literal
                               best-literal-to-delete-clause-index clause-index
                               best-literal-to-delete-literal-index last-literal-index)))
                     (setf (literal-deleted? last-literal) nil)
                     (undelete-literal name clause-index last-literal-index))))
               most-correct)
        (:apply (when best-literal-to-delete
                  (setf (literal-deleted? best-literal-to-delete) t)
                  (delete-literal name best-literal-to-delete-clause-index best-literal-to-delete-literal-index)
                  concept-description)))))
  )

;;;______________________________________________________________________________
;;; delete-any-literal
;;;
;;;  :test Independently marks the each undeleted literal in concept-description
;;;        as deleted and determines the number of examples correctly classified.
;;;        Records the deletion which leads to the most correctly classified
;;;        examples, and returns that number.
;;;
;;;  :apply (called after :test) Deletes the literal from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.

(let ((best-literal-to-delete nil)
      (best-literal-to-delete-clause-index nil)
      (best-literal-to-delete-literal-index nil))
  (defun delete-any-literal (concept-description-head concept-description pos-tuples neg-tuples op)
    (let ((most-correct 0)
          (current-correct 0)
          (name (r-name concept-description-head))
          (vars (r-vars concept-description-head))
          (mode (r-mode concept-description-head))
          (function (r-prolog-function concept-description-head))
          clause-body)
      (case op
        (:test (do ((clauses concept-description (rest clauses))
                    (clause-index 0 (incf clause-index)))
                   ((null clauses))
                 (when (setq clause-body (first clauses))
                   (do ((literal clause-body (literal-next literal))
                        (literal-index 0 (incf literal-index)))
                       ((null literal))
                     (unless (literal-deleted? literal)
                       (setf (literal-deleted? literal) t)
                       (delete-literal name clause-index literal-index)
                       (when (all-variables-in-clause-bound-p clause-body vars mode)
                         (setq current-correct (number-correct function pos-tuples neg-tuples))
                         (when (> current-correct most-correct)
                           (setq most-correct current-correct
                                 best-literal-to-delete literal
                                 best-literal-to-delete-clause-index clause-index
                                 best-literal-to-delete-literal-index literal-index)))
                       (setf (literal-deleted? literal) nil)
                       (undelete-literal name clause-index literal-index)))))
               most-correct)
        (:apply (when best-literal-to-delete
                  (setf (literal-deleted? best-literal-to-delete) t)
                  (delete-literal name best-literal-to-delete-clause-index best-literal-to-delete-literal-index)
                  concept-description)))))
  )

;;;______________________________________________________________________________
;;; drop-any-clause
;;;
;;;  :test Independently drops each clause in concept-description and
;;;        determines the number of examples correctly classified.
;;;        Records the modification which leads to the most correctly
;;;        classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Drops the clause from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.

(let ((best-clause-to-drop nil)
      (best-clause-to-drop-clause-index nil))
  (defun drop-any-clause (concept-description-head concept-description pos-tuples neg-tuples op)
    (let ((most-correct 0)
          (current-correct 0)
          (name (r-name concept-description-head))
          (function (r-prolog-function concept-description-head))
          (clause-index -1))
      (case op
        (:test (dolist (clause concept-description)
                 (incf clause-index)
                 (when clause
                   (delete-clause name clause-index)
                   (setq current-correct (number-correct function pos-tuples neg-tuples))
                   (when (> current-correct most-correct)
                     (setq most-correct current-correct
                           best-clause-to-drop clause
                           best-clause-to-drop-clause-index clause-index))
                   (undelete-clause name clause-index)))
               most-correct)
        (:apply (when best-clause-to-drop
                  (delete-clause name best-clause-to-drop-clause-index)
                  (nsubstitute nil best-clause-to-drop concept-description))))))
  )






(defvar *mode-for-rep* nil)
(defvar *rep-name* nil)
(defvar *beat-this* nil)

;;; NUMBER-wrong

(defun number-wrong (pos-set neg-set)
  (declare (ignore neg-set))
  ;;assumes mode
  (let ((wrong 0))
    (dolist (p pos-set wrong)
      (unless (eq (third (prove-goal (list *rep-name* (car p) '?v)))
                  (cadr p))
        (incf wrong)
        (if (and *beat-this* 
                 (> wrong *beat-this*) )
          (return-from number-wrong wrong))))))

;;;______________________________________________________________________________
;;; REDUCED-ERROR-PRUNING
;;;
;;;   Destructively prunes concept-description to increase its accuracy
;;;   on the set { pos-set neg-set }.  During one pass the pruning algorithm
;;;   independently applies each operator in all posisble ways and retains
;;;   the single modification which leads to the greatest improvement in
;;;   accuracy.  Mulitple passes are made over the concept-description until
;;;   all operators, if applied, would result in a decrease in accuracy on
;;;   the pruning set.  At this point pruning terminates and the pruned 
;;;   concept-description is returned.

(defun Compiled-Reduced-Error-Pruning (&key (concept-description-head *learned-description-head*)
                                            concept-description
                                            pos-set
                                            neg-set
				            (mode nil)
				            (operators (list #'drop-any-clause-compiled
                                                             #'delete-last-literal-compiled)))
  (let ((*mode-for-rep* mode)
        (*rep-name* (r-name concept-description-head)))
    (do ((*beat-this* (number-wrong pos-set neg-set))
         (best-operator nil nil)
         (terminate nil))
        (terminate concept-description)
      
      (setf terminate t)
      (dolist (operator operators)
        (let ((wrong (funcall operator concept-description-head concept-description pos-set neg-set :test)))
          (if (< wrong *beat-this*)
            (setf *beat-this* wrong
                  best-operator operator
                  terminate nil))))
      (format t "~a" *beat-this*)
      
      (if (not terminate)
        (setf concept-description 
              (funcall best-operator concept-description-head concept-description pos-set neg-set :apply))))))



;;;______________________________________________________________________________
;;;  REDUCED-ERROR-PRUNING OPERATORS
;;;______________________________________________________________________________

;;;______________________________________________________________________________
;;; delete-last-literal
;;;
;;;  :test Independently marks the last undeleted literal of each clause in
;;;        concept-description as deleted and determines the number of examples
;;;        correctly classified.  Records the deletion which leads to the least
;;;        correctly classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Deletes the literal from concept-description
;;;         which leads to the least correctly classified examples.  Returns
;;;         the modified concept-description.
;;;
;;; NOTE: This might cause problems if the last undeleted literal is the only
;;;       undeleted literal.  In theory this would cause the body to be empty
;;;       (ie, no antecedents) and therefore be monotonically true.  This
;;;       would cause the concept description to always return true.
;;;
;;; NOTE: If the last literal of a clause is a negated conjunctions of literals,
;;;       delete-last-literal will delete the entire conjunction

(let ((best-literal-to-delete nil))
  (defun delete-last-literal-compiled (clause-head concept-description pos-set neg-set op)
    (case op
      (:test (let ((least-wrong ( + *beat-this* 1))
                   (current-wrong 0)
                   (clause-number 0)
                   (last-literal nil)
                   (name (make-predicate *rep-name* (r-arity clause-head)))
                   )
               (dolist (clause-body concept-description)
                 (unless (clause-deleted name clause-number)
                   (setq last-literal (last-undeleted-compiled-literal name clause-number (- (length clause-body) 1)))
                   (when last-literal
                     (delete-literal name clause-number last-literal)
                     (setq current-wrong (number-wrong pos-set neg-set))
                     (if (< current-wrong least-wrong)
                       (setq least-wrong current-wrong
                             best-literal-to-delete (list clause-number last-literal)))
                     (undelete-literal name clause-number last-literal))
                   (incf clause-number)))
               least-wrong))
      
      (:apply (when best-literal-to-delete
                (delete-literal (make-predicate *rep-name* (r-arity clause-head))
                                (car best-literal-to-delete)(cadr best-literal-to-delete))
                concept-description))))
  )

(defun last-undeleted-compiled-literal(name clause-number literal-number)
  (if (< literal-number 0) nil
      (if (literal-deleted name clause-number literal-number)
        (last-undeleted-compiled-literal name clause-number (- literal-number 1))
        literal-number)))



;;;______________________________________________________________________________
;;; drop-any-clause
;;;
;;; drop-any-clause marks the last undeleted literal of a clause as
;;;       deleted.
;;;  :test Independently drops each clause in concept-description and
;;;        determines the number of examples wrongly classified.
;;;        Records the modification which leads to the least wrongly
;;;        classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Drops the clause from concept-description
;;;         which leads to the least wrongly classified examples.  Returns
;;;         the modified concept-description.
;;;
;;; NOTE: This might cause problems if the last undeleted literal is the
;;;       only undeleted literal.  In theory this would cause the concept-
;;;       description to be empty and therefore be monotonically false.

(let ((best-clause-to-drop nil))
  (defun drop-any-clause-compiled (clause-head concept-description pos-set neg-set op )
    (case op
      (:test (let ((least-wrong ( + *beat-this* 1))
                   (name (make-predicate *rep-name* (r-arity clause-head)))
                   (current-wrong 0))
               (dotimes  (i (- (length concept-description) 1))  ;never delete last
                 (unless (clause-deleted name i)
                   (delete-clause name i)
                   (setf current-wrong (number-wrong pos-set neg-set))
                   (undelete-clause name i)
                   (if (< current-wrong least-wrong)
                     (setf least-wrong current-wrong
                           best-clause-to-drop i)))
                 )
               least-wrong))
      
      (:apply (delete-clause (make-predicate *rep-name* (r-arity clause-head)) best-clause-to-drop) concept-description)))
  )




#|
;;;______________________________________________________________________________
;;; CREATE-PRUNE-SET-FROM-R-STRUCT
;;;
;;;  Finds the r-struct associated with name and removes percent-prune-set
;;;  of the positive and percent-prune-set of the negative examples at random.
;;;  The sets of removed examples are returned as the prune set.

(defun create-prune-set-from-r-struct (name &key (percent-prune-set 33.33))
  (let* ((r-struct (get-r-struct  name))
         (example-set-pos (r-pos r-struct))
         (example-set-neg (r-neg r-struct))
         (length-example-set-pos (length example-set-pos))
         (length-example-set-neg (length example-set-neg))
         (prune-set-pos nil)
         (prune-set-neg nil)
         (length-prune-set-pos (round (* length-example-set-pos percent-prune-set) 100))
         (length-prune-set-neg (round (* length-example-set-neg percent-prune-set) 100))
         (cons-cell nil)
         (position 0))

      ;; Extract Positive Prune Set
      (dotimes (i length-prune-set-pos)
        (setf position (- (random length-example-set-pos) 1))
        (cond ((= position -1)
               (setf prune-set-pos (push (car example-set-pos) prune-set-pos)
                     example-set-pos (cdr example-set-pos)))
              (t
               (setf cons-cell (nthcdr position example-set-pos)
                     prune-set-pos (push (cadr cons-cell) prune-set-pos))
               (rplacd cons-cell (cddr cons-cell))))
        (decf length-example-set-pos 1))

      ;; Extract Negative Prune Set
      (dotimes (i length-prune-set-neg)
        (setf position (- (random length-example-set-neg) 1))
        (cond ((= position -1)
               (setf prune-set-neg (push (car example-set-neg) prune-set-neg)
                     example-set-neg (cdr example-set-neg)))
              (t
               (setf cons-cell (nthcdr position example-set-neg)
                     prune-set-neg (push (cadr cons-cell) prune-set-neg))
               (rplacd cons-cell (cddr cons-cell))))
        (decf length-example-set-neg 1))

      ;; Redefine r-struct with training-set = (example-set - prune-set)
      (eval `(def-pred ,name
               :pos ,example-set-pos
               :neg ,example-set-neg
               :type ,(r-type r-struct)
               :constraint ,(r-constraint r-struct)
               :mode ,(r-mode r-struct)
               :commutative ,(r-commutative r-struct)
               :induction ,(r-induction r-struct)
               :vars ,(r-vars r-struct)
               :questions ,(r-questions r-struct)))

      (values prune-set-pos prune-set-neg)))

;;;______________________________________________________________________________
;;; RESTORE-ORIGINAL-R-STRUCT
;;;
;;;  Adds the examples in prune-set to the examples in name's r-struct.
;;;  Note: Applying CREATE-PRUNE-SET-FROM-R-STRUCT followed by
;;;        RESTORE-ORIGINAL-R-STRUCT will restore the examples in the
;;;        orginal r-struct (assuming the proper arguments are given),
;;;        but the order will probably have changed.

(defun restore-original-r-struct (name prune-set-pos prune-set-neg)
  (let* ((r-struct (get-r-struct name))
         (example-set-pos (r-pos r-struct))
         (example-set-neg (r-neg r-struct)))

    ;; Redefine r-struct with example-set = (training-set + prune-set)
    (eval `(def-pred ,name
             :pos ,(nconc example-set-pos prune-set-pos)
             :neg ,(nconc example-set-neg prune-set-neg)
             :type ,(r-type r-struct)
             :constraint ,(r-constraint r-struct)
             :mode ,(r-mode r-struct)
             :commutative ,(r-commutative r-struct)
             :induction ,(r-induction r-struct)
             :vars ,(r-vars r-struct)
             :questions ,(r-questions r-struct)))))
|#




