
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

;;;______________________________________________________________________________
;;; GARBAGE-COLLECT

(defun garbage-collect ()
  #+:ccl (gc)
  #+:allegro (gc t)
  #+:kcl (gc)       ;;  <-  I have no idea if this is right
  #+:lucid (gc)
  )

;;;______________________________________________________________________________
;;; INTRODUCE-CLASS-NOISE
;;;
;;;  NOTE: The noise examples are NOT randomly distributed.  The misclassified
;;;        negative examples appear at the end of the noisy-pos list, while the
;;;        misclassified positive examples appear at the begining of the noisy-
;;;        pos list.

(defun introduce-class-noise (real-pos real-neg percent-noise)
  (let ((noise-level (* percent-noise 100))
        (noisy-pos nil)
        (noisy-neg nil))

    (dolist (p real-pos)                        ;; Misclassify some of the positives
      (if (< (random 10000) noise-level)
        (if (< (random 100) 50)
          (push p noisy-neg)
          (push p noisy-pos))
        (push p noisy-pos)))

    (dolist (n real-neg)                        ;; Misclassify some of the negatives
      (if (< (random 10000) noise-level)
        (if (< (random 100) 50)  
          (push n noisy-pos)
          (push n noisy-neg))
        (push n noisy-neg)))

    (let ((l-noisy-pos (length noisy-pos))      ;; Scramble examples
          (l-noisy-neg (length noisy-neg)))
      (dotimes (i l-noisy-pos)
        (swap noisy-pos i (random l-noisy-pos))
      (dotimes (i l-noisy-neg)
        (swap noisy-neg i (random l-noisy-neg)))))

    (values noisy-pos noisy-neg)))

;;;______________________________________________________________________________
;;; SWAP  (swap the pos1 element with the pos2 element in a list)

(defun swap (list pos1 pos2)
  (let ((cons1 (nthcdr pos1 list))
        (cons2 (nthcdr pos2 list))
        (temp nil))
    (setf temp (car cons1))
    (rplaca cons1 (car cons2))
    (rplaca cons2 temp))
  list)

;;;______________________________________________________________________________
;;; PARTITION-EXAMPLES

(defun partition-examples (percent-prune-set pos neg)
  (let* 
    ((percent-prune-set-size (/ percent-prune-set 100))
     (u-pos (copy-list pos))
     (u-neg (copy-list neg))
     (p-pos nil)
     (p-neg nil)
     (length-pos (length pos))
     (length-neg (length neg))
     (p-set-size (round (* percent-prune-set-size (+ length-pos length-neg))))
     (p-pos-size (round (* percent-prune-set-size length-pos)))
     (p-neg-size (- p-set-size p-pos-size)))
    
    (do* ((i 1)
          (position (random length-pos) (random length-pos))
          (element (nth position u-pos) (nth position u-pos)))
         ((> i p-pos-size))
      (setf (nth position u-pos) nil)
      (when element
        (setf p-pos (push element p-pos))
        (incf i)))

     (do* ((i 1)
          (position (random length-neg) (random length-neg))
          (element (nth position u-neg) (nth position u-neg)))
         ((> i p-neg-size))
      (setf (nth position u-neg) nil)
      (when element
        (setf p-neg (push element p-neg))
        (incf i)))
          
     (values p-pos p-neg (delete nil u-pos) (delete nil u-neg))))



;;;______________________________________________________________________________
;;; CLASS-NOISE-LEAVE-ONE-OUT-TESTING
;;;
;;; [XXXX]  This function is hardwired to test
;;;           No Stopping   -  Standard FOCL
;;;           Stopping      -  FOIL's stopping criteria
;;;           REP           -  Reduced Error Pruning

(defun class-noise-leave-one-out-testing (concept 
                                          percent-class-noise
                                          percent-prune-set-size
                                          &rest keys
                                          &key the-ignored-key &allow-other-keys)
  (declare (ignore the-ignored-key))
  (let* ((s (get-r-struct concept))
         (real-pos (r-pos s))
         (real-neg (r-neg s))
         (arity (r-arity s))
         (number-of-examples (+ (length real-pos) (length real-neg)))

         (std-MCP 0)  (std-MCN 0)  (std-CCP 0)  (std-CCN 0)
         (foil-MCP 0) (foil-MCN 0) (foil-CCP 0) (foil-CCN 0)
         (rep-MCP 0)  (rep-MCN 0)  (rep-CCP 0)  (rep-CCN 0))

    (dolist (p real-pos)
      (multiple-value-bind
        (noisy-pos noisy-neg)
        (introduce-class-noise (remove p real-pos :test #'equal)
                               real-neg
                               percent-class-noise)
        (eval `(def-pred ,concept
                 :induction nil
                 :pos ,noisy-pos
                 :neg ,noisy-neg))
        
        ;;;  STANDARD FOCL
        (let ((learned-concept (apply #'focl concept (append '(:noise-tolerance nil) keys))))
          (if (prove-function? (make-concept-function 'learned-concept arity learned-concept) nil nil p)
            (incf std-CCP)
            (incf std-MCP)))

        ;;;  FOCL USING FOIL'S STOPPING CRITERIA
        (let ((learned-concept (apply #'focl concept (append '(:noise-tolerance :FOIL) keys))))
          (if (prove-function? (make-concept-function 'learned-concept arity learned-concept) nil nil p)
            (incf foil-CCP)
            (incf foil-MCP)))
 
        ;;;  REDUCED ERROR PRUNING
        (multiple-value-bind 
          (prune-pos 
           prune-neg
           train-pos
           train-neg)
          (partition-examples percent-prune-set-size
                              noisy-pos
                              noisy-neg)
          (eval `(def-pred ,concept
                   :induction nil
                   :pos ,train-pos
                   :neg ,train-neg))
          (let* ((learned-concept (apply #'focl concept (append '(:noise-tolerance nil) keys)))
                 (pruned-concept (Reduced-Error-Pruning :concept-description learned-concept
                                                        :pos-set prune-pos
                                                        :neg-set prune-neg)))
          (if (prove-function? (make-concept-function 'learned-concept arity pruned-concept) nil nil p)
              (incf rep-CCP)
              (incf rep-MCP))))))


    (dolist (n real-neg)
      (multiple-value-bind
        (noisy-pos noisy-neg)
        (introduce-class-noise real-pos
                               (remove n real-neg :test #'equal)
                               percent-class-noise)
         (eval `(def-pred ,concept
                 :induction nil
                 :pos ,noisy-pos
                 :neg ,noisy-neg))

        ;;;  STANDARD FOCL
        (let ((learned-concept (apply #'focl concept (append '(:noise-tolerance nil) keys))))
          (if (prove-function? (make-concept-function 'learned-concept arity learned-concept) nil nil n)
            (incf std-MCN)
            (incf std-CCN)))

        ;;;  FOCL USING FOIL'S STOPPING CRITERIA
        (let ((learned-concept (apply #'focl concept (append '(:noise-tolerance :FOIL) keys))))
          (if (prove-function? (make-concept-function 'learned-concept arity learned-concept) nil nil n)
            (incf foil-MCN)
            (incf foil-CCN)))

        ;;;  REDUCED ERROR PRUNING
        (multiple-value-bind 
          (prune-pos
           prune-neg
           train-pos
           train-neg)
          (partition-examples percent-prune-set-size
                              noisy-pos
                              noisy-neg)
          (eval `(def-pred ,concept 
                   :induction nil 
                   :pos ,train-pos 
                   :neg ,train-neg))
          (let* ((learned-concept (apply #'focl concept (append '(:noise-tolerance nil) keys)))
                 (pruned-concept (Reduced-Error-Pruning :concept-description learned-concept
                                                        :pos-set prune-pos
                                                        :neg-set prune-neg)))
            (if (prove-function? (make-concept-function 'learned-concept arity pruned-concept) nil nil n)
              (incf rep-MCN)
              (incf rep-CCN))))))

    (setf std-CCP (/ std-CCP number-of-examples)
          std-CCN (/ std-CCN number-of-examples)
          std-MCP (/ std-MCP number-of-examples)
          std-MCN (/ std-MCN number-of-examples)

          foil-CCP (/ foil-CCP number-of-examples)
          foil-CCN (/ foil-CCN number-of-examples)
          foil-MCP (/ foil-MCP number-of-examples)
          foil-MCN (/ foil-MCN number-of-examples)

          rep-CCP (/ rep-CCP number-of-examples)
          rep-CCN (/ rep-CCN number-of-examples)
          rep-MCP (/ rep-MCP number-of-examples)
          rep-MCN (/ rep-MCN number-of-examples))
          
    (format t "~%;CCP~aCCN~aMCP~aMCN~aSIZE" #\tab #\tab #\tab #\tab)
    (format t "~%;---~A---~A---~A---~A----" #\tab #\tab #\tab #\tab)
    (format t "~%~5f~A~5f~A~5f~A~5f~A~5f~A No Stopping"
            std-CCP #\tab std-CCN #\tab std-MCP #\tab std-MCN #\tab number-of-examples #\tab)
    (format t "~%~5f~A~5f~A~5f~A~5f~A~5f~A Stopping"
            foil-CCP #\tab foil-CCN #\tab foil-MCP #\tab foil-MCN #\tab number-of-examples #\tab)
    (format t "~%~5f~A~5f~A~5f~A~5f~A~5f~A REP (~a)"
            rep-CCP #\tab rep-CCN #\tab rep-MCP #\tab rep-MCN #\tab number-of-examples #\tab percent-prune-set-size)
 
    (eval `(def-pred ,concept             ;; Reset Initial Examples
             :induction nil 
             :pos ,real-pos 
             :neg ,real-neg))))






;;;______________________________________________________________________________
;;; DROP-ONE-STUDENT-TESTING

(defun drop-one-student-testing ()
  (format t "~%~%~%~%;; Student Loan 20% classification noise REP 33.33 Prune Set")
  (dotimes (i 10)
    (format t "~%;; RANDOM-STATE : ~A" (setf *random-state* (make-random-state t)))
    (class-noise-leave-one-out-testing 'not_in_default 20 33.33
                                       :clauses nil
                                       :reset-hash-tables t
                                       :max-new-variables 2
                                       :operationalize-intensional nil
                                       :intensional-induction t
                                       :trace nil
                                       :simplify-clauses t
                                       :reset-statistics t  
                                       :save-examples nil  
                                       :prefer-theory nil
                                       :proof-vars-available t
                                       :partial-dt-0-gain nil
                                       :simplify-operationalizations nil
                                       :batch-mode nil)))


;;;______________________________________________________________________________
;;; JUDGE-CONCEPT
;;;
;;;   This function can be used two ways:
;;;      1.  To determine the accuracy of concept-description at classifying
;;;          a set of correctly classified positive and negative examples.
;;;      2.  To determine the accuracy of a set of positive and negative examples
;;;          given a correct and complete concept-description.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Modified to work with the rule compiler
;;;  01  MP      7/10/91  allow concept function to be passed in
;;;                       useful for judging domain theory
(defun judge-concept (concept-description pos-examples neg-examples 
					  &optional (concept-function
						     (make-concept-function 'concept-description 
									    (length (or (first pos-examples) 
											(first neg-examples)))
									    concept-description)))
  (let ((pos-correct 0)			; number of pos-examples concept-description correctly classifies
        (neg-correct 0)			; number of neg-examples concept-description correctly classifies
        (pos-missed 0)			; number of pos-examples concept-description incorrectly classifies
        (neg-missed 0)			; number of neg-examples concept-description incorrectly classifies
        )
    (dolist (p pos-examples)
	    (if (prove-function? concept-function nil nil p)
		(incf pos-correct)
	      (incf pos-missed)))
    (dolist (n neg-examples)
	    (if (prove-function? concept-function nil nil n)
		(incf neg-missed)
	      (incf neg-correct)))
    (values pos-correct neg-correct pos-missed neg-missed)))


;;;______________________________________________________________________________
;;; QUALITATIVE-FEATURES
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   6/13/91  

(defun qualitative-features (concept-description)
  (let ((number-of-clauses (length concept-description))
        (number-of-non-negated-literals 0)
        (number-of-negated-literals 0))
    (dolist (clause concept-description)
      (do* ((literal clause (literal-next literal)))
           ((null literal))
        (if (literal-negated? literal)
          (incf number-of-negated-literals
                (count-literals (literal-negated-literals literal)))
          (incf number-of-non-negated-literals 1))))
    (values number-of-clauses 
            (+ number-of-non-negated-literals number-of-negated-literals)
            number-of-negated-literals)))


;;;______________________________________________________________________________
;;; GENERATE-NOISY-EXAMPLE-SET
;;;
;;;    generates or augments a set of positive (in) and negative (out) examples
;;;    which contain percent-class-noise percent noise and which are composed of
;;;    percent-positive postive examples (:from-domain uses the distribution
;;;    obtained from the example generator).
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   3/3/91   Correctly implement noise (used to be error rate)
;;;                       now its 50% error rate.

(defun generate-noisy-example-set (n          ; number of examples to generate
                             &key 
                             (in nil)   ; list of positive examples (possibly containing noise)
                             (out nil)  ; list of negative examples (possibly containing noise)
                             (percent-positive :from-domain)
                             (percent-class-noise 0)
                             (percent-tuple-noise 0))

  (let ((p-class-noise percent-class-noise)
        (n-class-noise percent-class-noise)
        (tuple-noise-level (* percent-tuple-noise 1000))
        (domain-range 8))         ;; [XXXX] Hacked for illegal

    (labels 
      ((add-t-noise (example)
                        (cond ((null example) nil)
                              ((< (random 100000) tuple-noise-level)
                               (rplaca example (+ (random domain-range) 1))
                               (add-t-noise (cdr example)))
                              (t (add-t-noise (cdr example)))))
       (add-tuple-noise (example)
                        (if (> tuple-noise-level 0)
                          (add-t-noise example))
                        example))

      (if (equal percent-positive :from-domain)
        (dotimes (i n)                            ; Select examples from the domain at random
          (let* ((example (generate-example)))
            (if (classify-example example)
              (if (< (random 100) p-class-noise)  ; Introduce class noise into positive examples
                (if (< (random 100) 50)           ;   Introduce error with probablity .5
                  (push (add-tuple-noise example) out)              
                  (push (add-tuple-noise example) in))
                (push (add-tuple-noise example) in))
              (if (< (random 100) n-class-noise)  ; Introduce class noise into negative examples
                (if (< (random 100) 50)           ;   Introduce error with probablity .5   
                  (push (add-tuple-noise example) out)
                  (push (add-tuple-noise example) in))
                (push (add-tuple-noise example) out)))))
        
        (dotimes (i  n)                              ; Select examples to conform to percent-positive
          (if (< (random 100) percent-positive)      ; Choose class of example (pos or neg)
            (if (< (random 100) p-class-noise)       ; Introduce class noise into positive examples
              (if (< (random 100) 50)                ;   Introduce error with probablity .5
                (push (add-tuple-noise (return-positive-example)) out)              
                (push (add-tuple-noise (return-positive-example)) in))
              (push (add-tuple-noise (return-positive-example)) in))
            (if (< (random 100) n-class-noise)       ; Introduce class noise into negative examples
              (if (< (random 100) 50)                ;   Introduce error with probablity .5   
                (push (add-tuple-noise (return-negative-example)) out)
                (push (add-tuple-noise (return-negative-example)) in))
              (push (add-tuple-noise (return-negative-example)) out)))))
      
      (values in out))))

;;;______________________________________________________________________________
;;; CLASSIFY-EXAMPLE

(defun classify-example (example)
    (illegal-position example))


;;;______________________________________________________________________________
;;; judge-example-set
;;;     judges the accuracy of a set of examples with respect to illegal-position.
;;;
;;; Note:  This could be replaced by a call to judge with a correct and complete
;;;        concept-description
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun judge-example-set (in out)
  (let ((pos-correct 0)   ; number of in which are actually positive examples
        (neg-correct 0)   ; number of out which are actually negative examples
        (pos-missed 0)    ; number of in which are actually negative examples
        (neg-missed 0))   ; number of out which are actually positive examples
    (dolist (example in)
      (if (illegal-position example)
        (incf pos-correct)
        (incf pos-missed)))
    (dolist (example out)
      (if (illegal-position example)
        (incf neg-missed)
        (incf neg-correct)))
    (values pos-correct neg-correct pos-missed neg-missed)))


;;;______________________________________________________________________________
;;; GENERATOR-ILLEGAL-DEF-PRED
;;;
;;;     generates and evaluated a def-pred for illegal containing in and out
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun generator-illegal-def-pred (in
                                   out
                                   &key (typed? nil))
  (if typed? 
    (eval `(def-pred illegal 
             :induction nil 
             :type (:row :column :row :column :row :column)
             :pos ,in 
             :neg ,out))
    (eval `(def-pred illegal 
             :induction nil
             :type (:thing :thing :thing :thing :thing :thing)
             :pos ,in 
             :neg ,out))))


;;;_____________________________________________________________________________
;;; PRINT-HEADER

(defun print-header ()
  (format t "~%~%;P P +~AN N +~AMCP -~AMCN -~A Cs ~A Ls ~AN-Ls~ASize~A%Pos~A%Noise~ANVs~A Alogithm"
          #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab)
  (format t "~%;_____~A_____~A_____~A_____~A____~A____~A____~A____~A____~A______~A___~A__________"
          #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab #\tab))

;;;_____________________________________________________________________________
;;; PRINT-DATUM

(defun print-datum (PC NC MCP MCN test-size Cs Ls N-Ls
                    size percent-positive noise max-new-vars alogithm &optional v)
  (format t "~%~5f~A~5f~A~5f~A~5f~A~A~A~A~A~A~A~A~A~A~A~A~A~A~A~A"
          (/ pc test-size) #\tab 
          (/ nc test-size) #\tab
          (/ mcp test-size) #\tab 
          (/ mcn test-size) #\tab
          Cs #\tab
          Ls #\tab
          N-Ls #\tab
          size #\tab
          percent-positive #\tab
          noise #\tab
          max-new-vars #\tab
          alogithm)
  (when v (format t "~A~A" #\tab v)))


(defun noise-level-test ()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :noise-levels '(2 5 10 20 50)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun prune-set-size-test ()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(5 10 20 30 33.33 40 50 70 90)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun maximum-new-variables-test ()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(0 1 2)
              :iterations 10
              :starting-state t))


(defun example-set-size-test ()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(50)
              :example-set-sizes '(10 20 40 80 160 320)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))

(defun test-amm()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :nt-alogorithms '(:amm :STANDARD :rep)
	       :example-set-sizes '(40 80 160 320)
	       :maximum-new-variables-values '(1)
	       :iterations 20
	       :starting-state t))

(defun distribution-test ()
  (reset-preds)
  (load-source-test-file "illegal-preds-untyped")
  (noise-test :noise-levels '(20)
              :percent-positive-values '(10 20 :from-domain 50 70 90)
              :example-set-sizes '(250)
              :test-set-size 1000
              :nt-alogorithms '(:STANDARD :FOIL :REP)
              :percent-pruning-set-sizes '(33.33)
              :maximum-new-variables-values '(2)
              :iterations 10
              :starting-state t))



;;;_____________________________________________________________________________
;;; NOISE-TEST

(defun noise-test (&key (noise-levels '(20))
                        (percent-positive-values '(50))
                        (example-set-sizes '(250))
                        (test-set-size 1000)
                        (nt-alogorithms '(:STANDARD :FOIL :REP :AMM))
			(voters '(7))
                        (percent-pruning-set-sizes '(33.33))
                        (maximum-new-variables-values '(0))
                        (iterations 10)
                        (starting-state t)
                        )
  (let ((learn-pred-name 'illegal)
        (focl-parameters nil)
        (current-example-set-size 0)
        (pos nil)
        (neg nil)
        (in nil)
        (out nil))
    (setf (getf focl-parameters :clauses) nil
          (getf focl-parameters :reset-hash-tables) t
          (getf focl-parameters :operationalize-intensional) t 
          (getf focl-parameters :intensional-induction) nil
          (getf focl-parameters :trace) nil
          (getf focl-parameters :simplify-clauses) t
          (getf focl-parameters :reset-statistics) t
          (getf focl-parameters :save-examples) nil
          (getf focl-parameters :prefer-theory) t
          (getf focl-parameters :proof-vars-available) t
          (getf focl-parameters :partial-dt-0-gain) nil
          (getf focl-parameters :simplify-operationalizations) nil
          (getf focl-parameters :batch-mode) nil

          *random-state* (make-random-state starting-state))

    (format t "~%~%#|")
    (format t "~%~%;;;ILLEGAL NOISE-TEST")
    (format t "~%;;;")
    (format t "~%;;;  NOISE-LEVELS :            ~A" noise-levels)
    (format t "~%;;;  PERCENT POSITIVE :        ~A" percent-positive-values)
    (format t "~%;;;  EXAMPLE-SET-SIZES :       ~A" example-set-sizes)
    (format t "~%;;;  TEST-SET-SIZES :          ~A" test-set-size)
    (format t "~%;;;  ALGORITHMS :              ~A" nt-alogorithms)
    (format t "~%;;;  PERCENT-PRUNE-SET-SIZES : ~A" percent-pruning-set-sizes)
    (format t "~%;;;  MAX-NEW-VARIABLES :       ~A" maximum-new-variables-values)
    (format t "~%;;;")    
    (format t "~%;;;  ITERATIONS :              ~A" iterations)
    (format t "~%;;;  INITIAL RANDOM STATE :    ~A" *random-state*)
    (format t "~%~%|#")

    (dotimes (i iterations)
      (garbage-collect)
      (dolist (percent-positive percent-positive-values)
        (multiple-value-bind (pos-test neg-test)
                             (generate-noisy-example-set test-set-size 
                                                         :percent-positive percent-positive
                                                         :percent-tuple-noise 0
                                                         :percent-class-noise 0)
          (setf current-example-set-size 0
                pos nil
                neg nil)
          (dolist (example-set-size example-set-sizes)        
            (multiple-value-setq (pos neg)
              (generate-noisy-example-set (- example-set-size current-example-set-size)
                                          :in pos
                                          :out neg
                                          :percent-positive percent-positive
                                          :percent-tuple-noise 0
                                          :percent-class-noise 0))
            (setf current-example-set-size example-set-size)
            
            (dolist (noise-level noise-levels)
              (multiple-value-setq (in out)
                (introduce-class-noise pos neg noise-level))
              
              (let ((length-in (length in))
                    (length-out (length out)))
                
                (print-header)
                (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                     (judge-example-set in out)
                  (print-datum pos-correct neg-correct pos-missed neg-missed
                               example-set-size "" "" ""
                               example-set-size percent-positive noise-level
                               "" "EXAMPLE SET"))
                
                (dolist (max-new-variables maximum-new-variables-values)
                  (dolist (nt-alogorithm nt-alogorithms)
                    (case nt-alogorithm
                      
                      ((:STANDARD :RECURSIVE :FOIL)
                       (generator-illegal-def-pred in out)
                       (setf (getf focl-parameters :noise-tolerance) (if (eql :STANDARD nt-alogorithm)
                                                                       nil
                                                                       nt-alogorithm)
                             (getf focl-parameters :max-new-variables) max-new-variables)
                       (let ((learned-concept-description 
                              (apply #'focl learn-pred-name focl-parameters)))
                         
                         (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                              (judge-concept learned-concept-description 
                                                             pos-test 
                                                             neg-test)
                           (multiple-value-bind (clauses literals negated-literals)
                                                (qualitative-features learned-concept-description)
                             
                             (print-datum pos-correct neg-correct pos-missed neg-missed
                                          test-set-size clauses literals negated-literals
                                          example-set-size percent-positive noise-level 
                                          max-new-variables nt-alogorithm)))))
                      
		      (:amm
		       (dolist (v voters)
                         (generator-illegal-def-pred in out)
                         (setf (getf focl-parameters :noise-tolerance) nil
                               (getf focl-parameters :max-new-variables) max-new-variables)
                         (let ((learned-concept-description 
                                (apply #'amm v learn-pred-name focl-parameters)))
			   (make-voting-rule 'lcd 6 learned-concept-description)
                           
                           (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                                (judge-concept nil
                                                               pos-test 
                                                               neg-test
							       (r-prolog-function (get-r-struct 'lcd)))
                             (multiple-value-bind (clauses literals negated-literals)
                                                  (qualitative-features (first learned-concept-description))
                               
                               (print-datum pos-correct neg-correct pos-missed neg-missed
                                            test-set-size clauses literals negated-literals
                                            example-set-size percent-positive noise-level 
                                            max-new-variables nt-alogorithm v))))))
                      
                      (:REP
                       (dolist (percent-pruning-set-size percent-pruning-set-sizes)
                         (let* 
                           ((pos-pruning-set-size (round (* percent-pruning-set-size length-in) 100))
                            (neg-pruning-set-size (round (* percent-pruning-set-size length-out) 100))
                            (pos-training-set (butlast in pos-pruning-set-size))
                            (pos-pruning-set (nthcdr (- length-in pos-pruning-set-size) in))
                            (neg-training-set (butlast out neg-pruning-set-size))
                            (neg-pruning-set (nthcdr (- length-out neg-pruning-set-size) out))
                            (pruning-set-size (+ pos-pruning-set-size neg-pruning-set-size))
                            (percent-training-set-size (- 100 percent-pruning-set-size))
                            (training-set-size (- example-set-size pruning-set-size)))
                           
                           (when (> pruning-set-size 0)
                             
                             (generator-illegal-def-pred pos-training-set neg-training-set)
                             
                             (setf (getf focl-parameters :noise-tolerance) nil
                                   (getf focl-parameters :max-new-variables) max-new-variables)
                             (let ((learned-concept-description 
                                    (apply #'focl learn-pred-name focl-parameters)))
                               
                               (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                                    (judge-concept learned-concept-description 
                                                                   pos-test 
                                                                   neg-test)
                                 (multiple-value-bind (clauses literals negated-literals)
                                                      (qualitative-features learned-concept-description)
                                   
                                   (print-datum pos-correct neg-correct pos-missed neg-missed
                                                test-set-size clauses literals negated-literals
                                                training-set-size percent-positive noise-level 
                                                max-new-variables 
                                                (format nil "~A% STANDARD" percent-training-set-size))))
                               
                               (setf learned-concept-description 
                                     (Reduced-Error-Pruning 
                                      :concept-description learned-concept-description
                                      :pos-set pos-pruning-set
                                      :neg-set neg-pruning-set))
                               
                               (case learned-concept-description
                                 ((:failed-true :true)
                                  (print-datum (length pos-test) 0 0 (length neg-test)
                                               test-set-size 1 1 0
                                               example-set-size percent-positive noise-level 
                                               max-new-variables 
                                               (format nil "~A% ~A" percent-pruning-set-size
                                                       nt-alogorithm)))
                                 
                                 ((:failed-empty :empty)
                                  (print-datum  0 (length neg-test) (length pos-test) 0 
                                                test-set-size 1 1 1
                                                example-set-size percent-positive noise-level 
                                                max-new-variables 
                                                (format nil "~A% ~A" percent-pruning-set-size
                                                        nt-alogorithm)))
                                 
                                 (otherwise
                                  (multiple-value-bind (pos-correct neg-correct pos-missed neg-missed)
                                                       (judge-concept learned-concept-description 
                                                                      pos-test
                                                                      neg-test)
                                    (multiple-value-bind (clauses literals negated-literals)
                                                         (qualitative-features learned-concept-description)
                                      
                                      (print-datum  pos-correct neg-correct pos-missed neg-missed
                                                    test-set-size clauses literals negated-literals
                                                    example-set-size percent-positive noise-level 
                                                    max-new-variables 
                                                    (format nil "~A% ~A" percent-pruning-set-size
                                                            nt-alogorithm)))))
                                 ))))))
                      )))))))))))
  
