/* PD624 Code: answer to exercise 7.3 */

/*    Quantitative Dynamic Model of a simple plumbing system              */

/* For historical reasons, this file is referred to at exercise 7.3 in the
course.  However, it makes use of features (in particular demons)
about which you will probably be more confident AFTER doing exercise 7.4 ! */

/* Class definitions for:                                                    */
/* vessels - which can hold water.                                           */
/* tanks   - a subclass of vessels which have an inlet and outlet.           */
/* conduits - components with an inlet and outlet which convey water.        */
/* pipes - a subclass of conduits.                                           */
/* taps - a subclass of conduits with a variable outlet aperture.            */
/* ballcocks - a subclass of conduits with a variable outlet aperture which  */
/*             is controlled by the height of the ballcocks float.           */
/*                                                                           */
/* See text for description of the model defined in this file.               */
/*                                                                           */
/* All linear measurements in cm's except diameters of pipes which are in mm */
/* Volumes and flow_rates are in ml's.                                       */

vessels subclass_of plumbing_components with
  height: unknown,
  width: unknown,
  depth: unknown,
  volume_of_contents:
   [value: Quantity,
       /* Checks to ensure that the quantity of water in the */
       /* vessel doesn't exceed its physical capacity */
     change_rule: (if the depth of ?self is D &
                      the width of ?self is W &
                      the height of ?self is H &
                      prolog(Capacity is D * W * H) &
                      prolog(smallest_of(Quantity, Capacity, Vol))
                            /* smallest_of is defined at end of this file */
                   then
                      make_value Vol)],
  level:
   [value: unknown,
       /* Simply the volume of water divided by the area of the vessel */
     access_rule: (if the depth of ?self is D &
                      the width of ?self is W &
                      the volume_of_contents of ?self is V &
                      prolog(Area is W * D) &
                      prolog(Level is V / Area)
                  then
                      make_value Level)].



tanks subclass_of vessels with
  inlet: unknown,
  outlet: unknown,
  height_of_inlet: unknown,
  outlet_pressure:
   [value: unknown,
     /* Proportional to the level of water in the tank */
    access_rule : (if the level of ?self is L then  make_value L)],
  flow_rate:
   [value: unknown,
     /* Depends on the flow_rate through the outlet component */
    access_rule : (if the outlet of ?self is Outlet &
                      the flow_rate of Outlet is Flow
                   then make_value Flow)].

conduits subclass_of plumbing_components with
  inlet: unknown,
  outlet: unknown,
  vertical_height: unknown,
  diameter: unknown,
  outlet_pressure:
   [value: unknown,
    /* Equivalent to the outlet pressure of the previous component */
    /* plus the vertical height of this piece of conduit */
    access_rule: (if the inlet of ?self is Inlet &
                     the outlet_pressure of Inlet is Inlet_pressure &
                     the vertical_height of ?self is Vertical_height &
                     prolog(Own_pressure is Inlet_pressure + Vertical_height)
                 then
                     make_value Own_pressure)],
  outlet_area:
   [value: unknown,
    /* Use a crude approximation, simply square the diameter */
    /* of the conduit - assumes that they are square pipes!!! */
    access_rule : (if the diameter of ?self is D &
                      prolog(Area is D * D)
                   then
                      make_value Area)],
  flow_rate: unknown.


pipes subclass_of conduits with
  flow_rate:
   [value: unknown,
    /* Flow rate depends on flow rate of the next item in line */
    access_rule : (if the outlet of ?self is Outlet &
                      the flow_rate of Outlet is Flow
                   then
                      make_value Flow)].

taps subclass_of conduits with
  vertical_height: 0,        /* Default */
  outlet_area: 0,            /* Default assumes that taps are shut, value */
                             /* can be between 0 and D^2 */
  flow_rate:
   [value: unknown,
    /* Another crude approximation: */
    /* flow rate ~ outlet pressure times outlet area */
    access_rule : (if the outlet_pressure of ?self is Pressure &
                      the outlet_area of ?self is Area &
                      prolog(Flow is Pressure * Area)
                   then
                      make_value Flow)].


ballcocks subclass_of conduits with
  arm_length: unknown,      /* the length of the ballcock arm */
  float_drop:
   [value: unknown,
     /* the level at which the float is below the ballcock inlet */
     /* depends on the level of water in the tank, */
     /* unless of course the water level is lower than */
     /* the length that the ballcock arm can reach. */
    access_rule : (if the outlet of ?self is Tank &
                      the height_of_inlet of Tank is Height &
                      the level of Tank is Level &
                      the arm_length of ?self is Arm_length &
                      prolog(Diff is Height - Level) &
                      prolog(smallest_of(Diff,Arm_length,Drop))
                        /* smallest_of is defined at end of this file */
                  then
                      make_value Drop)],
  outlet_area:
   [value: unknown,
    /* The area is the same as for other conduits (the diameter */
    /* squared), except that it may be modified depending on the */
    /* angle of the ballcock arm. Hence the potential outlet */
    /* area is multiplied by the ratio of the float drop to the */
    /* length in order to obtain the actual outlet area */
    access_rule: (if the diameter of ?self is D &
                     the arm_length of ?self is AL &
                     the float_drop of ?self is H &
                     prolog(Area is D * D) &
                     prolog(OA is Area * H / AL)
                  then
                     make_value OA)],
  flow_rate:
   [value: unknown,
    /* Again this depends on the outlet pressure and outlet area */
    access_rule: (if the outlet_pressure of ?self is Pressure &
                     the outlet_area of ?self is Area &
                     prolog(Flow is Pressure * Area)
                  then
                     make_value Flow)].



/* The definitions for our simple plumbing installation */

/* The main tank fed from the mains at a rate controlled by ballcock1 */
tank1 instance_of tanks with
  height: 100,
  width: 100,
  depth: 100,
  inlet: ballcock1,
  outlet: pipe3,
  height_of_inlet: 75,
  volume_of_contents: 30000.     /* 30 litres */


/* Controls inlet to tank1 */
ballcock1 instance_of ballcocks with
  inlet: pipe2,
  outlet: tank1,
  vertical_height: 0,
  diameter: 15,
  arm_length: 50.

/* The pipe that connects pipe1 to ballcock1 the inlet to tank1. */
pipe2 instance_of pipes with
  inlet: pipe1,
  outlet: ballcock1,
  vertical_height: 0,
  diameter: 15.

/* The pipe that connects the mains supply to pipe2 */

pipe1 instance_of pipes with
  inlet: mains_supply,
  outlet: pipe2,

/*                      COMPATIBILITY NOTICE
In the original version of OU Prolog, negative numbers were preceded by a
tilde (~) as opposed to the more normal dash (-).  However, if you are using
a 'turnkey' version of MIKE ('MIKE.EXE'), or if you are using a source
code version of MIKE ('MIKE.PL') running on top of 'Prolog-2 Public Domain'
(or most other standard Edinburgh syntax dialects of Prolog), then you
can leave the code alone (i.e. it as already been corrected). */

  vertical_height: -1000,     /* for most Prologs, this line is OK */
/* vertical_height: ~1000, */ /* for (old) OU Prolog, use this line instead */
  diameter: 15.


/* The mains supply provides water at a pressure head of 20 metres. */

mains_supply instance_of pipes with
  outlet: pipe1,
  diameter: 15,
  outlet_pressure: 2000.

/* The pipe that leads out of tank1 */
pipe3 instance_of pipes with
  inlet: tank1,
  outlet: tap1,
  vertical_height: 500,
  diameter: 10.


/* Connected to the end of pipe2, this tap empties into tank2
(a basin or bath) */
tap1 instance_of taps with
  inlet: pipe3,
  outlet: basin1,
  outlet_area: 50,
  diameter: 10.


/* tank2 represents a basin or bath in our model,
it has a deliberately blocked outlet */

tank2 instance_of tanks with
  height: 30,
  width: 50,
  depth: 50,
  inlet: tap1,
  outlet: blocked,
  volume_of_contents: 0.

/* Used to represent a blocked or terminated outlet. */

blocked instance_of pipes with
  inlet: dont_care,
  outlet: dont_care,
  flow_rate: 0.

/* ===================================================================== */
/* The rules required to run a simulation using the model defined above. */

/* Selects refractoriness as the conflict resolution strategy
and starts the simulation. */

rule init forward
  if
     start
  then
     remove start &
     strategy [refractoriness] &   /* ESSENTIAL NOT TO HAVE SPECIFICITY */
     add simulate.


rule stop forward
  if
     simulate &
     the volume_of_contents of Vessel is V &
     the height of Vessel is Height &
     the width of Vessel is Width &
     the depth of Vessel is Depth &
     prolog(Volume is Height * Width * Depth) &
     prolog(V >= Volume)
  then
   remove simulate &
   announce [Vessel,' full to overflowing with volume_of_contents: ', V, nl] &
   strategy [refractoriness, recency, specificity] & /* restore defaults */
   halt.

/* This rule will be applied to every tank in the model. Each firing of the */
/* rule represents a discrete time event in the model simulation for the */
/* tank which it acts upon. Using the existing contents of the tank, and */
/* flow rates into and out of the tank, it calculates the new volume of */
/* the tanks contents. It also displays this information on the terminal. */

rule calculate_new_volume forward
 if
   simulate &
   Vessel instance_of tanks &
   the volume_of_contents of Vessel is V &
   the inlet of Vessel is Inlet &
   the outlet of Vessel is Outlet &
   the flow_rate of Inlet is In_flow &
   the flow_rate of Outlet is Out_flow
  then
    prolog(New_vol is V + In_flow - Out_flow) &
    note the volume_of_contents of Vessel is New_vol &
/* next line retrieves just-noted value in case the change_rule demon
 associated with class 'vessel' knocks the number back down to
 prevent overflow.
 The 'prolog' is needed because it is doing a frame retrieval,
 which is NORMALLY done on the LEFT hand side of a rule, not the
 RIGHT!! */
    prolog(the volume_of_contents of Vessel is Adjusted_vol) &
    prolog(Vol_in_litres is Adjusted_vol/1000) &
    announce ['Contents of ', Vessel, ': ', Vol_in_litres, ' litres .', nl].

/* Simple prolog predicate to return the smallest of two numbers */
/* Used by the float_drop demon of ballcocks and volume_of_contents demon */
/* of vessels */

smallest_of(N1,N2,N1) :-
        N1 < N2, !.

smallest_of(_,N2,N2).
