
#|----------------------------------------------------------------------------
Artificial Intelligence, Second Edition
Elaine Rich and Kevin Knight
McGraw Hill, 1991

This code may be freely copied and used for educational or research purposes.
All software written by Kevin Knight.
Comments, bugs, improvements to knight@cs.cmu.edu
----------------------------------------------------------------------------|#

#|----------------------------------------------------------------------------
	    CANDIDATE ELIMINATION ALGORITHM FOR VERSION SPACES
			     "vs.lisp"
----------------------------------------------------------------------------|#

;; Version space learning with the candidate elimination algorithm.
;; Concepts are represented as lists of attribute-value pairs.  See
;; bottom of file for examples.

(defvar *s-set*)
(defvar *g-set*)
(defvar *attribute-values*)
(defvar *most-general-concept* nil)


;; Function INITIALIZE-S-AND-G accepts a positive instance of the concept
;; to be learned, and initializes the S and G sets.

(defun initialize-s-and-g (first-positive-example)
  (setq *g-set* (list *most-general-concept*))
  (setq *s-set* (list (order-attributes first-positive-example)))
  (status-report))

;; Function ACCEPT-INSTANCE incrementally accepts positive (t) and negative
;; (nil) instances of the target concept.

(defun accept-instance (inst positive?)
  (if positive?
      (accept-positive-instance inst)
      (accept-negative-instance inst))
  (status-report))

(defun accept-positive-instance (inst)
  (setq *g-set* (remove-if #'(lambda (g-elt)
				 (not (more-general-than g-elt inst)))
			   *g-set*))
  (setq *s-set* (mapcar #'(lambda (s-elt)
				(generalize s-elt inst))
			*s-set*)))

(defun accept-negative-instance (inst)
  (setq *s-set* (remove-if #'(lambda (s-elt)
				(more-general-than s-elt inst))
			   *s-set*))
  (setq *g-set* (mapcan #'(lambda (g-elt)
			     (specialize g-elt inst))
			*g-set*)))

(defun status-report ()
  (format t "~%~%New S set: ~d. ~%~%" *s-set*)
  (format t "New G set: ~d. ~%~%" *g-set*)
  (when (equal *s-set* *g-set*)
     (format t "Done. Learned concept = ~d. ~%~%" (car *s-set*)))
  (when (or (null *s-set*) (null *g-set*))
     (format t "Inconsistent data. ~%~%"))))


;; Function MORE-GENERAL-THAN returns t iff concept C1 is a more general 
;; description than C2.

(defun more-general-than (c1 c2)
   (subsetp c1 c2 :test #'equal))

;; Function ORDER-ATTRIBUTES reorders the attributes in a concept to an 
;; alphabetical standard.

(defun order-attributes (c)
  (sort (copy-tree c)
	#'(lambda (attr1 attr2)
	     (string<
		(format nil "~d" (car attr1))
		(format nil "~d" (car attr2))))))

;; Function GENERALIZE returns the generalization of two concepts.

(defun generalize (s-elt inst)
   (order-attributes (intersection s-elt inst :test #'equal)))

;; Function SPECIALIZE returns a list of specializations of G-ELT that do 
;; not cover the negative instance INST.

(defun specialize (g-elt inst)
  (mapcar #'order-attributes
	  (let* ((attributes-to-specialize
		     (set-difference (all-attribute-names) 
                                     (mapcar #'car g-elt)
				     :test #'equal))
		(possible-specializations
		     (mapcan #'(lambda (attrib)
				  (mapcar #'(lambda (value)
						(cons (list attrib value)
						      g-elt))
					  (all-attribute-values attrib)))
			     attributes-to-specialize)))
	     (remove-if #'(lambda (potential-g-elt)
			      (or (not (some #'(lambda (s-elt)
						   (more-general-than
							potential-g-elt
							s-elt))
					     *s-set*))
				  (more-general-than potential-g-elt inst)))
		        (cons g-elt possible-specializations)))))


;; --------------------------------------------------------------------------
;;  EXAMPLE
;;
;;  Positive and negative examples of the concept "Japanese economy car".
;;  cf. page 467.

(defvar *johns-car*)     ; +
(defvar *marys-car*)     ; -
(defvar *herbs-car*)     ; +
(defvar *allens-car*)    ; -
(defvar *marvins-car*)   ; +


(setq *johns-car*
   '((origin japan) (mfr honda) (color blue) (decade 1980) (type economy)))

(setq *marys-car*
   '((origin japan) (mfr toyota) (color green) (decade 1970) (type sports)))

(setq *herbs-car*
   '((origin japan) (mfr toyota) (color blue) (decade 1990) (type economy)))

(setq *allens-car*    ; corrected from first printing (blue, not red)
   '((origin usa) (mfr chrysler) (color blue) (decade 1980) (type economy)))

(setq *marvins-car*
   '((origin japan) (mfr honda) (color white) (decade 1980) (type economy)))

(setq *attribute-values*
   '((origin japan usa britain germany italy)
     (mfr honda toyota ford chrysler jaguar bmw fiat)
     (color blue green red white)
     (decade 1950 1960 1970 1980 1990 2000)
     (type economy luxury sports)))

(defun all-attribute-values (attrib)
  (cdr (assoc attrib *attribute-values*)))

(defun all-attribute-names ()
  (mapcar #'car *attribute-values*))


;; To test, evaluate the following five expressions:
;;
;; (initialize-s-and-g *johns-car*)
;; (accept-instance *marys-car* nil)
;; (accept-instance *herbs-car* t)
;; (accept-instance *allens-car* nil)
;; (accept-instance *marvins-car* t)

