
/* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<  >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> */
/* file containing all structure definitions, most of which are shared
across more than one file */
/* <<<<<<<<<<<<<<<<<<<<<<<<<<<<<  >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> */


#include "env_defs.h"
#include "env_spec.h"
#include "animal_defs.h"
#include "../asm/asm_defs.h"

/* specification of structure containing a vector of pointers and a count to
be used to contain information for each area of the environment grid */
typedef struct env_square
{

  int physical_f_count;                 /* number of physical features in
										square */ 
  void *p_arr[NUM_FEATURE_TYPES];       /* pointers to features or NULLS */
  /* all the following are dependent on the type of feature in the square */
  double temp_factor;                    /* reduction in temperature effect */
  double perception_factor;              /* reduction in effectiveness of
                                        perception into square */
  double escape_factor;                  /* increase in likelihood of being
                                        able to escape predator */
} ENV_SQUARE;


/* specification of a structure containing a count and vectors for pointers
to features and corresponding array positions (for fast access to all
features of the same type) */
typedef struct feature_list
{

  int count;
  void *f_ptr[MAX_NUM_ANY_FEATURE_TYPE];
  int row[MAX_NUM_ANY_FEATURE_TYPE];
  int column[MAX_NUM_ANY_FEATURE_TYPE];
}FEATURE_LIST;


/* specifications of structures to be used for different environmental
features */
typedef struct cereal_food
{

  double conditions;             /* 0.2 to 1.0 */
  double value;                  /* 0.0 to 1.0 */
  double toxicity;               /* 0.0 to 1.0 */
  int toxicity_delay;           /* number of timesteps after which illhealth
                                takes effect */
}CEREAL_FOOD;

typedef struct cover
{

  double thickness;              /* 0.2 to 1.0 */
} COVER;

typedef struct dangerous_place
{

int x;
}DANGEROUS_PLACE;

typedef struct den
{

  int x;
}DEN;

typedef struct fruit_food
{

  double conditions;             /* 0.2 to 1.0 */
  int cycle_length;             /* in days (5-10) */
  double ripe_time;              /* in days (2-5) */
  int cycle_state;              /* in days (1-cycle_length) */
  double value;                  /* 0.0 to 1.0 */
  double toxicity;               /* 0.0 to 1.0 */
  int toxicity_delay;           /* number of timesteps after which illhealth
                                takes effect */
} FRUIT_FOOD;

typedef struct irrelevant
{

  double north_prob;
  double east_prob;
  double south_prob;
  double west_prob;

  int pack_leader;              /* `boolean' to denote whether instance is
                                leader of pack or not */

  int pack_number;              /* index to structure for pack information */

  struct irrelevant *next_ptr;	/* pointer to another instance of this
								feature in the square (if there is one) */
} IRRELEVANT;

typedef struct landmark
{

  int number;
} LANDMARK;

typedef struct mate
{

  int receptive;				/* can be TRUE or FALSE */
  int courted;					/* can be TRUE or FALSE */

  double north_prob;
  double east_prob;
  double south_prob;
  double west_prob;
   
  struct mate *next_ptr;		/* pointer to another instance of this
								feature in the square (if there is one) */
} MATE;

typedef struct predator_1
{

  double north_prob;
  double east_prob;
  double south_prob;
  double west_prob;

  int prev_animal_direction;    /* where the predator last perceived */
  int prev_animal_distance;     /* the animal to be */    

  double size;
  double damage;
  double persistence;
  double time_chasing;
  int animal_spotted;
  int chasing_animal;
  int hungry;

  int pack_leader;              /* `boolean' to denote whether instance is
                                leader of pack or not */

  int pack_number;              /* index to structure for pack information */

  struct predator_1 *next_ptr;	/* pointer to another instance of this
								feature in the square (if there is one) */
} PREDATOR_1;

typedef struct predator_2
{

  double north_prob;
  double east_prob;
  double south_prob;
  double west_prob;

  int prev_animal_direction;    /* where the predator last perceived */
  int prev_animal_distance;     /* the animal to be */    

  double damage;
  double persistence;
  double time_chasing;
  int animal_spotted;
  int chasing_animal;
  int hungry;

  struct predator_2 *next_ptr;	/* pointer to another instance of this
								feature in the square (if there is one) */
} PREDATOR_2;

typedef struct prey
{

  double north_prob;
  double east_prob;
  double south_prob;
  double west_prob;
  double value;
   
  struct prey *next_ptr;		/* pointer to another instance of this
								feature in the square (if there is one) */
} PREY;

typedef struct shade
{

  double thickness;              /* 0.2 to 0.8 */
} SHADE;

typedef struct water
{

  double toxicity;               /* 0.0 to 1.0 */
  double average_size;           /* 0.2 to 1.0 */
  double value;                  /* 0.0 to 1.0 */
  int toxicity_delay;           /* number of timesteps after which illhealth
                                takes effect */
} WATER;

typedef struct animal
{

  struct animal *next_ptr;		/* pointer to another instance of this
								feature in the square (if there is one) */
} ANIMAL;


/* structures to contain information about packs */
typedef struct irrelevant_pack_struct
{

  int pack_leader_c;
  int pack_leader_r;

  int num_pack_in_env;    /* number of pack members that are still in
                                the environment */

  int pack_leader_in_env; /* `boolean' to show whether the pack leader
                                is still in the environment */
} IRRELEVANT_PACK_STRUCT;

typedef struct predator_1_pack_struct
{

  int pack_leader_c;
  int pack_leader_r;

  int num_pack_in_env;    /* number of pack members that are still in
                                the environment */

  int pack_leader_in_env; /* `boolean' to show whether the pack leader
                                is still in the environment */
} PREDATOR_1_PACK_STRUCT;


/* structure to form part of a linked list for perception */
typedef struct sqr_c_lle
{

  int f_num;    		/* position of this feature in perc_fs */

  struct sqr_c_lle *next_ptr; 	/* pointer to next instance of this struct */
}SQR_C_LLE;


/* structure to contain perceivable information about a feature */
typedef struct perc_feature
{

  int c;
  int r;
  int nft;
  int number;
  int courted;
  int receptive;

  double value;
  double thickness;
  double conditions;
}PERC_FEATURE;


/* definition of structure containing everything that the animal can perceive*/
typedef struct perc
{
        
  /* counter for number of features copied into the perception structure
  (after perception module run will equal number of perceived features) */
  int perc_f_count;

  /* list of features and associated properties */
  PERC_FEATURE perc_fs[MAX_NUM_PERC_FEATURES];

  /* array of indexs to instances of feature types in perc_fs */
  int feature_instance_indexs[NUM_FEATURE_TYPES][MAX_NUM_SINGLE_PERC_FEATURE];

  /* vector of counters for each type of feature (counted over everything
  that can be perceived, not just one square) */
  int f_counter[NUM_FEATURE_TYPES];

  /* 1 int for each type of feature in each square.  Each int is counter of
  number of instances of that type in the square */
  int counters[2*M_A_P+1][2*M_A_P+1][NUM_FEATURE_TYPES];

  /* correct internal and external perception probabilities for each
  perceived square */
  double ext_perc_probs[2*M_A_P+1][2*M_A_P+1];
  double int_perc_probs[2*M_A_P+1][2*M_A_P+1];

  /* beginning of linked list for each type of feature in each square */
  SQR_C_LLE *contents[2*M_A_P+1][2*M_A_P+1][NUM_FEATURE_TYPES];
} PERC;


/* structure to hold details of an individual instance of toxicity */
typedef struct toxic_inst
{

  int delay;		/* time before it takes effect */
  double strength;	/* how harmful the toxicity will be */
  int type;			/* number designating what was eaten/drunk to cause this*/
} TOXIC_INST;


/* the animal's memory of a particular feature in a particular location.
The type of feature is not mistakable.  The position and values of feature
attributes are only known approximately, but with increasing accuracy as the
instance is visited more and more often */
typedef struct place_memory
{

  int nft;
  int num_recognitions;
  double memory_strength;
  double variance;
  double est_c;
  double est_r;

  int est_number;
  double est_value;
  double est_thickness;
  double est_conditions;
  double est_toxicity;
  int time_since_consumed;
  int time_since_unsuccessful_visit;
} PLACE_MEMORY;

/* structure to hold work information to be sent out to slaves */
typedef struct work_struct
{

  int indvdl_number;
  int slave_type;
} WORK_STRUCT;

/* structure to hold information to be sent back by slaves to master */
typedef struct gf_struct
{

  int indvdl_number;
  int slave_type;
  double gf;
  double lifespan;
  char slave_name[50];
} GF_STRUCT;
