/*
 * partition.cpp
 *
 *  Created on: Aug 30, 2011
 *      Author: Tony
 */

#include "partition.h"
#include <algorithm>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static char base64map[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";


//---------------------------------------------------------------
// Partition
partition::partition(const std::vector<bool>& pvec) {
	this->parts = pvec;
}

partition::partition(const partition& part) {
	this->parts = part.parts;
}

partition::~partition() {
}

const std::string partition::getBase64String() {
	int n = this->parts.size()/6 + (this->parts.size() % 6 > 0 ? 1 : 0);
	char* pBase64Str;
	pBase64Str = (char*)malloc(sizeof(char)*(n+1));
	memset(pBase64Str, 0, n+1);

	int base64key = 0;
	for (int i = 0; i < (int)this->parts.size(); ++i) {

		base64key = base64key << 1;
		base64key += (this->parts[i] ? 1 : 0);

		if (i % 6 == 5 || i+1 == (int)this->parts.size()) {
			pBase64Str[i/6] = base64map[base64key];
			base64key = 0;
		}
	}

	std::string base64string(pBase64Str);
	free(pBase64Str);

	return base64string;
}


// Obtain base 2 string
const std::string partition::getBase2String() {
	char* pBase2Str;
	pBase2Str = (char*)malloc(sizeof(char)*(this->parts.size()+1));
	memset(pBase2Str, 0, this->parts.size()+1);

	for (int i = 0; i < (int)this->parts.size(); ++i) {
		pBase2Str[i] = (this->parts[i]) ? (char)'1' : (char)'0';
	}
	std::string base2string(pBase2Str);
	free(pBase2Str);

	return base2string;
}

// Return size of the partition
size_t partition::size() {
	return this->parts.size();
}

// Return the indices set of the smaller partition of the bipartition
std::vector<int> partition::getSmallPartition() {
	std::vector<int> p1;
	std::vector<int> p0;

	for (int i = 0; i < (int)this->parts.size(); ++i) {
		if (parts[i]) {
			p1.push_back(i);
		} else {
			p0.push_back(i);
		}
	}

	if (p1.size() < p0.size()) {
		return p1;
	} else {
		return p0;
	}
}

// Return whether the partition is null
bool partition::isNull() const {
	return this->parts.size() == 0;
}

// Return whether the partition is null or empty.  E.g. every bit is the same
bool partition::isEmpty() const {
	bool isEmpty = isNull();

	if (!isEmpty) {
		bool bit = parts[0];
		for (int i = 0; i < this->parts.size(); ++i) {
			if (parts[i] != bit) {
				isEmpty = 0;
				break;
			}
		}
	}

	return isEmpty;
}

// Return whether the partition is the same or not
bool partition::isEqual(const partition& part) const {
	bool isSame = 1;

	if (!isNull()) {
		if (part.parts.size() != this->parts.size()) {
			isSame = 0;
		} else {
			bool sign = (this->parts[0] == part.parts[0]);
			for (int i = 0; i < (int)this->parts.size(); ++i) {
				if (sign) {
					if (this->parts[i] != part.parts[i]) {
						isSame = 0;
						break;
					}
				} else {
					if (this->parts[i] == parts[i]) {
						isSame = 0;
						break;
					}
				}
			}
		}
	} else {
		isSame = part.isNull();
	}

	return isSame;
}

// Get
const bool partition::get (unsigned i) const {
	return this->parts.at(i);
}

// Set
void partition::set(unsigned i, bool val) {
	this->parts[i] = val;
}

/// Overriding == operator
bool partition::operator ==(const partition& b) {

	bool eq = 1;
	bool sign = this->parts[0] == b.get(0);
	for (int i = 0; i < (int)this->parts.size(); ++i) {
		if (sign && this->parts[i] != b.get(i)) {
			eq = 0;
			break;
		} else if (!sign && this->parts[i] == b.get(i)) {
			eq = 0;
			break;
		}
	}
	return eq;
}

//--------------------------------------------------------------------
// Model Partition
model_partition::model_partition(const partition& partion, int weight) :
	partition(partion),
	weight(weight)
{
}

model_partition::~model_partition() {
}

int model_partition::getWeight() {
	return this->weight;
}

//--------------------------------------------------------------------
// Masked Model Partition
masked_model_partition::masked_model_partition(const partition& part, int weight, const std::vector<bool>& mask) :
	model_partition(part, weight),
	mask(mask)
{
	for (int i = 0; i < (int)mask.size(); ++i) {
		if (mask[i]) {
			pmap.push_back(i);
		}
	}
}

masked_model_partition::~masked_model_partition() {
}
//
//// Obtain base 64 string
//const std::string masked_model_partition::getBase64String() {
//	if (this->mask.size() > 0) {
//		int n = this->pmap.size()/6 + (this->pmap.size() % 6 > 0 ? 1 : 0);
//		char* pBase64Str;
//		pBase64Str = (char*)malloc(sizeof(char)*(n+1));
//		memset(pBase64Str, 0, n+1);
//
//		int base64key = 0;
//		for (int i = 0; i < (int)this->pmap.size(); ++i) {
//
//			base64key = base64key << 1;
//			base64key += (this->parts[pmap[i]] ? 1 : 0);
//
//			if (i % 6 == 5 || i+1 == (int)this->pmap.size()) {
//				pBase64Str[i/6] = base64map[base64key];
//				base64key = 0;
//			}
//		}
//
//		std::string base64string(pBase64Str);
//		free(pBase64Str);
//
//		return base64string;
//	} else {
//		return model_partition::getBase64String();
//	}
//}
//
//// Obtain base 2 string
//const std::string masked_model_partition::getBase2String() {
//	if (mask.size() > 0) {
//		char* pBase2Str;
//		pBase2Str = (char*)malloc(sizeof(char)*(this->pmap.size()+1));
//		memset(pBase2Str, 0, this->pmap.size()+1);
//
//		for (int i = 0; i < (int)this->pmap.size(); ++i) {
//			pBase2Str[i] = (this->parts[pmap[i]]) ? (char)'1' : (char)'0';
//		}
//		std::string base2string(pBase2Str);
//		free(pBase2Str);
//
//		return base2string;
//	} else {
//		return model_partition::getBase2String();
//	}
//}

// Get boolean
const bool masked_model_partition::get(unsigned i) const {
	if (mask.size() > 0) {
		return model_partition::get(pmap[i]);
	} else {
		return model_partition::get(i);
	}
}

// Set boolean
void masked_model_partition::set(unsigned i, bool val) {
	if (mask.size() > 0) {
		model_partition::set(pmap[i], val);
	} else {
		model_partition::set(i, val);
	}
}

// Return size of the partition
size_t masked_model_partition::size() {
	if (mask.size() > 0) {
		return pmap.size();
	} else {
		return model_partition::size();
	}
}

// Get Mask
const std::vector<bool> masked_model_partition::getMask() {
	return this->mask;
}

// Has Mask
bool masked_model_partition::hasMask() {
	return mask.size() > 0;
}

//---------------------------------------------------------------------------
// Model Partition Set

// Helpers
bool compareParts(model_partition* p1, model_partition* p2) {
	return p1->getWeight() > p2->getWeight();
}

masked_model_partition_set::masked_model_partition_set(const std::vector<bool>& mask) :
	mask(mask)
{
}

masked_model_partition_set::~masked_model_partition_set() {
	for (int i = 0; i < (int)this->models.size(); ++i) {
		delete models[i];
	}
	models.empty();
}

// Add partition to set
void masked_model_partition_set::add_model_partition(partition* pParts, int weight) {
	this->models.push_back(new model_partition(*pParts, weight));
}

// Get partition index
model_partition* masked_model_partition_set::getPartition(int idx) {
	return this->models[idx];
}

// Get size of the models
size_t masked_model_partition_set::size() {
	return this->models.size();
}

// Set null partition
void masked_model_partition_set::setNullWeight(int nullWt) {
	this->nullWeight = nullWt;
}

// Get null partition weight
int masked_model_partition_set::getNullWeight() {
	return this->nullWeight;
}

// Return output string
const std::string masked_model_partition_set::toString() {
	std::string str;
	int n;
	char buffer[50];

	for (int i = 0; i < (int)mask.size(); ++i) {
		str.append(mask[i] ? "1" : "0");
	}
	str.append("\n");

	for (int i= 0; i < (int)models.size(); ++i) {
		n = sprintf(buffer, "%d ", this->models[i]->getWeight());
		str.append(buffer, n);
	}
	n = sprintf(buffer, "%d\n", getNullWeight());
	str.append(buffer,n);

	// output data
	for (int i = 0; i < (int)models.size(); ++i) {
		str.append(models[i]->getBase2String().data());
		str.append("\n");
	}

	return str;
}

// Return a list of sorted models
void masked_model_partition_set::sort() {
	std::sort(models.begin(), models.end(), compareParts);
}

// Mask
const std::vector<bool> masked_model_partition_set::getMask() {
	return this->mask;
}
