/*
 * admixture_model.h
 *
 *  Created on: Sep 6, 2011
 *      Author: Tony
 */

#ifndef ADMIXTURE_MODEL_H_
#define ADMIXTURE_MODEL_H_

// Include
#include <string>
#include <vector>

// Forward declaration
class masked_model_partition_set;

class admixture_hyperparameter {
public:
	//------------------------------------------------------------------
	// Enum
	enum Type { Time, Admixture, Theta };

public:
	//------------------------------------------------------------------
	// Members
	int eventIdx;
	Type eventType;
	double sigma;
	double sigma_min;
	double sigma_max;

public:
	//------------------------------------------------------------------
	// Constructor
	admixture_hyperparameter();
	admixture_hyperparameter(int eventIdx, Type eventType, double sigma, double sigma_min, double sigma_max);
	admixture_hyperparameter(const admixture_hyperparameter& hyperparameters);
};

class admixture_hyperparameters {
public:
	//------------------------------------------------------------------
	// Members
	double error_width;
	admixture_hyperparameter theta;
	std::vector<admixture_hyperparameter*> time;
	std::vector<admixture_hyperparameter*> alpha;

public:
	//------------------------------------------------------------------
	// Constructors
	admixture_hyperparameters();
	admixture_hyperparameters(const admixture_hyperparameters& hyperparameters);
};



class event {
public:
	//------------------------------------------------------------------
	// Members
	double t;
	double alpha;
	int p1;
	int p2;
	int pa;

public:
	//------------------------------------------------------------------
	// Constructors
	event(double t, double alpha, int p1, int p2, int pa);
	event(const event& e);

public:
	//------------------------------------------------------------------
	// operators
	event& operator=(const event& e);
};


class admixture_parameters {
protected:
	//------------------------------------------------------------------
	// Members

	// theta = N*mu*l
	double theta;

	// evolutionray events
	std::vector<event*> events;

public:
	//------------------------------------------------------------------
	// Constructors
	admixture_parameters();
	admixture_parameters(int nEvents);
	admixture_parameters(const admixture_parameters& params);
	virtual ~admixture_parameters();

public:
	//------------------------------------------------------------------
	// Operators

	// Copies admixture parameters
	void copy(admixture_parameters* pRef);

	//------------------------------------------------------------------
	// Methods


	// Add Event to the admixture model
	void addEvent(const event& e);

	// Get idx-th event
	event* getEvent(int idx);

	// Get the number of events in the model
	size_t getNumberOfEvent();

	// Set theta
	void setTheta(double theta);

	// Get theta
	double getTheta();

	// Return a string describing the parameters
	const std::string toString();

	// Return a simple string containing time alpha p1 p2 pa
	const std::string toStringSimple();
};

class admixture_output {
public:
	//------------------------------------------------------------------
	// Members
	//double theta;
	//std::vector<double> time;
	//std::vector<double> alpha;
	std::vector<event*> events;
	double theta;

public:
	//------------------------------------------------------------------
	// Constructors
	admixture_output(admixture_parameters* pParams);
	admixture_output(int n);
};

class admixture_data {
private:
	//------------------------------------------------------------------
	// Members
	std::vector<int> weights;
	std::vector<int> pop_assign;
	std::vector<masked_model_partition_set*> models;
	int k;


public:
	//------------------------------------------------------------------
	// Constructors
	admixture_data(int k, const std::vector<int>& pop_assign, const std::vector<masked_model_partition_set*>& models_set);
	virtual ~admixture_data();

public:
	//------------------------------------------------------------------
	// Methods

	// Get the idx-th model bipartition
	masked_model_partition_set* getModelSet(int idx);

	// Get the number of model bipartition in the model
	size_t getNumberOfModelSets();

	// Get population assignments
	const std::vector<int> getPopAssignment();

	// Get the number of populations in the dataset
	int getNumOfPopulation();

	// Get the number of individual in population i;
	int getNumOfIndidivuals(int pop);
};

class admixture_model_event {
public:
	//------------------------------------------------------------------
	// Members
	std::vector<int> p1_candidates;
	std::vector<int> p2_candidates;
	std::vector<int> pa_candidates;

public:
	//------------------------------------------------------------------
	// Constructors
	admixture_model_event();
	admixture_model_event(const admixture_model_event& e);

public:
	//------------------------------------------------------------------
	// Methods

	// Add potential candidate to P1
	void addCandidateToP1(int popid);

	// Add potential candidate to P1
	void addCandidateToP2(int popid);

	// Add potential candidate to P1
	void addCandidateToPa(int popid);

	// Return whether this event can also be a divergence event
	bool isNotAdmixable();
};

class admixture_model {
private:
	//------------------------------------------------------------------
	// Members

	// Admixture input data
	admixture_data* pData;

	// Set of admixture model events
	std::vector<admixture_model_event*> events;


public:
	//------------------------------------------------------------------
	// Constructor
	admixture_model(int k, const std::vector<int>& pop_assign, const std::vector<masked_model_partition_set*>& model_sets);
	virtual ~admixture_model();

public:
	//------------------------------------------------------------------
	// Methods

	// add Model event
	void addModelEvent(admixture_model_event* event);

	// return admixture model event
	admixture_model_event* getModelEvent(int i);

	// return the number of events
	int getNumberOfEvents();

	// Gets admixure data
	admixture_data* getData();

	// Output string
	const std::string toString();
};
#endif /* ADMIXTURE_MODEL_H_ */
