/*
 * admixture_model.cpp
 *
 */

// My Include
#include "admixture_model.h"

// Includes
#include <stdio.h>
#include "partition.h"

//------------------------------------------------------------------
// Hyperparameter
admixture_hyperparameter::admixture_hyperparameter(int eventIdx, Type eventType, double sigma, double sigma_min, double sigma_max) :
	eventIdx(eventIdx),
	eventType(eventType),
	sigma(sigma),
	sigma_min(sigma_min),
	sigma_max(sigma_max)
{

}
admixture_hyperparameter::admixture_hyperparameter(const admixture_hyperparameter& hyperparameters) :
	eventIdx(hyperparameters.eventIdx),
	eventType(hyperparameters.eventType),
	sigma(hyperparameters.sigma),
	sigma_min(hyperparameters.sigma_min),
	sigma_max(hyperparameters.sigma_max)
{

}
admixture_hyperparameter::admixture_hyperparameter() {
}


//------------------------------------------------------------------------
// Admixture Hyperparameters
admixture_hyperparameters::admixture_hyperparameters() {

}


admixture_hyperparameters::admixture_hyperparameters(const admixture_hyperparameters& hyperparameters) :
	error_width(hyperparameters.error_width),
	theta(hyperparameters.theta)
{
	for (int i = 0; i < (int)hyperparameters.time.size(); ++i) {
		this->time.push_back(new admixture_hyperparameter(*hyperparameters.time[i]));
	}
	for (int i = 0; i < (int)hyperparameters.alpha.size(); ++i) {
		this->alpha.push_back(new admixture_hyperparameter(*hyperparameters.alpha[i]));
	}

}

//------------------------------------------------------------------------
// Events

event::event(double t, double alpha, int p1, int p2, int pa) :
	t(t),
	alpha(alpha),
	p1(p1),
	p2(p2),
	pa(pa) {
}

event::event(const event& e) :
	t(e.t),
	alpha(e.alpha),
	p1(e.p1),
	p2(e.p2),
	pa(e.pa) {
}

event& event::operator=(const event& e) {
	this->t = e.t;
	this->alpha = e.alpha;
	this->p1 = e.p1;
	this->p2 = e.p2;
	this->pa = e.pa;
	return *this;
}

//------------------------------------------------------------------------
// Admixture Parameters
admixture_parameters::admixture_parameters() {
}

admixture_parameters::admixture_parameters(int nEvents) :
	events(nEvents)
{
	for (int i = 0; i < (int)nEvents; ++i) {
		events[i] = new event(0,0,0,0,0);
	}
}

admixture_parameters::admixture_parameters(const admixture_parameters& params) :
	theta(params.theta),
	events(params.events.size())
{
	for (int i = 0; i < (int)params.events.size(); ++i) {
		events[i] = new event(*params.events[i]);
	}
}

admixture_parameters::~admixture_parameters() {
	for (int i = 0; i < (int)events.size(); ++i) {
		delete events[i];
	}
	events.clear();
}

// Copies admixture parameters
void admixture_parameters::copy(admixture_parameters* pRef) {
	setTheta(pRef->getTheta());
	for (int j = 0; j < (int)pRef->events.size(); ++j) {
		if ((int)events.size() < j+1) {
			this->addEvent(*pRef->events[j]);
		}
		*events[j] = *pRef->events[j];
	}
}


void admixture_parameters::addEvent(const event& e) {
	event* pEvent = new event(e);
	events.push_back(pEvent);
}

// Get idx-th event
event* admixture_parameters::getEvent(int idx) {
	if (idx < (int)events.size()) {
		return events[idx];
	} else {
		return NULL;
	}
}

// Get the number of events in the model
size_t admixture_parameters::getNumberOfEvent() {
	return events.size();
}

// Set theta
void admixture_parameters::setTheta(double theta) {
	this->theta = theta;
}

// Get theta
double admixture_parameters::getTheta() {
	return theta;
}

// Return a string describing the parameters
const std::string admixture_parameters::toString() {
	std::string str;
	char buffer[50];
	int n;
	for (int i = 0; i < (int)this->events.size(); ++i) {
		if (i > 0) {
			str.append("->");
		}
		n = sprintf(buffer,"[%.2f,%.2f:%d<-%d", events[i]->t, events[i]->alpha, events[i]->p1, events[i]->pa);
		str.append(buffer, n);
		if (events[i]->p2 >= 0) {
			n = sprintf(buffer, "->%d", events[i]->p2);
			str.append(buffer, n);
		}
		str.append("]");
	}
	n = sprintf(buffer, " Theta:%.1f", this->theta);
	str.append(buffer,n );

	return str;
}


// Return a simple string containing time alpha p1 p2 pa
const std::string admixture_parameters::toStringSimple() {
	std::string str;
	char buffer[50];
	int n;
	for (int i = 0; i < (int)this->events.size(); ++i) {
		n = sprintf(buffer,"%.2f %.2f %d %d %d ", events[i]->t, events[i]->alpha, events[i]->p1, events[i]->p2, events[i]->pa);
		str.append(buffer, n);
	}
	return str;
}

//------------------------------------------------------------------------
// Admixture Output
admixture_output::admixture_output(admixture_parameters* pParams) :
	events(std::vector<event*>(pParams->getNumberOfEvent())),
	theta(pParams->getTheta())
{
	for (int i = 0; i < (int)pParams->getNumberOfEvent(); ++i) {
		events[i] = new event(*pParams->getEvent(i));
	}
}

admixture_output::admixture_output(int n) :
	events(std::vector<event*>(n))
{
	for (int i = 0; i < n; ++i) {
		events[i] = new event(0.0, 0.0, -1, -1, -1);
	}
}
//------------------------------------------------------------------------
// Admixture Data

admixture_data::admixture_data(int k, const std::vector<int>& pop_assign, const std::vector<masked_model_partition_set*>& models_set) :
	pop_assign(pop_assign),
	models(models_set),
	k(k){
}

admixture_data::~admixture_data() {
	for (int i = 0; i < (int)models.size(); ++i) {
		delete models[i];
	}
	models.clear();
}

// Get the idx-th model bipartition
masked_model_partition_set* admixture_data::getModelSet(int idx) {
	if (idx < (int)models.size()) {
		return models[idx];
	} else {
		return NULL;
	}
}

// Get the number of model bipartition in the model
size_t admixture_data::getNumberOfModelSets() {
	return models.size();
}

// Get population assignments
const std::vector<int> admixture_data::getPopAssignment() {
	return this->pop_assign;
}

// Get the number of populations in the dataset
int admixture_data::getNumOfPopulation() {
	return k;
}

// Get the number of individual in population i;
int admixture_data::getNumOfIndidivuals(int pop) {
	int nind = 0;
	for (int i = 0; i < (int)pop_assign.size(); ++i) {
		if (pop_assign[i] == pop) {
			++nind;
		}
	}
	return nind;
}

//------------------------------------------------------------------
// Constructors
admixture_model_event::admixture_model_event()
{
}

admixture_model_event::admixture_model_event(const admixture_model_event& e)
{
}

// Add potential candidate to P1
void admixture_model_event::addCandidateToP1(int popid) {
	this->p1_candidates.push_back(popid);
}

// Add potential candidate to P1
void admixture_model_event::addCandidateToP2(int popid) {
	this->p2_candidates.push_back(popid);
}

// Add potential candidate to P1
void admixture_model_event::addCandidateToPa(int popid) {
	this->pa_candidates.push_back(popid);
}

// Return whether this event can also be a divergence event
bool admixture_model_event::isNotAdmixable() {

	bool isNotAdmixable = 0;
	for (int i = 0; i < (int)p2_candidates.size(); ++i) {
		if (p2_candidates[i] == -1) {
			isNotAdmixable = 1;
			break;
		}
	}
	return isNotAdmixable;
}

//------------------------------------------------------------------
// Admixture Model
admixture_model::admixture_model(int k, const std::vector<int>& pop_assign, const std::vector<masked_model_partition_set*>& model_sets)
{
	pData = new admixture_data(k, pop_assign, model_sets);
}

admixture_model::~admixture_model() {
	delete pData;
}

// return whether we know what populations were involved
void admixture_model::addModelEvent(admixture_model_event* event) {
	this->events.push_back(event);
}


// return admixture model event
admixture_model_event* admixture_model::getModelEvent(int i) {
	if ((int)events.size() > i) {
		return events[i];
	}
	return NULL;
}

// return the number of events
int admixture_model::getNumberOfEvents() {
	return (int)this->events.size();
}

// Gets admixure data
admixture_data* admixture_model::getData() {
	return pData;
}


// Output string
const std::string admixture_model::toString() {
	char buffer[10];
	std::string str;
	str.append("------------------------------------\nEvents:\n");
	for (int i = 0; i < this->getNumberOfEvents(); ++i) {
		str.append("[P1={");
		for (int j = 0; j < (int)this->events[i]->p1_candidates.size(); ++j) {
			int n = sprintf(buffer, "%d", this->events[i]->p1_candidates[j]);
			if (j > 0) { str.append(","); }
			str.append(buffer, n);
		}
		str.append("}, P2={");
		for (int j = 0; j < (int)this->events[i]->p2_candidates.size(); ++j) {
			int n = sprintf(buffer, "%d", this->events[i]->p2_candidates[j]);
			if (j > 0) { str.append(","); }
			str.append(buffer, n);
		}
		str.append("}, PA={");
		for (int j = 0; j < (int)this->events[i]->pa_candidates.size(); ++j) {
			int n = sprintf(buffer, "%d", this->events[i]->pa_candidates[j]);
			if (j > 0) { str.append(","); }
			str.append(buffer, n);
		}
		str.append("}]\n");
	}
	str.append("Population Assignments\n");
	for (int i = 0; i < this->pData->getNumOfPopulation(); ++i) {
		int n = sprintf(buffer, "%d: ", i);
		str.append(buffer, n);
		bool first = 1;
		for (int j = 0; j < (int)this->pData->getPopAssignment().size(); ++j) {
			if (this->pData->getPopAssignment()[j] == i) {
				if (!first) {
					str.append(",");
				} else {
					first = 0;
				}
				n = sprintf(buffer, "%d", j);
				str.append(buffer, n);
			}
		}
		str.append("\n");
	}
	str.append("------------------------------------\n");
	return str;
}
