/*
 * coalescence_algorithm.cpp
 *
 */

// Standard Include
#include "../cleax.h"

// My Include
#include "coalescence_algorithm.h"

// Standard Includes
#include <assert.h>
#include <limits>
#include <stdio.h>
#include <stdlib.h>
#include "../dmo/admixture_model.h"
#include "../dmo/genealogy.h"
#include "../dmo/partition.h"

//-----------------------------------------------------------------------------
// Constructors
coalescence_algorithm::coalescence_algorithm(const std::vector<int>& pop_assign, int num_pop) :
	pop_assign(pop_assign),
	num_pop(num_pop)
{
}

coalescence_algorithm::~coalescence_algorithm() {
}

//-----------------------------------------------------------------------------
// Methods

genealogy* coalescence_algorithm::generate_genalogy_fixed(admixture_parameters* pParams) {

	// pools = array of lineages belonging to the same population
	// pools_id[i] = id number for the i-th lineage pools
	// pools_idx[i] = last index of the i-th lineage pools
	// k = number of population at i-th event
	// n = number of lineages

	int k = num_pop;
	std::vector<std::vector<genealogy_branch*> > pools(k);
	std::vector<int> pools_idx(k, 0);
	std::vector<int> pools_id(k, 0);
	std::vector<double> times(k);
	int n = (int) pop_assign.size();
	int numEvents = pParams->getNumberOfEvent();

	genealogy* pGenealogy = new genealogy(2*n-1);

	for (int i = 0; i < k; ++i) {
		pools[i] = std::vector<genealogy_branch*>(n*k);
		pools_id[i] = i;
	}

	for (int i = 0; i < n; ++i) {
		genealogy_branch* pBranch = new genealogy_branch(i, n, pop_assign[i], 0);
		pools[pop_assign[i]][pools_idx[pop_assign[i]]] = (pBranch);
		pools_idx[pop_assign[i]]++;
	}

	for (int i = 0; i < numEvents+1; ++i) {
		double time1 = (i == 0) ? 0.0 : pParams->getEvent(i-1)->t;
		double time2 = (i < numEvents) ?  pParams->getEvent(i)->t : std::numeric_limits<double>::max();
		for (int j = 0; j < k; ++j) {
			fflush(stdout);
			int idx = pools_id[j];
			int n = pools_idx[idx];
			std::vector<double> time_vec(coalesce_times_fixed(time2-time1, n));

			if (isDebug) {
				printf("%d coalescence in iterval %d (%f, %f) for population %d\n", (int)time_vec.size(), i, time1, time2, idx);
				printf("[ ");
				for (int l = 0; l < (int)time_vec.size(); ++l) {
					printf("%.3f ", (time_vec[l]));
				}
				printf("] \n%d\n", pools_idx[idx]);
				fflush(stdout);
			}

			for (int l = 0; l < (int)time_vec.size(); ++l) {
				// randomly select
				int idx1 = rand() % pools_idx[idx];
				int idx2 = rand() % (pools_idx[idx]-1);
				idx2 = (idx2 >= idx1) ? idx2+1 : idx2;
				genealogy_branch* pChild1 = pools[idx][idx1];
				genealogy_branch* pChild2 = pools[idx][idx2];

				if (isDebug) {
					printf("ABCD");
					fflush(stdout);
					printf("%d=%d:%d, %d:%d\n", idx, idx1, pChild1->getUID(), idx2, pChild2->getUID());
					fflush(stdout);
				}

				double t = time1+time_vec[l];
				genealogy_branch* pParent = new genealogy_branch(pChild1, pChild2, t, idx, i);
				pChild1->setEndTime(t);
				pChild2->setEndTime(t);

				if (pChild1->getEndTime() <= pChild1->getStartTime()) {
					printf("Something is not right here (%f-%f) \n", pChild1->getEndTime()*40000, pChild1->getStartTime()*40000);
					fflush(stdout);
				}
				if (pChild2->getEndTime() <= pChild2->getStartTime()) {
					printf("Something is not right here (%f-%f) \n", pChild2->getEndTime()*40000, pChild2->getStartTime()*40000);
					fflush(stdout);
				}

				pools_idx[idx]--;
				if (idx1 > idx2) {
					pools[idx][idx2] = pParent;
					pools[idx][idx1] = pools[idx][pools_idx[idx]];
				} else {
					pools[idx][idx1] = pParent;
					pools[idx][idx2] = pools[idx][pools_idx[idx]];
				}
				pGenealogy->addBranch(pChild1);
				pGenealogy->addBranch(pChild2);

				// DEBUG, make sure no duplicates
				if (isDebug) {
					std::vector<int> flag(2*(pop_assign.size())-1, 0);
					for (int r = 0; r < k; ++r) {
						int id = pools_id[r];
						for (int l = 0; l < pools_idx[id]; ++l) {
							int uid = pools[id][l]->getUID();
							if (flag[uid] > 0) {
								printf("Duplicate %d, %d, %d (UID: %d)\n", id, flag[uid], l, uid);
								fflush(stdout);
							}
							flag[uid] = l;
						}
					}
				}
			}

			if (isDebug) {
				printf("Lineages: %d\n", pools_idx[idx]);
			}
		}

		if (i < numEvents) {
			// Perform merging
			double alpha = pParams->getEvent(i)->alpha;
			for (int j = 0; j < pools_idx[pParams->getEvent(i)->pa]; ++j) {
				int assign = -1;
				if (alpha == 0) {
					assign = pParams->getEvent(i)->p2;
				} else if (alpha == 1) {
					assign = pParams->getEvent(i)->p1;
				} else {
					assign = (((double)rand())/RAND_MAX < alpha) ? pParams->getEvent(i)->p1 : pParams->getEvent(i)->p2;
				}
				pools[assign][pools_idx[assign]++] = pools[pParams->getEvent(i)->pa][j];
			}
			int rmIdx = -1;
			for (int j = 0; j < k; ++j) {
				if (pools_id[j] == pParams->getEvent(i)->pa) {
					rmIdx = j;
					break;
				}
			}
			k--;
			pools_id[rmIdx] = pools_id[k];

//			// DEBUG, make sure no duplicates
//			std::vector<int> flag(2*(pop_assign.size())-2, 0);
//			for (int j = 0; j < k; ++j) {
//				int id = pools_id[j];
//				for (int l = 0; l < pools_idx[id]; ++l) {
//					int uid = pools[id][l]->getUID();
//					if (flag[uid] > 0) {
//						printf("Duplicate %d, %d, %d (UID: %d)\n", flag[uid], id, l, uid);
//					}
//					flag[uid] = l;
//				}
//			}
		} else {
			int idx = pools_id[0];
			genealogy_branch* pRoot = pools[idx][0];
			pRoot->setEndTime(pRoot->getStartTime());
			pGenealogy->addBranch(pRoot);
		}
	}

	return pGenealogy;
}


// Returns a set of expected coalescence time between t1 and t2;
std::vector<double> coalescence_algorithm::coalesce_times_fixed(double t, int n) {

	int m = n;
	int k = 0;

	if (n > 1) {
		double ti = 0;
		while (ti <= t && n > 1) {
			ti += (1.0/((double)n*(double)(n-1)));
			k++;
			n--;
		}
		if (ti > t) {
			k--;
		}

		std::vector<double> time_vec(k, 0);
		ti = 0;
		for (int i = 0; i < k; ++i) {
			double tmp = (1.0/((double)(m-i)*(double)(m-i-1)));
			ti += tmp;
			time_vec[i] = ti;
		}

		return time_vec;
	} else {
		return std::vector<double>();
	}
}
