#!/usr/bin/env python

# Wiggle Circle Animator for Python v0.09
# Peter Barnum <pbarnum@cs.cmu.edu>
# 
# Copyright (c) 2010, Peter Barnum
# 
# Distributed under the MIT license:
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

# The SnakeManager holds various Snake groups, each of which is composed of
# various circles.

import math
import random

import pygame, sys, os
from pygame.locals import * 

class SnakeManager():
    
    def __init__(self, sharedData):
        self.sharedData = sharedData
        self.snakes = [Snake()]
        """newCircle = {'position':[200,300],
                     'radius':10,
                     'color':[255,0,0]}
        self.snakes[0].circles.append(newCircle)"""
        self.whichSnake = 0
        
    def drawSnakes(self, surface, phase):
        for ind, snake in enumerate(self.snakes):
            if ind == self.whichSnake:
                isThisSnakeSelected = True
            else:
                isThisSnakeSelected = False
                
            snake.draw(surface, phase, self.sharedData.paused, isThisSnakeSelected)
                
    
    def setCurSnake(self, whichSnake):
        if whichSnake < 0:
            whichSnake = 0
            
        if whichSnake+1 > len(self.snakes):
            self.snakes.append(Snake())
            whichSnake = len(self.snakes)-1
            
        self.whichSnake = whichSnake
    
    def addCircle(self, position):
        newCircle = {'position':position[:],
                     'radius':self.sharedData.newCircleRadius,
                     'color':self.sharedData.circleColor[:]}
        self.snakes[self.whichSnake].circles.append(newCircle)
        
    def removeCircle(self, position):
        curSnakeCircles = self.snakes[self.whichSnake].circles
        newSnakeCircles = []
        found = False
        for ind, circle in enumerate(curSnakeCircles):
            distance = math.sqrt(math.pow(position[0]-circle['position'][0],2)+\
                                 math.pow(position[1]-circle['position'][1],2))
            if found == False and distance <= circle['radius']:
                found = True
            else:
                newSnakeCircles.append(circle)
                
        self.snakes[self.whichSnake].circles = newSnakeCircles
        
    def getCurSnake(self):
        return self.snakes[self.whichSnake]
        
class Snake():
    def __init__(self):
        self.spatialFrequency = 50
        self.temporalFrequency = 5
        self.phaseOffset = 0
        self.amplitude = 50
        self.circles = []
        
    def draw(self, surface, phase, paused, isThisSnakeSelected):
        for circle in self.circles:
            #currentPhase = float(phase) - float(self.phaseOffset) + math.pi*circle['position'][0]/float(self.spatialFrequency)
            if self.spatialFrequency == 1000:
                currentPhase = float(phase) + float(self.phaseOffset)*math.pi/180.
            else:
                currentPhase = float(phase) + float(circle['position'][0])/float(self.spatialFrequency)/(.5*math.pi) + float(self.phaseOffset)*math.pi/180.
                
            currentPhase = (1-paused)*currentPhase
            location = [int(circle['position'][0]),
                        int(float(circle['position'][1])+\
                            float(self.amplitude)*math.sin(float(self.temporalFrequency)*currentPhase))]
            
            pygame.draw.circle(surface, circle['color'], location, int(circle['radius']), 0)
            
            if isThisSnakeSelected:
                pygame.draw.circle(surface, [0,0,0], location, int(circle['radius'])-1, 1)
                pygame.draw.circle(surface, [255,255,255], location, int(circle['radius']), 1)
                
    def __getstate__(self):
        odict = self.__dict__.copy()
        return odict

    def __setstate__(self, dict):
        self.__dict__.update(dict)
            