/** CFile **************************************************************

  FileName    [ specBasic.c ]

  PackageName [ spec ]

  Synopsis    [ Basic data manipulation routines ]

  Description [ This file contains the basic data manipulation
  routines for the type spec_Node_t: constructor, destructor,
  query and set fields. spec_Node_t is a pointer type to
  structures that hace a next field, a generic info field,
  a generic value field, and a string name field. 
  This data type is used to represent abbreviations and
  properties of a CV spec file. ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Log$ ]

***********************************************************************/

#include "specInt.h"

#include <stdarg.h>
#include <stdio.h>

static char rcsid [] = "$Log$";
/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/
typedef struct spec_NodeStruct_t_ {
  spec_Node_t next;
  pointer info;
  pointer value;
  char * name;
} spec_NodeStruct_t;

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
static rec_mgr _memory_manager;

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ spec_Node_t constructor ]
  Description [ Allocates a new node to represent the data parsed
  by the spec compiler ]
  SideEffects [ Memory is allocated via the mem(3) package, using the
  special purpose record manager _memory_manager, declared static. ]
  SeeAlso     [ spec_BasicFreeNode ]
 */
spec_Node_t
spec_BasicNew
(pointer info,
 char * name,
 spec_Node_t next,
 pointer value) 
{
  register spec_Node_t res;
  if (_memory_manager == 0) {
    _memory_manager = mem_new_rec_mgr(sizeof(struct spec_NodeStruct_t_));
  }
  res = (spec_Node_t) mem_new_rec(_memory_manager);
  res->next = next;
  res->info = info;
  res->value = value;
  res->name = name;
  return res;
}

/** Function **
  Synopsis    [ spec_Node_t destructor ]
  SideEffects [ Memory is disallocated. ]
  SeeAlso     [ spec_BasicNewNode ]
 */
void
spec_BasicFree
(spec_Node_t n)
{
  mem_free_rec(_memory_manager, n);
}

/** Function **
  Synopsis    [ Query info field ]
  SideEffects [ None. ]
 */
pointer
spec_BasicQInfo
(spec_Node_t n)
{ /* this field is aligned */
  return n->info;
}

/** Function **
  Synopsis    [ Query name field ]
  SideEffects [ None. ]
 */
char *
spec_BasicQName
(spec_Node_t n)
{
  return n->name;
}

/** Function **
  Synopsis    [ Query value field ]
  SideEffects [ None. ]
 */
spec_Node_t
spec_BasicQValue
(spec_Node_t n)
{
  return n->value;
}

/** Function **
  Synopsis    [ Query next field ]
  SideEffects [ None. ]
 */
spec_Node_t
spec_BasicQNext
(spec_Node_t n)
{
  return n->next;
}

/** Function **
  Synopsis    [ Sets info field ]
  SideEffects [ Writes on info field of n. ]
 */
void
spec_BasicSInfo
(spec_Node_t n,
 pointer info)
{
  n->info = info;
}

/** Function **
  Synopsis    [ Sets name field ]
  SideEffects [ Writes on name field of n. ]
 */
void 
spec_BasicSName
(spec_Node_t n,
 char * name)
{
  n->name = name;
}

/** Function **
  Synopsis    [ Sets next field ]
  SideEffects [ Writes next field of n. ]
 */
void
spec_BasicSNext
(spec_Node_t n,
 spec_Node_t next)
{
  n->next = next;
}

/** Function **
  Synopsis    [ Sets value field ]
  SideEffects [ Writes value field of n. ]
 */
void
spec_BasicSValue
(spec_Node_t n,
 pointer value)
{
  n->value = value;
}

/** Function **
  Synopsis    [ Appends l1 to l2. ]
  SideEffects [ Writes next field of last element of l1. ]
 */
spec_Node_t
spec_BasicAppend
(spec_Node_t l1,
 spec_Node_t l2)
{
  spec_Node_t ptr;
  if (l1 == 0) return l2;
  for (ptr = l1; spec_BasicQNext(ptr); ptr = spec_BasicQNext(ptr));
  spec_BasicSNext(ptr, l2);
  return l1;
}

/** Function **
  Synopsis    [ Display routine ]
  SideEffects [ Sends data to outstream ]
 */
void
spec_BasicDisplay
(FILE * outstream,
 spec_Node_t n)
{
  fprintf(outstream, "%s: ", spec_BasicQName(n));
  ctl_DisplayText(outstream, (ctl_term) spec_BasicQValue(n), 
                  (ctl_TerminalDisplayFn_t) & cvn_display);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

