/** CFile **************************************************************

  FileName    [ specAction.c ]

  PackageName [ spec ]

  Synopsis    [ Actions performed while parsing CV spec files. ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Log$ ]

***********************************************************************/

#include "specInt.h"

#include <stdlib.h>
#include <Errors.h>

static char rcsid [] = "$Log$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
static const tOperator spec_op_to_vhdl_op_tab[] =
  {kNot, 
   kLe, kAnd, kOr, kXor, kXnor, kNand, kNor,  /* implies = '<=' */
   kEq, kNeq, kLe, kLt, kGe, kGt,
   0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0};

static const ctl_Operator_t spec_op_to_ctl_op_tab[] =
  {ctl_Not, ctl_Implies, ctl_And, ctl_Or, ctl_Xor, ctl_Xnor,
   ctl_Nand, ctl_Nor, 0, 0, 0, 0, 0, 0,
   ctl_EX, ctl_AX, ctl_EF, ctl_AF, ctl_EF, ctl_AG,
   ctl_EU, ctl_AU, ctl_EW, ctl_AW, ctl_Stable};

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/
/* * Macro **
  Synopsis    [ Predicate on temporal operators ]
  Description [ Given any specification operator 
  (type spec_Operator_t) returns 1 if it is a temporal
  operator (EX, ...AW), 0 otherwise. ]
  SideEffects [ None ]
*/
#define /* int */ _tl_op_p(/* spec_Operator_t */ op) \
 (((op) >= spec_EX ? 1 : 0) || ((op) == spec_Implies))



/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int _type_compatible(cvn T1, cvn T2);
static cvn _declaration_get(char * id);
static cvn __declaration_get(cvn node, char * id);
static cvn _binary_vhdl(tOperator op, cvn l, cvn r, int * nberrors, tPosition pos);
static cvn _unary_vhdl(tOperator op, cvn arg, int * nberrors, tPosition pos);
static void _check_invariant(cvn expression, int * nberrors, tPosition pos);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Computes ctl_term attribute for binary expressions ]
  Description [ A CTL binary term is either any binary boolean or
  temporal operator applied to two CTL terms or a temporal operator 
  applied to any two terms. ]
  SideEffects [ Memory is allocated to create the result ]
  SeeAlso     [ spec_ActionBinaryVHDLTerm, ctl ]
 */
ctl_term
spec_ActionBinaryCTLTerm
(spec_Operator_t op /* the operator of the expression */,
 ctl_term ctl1 /* ctl_term attribute of first argument */,
 cvn vhdl1 /* vhdl_term attribute of first argument */,
 ctl_term ctl2 /* ctl_term attribute of second argument */,
 cvn vhdl2 /* vhdl_term attribute of second argument */,
 int * nberrors /* address of number of parsing errors so far */,
 tPosition pos       /* position of the expression in file */)
{
  ctl_Operator_t ctl_op;

  /* Returns 0 if there is no temporal logic */
  if (!ctl1 && !ctl2 && !_tl_op_p(op)) {
    return 0;
  }
  /* Builds the atomic expressions of the CTL term if necessary */
  if (!ctl1) {
    ctl1 = ctl_BasicNew(0, 0, ctl_AtomicTerm, (pointer) vhdl1);
  }
  if (!ctl2) {
    ctl2 = ctl_BasicNew(0, 0, ctl_AtomicTerm, (pointer) vhdl2);
  }
  /* Checks that spec operator correspond to a ctl operator */
  if ((ctl_op = spec_op_to_ctl_op_tab[(int) op]) == 0) {
    ++*nberrors;
    Message("Invalid operator on CTL terms", xxError, pos);
  }
  /* Builds and returns result */
  return ctl_BasicNew(0, 0, ctl_BinaryTerm, ctl_op, ctl1, ctl2);
}

/** Function **
  Synopsis    [ Computes vhdl_term attribute for binary expressions ]
  Description [ A VHDL binary expression is a binary vhdl operator 
  applied to two vhdl expressions. Exception: the implies operator
  is considered a vhdl operator although it is not one, it is treated
  as the lower or equal operator (FALSE le FALSE, FALSE le TRUE, 
  TRUE le TRUE). ]
  SideEffects [ Memory is allocated to create the result ]
  SeeAlso     [ spec_ActionBinaryVHDLTerm ]
 */
cvn
spec_ActionBinaryVHDLTerm
(spec_Operator_t op /* the operator of the expression */,
 ctl_term ctl1 /* ctl_term attribute of first argument */,
 cvn vhdl1 /* vhdl_term attribute of first argument */,
 ctl_term ctl2 /* ctl_term attribute of second argument */,
 cvn vhdl2 /* vhdl_term attribute of second argument */,
 int * nberrors /* address of number of parsing errors so far */,
 tPosition pos /* position of the expression in file */)
{
  tOperator vhdl_op;
  /* Returns 0 if there is temporal logic */
  if (ctl1 || ctl2 || _tl_op_p(op)) {
    return 0;
  }
  /* Checks that spec operator correspond to a ctl operator */
  if ((vhdl_op = spec_op_to_vhdl_op_tab[(int) op]) == 0) {
    ++*nberrors;
    Message("Invalid operator in VHDL expression", xxError, pos);
  }
  /* Builds and returns result */
  return _binary_vhdl(vhdl_op, vhdl1, vhdl2, nberrors, pos);
}

/** Function **
  Synopsis    [ Computes ctl_term attribute for unary expressions ]
  Description [ A CTL unary term is either a unary operator applied
  to a CTL term or a unary temporal operator applied to any term. ]
  SideEffects [ Memory is allocated to create the result ]
  SeeAlso     [ spec_ActionBinaryVHDLTerm, ctl ]
 */
ctl_term
spec_ActionUnaryCTLTerm
(spec_Operator_t op /* the operator of the expression */,
 ctl_term ctl /* ctl_term attribute of first argument */,
 cvn vhdl /* vhdl_term attribute of first argument */,
 int * nberrors /* address of number of parsing errors so far */,
 tPosition pos /* position of the expression in file */)
{
  ctl_Operator_t ctl_op;

  /* Returns 0 if there is no temporal logic */
  if (!ctl && !_tl_op_p(op)) {
    return 0;
  }
  /* Builds the atomic expressions of the CTL term if necessary */
  if (!ctl) {
    ctl = ctl_BasicNew(0, 0, ctl_AtomicTerm, (pointer) vhdl);
  }
  /* Checks that spec operator correspond to a ctl operator */
  if ((ctl_op = spec_op_to_ctl_op_tab[(int) op]) == 0) {
    ++*nberrors;
    Message("Invalid operator in CTL expression", xxError, pos);
  }
  /* Builds and returns result */
  return ctl_BasicNew(0, 0, ctl_UnaryTerm, ctl_op, ctl);
}

/** Function **
  Synopsis    [ Computes ctl_term attribute for Macro expressions ]
  Description [ A CTL Macro term will be replaced by other formulas
                in model checker ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
ctl_term
spec_ActionMacroCTLTerm
(spec_Operator_t op /* the operator of the expression */,
 int * nberrors /* address of number of parsing errors so far */,
 tPosition pos /* position of the expression in file */)
{
  ctl_Operator_t ctl_op;

  /* Returns 0 if there is no temporal logic */
  if(!_tl_op_p(op)) {
    return 0;
  }
  if ((ctl_op = spec_op_to_ctl_op_tab[(int) op]) == 0) {
    ++*nberrors;
    Message("Invalid operator in CTL expression", xxError, pos);
  }
  return ctl_BasicNew(0,0, ctl_MacroTerm, ctl_op);
}

/** Function **
  Synopsis    [ Computes vhdl_term attribute for unary expressions ]
  Description [ A VHDL unary expression is a vhdl unary operator 
  applied to a vhdl expression. ]
  SideEffects [ Memory is allocated to create the result ]
  SeeAlso     [ spec_ActionBinaryVHDLTerm ]
 */
cvn
spec_ActionUnaryVHDLTerm
(spec_Operator_t op /* the operator of the expression */,
 ctl_term ctl /* ctl_term attribute of first argument */,
 cvn vhdl /* vhdl_term attribute of first argument */,
 int * nberrors /* address of number of parsing errors so far */,
 tPosition pos /* position of the expression in file */)
{
  tOperator vhdl_op;

  /* Returns 0 if there is temporal logic */
  if (ctl || _tl_op_p(op)) {
    return 0;
  }
  /* Checks that spec operator correspond to a ctl operator */
  if ((vhdl_op = spec_op_to_vhdl_op_tab[(int) op]) == 0) {
    ++*nberrors;
    Message("Invalid operator in VHDL expression", xxError, pos);
  }
  /* Builds and returns result */
  return _unary_vhdl(vhdl_op, vhdl, nberrors, pos);
}

/** Function **
  Synopsis    [ Computes vhdl_term attribute for identifiers ]
  Description [ If the identifier denotes an abbreviation,
  then the result is the corresponding abbreviated expression. 
  If the identifier denotes a signal of the specified design,
  a cvn node is built and returned.
  Otherwise this is an error, the error counter is incremented
  and a message is displayed.]
  SideEffects [ None (if the identifier is an abbreviation), or
  memory is allocated (if the identifier denotes a signal), or
  an error message is printed to stdout. ]
 */
cvn
spec_ActionIdentifierTerm
(char * identifier,
 int * nberrors,
 tPosition pos)
{
  /* trying to repair errors, the parser may introduce null identifier.
     i do not know if there is something more clever to do than just
     returning 0... */
  if (!identifier) return 0;

  /* if identifier denotes an abbreviation then return the term */
  {
    spec_Node_t a;
    a = spec_abbreviations;
    while (a && strcmp(identifier, spec_BasicQName(a))) {
      a = spec_BasicQNext(a);
    }
    if (a) {
      return (cvn) spec_BasicQValue(a);
    }
  }
  /* otherwise, create a cvn node */
  {
    cvn decl;
    decl= _declaration_get(identifier);
    if (decl) {
      if ((qKind(decl) == kEnumeratedLiteral) ||
          (qKind(decl) == kCharacterLiteral)) {
        return decl;
      } else {
        return mIdentifier(0, identifier, decl);
      }
    } else {
      char * message;
      message = (char *) 
        mem_get_block(sizeof(char) *
                      (strlen(identifier)+ strlen(" not declared")+1));
      sprintf(message, "%s not declared", identifier);
      Message(message, xxError, pos);
      ++*nberrors;
      mem_free_block(message);
      return 0;
    }
  }
}

/** Function **
  Synopsis    [ Computes vhdl_term attribute for a character literal ]
  Description [ If the identifier literal denotes a legal literal in
  the specified design, a cvn node is built and returned.
  Otherwise this is an error, the error counter is incremented
  and a message is displayed.]
  SideEffects [ Memory is allocated or an error message is printed
  to stdout. ]
 */
cvn
spec_ActionCharLiteralTerm
(char literal,
 int * nberrors,
 tPosition pos)
{
  cvn node;
  char str[4] ;
  sprintf(str, "'%c'", literal);
  node = _declaration_get(str);
  if (!node) {
    unsigned length = strlen("character literal  not declared") + 2;
    char * message = (char *) mem_get_block(length * sizeof(char));
    sprintf(message, "character literal %c not declared", literal);
    Message(message, xxError, pos);
    mem_free_block(message);
    ++*nberrors;
  }
  return node;
}

/** Function **
  Synopsis    [ Prints a starting banner ]
  SideEffects [ Data is sent to stdout ]
 */
void
spec_ActionStartSpecification 
(char * name)
{
  printf("loading specification...\n");
}

/** Function **
  Synopsis    [ Prints an ending banner ]
  SideEffects [ Data is sent to stdout ]
 */
void
spec_ActionEndSpecification 
(void)
{
  printf("...specification loaded\n");
  printf("name: %s\n", spec_name);
  printf("design:");
  {
    if (IsA(spec_design, kEntityDeclaration)) 
      printf("entity declaration %s\n", qName(spec_design));
    else
      printf("architecture body %s(%s)\n", qName(qPrimary(spec_design)), 
	     qName(spec_design));
  }
}

/** Function **
  Synopsis    [ Loads the .cv file of the specified entity ]
  SideEffects [ Sets the global variable spec_design to the
  root node of the design intermediate format tree. Sets the
  global variable spec_design_library to the parameter string
  library. A .cv file is opened. ]
 */
cvn
spec_ActionLoadDesign
(char * library,
 char * entity,
 char * architecture,
 tPosition pos)
{
  spec_design_library = library;
  spec_design = cv_load_unit(library, entity, architecture);
  if (NullNode(spec_design)) {
    char * message;
    message = (char *)
      mem_get_block((SIZE_T) sizeof(char) *
          (strlen("Cannot load library unit .().") +
           strlen(library) +
           strlen(entity) +
           strlen(architecture)));
    sprintf(message, "Cannot load library unit %s.%s(%s)", library, entity, architecture);
    Message(message, xxFatal, pos);
  }
  return spec_design;
}

/** Function **
  Synopsis    [ Checks if an identifier can serve as an abbreviation ]
  Description [ If the given identifier id does not denote an already
  parsed abbreviation or a signal of the top-level entity declaration
  of the specified design, returns 1.
  Otherwise an error message is sent to stdout, the error counter is
  incremented and the value 0 is returned. ]
  SideEffects [ See description ]
 */
int
spec_ActionCheckAbbreviation
(char * id,
 int * nberrors,
 tPosition pos)
{
  /* first check if there is an abbreviation labeled id */
  {
    spec_Node_t a = spec_abbreviations;
    while (a && strcmp(spec_BasicQName(a), id)) a= spec_BasicQNext(a);
    if (a) {
      char * message = (char *) 
        mem_get_block(sizeof(char) *
                      (1 + strlen(spec_BasicQName(a))+ 
                       strlen(" already denotes an abbreviation")));
      sprintf(message, "%s already denotes an abbreviation", spec_BasicQName(a));
      Message(message, xxError, pos) ;
      mem_free_block(message);
      ++*nberrors;
      return 0;
    }
  }
  /* then check among the top-level signals of the VHDL design */
  {
    cvn decl = IsA(spec_design, kEntityDeclaration) ? qPorts(spec_design) :
      qPorts(qPrimary(spec_design));
    while (decl && strcmp(qName(decl), id)) decl= qNext(decl);
    if (decl) {
      char * message = (char *) 
        mem_get_block(sizeof(char) *
                (1 + strlen(id)+ strlen(" already denotes a signal of the entity")));
      sprintf(message, "%s already denotes a signal of the entity", id);
      Message(message, xxError, pos) ;
      mem_free_block(message);
      ++*nberrors;
      return 0;
    }
  }
  return(1);
}

/** Function **
  Synopsis    [ Checks that an expression is a condition ]
  Description [ If expression is not equal to null and does not
  correspond to a VHDL condition (i.e. of type BOOLEAN), then
  an error message is displayed and the error counter is
  incremented ]
  SideEffects [ nberrors may be incremented, output may be sent
  to stdout ]
 */
void
spec_ActionCheckCondition
(cvn expression,
 int * nberrors,
 tPosition pos)
{
  if (expression == 0)
    return;
  if (cv_eq(qBaseType(qSubtype(expression)), boolean_type))
    return;
  ++*nberrors;
  Message("condition expected", xxError, pos) ;
}

/** Function **
  Synopsis    [ Checks that an expression is a valid invariant ]
  Description [ A valid invariant is a condition on signals of
  mode IN. ]
  SideEffects [ nberrors may be incremented, output may be sent
  to stdout ]
 */
void
spec_ActionCheckInvariant
(cvn expression,
 int * nberrors,
 tPosition pos)
{
  if (!cv_eq(qBaseType(qSubtype(expression)), boolean_type)) {
    ++*nberrors;
    Message("invariant must be a condition", xxError, pos) ;
    return;
  }
  _check_invariant(expression, nberrors, pos);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Checks if two VHDL subtypes are compatible. ]
  Description [ If two VHDL subtypes are compatible (e.g. can 
  the types of the two arguments of an equality), if they
  have the same base type (? check that...). ]
  SideEffects [ None ]
 */
static int
_type_compatible
(cvn T1,
 cvn T2)
{
  return (cv_eq(qBaseType(T1), qBaseType(T2)));
}

/** Function **
  Synopsis    [ Finds a declaration matching a string ]
  SideEffects [ none ]
  SeeAlso     [ __declaration_get ]
 */
static cvn
_declaration_get
(char * id)
{
  return __declaration_get(spec_design, id); 
}

/** Function **
  Synopsis    [ Finds a declaration matching a string ]
  Description [ Goes down in the hierarchy of the VHDL
  entity represented by node and returns the first signal
  declaration or enumerated whose name is id. If there is
  no declaration matching this criterion, returns 0. ]
  SideEffects [ ]
  SeeAlso     [ _declaration_get ]
 */
static cvn
__declaration_get
(cvn node,
 char * id)
{
  cvn res;
  if (node == 0) return 0;
  if (id == 0) return 0;
  switch(qKind(node)) {
  case kUseClause:
    res = __declaration_get(qSelectedNames(node), id);
    if (res) return res;
    return __declaration_get(qNext(node), id);
  case kArchitectureBody:
    res = __declaration_get(qPrimary(node), id);
    if (res) return res;
    res = __declaration_get(qDeclarations(node), id);
    if (res) return res;
    res = __declaration_get(qStatements(node), id);
    if (res) return res;
    return __declaration_get(qNext(node), id);
  case kPackageDeclaration:
    return __declaration_get(qDeclarations(node), id);
  case kEntityDeclaration:
    res = __declaration_get(qPorts(node), id) ;
    if (res) return res;
    res = __declaration_get(qDeclarations(node), id) ;
    if (res) return res;
    return __declaration_get(qNext(node), id);
  case kBlockStatement:
    res = __declaration_get(qDeclarations(node), id);
    if (res) return res;
    return __declaration_get(qStatements(node), id);
  case kSignalAssignmentStatement : 
  case kConditionalSignalAssignmentStatement :
  case kProcessStatement:
    return __declaration_get(qNext(node), id);
  case kIdentifier: 
  case kEnumeratedLiteral: 
  case kCharacterLiteral: 
    if (strcasecmp(qName(node), id) == 0) {
      return node;
    } else {
      return 0;
    }
  case kSignalDeclaration:
    if (strcasecmp(qName(node), id) == 0) return node;
    return __declaration_get(qNext(node), id);
  case kTypeDeclaration:
    if (IsA(qBaseType(qSubtype(node)), kEnumerationType)) {
      res = __declaration_get(qElements(qBaseType(qSubtype(node))), id);
      if (res) return res;
    }
    return __declaration_get(qNext(node), id);
  case kList:
    res = __declaration_get(qValue(node), id);
    if (res) return res;
    return __declaration_get(qNext(node), id);
  default:
    return 0;
  }
}

/** Function **
  Synopsis    [ Builds a binary VHDL expression. ]
  Description [ op, l, and r are the operator, first and
  second argument of a VHDL expression. This routine
  checks if the expression "l op r" is valid VHDL. If yes,
  builds and returns the node representing this expression.
  If no, prints an error message, increments the error
  counter and returns 0. ]
  SideEffects [ Allocate memory or send data to stdout 
  (see description). ]
 */
static cvn
_binary_vhdl
(tOperator op,
 cvn l,
 cvn r,
 int * nberrors,
 tPosition pos)
{
  switch (op) {
    /* for the relations, the result is boolean, and the arguments
       must be of the same type */
  case kEq : case kNeq : case kGt : case kGe : case kLt : case kLe :
    if (l && r) {
      if (!_type_compatible(qSubtype(l), qSubtype(r))) {
         Message("incompatible types", xxError, pos) ;
         ++*nberrors;
      }
    }
    return mBinary(0, mSubtype(0, 0, boolean_type, 0), op, l, r);
  /* for the logical operations, the arguments must be of the same
     type and the result type is this type. Whether the type actually
     has an associated "and" operator is not checked, for we only
     accept BIT and BOOLEAN types. */
  case kAnd : case kOr : case kNand : case kNor : case kXor : case kXnor :
    if (l && r) {
      if (!_type_compatible(qSubtype(l), qSubtype(r))) {
        Message("incompatible types", xxError, pos);
        ++*nberrors;
      }
      return mBinary(0, qSubtype(l), op, l, r);
    }
    if (l)
      return mBinary(0, qSubtype(l), op, l, r);
    if (r)
      return mBinary(0, qSubtype(r), op, l, r);
    return 0;
    /* for the other operations, issue an error message and return
       a node of the type of the left operand */
  default :
     Message("invalid operator in expression",
             xxError, pos);
     if (l)
       return mBinary(0, qSubtype(l), op, l, r);
     if (r)
       return mBinary(0, qSubtype(r), op, l, r);
     return 0;
  }
}

/** Function **
  Synopsis    [ Builds a unary VHDL expression. ]
  Description [ op and arg are the operator and the argument
  of a VHDL expression. This routine checks if the expression 
  "op arg" is valid VHDL. If yes, builds and returns the node
  representing this expression.
  If no, prints an error message, increments the error
  counter and returns 0. ]
  SideEffects [ Allocate memory or send data to stdout 
  (see description). ]
 */
static cvn
_unary_vhdl
(tOperator op,
 cvn arg,
 int * nberrors,
 tPosition pos)
{
  switch (op) {
    /* for the logical operations, the arguments must be of the same
       type and the result type is this type. Whether the type actually
       has an associated "and" operator is not checked, for we only
       accept BIT and BOOLEAN types. */
  case kNot:
    if (arg)
      return mUnary(0, qSubtype(arg), op, arg) ;
  /* for the other operations, issue an error message and return
     a node of the type of the left operand */
  default :
    Message("invalid operator in expression", xxError, pos) ;
    ++*nberrors;
    if (arg)
      return mUnary(0, qSubtype(arg), op, arg) ;
    return 0;
  }
}

/**Function**
  Synopsis           [ Auxiliary function for spec_ActionCheckInvariant ]
  Description        [ Checks that all identifiers in expression are
  signal of mode "in". Prints an error message and increments the error
  counter if this is not true. ]
  SideEffects        [ If check fails, nberrors is incremented and an
  error message is sent to stderr ]
  SeeAlso            [ spec_ActionCheckInvariant ]
**/
static void
_check_invariant
(cvn expression,
 int * nberrors,
 tPosition pos)
{
  switch (qKind(expression)) {
    case kIdentifier:
      if ((!IsA(qDeclaration(expression), kSignalDeclaration)) ||
          (qMode(qDeclaration(expression)) != kIn)) {
        char * message;
        message = mem_get_block((SIZE_T) sizeof(char) *
                                (strlen(qName(expression)) + 
                                 strlen("invariant depends on , which is not a signal of mode \042in\042") + 1));
        sprintf(message, 
                "invariant depends on %s, which is not a signal of mode \042in\042", 
                 qName(expression));
        ++*nberrors;
        Message(message, xxError, pos);
      }
      return;
    case kUnary:
      _check_invariant(qOperand(expression), nberrors, pos);
      return;
    case kBinary:
      _check_invariant(qLeft(expression), nberrors, pos);
      _check_invariant(qRight(expression), nberrors, pos);
      return;
    default:
      return;
  }
}
