/** CFile **************************************************************
  FileName    [ modelModify.c ]
  PackageName [ model ]
  Synopsis    [ Modifications on a complete model ]
  Description [ Routines to modify a model after its representation
                has been built. This routines implement heruistic to
                optimize the transition representation in order to
                accelerate the traversal of the state space of the
                models. ]
  SeeAlso     [ modelStruct_t modelReachability.c modelCheck.c ]
  Author      [ David Deharbe ]
  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]
  Revision    [ $Id$ ]
***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "modelInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Reduce the transition representation under some assumption ]
  Description [ Simplify the transition representation under the
                assumption that the transitions are taken from 
                states that verify the given predicate using the
                so-called generalized cofactor. A useful example
                predicate would be the characteristic function of
                the set of reachable states.
                Returns the difference between the size of the
                transition representation before and after the
                reduction.
                This operation might take a noticeable amount of
                time but the subsequent call to traversal routines
                are much faster. ]
  SideEffects [ Modifies the transition representation (the field
                tf) of the model. It is not possible to come back
                to the original transition function. ]
  SeeAlso     [ ]
 */

long 
model_ModifyReduceTransition (
  Model m /* model to reduce */,
  FILE * outstream /* output stream where the operation is traced */,
  bdd f /* assumption predicate */)
{
  /* declarations */
  long before, after;
  bdd_manager bddm;
  /* statements */
  bddm = m->bddm;
  before = model_MeasureSizeTransition(m);
  printf("reducing model...\n");
  ant_ReduceCareSet(m->ant, f);
  img_ReduceCareSet(m->img, f);
  printf("...model reduction done\n");
  after = model_MeasureSizeTransition(m);
  return (before - after);
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */

void 
model_ModifyReduceSCC 
(Model m /* model to reduce */)
{
  /* statements */
  img_ReduceSCC(m->img, m->init);
}

/**Function**
  Synopsis           [required]
  Description        [optional]
  SideEffects        [required]
  SeeAlso            [optional]
**/
void
model_ModifyStillInputs
(Model m)
{
  bdd_manager bddm = m->bddm;
  bdd event, stable, init, condition;

  m->still_inputs = 1;
  /* compute event: exists an event on a signal of mode in */
  {
    ModelSignalStruct_t * sig;
    int idx;
    event = bdd_identity(bddm, bdd_zero(bddm));
    sig = m->signals;
    for (idx = m->nb_signals; idx; --idx) {
      if ((qMode(sig->declaration) == kIn) && sig -> previous) {
        event = bdd_orup2(bddm, sig->event, event);
      }
      ++sig;
    }
  }
  /* compute stable */
  stable= m->stable;

  /* compute init */
  init= m->init;

  condition = bdd_orup2(bddm, init,
                        bdd_orup(bddm,
                                 bdd_and(bddm, event, stable),
                                 bdd_andup(bddm, bdd_not(bddm, event),
                                           bdd_not(bddm, stable))));
  model_RepresentationSetInvariant(m, condition);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

