/** CFile **************************************************************

  FileName    [ modelMeasure.c ]

  PackageName [ model ]

  Synopsis    [ Measures certain characteristic of models ]

  Description [ This file contains routines that measure the 
                size of the models. ]

  SeeAlso     [ ModelStruct_t ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "modelInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Size of the transition representation ]
  Description [ Returns the actual number of BDD nodes
                (i.e. considering sharing) used for the
                representation of the transitions. ]
  SideEffects [ none ]
  SeeAlso     [ ]
 */

long
model_MeasureSizeTransition 
(Model m /* measured model */)
{
  bdd_manager bddm;
  bddm = m->bddm;
  return 0l;
}

/** Function **
  Synopsis    [ Number of states in the model ]
  Description [ Computes and returns the number of states in the
                model. This number is 2 to the power of the number
                of atomic propositions and some of these states 
                may not be reachable.  ]
  SideEffects [ none ]
  SeeAlso     [ model_MeasureNbReachableStates 
                model_MeasureNbStableStates ]
 */
double
model_MeasureNbStates
(Model m /* measured model */)
{
  return model_MeasureSetCardinality(m, bdd_one(m->bddm));
}

/** Function **
  Synopsis    [ Number of reachable states in the model ]
  Description [ If the set of reachable states was not 
                computed before returns 0, otherwise returns
                the number of reachable states. ]
  SideEffects [ none ]
  SeeAlso     [ model_MeasureNbReachableStates 
                model_MeasureNbStableStates ]
 */
double
model_MeasureSetCardinality
(Model m /* measured model */,
 bdd set)
{
  return bdd_satisfying_number(m->bddm, set,
                               m->nb_propositions);
}

/** Function **
  Synopsis    [ Number of reachable states in the model ]
  Description [ If the set of reachable states was not 
                computed before returns 0, otherwise returns
                the number of reachable states. ]
  SideEffects [ none ]
  SeeAlso     [ model_MeasureNbReachableStates 
                model_MeasureNbStableStates ]
 */
double
model_MeasureNbReachableStates
(Model m /* measured model */)
{
  if (! m->reachability_analysis.done)
    return 0;

  return model_MeasureSetCardinality(m, 
                                     m->reachability_analysis.reachable);
}

/** Function **
  Synopsis    [ Number of stable reachable states in the model ]
  Description [ If the set of reachable states was not 
                computed before returns 0, otherwise returns
                the number of reachable states that satisfy the
                stability predicate.
                Remark that this does not include the initial
                states. ]
  SideEffects [ none ]
  SeeAlso     [ model_MeasureNbReachableStates 
                model_MeasureNbStableStates ]
 */
double
model_MeasureNbStableStates 
(Model m /* measured model */)
{
  if (! m->reachability_analysis.done)
    return 0;

  return model_MeasureSetCardinality(m, 
                                     bdd_and(m->bddm, 
                                             m->stable,
                                             m->reachability_analysis.reachable));
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

