/** CFile **************************************************************

  FileName    [ modelExplicit.c ]

  PackageName [ model ]

  Synopsis    [ Obtains explicit information for diagnosis purposes ]

  Description [ This file contains routines that permit to extract
  explicit information out of the implicit BDD-based representations.
  This information is to be used for reporting counterexamples or
  witnesses. 
  A configuration of the model is a predicate that corresponds to 
  the conjunction of exactly one value per selected VHDL object.
  It is possible to select input variables, state variables or
  both.
  The type ModelConfig is a pointer to a structure with the BDD of
  a configuration and an array of CV nodes. For each even position in
  this array is a node of VHDL object declaration, and at the following
  odd position is the node of the value it takes in the configuration. ]

  SeeAlso     [ modelDiagnose.c ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include "modelInt.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _ComputeValuations(Model m, bdd c, int selector, cvn ** configs, int * nbconfigs);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Displays a set of configurations. ]
  Description [ The parameters are a model m, the characteristic
  function of a set of states of m and an output stream and the
  object selector.
  Prints a human-readable format of the configurations to the
  output stream.
  This routine is useful to print small sets since the listing
  gets long very quickly otherwise. It is meant to be used for
  debugging purposes. ]
  SideEffects [ Sends output to the stream. Manipulates new BDDs. ]
  SeeAlso     [ model_ExplicitDisplayConfig ]
 */
void
model_ExplicitDisplaySet
(Model m,
 FILE * outstream,
 bdd set,
 int selector)
{
  bdd_manager bddm;
  bdd tmp;
  bddm = m->bddm;
  bdd_assoc(bddm, _SelectAssoc(m, selector));
  tmp = bdd_project(bddm, set);
  fprintf(outstream, "\n");
  while (tmp != bdd_zero(bddm)) {
    ModelConfig config;
    bdd config_fn;
    config_fn = model_ExplicitComputeCondition(m, tmp, selector);
    config = model_ExplicitNewConfig(m, config_fn, selector);
    model_ExplicitDisplayConfig(outstream, config);
    fprintf(outstream, "\n");
    tmp = bdd_andup(bddm, tmp, bdd_notup(bddm, config_fn));
    model_ExplicitFreeConfig(config);
  }
  /* tmp = bdd_zero(bddm) needs not to be freed */
}

/** Function **
  Synopsis    [ Computes a configuration ]
  Description [ Given the BDD representing a set of states of the
  model m, returns a BDD with exactly one node at each level that
  corresponds to an atomic proposition that codes a selected VHDL
  object. It is possible to select to disjoint classes of VHDL
  objects: inputs (i.e signals of mode in) and state variables
  (i.e. other signals, variables, and implicit variables). 
  MODEL_INPUTMASK, MODEL_STATEMASK, 
  MODEL_INPUTMASK | MODEL_STATEMASK are the three possible
  combinations for the parameter selector. ]
  SideEffects [ Modifies the state of the BDD data structure. ]
  SeeAlso     [ ]
 */
bdd 
model_ExplicitComputeCondition
(Model m,
 bdd c,
 int selector /* MODEL_INPUTMASK, MODEL_STATEMASK, (MODEL_INPUTMASK | MODEL_STATEMASK) */)
{
  bdd_assoc(m->bddm, _SelectAssoc(m, selector));
  return bdd_satisfy_support_rand(m->bddm, c, 1, 1);
}

/** Function **
  Synopsis    [ Creates a new configuration representation ]
  Description [ Individual configurations of the symbolic model can be
  manipulated individually. This routine creates a new configuration
  representation, such that the characteristic function is a given BDD c.
  This routine also computes and stores in this representation the values
  that take the various VHDL objects that are represented as state
  variables in the model. 
  The memory used by result representation can be returned to the memory
  pool with the routine model_ExplicitFreeConfig. ]
  SideEffects [ Memory is allocated to represent the result. ]
  SeeAlso     [ ]
 */
ModelConfig
model_ExplicitNewConfig
(Model m,
 bdd c,
 int selector)
{
  cvn * ov;
  int nb_ov;
  ModelConfig res;
  _ComputeValuations(m, c, selector, &ov, &nb_ov);
  res = (ModelConfig) mem_new_rec(ModelConfigRecMgr);
  res->code = c;
  res->object_valuations = ov;
  res->nbvaluations = nb_ov;
  return res;
}

/** Function **
  Synopsis    [ Deletes a configuration ]
  Description [ Deletes a configuration created with 
  model_ExplicitNewConfig ]
  SideEffects [ Disallocates memory. ]
  SeeAlso     [ ]
*/
void
model_ExplicitFreeConfig
(ModelConfig c)
{
  mem_free_block((pointer) c->object_valuations);
  mem_free_rec(ModelConfigRecMgr, (pointer) c);
}

/** Function **
  Synopsis    [ Displays a state ]
  Description [ Displays a human-friendly format of the state to
  the given output stream. ]
  SideEffects [ Output is sent to the stream. ]
  SeeAlso     [ ]
 */

void
model_ExplicitDisplayConfig
(FILE * outstream,
 ModelConfig s)
{
  _DisplayObjectValues(outstream, s->object_valuations, s->nbvaluations);
}

/** Function **
  Synopsis    [ Predicate of a configuration ]
  SideEffects [ None ]
*/
bdd
model_ExplicitQCode
(ModelConfig c)
{
  return c->code;
}

/** Function **
  Synopsis    [ Creates a new path representation ]
  Description [ Individual paths of the symbolic model can be manipulated
  individually. This routine creates a new path representation with the
  given state being its starting point.
  The memory used by result representation can be returned to the memory
  pool with the routine model_ExplicitFreePath. ]
  SideEffects [ Memory is allocated to represent the result. ]
  SeeAlso     [ ModelPath ]
 */

ModelPath
model_ExplicitNewPath
(void)
{
  ModelPath res;
  const int SIZEINIT = 8;
  res = (ModelPath) mem_new_rec(ModelPathRecMgr);
  res->transitions = (bdd *) mem_get_block(SIZEINIT * sizeof(bdd));
  res->nbtransitions = 0;
  res->nbmaxtransitions = SIZEINIT;
  res->loopstart = 0;
  return res;
}

/** Function **
  Synopsis    [ Deletes a path representation ]
  Description [ Deletes a path representation previously created
  with model_ExplicitNewPath ]
  SideEffects [ Disallocates memory. ]
  SeeAlso     [ ]
 */
void
model_ExplicitFreePath
(ModelPath t)
{
  mem_free_block((pointer) t->transitions);
  mem_free_rec(ModelPathRecMgr, (pointer) t);
}

/** Function **
  Synopsis    [ Indicates that a state starts a loop ]
  Description [ The given model configuration c is used to mark
  the start of a loop in the path. This function should be called
  in order to build an infinite path. An infinite path has a finite
  prefix followed by a loop. The given configuration should be
  the first state on the path after the prefix. ]
  SideEffects [ Writes on the field loopstart of the given path.
  c is a pointer thus it might be a cause of bugs to disallocate
  the referenced memory before the path itself. ]
  SeeAlso     [ ]
 */
ModelPath
model_ExplicitLoopPath
(ModelPath p,
 bdd c)
{
  p->loopstart = c;
  return p;
}

/** Function **
  Synopsis    [ Appends a transition at the end of a path ]
  Description [ Given a path, an input vector, and a state, appends
  the input and state at the end of the path. This routine together
  with the routine model_ExplicitNewPath, are the constructors of the
  type ModelPath. ]
  SideEffects [ The representation of the path is modified to 
  incorporate the new transition. Memory may be allocated to extend
  the representation of the transitions of the path. ]
  SeeAlso     [ ]
 */

void
model_ExplicitAppendTransitionToPath
(ModelPath p,
 bdd c)
{
  if ((p->nbtransitions) == p->nbmaxtransitions) {
    p->nbmaxtransitions *= 2;
    p->transitions = (bdd *) 
      mem_resize_block((pointer) p->transitions,
                       p->nbmaxtransitions * sizeof(bdd));
  }
  p->transitions[p->nbtransitions++] = c;
}

/** Function **
  Synopsis    [ Appends a path at the end of a path ]
  Description [ Appends the second parameter path (suffix) at the
  end of the first parameter path. ]
  SideEffects [ The representation of the first path parameter is 
  modified to incorporate the new transitions. Memory may be 
  allocated to extend the representation of the transitions of the 
  path. ]
  SeeAlso     [ model_ExplicitAppendTransitionToPath ]
 */

void
model_ExplicitAppendPaths
(ModelPath path,
 ModelPath suffix)
{
  bdd * ptr;
  int count;
  for (count = 0, ptr = suffix->transitions; 
       count < suffix->nbtransitions;
       ++count, ++ptr) {
    model_ExplicitAppendTransitionToPath(path, *ptr);
  }
}

/** Function **
  Synopsis    [ Returns the last state of a path ]
  Description [ If the given path is finite, the result is the
  last state of the given path. If the given path is infinite,
  the result is 0. ]
  SideEffects [ None ]
  SeeAlso     [ ]
 */

bdd
model_ExplicitQPathDestination
(ModelPath t)
{
  /* case of an empty path */
  if (t->nbtransitions == 0) {
    return 0;
  }
  /* case of an infinite path */
  if (t->loopstart != 0) {
    return 0;
  }
  /* case of a finite path */
  return t->transitions[t->nbtransitions - 1];
}

/** Function **
  Synopsis    [ Displays a human-friendly format of a path ]
  Description [ Displays a human-friendly format of the path to 
  the given output stream. ]
  SideEffects [ Output is sent to the stream. ]
  SeeAlso     [ optional ]
 */

void
model_ExplicitDisplayPath
(Model m,
 FILE * outstream,
 ModelPath t)
{
  int infinite, idx;
  bdd loopstart, * ptr;

  if (t->loopstart != 0) {
    infinite = 1;
    loopstart = t->loopstart;
  } else {
    infinite = 0;
    loopstart = 0;
  }
    
  for (idx = 1, ptr = t->transitions; idx < t->nbtransitions; ++idx, ++ptr) {
    ModelConfig state, input;
    bdd tmp;
    int is_init, is_stable;
    state = model_ExplicitNewConfig
      (m, model_ExplicitComputeCondition (m, *ptr, MODEL_STATEMASK),
       MODEL_STATEMASK);
    tmp = bdd_and(m->bddm, state->code, m->init);
    is_init = (tmp != bdd_zero(m->bddm)); 
    bdd_free(m->bddm, tmp);
    tmp = bdd_and(m->bddm, state->code, m->stable);
    is_stable = (tmp != bdd_zero(m->bddm)); 
    bdd_free(m->bddm, tmp);
    if (idx == 1) {
      fprintf(outstream, "-- path begins\n");
    }
    if (infinite && (* ptr == loopstart)) {
      fprintf(outstream, "-- loop begins\n");
    }
    if ((m->still_inputs == 0) || is_stable || (idx == 1)) {
      input = model_ExplicitNewConfig
        (m, model_ExplicitComputeCondition
          (m, *ptr, MODEL_INPUTMASK), MODEL_INPUTMASK);
    }
    model_ExplicitDisplayConfig(outstream, state);
    if (is_init) {
      fprintf(outstream, "[initial]\n");
    } else if (is_stable) {
      fprintf(outstream, "[stable]\n");
    } 
    model_ExplicitFreeConfig(state);
    if ((m->still_inputs == 0) || is_stable || (idx == 1)) {
      fprintf(outstream, "input values:\n");
      model_ExplicitDisplayConfig(outstream, input);
      model_ExplicitFreeConfig(input);
    }
    fprintf(outstream, "transitions to:\n");
  }
  {
    ModelConfig state;
    state = model_ExplicitNewConfig
      (m, model_ExplicitComputeCondition (m, *ptr, MODEL_STATEMASK),
       MODEL_STATEMASK);
    model_ExplicitDisplayConfig(outstream, state);
    if (infinite) {
      fprintf(outstream, "-- loop ends\n");
    }
    fprintf(outstream, "-- path ends\n");
    model_ExplicitFreeConfig(state);
  }
}

/** Function **
  Synopsis    [ Output a path as a sequence of VHDL statements. ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
void
model_ExplicitDisplayVHDL
(Model m,
 FILE * outstream,
 ModelPath t)
{
  int idx, infinite;
  bdd loopstart, *ptr;
  if(t->loopstart != 0) {
    infinite = 1;
    loopstart = t->loopstart;
  } else {
    infinite = 0;
    loopstart = 0;
  }
  for (idx = 1, ptr = t->transitions; idx < t->nbtransitions; ++idx, ++ptr) {
    ModelConfig state, input;
    bdd tmp;
    int is_init, is_stable;
    state = model_ExplicitNewConfig
      (m, model_ExplicitComputeCondition (m, *ptr, MODEL_STATEMASK),
       MODEL_STATEMASK);
    tmp = bdd_and(m->bddm, state->code, m->init);
    is_init = (tmp != bdd_zero(m->bddm));
    bdd_free(m->bddm, tmp);
    tmp = bdd_and(m->bddm, state->code, m->stable);
    is_stable = (tmp != bdd_zero(m->bddm));
    bdd_free(m->bddm, tmp);
    if(infinite && (*ptr == loopstart)) {
      fprintf(outstream, "    while TRUE loop\n");
    }
    if ((m->still_inputs == 0) || is_stable || (idx == 1)) {
      input = model_ExplicitNewConfig
        (m, model_ExplicitComputeCondition
          (m, *ptr, MODEL_INPUTMASK), MODEL_INPUTMASK);
      _VHDLDisplayValues(outstream, input->object_valuations,
                         input->nbvaluations);
      model_ExplicitFreeConfig(input);
      if (is_stable) {
        fprintf(outstream, "    wait for 10 ns;\n\n");
      } else {
        fprintf(outstream, "    wait for 0 ns;\n\n");
      }
    }
 
  }
  if(infinite) {
    fprintf(outstream, "    end loop;\n");
  }
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Computes the value of a VHDL object ]
  Description [ Given the BDD of the characteristic function of
  a state or input vector (with at most one node at each level),
  given a VHDL object declaration, and given the BDD-based coding 
  of this object, returns the value of the VHDL object in this
  state or input.
  This routine uses the function cv_subtype_nth
  of the package cvu. ]
  SideEffects [ Memory is allocated to represent the result. ]
  SeeAlso     [ cv_subtype_nth ]
 */
cvn _ComputeObjectValue
(bdd_manager bddm,
 cvn decl,
 vbdd coding,
 bdd c)
{
  int ordinal;
  bdd valuation;
  /* creates an assoc and sets the current assoc to it */
  bdd_temp_assoc(bddm, vbdd_elements(coding), 0);
  bdd_assoc(bddm, -1);
  /* we got it : the valuation of the variables coding decl in c */
  valuation = bdd_satisfy_support_rand(bddm, c, 1, 1);
  ordinal = vbdd_decode_ordinal(bddm, valuation, coding);
  return cv_subtype_nth(qSubtype(decl), ordinal);
}

/** Function **
  Synopsis    [ Displays formatted object valuations. ]
  Description [ The parameters of this function include a vector
  of CV nodes, actually composed with couples object declarations
  (at even positions) and values (at odd positions). nb_ov gives
  the number of such couples. 
  A formatted display of equalities (4 per line) object-value is
  sent to outstream. 
  This routine uses the function cvn_display of the cv package. ]
  SideEffects [ Output is sent to stream. ]
  SeeAlso     [ cvn_display ]
 */
void
_DisplayObjectValues 
(FILE * outstream, cvn * ov, int nb_ov)
{
  int idx;
  for (idx = 0; idx < nb_ov; idx++) {
    if (idx && ((idx % 3) == 0)) {
      fprintf(outstream, "\n");
    } 
    fprintf(outstream, "(%s = ", qName(*ov++));
    cvn_display(outstream, *ov++);
    fprintf(outstream, ")");
    if (idx < (nb_ov - 1)) {
    fprintf(outstream, " and ");
    }
  }
  fprintf(outstream, "\n");
}

/** Function **
  Synopsis    [ Computes the values of the VHDL objects in a given configuration  ]
  Description [ Given a predicate that represents a configuration of the
                model (as returned by model_ExplicitQConfigurationCondition)
                computes the values of the signals and variables of
                the modeled architecture in that configuration.
                The result is stored in the parameters passed by
                reference: a vector state of object_valuation of length
                nb_states. 
                object_valuation is the type for a pair of cvn nodes,
                one is an object declaration, and one is a literal
                constant.
                The stored arrays are such that the conjuction of all
                the equalities "object = constant" matches the state. ]
  SideEffects [ Memory is allocated to represent the result with mem(3) ]
  SeeAlso     [ ]
 */
static void
_ComputeValuations
(Model m,
 bdd c,
 int selector,
 cvn ** configs,
 int * nbconfigs)
{
  int idx, configcounter;

  /* count the number of valuations */
  configcounter = 0;
  for (idx = 0; idx < m->nb_signals; ++idx) {
    /* == has higher precedence than & */
    if (((qMode(m->signals[idx].declaration) == kIn) &&
         (selector & MODEL_INPUTMASK)) ||
        ((qMode(m->signals[idx].declaration) != kIn) &&
         (selector & MODEL_STATEMASK))) {
      ++configcounter;
    }
  }
  if (selector & MODEL_STATEMASK) {
    configcounter+=m->nb_variables;
  }
  * nbconfigs = configcounter;

  /* allocate memory to store valuations */
  * configs = (cvn *) mem_get_block(2 * configcounter * sizeof(cvn));

  /* compute and store valuations */
  configcounter = 0;
  for (idx = 0; idx < m->nb_signals; ++idx) {
    ModelSignalStruct_t object;
    cvn declaration;
    object = m->signals[idx];
    declaration = object.declaration;
    if (((qMode(declaration) == kIn) && (selector & MODEL_INPUTMASK)) ||
        ((qMode(declaration) != kIn) && (selector & MODEL_STATEMASK))) {
      (* configs)[2*configcounter] = declaration;
      (* configs)[2*configcounter+1] = 
         _ComputeObjectValue(m->bddm, declaration, object.coding, c);
      ++configcounter;
    }
  }
  if (selector & MODEL_STATEMASK) {
    for (idx = 0; idx < m->nb_variables; ++idx) {
      ModelVariableStruct_t object;
      cvn declaration;
      object = m->variables[idx];
      declaration = object.declaration;
      (* configs)[2*configcounter] = declaration;
      (* configs)[2*configcounter+1] = 
         _ComputeObjectValue(m->bddm, declaration, object.coding, c);
      ++configcounter;
    }
  }
}


/** Function **
  Synopsis    [ Returns the BDD assoc of propositions coding selected objects ]
  SideEffects [ None ]
 */
int
_SelectAssoc
(Model m,
 int selector)
{
  switch (selector) {
  case MODEL_INPUTMASK | MODEL_STATEMASK:
    return bdd_new_assoc(m->bddm, vbdd_elements(m->vars), 0);
    break;
  case MODEL_INPUTMASK:
    return bdd_new_assoc(m->bddm, vbdd_elements(m->input_vars), 0);
    break;
  case MODEL_STATEMASK:
    return bdd_new_assoc(m->bddm, vbdd_elements(m->state_vars), 0);
    break;
  default:
    return 0;
  }
}

/** Function **
  Synopsis    [ Output a set of VHDL signal assignment statements. ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
void _VHDLDisplayValues
(FILE * outstream, cvn * ov, int nb_ov)
{
  int idx;
  for (idx = 0; idx < nb_ov; idx++) {
    fprintf(outstream, "    %s <= ", qName(*ov++));
    cvn_display(outstream, *ov++);
    fprintf(outstream, ";\n");
  }
}

