/** CFile **************************************************************

  FileName    [ modelCheck.c ]

  PackageName [ model ]

  Synopsis    [ Symbolic model checking of models of VHDL architectures. ]

  Description [ The routines model_Check<OP>, where <OP> is one of
                {EX, AX, EF, AF, EG, AG, EU, AU, EW, AW}, implement the
                symbolic model checking routines. These routines report
                their progress to an output stream.
                The result for these routines has the type 
                ModelCheckComputation and contain the solution and the
                successive results of the fixpoint computations. 
                Also a set of internal routines permit to allocate, 
                free and access the fields of these results by.
                The verification is actually carried out to three
                static routines _BasicCheck<OP>, where <OP> is one 
                of the three basic operators of CTL: {EX, EF, EG}. ]

  SeeAlso     [ Optional ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All right reserved. ]

  Revision    [ $Id: modelCheck.c,v 1.1 1996/05/14 16:55:19 david Exp david $ ]

***********************************************************************/
#include <stdio.h>

#include "modelInt.h"

static char rcsid [] = "$Id: modelCheck.c,v 1.1 1996/05/14 16:55:19 david Exp david $";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/
typedef ModelCheckComputation (* ctl_un_fn) (Model, FILE *, bdd);
typedef ModelCheckComputation (* ctl_bin_fn) (Model, FILE *, bdd, bdd);

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/
static ctl_un_fn ctl_un_op_to_fn_tab [] = {
  & model_CheckEX,
  & model_CheckAX,
  & model_CheckEF,
  & model_CheckAF,
  & model_CheckEG,
  & model_CheckAG,
  0, /* EU */
  0, /* AU */
  0, /* EW */
  0, /* AW */
  0, /* Implies */
  & model_CheckNot,
  0, /* And */
  0, /* Or */
  0, /* Xor */
  0, /* Xnor */
  0, /* Nand */
  0 /* Nor */
};

static ctl_bin_fn ctl_bin_op_to_fn_tab [] = {
  0,
  0,
  0,
  0,
  0,
  0,
  & model_CheckEU, /* EU */
  & model_CheckAU, /* AU */
  & model_CheckEW, /* EW */
  & model_CheckAW, /* AW */
  & model_CheckImplies, /* Implies */
  0,
  & model_CheckAnd, /* And */
  & model_CheckOr, /* Or */
  & model_CheckXor, /* Xor */
  & model_CheckXnor, /* Xnor */
  & model_CheckNand, /* Nand */
  & model_CheckNor /* Nor */
};

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

#define ctl_un_op_to_fn(op) \
   ctl_un_op_to_fn_tab[(unsigned) op]
#define ctl_bin_op_to_fn(op) \
   ctl_bin_op_to_fn_tab[(unsigned) op]

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static bdd _BasicCheckEX(Model m, FILE * outstream, bdd f);
static bdd _BasicCheckEU(Model m, FILE * outstream, bdd f, bdd g, vbdd frontiers);
static bdd _BasicCheckEG(Model m, FILE * outstream, bdd f, vbdd ** approximations);
static bdd _FairIteration(Model m, FILE * outstream, bdd g, cvn fairness_constraint, vbdd * approximations);
static bdd _FairIterationAux(Model m, FILE * outstream, bdd g, bdd fairness_constraint, vbdd * approximations);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Symbolic model checking for a CTL formula ]
  Description [ m is a Kripke structure. f is a ctl formula over 
  the state space of m. fn is a function invoked to evaluate
  CTL atomic propositions, in which case env is passed to fn as
  second argument. The predicate identifying the configurations
  where f is valid is stored in the field info of the CTL formula t. ]
  SideEffects [ The bdd value of the formula t in the model m
  is set to the field info of t. A trace of the computations is
  output to outstream. ]
  SeeAlso     [ ctl ]
 */
void
model_Check
(Model m,
 FILE * outstream,
 ctl_term t,
 terminal_check_fn fn,
 pointer env)
{
  ModelCheckComputation value;
  /* statement part */

  switch (ctl_BasicQType(t)) {
  case ctl_AtomicTerm: {
    bdd valid, code;
    if ((valid = model_ReachabilityQAll(m)) == 0) {
      valid = bdd_one(m->bddm);
    }
    code = bdd_and(m->bddm, valid, (* fn)(ctl_BasicQAtomicTerm(t), env));
    value = model_CheckAllocComputation(code, 0);
  }
    break;
  case ctl_UnaryTerm: {
    ctl_term arg;
    arg = ctl_BasicQArgument(t);
    model_Check(m, outstream, arg, fn, env);
    value =
      (* ctl_un_op_to_fn(ctl_BasicQOperator(t)))
        (m, outstream, 
         model_CheckQSolution ((ModelCheckComputation) ctl_BasicQInfo(arg)));
  }
    break;
  case ctl_BinaryTerm: {
    ctl_term arg1, arg2;
    arg1 = ctl_BasicQArgument(t);
    arg2 = ctl_BasicQArgument2(t);
    model_Check(m, outstream, arg1, fn, env);
    model_Check(m, outstream, arg2, fn, env);
    value =
      (* ctl_bin_op_to_fn(ctl_BasicQOperator(t)))
        (m, outstream,
         model_CheckQSolution ((ModelCheckComputation) ctl_BasicQInfo(arg1)),
         model_CheckQSolution ((ModelCheckComputation) ctl_BasicQInfo(arg2)));
  }
    break;
  case ctl_MacroTerm: {
    value = model_CheckAllocComputation(m->stable, 0);
  }
    break;
  default:
    Message("model_Check: inconsistency occured during execution - please report",
	    xxError, NoPosition);
    value = model_CheckAllocComputation(bdd_one(m->bddm), 0);
  }
  ctl_BasicSInfo(t, value);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for NOT formulas ]
  Description [ m is a Kripke structure. f is a predicate over 
  the state space of m.
  The predicate identifying the configurations where NOT f is
  valid is stored in the result. ]
  SideEffects [ Changes the state of the BDD data structure managed by
  the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckNot (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_not(m->bddm, f), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AND formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f AND g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckAnd (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_and(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for OR formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f OR g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckOr (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_or(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for IMPLIES formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f IMPLIES g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckImplies (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  if (f == bdd_zero(m->bddm)) {
    fprintf(outstream, "Warning: left hand side of => operator is false.\n");
  }
  return model_CheckAllocComputation(bdd_imply(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for NAND formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f NAND g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckNand (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_nand(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for NOR formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f NOR g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckNor (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_nor(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for XOR formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f XOR g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckXor (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_xor(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for XNOR formulas ]
  Description [ m is a Kripke structure.
                f and g are predicates over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where f XNOR g is valid. ]
  SideEffects [ Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckXnor (
  Model m /* a Kripke structure model */,
  FILE * outstream,
  bdd f /* the first argument of the formula */,
  bdd g /* the second argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(bdd_xnor(m->bddm, f, g), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for EX formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where EX f is valid. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The field solution can be obtained */
model_CheckEX (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* statements */
  return model_CheckAllocComputation(_BasicCheckEX(m, outstream, f), 0);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AX formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where AX f is valid. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */

ModelCheckComputation /* The field solution can be obtained */
model_CheckAX (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* declarations */
  bdd notf;
  ModelCheckComputation result, tmp;
  /* statements */
  notf = bdd_not(m->bddm, f);
  tmp = model_CheckEX(m, outstream, notf);
  bdd_free(m->bddm, notf);
  result = 
    model_CheckAllocComputation
      (bdd_not(m->bddm, model_CheckQSolution(tmp)), 0);
  model_CheckFreeComputation(tmp);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for EF formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where EX f is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution, fixpoint1 are valid */
model_CheckEF (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* statements */
  return model_CheckEU(m, outstream, bdd_one(m->bddm), f);
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AF formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where AF f is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution fixpoint1 are valid */
model_CheckAF (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* declarations */
  ModelCheckComputation result, tmp;
  bdd notf;
  /* statements */
  notf = bdd_not(m->bddm, f);
  tmp = model_CheckEG(m, outstream, notf);
  bdd_free(m->bddm, notf);
  result = model_CheckAllocComputation(
    bdd_notup(m->bddm, model_CheckQSolution(tmp)), 
              model_CheckQInfo(tmp));
  model_CheckFreeComputation(tmp);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for EG formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where EG f is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution, fixpoint1 are valid */
model_CheckEG (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* declarations */
  bdd solution;
  vbdd * approximations;
  ModelCheckComputation result;
  /* statements */
  approximations = 0;
  solution = _BasicCheckEG(m, outstream, f, & approximations);
  result = model_CheckAllocComputation(solution, (pointer) approximations);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AG formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where AG f is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solutio, fixpoint1 are valid */
model_CheckAG (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the argument of the formula */)
{
  /* declarations */
  ModelCheckComputation tmp, result;
  bdd notf;
  /* statements */
  notf = bdd_not(m->bddm, f);
  tmp = model_CheckEU(m, outstream, bdd_one(m->bddm), bdd_not(m->bddm, f));
  bdd_free(m->bddm, notf);
  result = model_CheckAllocComputation
   (bdd_notup(m->bddm, model_CheckQSolution(tmp)), model_CheckQInfo(tmp));
  model_CheckFreeComputation(tmp);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for EU formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where E\[fUg\] is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution, fixpoint1 are valid */
model_CheckEU (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the first argument of the formula */,
  bdd g /* the secont argument of the formula */)
{
  /* declarations */
  ModelCheckComputation result;
  vbdd fixpoint;
  bdd solution;
  /* statements */
  fixpoint = vbdd_new();
  solution = _BasicCheckEU(m, outstream, f, g, fixpoint);
  result = model_CheckAllocComputation(solution, (pointer) fixpoint);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AU formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where E\[fUg\] is valid and the
                successive steps of both involved fixpoint computations. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution, fixpoint1, fixpoint2 are valid */
model_CheckAU (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the first argument of the formula */,
  bdd g /* the secont argument of the formula */)
{
  /* declarations */
  ModelCheckComputation tmp1, tmp2, result;
  bdd notf, notg, notfandnotg;
  vbdd * info;
  /* statements */
  notf = bdd_not(m->bddm, f), 
  notg = bdd_not(m->bddm, g), 
  notfandnotg = bdd_and(m->bddm, notf, notg);
  tmp1 = model_CheckEU(m, outstream, notf, notfandnotg);
  bdd_free(m->bddm, notf);
  bdd_free(m->bddm, notfandnotg);
  tmp2 = model_CheckEG(m, outstream, notg);
  info = mem_get_block((SIZE_T) 2 * sizeof(vbdd));
  info[0] = (vbdd) model_CheckQInfo(tmp1);
  info[1] = (vbdd) model_CheckQInfo(tmp2);
  result = 
    model_CheckAllocComputation
     (bdd_andup(m->bddm, bdd_notup(m->bddm, model_CheckQSolution(tmp1)),
                         bdd_notup(m->bddm, model_CheckQSolution(tmp2))),
      (pointer) info);
  model_CheckFreeComputation(tmp1);
  model_CheckFreeComputation(tmp2);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for EW formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where E\[fWg\] is valid and the
                successive steps of both involved fixpoint computations. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* result fields solution, fixpoint1, fixpoint2 are valid */
model_CheckEW (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the first argument of the formula */,
  bdd g /* the secont argument of the formula */)
{
  /* declarations */
  ModelCheckComputation tmp1, tmp2, result;
  vbdd * info;
  /* statement part */
  tmp1 = model_CheckEU(m, outstream, f, g);
  tmp2 = model_CheckEG(m, outstream, f);
  info = mem_get_block((SIZE_T) 2 * sizeof(vbdd));
  info[0] = (vbdd) model_CheckQInfo(tmp1);
  info[1] = (vbdd) model_CheckQInfo(tmp2);
  result = 
    model_CheckAllocComputation
      (bdd_orup(m->bddm, bdd_notup(m->bddm, model_CheckQSolution(tmp1)),
       bdd_notup(m->bddm, model_CheckQSolution(tmp2))),
    (pointer) info);
  model_CheckFreeComputation(tmp1);
  model_CheckFreeComputation(tmp2);
  return result;
}

/** Function **
  Synopsis    [ Symbolic model checking routine for AW formulas ]
  Description [ m is a Kripke structure.
                f is a predicate over the state space of m. 
                The result is a predicate over the state space of m
                that identifies the states where E\[fUg\] is valid and the
                successive steps of the involved fixpoint computation. ]
  SideEffects [ Prints information to outstream.
                Changes the state of the BDD data structure managed by
                the BDD manager of m. ]
  SeeAlso     [ ModelCheckComputation ]
 */
ModelCheckComputation /* The fields solution, fixpoint1 are valid */
model_CheckAW (
  Model m /* a Kripke structure model */,
  FILE * outstream /* writable stream where output is sent */,
  bdd f /* the first argument of the formula */,
  bdd g /* the secont argument of the formula */)
{
  /* declarations */
  ModelCheckComputation tmp, result;
  bdd notf, notfandnotg;
  /* statements */
  notf = bdd_not(m->bddm, f), 
  notfandnotg = bdd_andup2(m->bddm, notf, bdd_not(m->bddm, g));
  tmp = model_CheckEU(m, outstream, notf, notfandnotg);
  bdd_free(m->bddm, notf);
  bdd_free(m->bddm, notfandnotg);
  result = model_CheckAllocComputation
    (bdd_notup(m->bddm, model_CheckQSolution(tmp)), model_CheckQInfo(tmp));
  model_CheckFreeComputation(tmp);
  return result;
}


/** Function **
  Synopsis    [ Checks delta convergence of model ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
void
model_CheckDeltaConvergence
(Model m,
 FILE * outstream)
{
  ModelCheckComputation tmp;
  tmp = model_CheckAF(m, outstream, m->stable);
  model_CheckAG(m, outstream, model_CheckQSolution(tmp));
  model_CheckFreeComputation(tmp);
}

/** Function **
  Synopsis    [ Allocates the result for a model checking computation ]
  Description [ Allocates memory data gathered during a model checking
                computation and stores the parameters in the result. 
                The reference counters of the stored BDD is _not_
                incremented. ]
  SideEffects [ Memory is allocated with the mem(3) package ]
  SeeAlso     [ model_CheckFreeComputation ]
 */
ModelCheckComputation /* the allocated data structure */
model_CheckAllocComputation
(bdd solution,
 pointer info)
{
  ModelCheckComputation result;
  result = (ModelCheckComputation) mem_new_rec(ModelCheckRecMgr);
  result -> solution = solution;
  result -> info = info;
  return result ;
}

/** Function **
  Synopsis    [ Frees the result of a model checking computation ]
  Description [ Frees the result of a call to a model checking routine.
                It is up to the caller to free the bdds contained in
                the freed data structure. ]
  SideEffects [ Memory is freed via the mem(3) package ]
  SeeAlso     [ model_CheckAllocComputation ]
 */
void model_CheckFreeComputation
(ModelCheckComputation data /* the data structure to disallocate */ )
{
  mem_free_rec(ModelCheckRecMgr, (pointer) data);
}


/** Function **
  Synopsis    [ Gets the solution of a model checking computation ]
  Description [ Returns the solution (the condition characterizing 
                the set of states) of a model checking computation. ]
  SideEffects [ None ]
  SeeAlso     [ model_CheckQInfo ]
 */
bdd /* the solution of a model checking computation */
model_CheckQSolution (
  ModelCheckComputation result /* result of a previous model checking 
                                  computation */)
{
  return result->solution;
}

/** Function **
  Synopsis    [ Gets the info associated to a model checking computation ]
  Description [ Returns the information stored during a model checking 
  computation.  The following summarizes what first fixpoint is computed
  for which CTL formula: 
     . EX, AX: none
     . EG f: (vbdd *) last fixpoint approximation sequences for fairness
     . AG f: (vbdd) fixpoint of E\[1 U ~f\]
     . EF f: (vbdd) fixpoint of E\[1 U f\]
     . AF f: (vbdd *) fixpoint of EG ~f
     . E\[fUg\]: (vbdd) least fixpoint of K = g | (f & EX K)
     . A\[fUg\]: (vbdd[2]) fixpoint of E\[~f U ~f&~g\] and fixpoint of EG~g
     . E\[fWg\]: (vbdd[2]) fixpoint of E\f[f U g\] and fixpoint EGf
     . A\[fWg\]: (vbdd) fixpoint of E\[~f U ~g\] ]
  SideEffects [ None ]
  SeeAlso     [ model_CheckQInfo ]
 */
pointer /* the VBDD containing the fixpoint computation steps */
model_CheckQInfo
(ModelCheckComputation result /* result of a previous model checking computation */)
{
  return result->info;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Semantics of CTL formula EXf ]
  Description [ Performs a preimage computation to determine the 
                value of the CTL formula EXf in the model m.
                Returns the BDD of the set of m-states where EXf
                stands.  
                The call is traced to outstream with the string
                  (EX) ]
  SideEffects [ Output is sent to outstream.
                The BDD data under the management of bddm is modified.  ]
  SeeAlso     [ _BasicCheckEU _BasicCheckEG ]
 */
static bdd /* the characteristic predicate of the solution set of states */
_BasicCheckEX (
  Model m /* Kripke structure model */, 
  FILE * outstream /* stream where a trace of the fixpoint computation is printed */,
  bdd f /*  argument of the CTL formula */)
{

  /* declarations */
   bdd solution, fair;

  /* statements */
   fprintf(outstream, "(EX"); fflush(outstream);
   if (m->fair_states) {
     fair = bdd_and(m->bddm, f, m->fair_states);
   } else {
     fair = bdd_identity(m->bddm, f);
   }
   solution = model_ReachabilityStepBackward(m, fair);
   bdd_free(m->bddm, fair);
   fprintf(outstream, ")\n"); fflush(outstream);
   return(solution) ;
}

/** Function **
  Synopsis    [ Semantics of CTL formula E\[fUg\] ]
  Description [ Performs a symbolic least fixpoint computation to 
                determine the value of the CTL formula E\[fUg\] in 
                the model m.
                Returns the BDD of the set of m-states where EGf
                stands.  
                The parameter frontiers must have been initialized 
                before the call (e.g. with vbdd_new). If frontiers 
                is a non null VBDD, the set of the new states found
                at the successive iterations are added to frontiers, 
                i.e.
                \[g; ~g & (f&EXg); ~g & ~(f&EXg) & (f&EX(f&EXg)); ...\] 
                If the set is empty (fixpoint reached), it is not 
                added to frontiers.
                Successive iterations are traced to outstream, a la 
                LaTeX: each time an iteration starts, a left square 
                bracket followed by the iteration number is printed, a 
                right square bracket is printed when the iteration 
                ends, i.e.
                  (EU \[1\] \[2\] \[3\] ...) ]
  SideEffects [ The contents of the passed-by-reference parameter frontiers 
                is modified.
                Output is sent to outstream.
                The BDD data under the management of bddm is modified.  ]
  SeeAlso     [ _BasicCheckEG _BasicCheckEX ]
 */
static bdd /* the characteristic predicate of the solution set of states */
_BasicCheckEU (
  Model m /* Kripke structure model */, 
  FILE * outstream /* stream where a trace of the fixpoint computation is printed */,
  bdd f, /* first argument of the CTL formula */
  bdd g, /* second argument of the CTL formula */
  vbdd frontiers /* reference-passing parameter storing the fixpoint steps */)
{
  /* declarations */
  bdd_manager bddm = m->bddm ;
  bdd solution, prev_solution;
  bdd reachable, fair;
  int step;

  /* statements */

   if (m->fair_states) {
     fair = bdd_and(bddm, g, m->fair_states);
   } else {
     fair = bdd_identity(bddm, g);
   }

  solution = bdd_identity(bddm, fair) ;
  if (frontiers)
    vbdd_add(bddm, frontiers, fair);

  /* if g stands globally, E[f U g] stands globally as well */

  if ((reachable = model_ReachabilityQAll(m)) == 0) {
    reachable = bdd_one(bddm);
  }
  if (solution == reachable)
    return solution;

  /* fixpoint */
  step= 0;
  prev_solution = bdd_identity(bddm, solution) ;
  fprintf(outstream, "(EU"); fflush(outstream);
  do {
    bdd_free(bddm, prev_solution) ;
    fprintf(outstream, " [%u", ++step); fflush(outstream);
    prev_solution = bdd_identity(bddm, solution) ;
    solution =
         bdd_orup(bddm, solution,
                  bdd_andup2(bddm, f, model_ReachabilityStepBackward(m, solution))) ;
    if (frontiers) {
      bdd tmp;
      tmp = bdd_andup2(bddm, solution, bdd_not(bddm, prev_solution));
  
      if (tmp != bdd_zero(bddm)) {
        vbdd_add(bddm, frontiers, tmp);
      }
      bdd_free(bddm, tmp);
    }
    fprintf(outstream, "]"); fflush(outstream);
  /* fixpoint found or the solution is the reachable states */
  } while ((solution != prev_solution) && (solution != reachable));

  bdd_free(bddm, prev_solution);
  bdd_free(bddm, fair);
  fprintf(outstream, ")\n");
  return(solution) ;
}


/** Function **
  Synopsis    [ Semantics of CTL formula EGf ]
  Description [ Performs a symbolic least fixpoint computation to 
  determine the value of the CTL formula EGf in the model m.
  Returns the BDD of the set of m-states where EGf stands.  
    gfp Y (f \land \bigwedge_{h\in H} EXE[f U h \land Y])
  H is the set of fairness constraints.
  Approximations must point to a memory location large enough 
  to store the last sequences of approximations in the nested
  fixpoint computations of the CheckFairEG algorithm:
          E[f U h\land Y]
  approximations will be computed as a null-terminated array
  of vectors of BDDs.
  Successive iterations are traced to outstream, a la LaTeX: each time
  an iteration starts, a left square bracket followed by the iteration
  number is printed, a right square bracket is printed when the iteration 
  ends, i.e. (EG \[1\] \[2\] \[3\] ...)
  Fixpoint computations to evaluate fairness constraints are also
  traced to outstream. ]
  SideEffects [ The contents of the passed-by-reference parameter 
  approximations is modified. Output is sent to outstream.
  The BDD data under the management of bddm is modified.  ]
  SeeAlso     [ _BasicCheckEU _BasicCheckEX ]
 */
static bdd /* the characteristic predicate of the solution set of states */
_BasicCheckEG (
  Model m /* Kripke structure model */, 
  FILE * outstream /* stream where a trace of the fixpoint computation is printed */,
  bdd f, /*  argument of the CTL formula */
  vbdd ** approximations /* reference-passing result containing nested fixpoint
                            computation sequences for fairness constraints */)
{
  /* declarations */
  bdd_manager bddm = m->bddm ;
  bdd solution, prev_solution ;
  int step;
  int nb_fairness_constraints;

  /* statements */
  {
    cvn constraint;
    int counter;
    for (nb_fairness_constraints = 0, constraint = m->fairness_constraints;
         constraint;
         ++nb_fairness_constraints,
         constraint = qNext(constraint)) ;
    * approximations =
      (vbdd *) mem_get_block((SIZE_T) (nb_fairness_constraints+1)*sizeof(vbdd));
    for (counter = 0; counter < nb_fairness_constraints; ++counter) {
      (* approximations)[counter] = vbdd_new();
    }
    (* approximations)[nb_fairness_constraints] = 0;
  }
  
  step= 0;
  solution= bdd_identity(bddm, f) ;

  /* the fixpoint */
  prev_solution = bdd_identity(bddm, solution) ;
  fprintf(outstream, "(EG"); fflush(outstream);
  do {
    bdd next; 
    bdd_free(bddm, prev_solution) ;
    if (step != 0) {
      int counter;
      for (counter = 0; counter < nb_fairness_constraints; ++counter) {
        vbdd_free(bddm, (* approximations)[counter]);
        (* approximations)[counter] = vbdd_new();
      }
    }
    fprintf(outstream, " [%u", ++step); fflush(outstream);
    prev_solution = bdd_identity(bddm, solution) ;
    next = _FairIteration(m, outstream, solution, 
                          m->fairness_constraints, * approximations);
    solution = bdd_andup(bddm, solution, next);
    next = model_ReachabilityStepBackward(m, solution);
    solution = bdd_andup(bddm, solution, next);
    fprintf(outstream, "]"); fflush(outstream);
  } while (solution != prev_solution);
  bdd_free(bddm, prev_solution);
  fprintf(outstream, ")\n");
  return solution;
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static bdd
_FairIteration
(Model m,
 FILE * outstream,
 bdd g,
 cvn fairness_constraint,
 vbdd * approximations)
{
  bdd first, rest;
  if (fairness_constraint == 0)
    return bdd_identity(m->bddm, bdd_one(m->bddm));

  first = 
    _FairIterationAux(m, outstream, g, 
                      (bdd) qToolInfo(qValue(fairness_constraint)),
                      approximations);
  rest = _FairIteration(m, outstream, g, qNext(fairness_constraint),
                        approximations+1);
  return bdd_andup(m->bddm, first, rest);
}

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static bdd
_FairIterationAux
(Model m,
 FILE * outstream,
 bdd g,
 bdd fairness_constraint,
 vbdd * approximations)
{
  bdd destination, result;
  destination = bdd_and(m->bddm, g, fairness_constraint);
  result =
    _BasicCheckEX(m, outstream,
                  _BasicCheckEU(m, outstream, g,
                                destination, *approximations));
  bdd_free(m->bddm, destination);
  return result;
}

