/** CFile **

  FileName    [ imgReduce.c ]

  PackageName [ img ]

  Synopsis    [ Reduction routines ]

  Description [ This file contains routines that can be
  called to decrease the size of the transition representations. ]

  SeeAlso     [ ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 3.1 ]

***********************************************************************/

#include "imgInt.h"

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/
typedef struct graph_ * graph;
typedef struct vertex_rec_ vertex_rec, * vertex;
typedef struct edge_rec_ edge_rec, * edge;
typedef void (* vertex_map) (vertex, void *);

typedef struct bdd_id_rec_ bdd_id_rec, * bdd_id;
typedef struct scc_rec_ scc_rec, * scc;

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

struct graph_ {
  long nb_vertices;
  vertex_rec * vertices;
  vertex * reverse;
  rec_mgr edge_mgr;
};

struct vertex_rec_ {
  long finish;
  int mark;
  long data;
  edge edges;
};

struct edge_rec_ {
  edge link;
  vertex source;
  vertex target;
};

struct bdd_id_rec_ {
  bdd_id link;
  long id;
};

struct scc_rec_ {
  scc link;
  rec_mgr bdd_id_mgr;
  bdd_id vertices;
};

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void _AddEdge(rec_mgr m, vertex s, vertex t);
static void _SccFree(scc s);
static void _GraphFree(graph g);
static void _GraphBuildReverse(graph g);
static graph _DependencyGraph(img_manager t);
static void _FinishingTimesAux(vertex v, long * finish);
static void _FinishingTimes(graph g);
static graph _TransposeGraph(img_manager t, graph g);
static void _AddId(vertex v, void * env);
static void _DFSVertex(vertex v, vertex_map f, void * env);
static void _DFSEdge(edge e, vertex_map f, void * env);
static scc _SCCBuild(graph g);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Reduce transitions on care set ]
  Description [ Reduces the transition functions by making the
  hypothesis that all transitions are taken from the care set. 
  If care is equal to bdd_zero, the routine does nothing. ]
  SideEffects [ If a BDD overflow occurs, invokes the routine
  bdd_resize_manager on the BDD manager of t. ]
 */
void
img_ReduceCareSet
(img_manager t,
 bdd care)
{
  bdd_manager bddm = t->bddm;
  Atom a;
  if (care == bdd_zero(bddm))
    return;
  for (a = t->transitions; a; a = a->link) {
    do {
      bdd tmp = bdd_reduce(bddm, a->f, care);
      if (bdd_overflow(bddm)) {
        bdd_resize_manager(bddm);
      } else {
        bdd_free(bddm, a->f);
        a->f = tmp;
        break;
      }
    } while (1);
  }
  imgBuildFunctionOfVar(t);
}

/** Function **
  Synopsis    [ Experimental simplification heuristic ]
  Description [ Let R be the set of states reachable from an initial
  set of states. Tries to find a set of states that is 1) easier to
  compute than R, 2) contains R, 3) can be used to simplify the
  transition representation. ]
  SideEffects [ Modifies the transition representation, manipulates
  BDDs and dynamically allocates and frees memory. ]
  SeeAlso     [ ]
 */
void 
img_ReduceSCC
(img_manager t /* the considered transition system */,
 bdd init /* the set of initial states */)
{
  bdd_manager bddm = t->bddm;
  graph dependency, transpose;
  scc SCCs;

  dependency = _DependencyGraph(t);
  /* Compute SCCs of dependency graph */
  _FinishingTimes(dependency);
  transpose = _TransposeGraph(t, dependency);
  _GraphFree(dependency);
  SCCs = _SCCBuild(transpose);

  /* Print information about the SCCs */

  /* if the number of SCCs is greater than one, reduce */  
  if (SCCs->link != 0) {

    /* do not consider SCCs of size 1 */
    {
      scc c, p, n;
      p = 0; 
      for (c = SCCs; c; c = n) {
        n = c->link;
        if (c->vertices->link == 0) {
          if (p == 0) {
            SCCs = n;
          } else {
            p->link = n;
          }
          _SccFree(c);
        } else {
          p = c;
        }
      }
    }
    
    while (SCCs) {
      RangeVector r;
      bdd approximation;
      bdd_id i;
      scc n;
      int count;
      r = imgRangeVectorNew(t);
      for (i = SCCs->vertices, count = 0; i; i = i->link, ++count) {
        long id = i->id;
        imgRangeVectorAdd(t, r, 
                          bdd_var_with_id(bddm, id), 
                          imgFunctionOfVar(t, id));
      }
      approximation = imgRangeVectorReachable(t, r, init);
      img_ReduceCareSet(t, approximation);
      bdd_free(bddm, approximation);
      img_Flush(t);
      n = SCCs->link;
      _SccFree(SCCs);
      SCCs = n;
    }
  }
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_AddEdge
(rec_mgr m, vertex s, vertex t)
{
  edge e = (edge) mem_new_rec(m);
  e->source = s;
  e->target = t;
  e->link = s->edges;
  s->edges = e;
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_SccFree
(scc s)
{
  mem_free_rec_mgr(s->bdd_id_mgr);
  mem_free_block(s);
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_GraphFree
(graph g)
{
  mem_free_rec_mgr(g->edge_mgr);
  mem_free_block(g->vertices);
  mem_free_block(g->reverse);
  mem_free_block(g);
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_GraphBuildReverse
(graph g)
{
  long i;
  vertex vp;
  for (i = 0, vp = & g->vertices[0]; i < g->nb_vertices; ++i, ++vp) {
    g->reverse[vp->data] = vp;
  }
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static graph
_DependencyGraph
(img_manager t)
{
  bdd_manager bddm = t->bddm;
  long nb_vertices;
  vertex vp;
  Atom a;
  graph result;
  SIZE_T size;

  for (a = t->transitions, nb_vertices = 0; a; a = a->link, ++nb_vertices);

  result = (graph) mem_get_block((SIZE_T) sizeof(struct graph_));
  result->nb_vertices = nb_vertices; 
  result->edge_mgr = mem_new_rec_mgr(sizeof(edge_rec));

  size = (SIZE_T) nb_vertices * sizeof(vertex_rec);
  result->vertices = (vertex_rec *) mem_get_block(size);
  mem_zero((pointer) result->vertices, size);

  size = (SIZE_T)  bdd_vars(bddm) * sizeof(vertex);
  result->reverse = (vertex *) mem_get_block(size);
  mem_zero((pointer) result->reverse, size);

  for (a = t->transitions, vp = & result->vertices[0]; a; a = a->link, ++vp) {
    vp->data = bdd_if_id(bddm, a->v);
  }

  _GraphBuildReverse(result);

  {
    bdd * support = bdd_new_support(bddm), * v; 

    for (a = t->transitions; a; a = a->link) {
      vertex source = result->reverse[bdd_if_id(bddm, a->v)];
      bdd_support(bddm, a->f, support);
      for (v = support; * v; ++v) {
        long vid = bdd_if_id(bddm, * v);
        if (imgFunctionOfVar(t, vid)) {
          _AddEdge(result->edge_mgr, source, result->reverse[vid]);
        }
      }
    }
  }

  return result;

}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_FinishingTimesAux
(vertex v,
 long * finish)
{
  if (v->mark == 0) {
    edge e;
    v->mark = 1; 
    ++*finish;
    for (e = v->edges; e; e = e->link) {
      _FinishingTimesAux(e->target, finish);
    }
    ++*finish;
    v->finish = *finish;
  }
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_FinishingTimes
(graph g)
{
  long finish, i;
  vertex v;
  finish = 0;
  for (i = 0, v = & g->vertices[0]; i < g->nb_vertices; ++i, ++v) {
    _FinishingTimesAux(v, & finish);
  }
  for (i = 0, v = & g->vertices[0]; i < g->nb_vertices; ++i, ++v) {
    v->mark = 0;
  }
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static graph
_TransposeGraph
(img_manager t,
 graph g)
{
  vertex gs, rs;
  long i, nb_vertices;
  graph result;
  SIZE_T size;

  nb_vertices = g->nb_vertices;
  /* copies g and its fields into result */
  result = (graph) mem_get_block((SIZE_T) sizeof(struct graph_));
  result->nb_vertices = nb_vertices;

  size = (SIZE_T) result->nb_vertices * sizeof(vertex_rec);
  result->vertices = (vertex_rec *) mem_get_block(size);
  mem_copy((pointer) result->vertices, (pointer) g->vertices, size);

  size = (SIZE_T) bdd_vars(t->bddm) * sizeof(vertex);
  result->reverse = (vertex *) mem_get_block(size);
  _GraphBuildReverse(result);
  
  result->edge_mgr = mem_new_rec_mgr((SIZE_T) sizeof(edge_rec));

  /* removes edges from result */
  for (i = 0, rs = & result->vertices[0]; i < nb_vertices; ++i, ++rs) {
    rs->edges = 0;
  }

 
  /* for each vertex in g, for each edge outgoing this vertex,
     add an edge in result */
  for (i = 0, gs = & g->vertices[0], rs = & result->vertices[0]; 
       i < nb_vertices;
       ++i, ++gs, ++rs) {
    edge e;
    for (e = gs->edges; e; e = e->link) {
      _AddEdge(result->edge_mgr, result->reverse[e->target->data], rs);
    }
  }

  return result;
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_AddId
(vertex v,
 void * env)
{
  scc c  = (scc) env;
  bdd_id r = (bdd_id) mem_new_rec(c->bdd_id_mgr);
  r->link = c->vertices;
  r->id = v->data;
  c->vertices = r;
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_DFSVertex
(vertex v,
 vertex_map f,
 void * env)
{
  if (v->mark) return;
  v->mark = 1;
  (* f)(v, env);
  _DFSEdge(v->edges, f, env);
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static void
_DFSEdge
(edge e,
 vertex_map f,
 void * env)
{
  if (e != 0) {
    _DFSVertex(e->target, f, env);
    _DFSEdge(e->link, f, env);
  }
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ ]
  SeeAlso     [ ]
 */
static scc
_SCCBuild
(graph g)
{
  long i, max;
  vertex v, elected;
  scc result;
  result = 0;
  do {
    max = -1; elected = 0;
    for (i = 0, v = & g->vertices[0]; i < g->nb_vertices; ++i, ++v) {
      if ((v->mark == 0) && (v->finish > max)) {
        max = v->finish;
        elected = v;
      }
    }
    if (elected) {
      scc c = (scc) mem_get_block((SIZE_T) sizeof(scc_rec));
      c->link = result;
      c->bdd_id_mgr = mem_new_rec_mgr((SIZE_T) sizeof(bdd_id_rec));
      c->vertices = 0;
      result = c;
      _DFSVertex(elected, (vertex_map) & _AddId, (void *) c);
    }
  } while (elected);
  return result;
}

