/** CFile **************************************************************

  FileName    [ imgCompute.c ]

  PackageName [ img ]

  Synopsis    [ Image computation ]

  Description [ This file contains routines that implement the
  image computation based on range computation (published by
  Coudert and Madre 1991). ]

  SeeAlso     [ ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 3.1 ]

***********************************************************************/

#include "imgInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static RangeClass _Partition(img_manager t, Atom vector);
static RangeClass _PartitionGetRangeClass(img_manager t, bdd v);
static void _PartitionSetRangeClass(img_manager t, RangeClass c, bdd * support);
static void _PartitionMergeRangeClass(img_manager t, RangeClass c1, RangeClass c2);
static bdd _RangeCache(img_manager t, RangeClass c);
static bdd _TopVariable(bdd_manager bddm, Atom vector);
static bdd _RangeRecurse(img_manager t, RangeClass c);

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Image computation ]
  Description [ Returns the image of set by img. ]
  SideEffects [ While BDD overflows occur, the function 
  bdd_resize_manager is repeatedly called. ]
  SeeAlso     [ ]
 */

bdd
img_Compute
(img_manager t,
 bdd set)
{
  bdd_manager bddm = t->bddm;
  Atom a;
  bdd result;
  RangeVector c;
  do {
    c = imgRangeVectorNew(t);
    for (a = t->transitions; a; a = a->link) {
      imgRangeVectorAdd(t, c, a->v, imgRangeRestrict(bddm, a->f, set));
    }
    result = imgRange(t, c);
    if (result == 0) {
      bdd_resize_manager(bddm);
    } else {
      break;
    }
  } while(1);
  bdd_gc(bddm);
  return result;
}

/** Function **
  Synopsis    [ Reachable states computation ]
  Description [ Returns the set of states reachable from init by
  repeatedly taking the transitions defined by t. ]
  SideEffects [ While BDD overflows occur, the function 
  bdd_resize_manager is repeatedly called. ]
  SeeAlso     [ ]
 */

bdd
img_ComputeReachableStates
(img_manager t,
 bdd init)
{
  bdd_manager bddm = t->bddm;
  bdd result = bdd_zero(bddm);
  return result;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Generalized cofactor of f and g ]
  SideEffects [ While BDD overflows occur, bdd_resize_manager
  is invoked on the BDD manager bddm ]
 */
bdd
imgRangeRestrict
(bdd_manager bddm,
 bdd f,
 bdd g)
{
  bdd tmp = bdd_cofactor(bddm, f, g);
  if (bdd_overflow(bddm)) {
    bdd_resize_manager(bddm);
    return imgRangeRestrict(bddm, f, g);
  } else {
    return tmp;
  }
}

/** Function **
  Synopsis    [ Range of a function vector ]
  Description [ Returns the range of vector. The range
  is the characteristic function of the set of values
  that are the result of the application of the transition
  functions contained in the vector. 
  An outline of the algorithm: using the support of the
  atomic transition functions, partition the vector of
  functions. Apply _RangeCache on the result subvectors, 
  return a conjunction of the result ranges with the
  BDD info of vector (that represents transition functions
  that could have been eliminated). ]
  SideEffects [ Uses an internal cache to store intermediate
  results. The range computation is complex and requires
  extensive operations on BDDs. While BDD overflows occur,
  the routine bdd_resize_manager is invoked. The result is
  therefore always significant. Also, vector is deallocated
  inside this routine. ]
  SeeAlso     [ _RangeCache _RangeRecurse ]
 */
bdd
imgRange
(img_manager t,
 RangeVector vector)
{
  bdd_manager bddm = t->bddm;
  bdd result;
  RangeClass classes;
  int overflow;
  classes = _Partition(t, vector->contents);
  result = bdd_identity(bddm, vector->info);
  overflow = 0;
  while (classes && !overflow) {
    RangeClass elem;
    elem = classes;
    classes = elem->link;
    result = bdd_andup(bddm, _RangeCache(t, elem), result);
    overflow = bdd_overflow(bddm);
  }
  imgRangeVectorFree(t, vector);
  if (overflow) {
    return 0;
  } else {
    return result;
  }
}

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ required ]
  Description [ optional ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */
static RangeClass
_Partition
(img_manager t,
 Atom vector)
{
  RangeClass result;
  bdd_manager bddm = t->bddm;
  result = 0;
  mem_zero((pointer) t->var_id_to_class,
           (SIZE_T) sizeof(RangeClass) * t->var_id_to_class_size);
  while (vector) {
    bdd * ptr, v = vector->v, f = vector->f;
    RangeClass f_class;
    bdd_support(bddm, f, t->support);
    f_class = 0;
    ptr = t->support;
    while (* ptr) {
      RangeClass v_class = _PartitionGetRangeClass(t, * ptr);
      if (v_class != 0) {
        if (v_class != f_class) {
          if (f_class == 0) {
            f_class = v_class;
          } else {
            /* remove v_class from partition */
            if (v_class == result) { 
              result = v_class->link;
            } else {
              v_class->prelink->link = v_class->link;
            }
            if (v_class->link) {
              v_class->link->prelink = v_class->prelink;
            }
            _PartitionMergeRangeClass(t, f_class, v_class);
            f_class = imgRangeClassMerge(t, f_class, v_class);
          }
        } 
      }
      ++ptr;
    }
    if (f_class == 0) {
      f_class = imgRangeClassNew(t);
      if (result) {
        result->prelink = f_class;
      }
      f_class->link = result;
      f_class->prelink = 0;
      result = f_class;
    }
    imgRangeClassAdd(t, f_class, v, f);
    _PartitionSetRangeClass(t, f_class, t->support);
    vector = vector->link;
  }
  return result;
}

/** Function **
  Synopsis    [ Returns the class in current partition ]
  Description [ optional ]
  SideEffects [ None ]
  SeeAlso     [ _Partition ]
 */
static RangeClass
_PartitionGetRangeClass
(img_manager t,
 bdd v)
{
  long id = bdd_if_id(t->bddm, v);
  if (id > t->var_id_to_class_size) {
    return 0;
  }
  return t->var_id_to_class[id];
}

/** Function **
  Synopsis    [ Associates a class to a set of variables ]
  Description [ For each BDD variable in support, sets the value 
  of its map t->var_id_to_class_size to c. ]
  SideEffects [ t->var_id_to_class_size is dynamically resized
  to accomodate new variables ]
  SeeAlso     [ _Partition ]
 */
static void
_PartitionSetRangeClass
(img_manager t,
 RangeClass c,
 bdd * support)
{
  while (* support) {
    long id = bdd_if_id(t->bddm, * support);
    if (id > (t->var_id_to_class_maxsize - 1)) {
      SIZE_T buffer_half_size;
      buffer_half_size = (SIZE_T) t->var_id_to_class_maxsize * sizeof(RangeClass);
      t->var_id_to_class = 
        mem_resize_block((pointer) t->var_id_to_class, 2 * buffer_half_size);
      mem_zero((pointer) (t->var_id_to_class + t->var_id_to_class_maxsize),
               buffer_half_size);
      t->var_id_to_class_maxsize *= 2;
    }
    if (id > (t->var_id_to_class_size - 1)) {
      t->var_id_to_class_size = id + 1;
    }
    t->var_id_to_class[id] = c;
    ++support;
  }
}

/** Function **
  Synopsis    [ Merge range classes in current partition ]
  Description [ For each BDD variable, if it is mapped to
  class c2 in t->var_id_to_class, maps it to c1 instead. ]
  SideEffects [ Modifies t->var_id_to_class ]
  SeeAlso     [ _Partition ]
 */
static void
_PartitionMergeRangeClass
(img_manager t,
 RangeClass c1,
 RangeClass c2)
{
  RangeClass * ptr;
  long i;
  if ((c1 == 0) || (c2 == 0)) {
    return;
  }
  for (ptr = t->var_id_to_class, i = t->var_id_to_class_size;
       i;
       ++ptr, --i) {
    if (* ptr == c2) {
      * ptr = c1;
    }
  }
}

/** Function **
  Synopsis    [ Looks for result in cache ]
  Description [ If the contents length of c is 0 or 1, the
  result is bdd_one. Otherwise, looks up if the cache contains
  the result for an equivalent class. If yes, returns it. If no
  or if an overflow has occured (in which case the result might
  not be significant and the cache reset), apply _RangeRecurse
  to actually compute the range. If an overflow occurs, the
  routine bdd_resize_manager is invoked. ]
  SideEffects [ If c is not put in the cache, it is deallocated.
  BDD operations occur. If an overflow happens, the routine 
  bdd_resize_manager is invoked. ]
  SeeAlso     [ imgRange _RangeRecurse ]
 */
static bdd
_RangeCache
(img_manager t,
 RangeClass c)
{
  RangeClass in_cache;
  bdd_manager bddm = t->bddm;
  bdd result;
  int overflow;
  if ((c->contents == 0) || (c->contents->link == 0)) {
    result = bdd_one(bddm);
    imgRangeClassFree(c, t);
  } else {
    if ((in_cache = (RangeClass) cache_1Find(t->cache, (void *) c)) != 0) {
      bdd_temp_assoc(bddm, t->support, 1);
      bdd_assoc(bddm, -1);
      result = bdd_substitute(bddm, in_cache->range);
    }
    if ((overflow = bdd_overflow(bddm)) != 0) {
      bdd_resize_manager(bddm);
    }
    if (overflow || (in_cache == 0)) {
      c->range = bdd_identity(bddm, _RangeRecurse(t, c));
      cache_1Insert(t->cache, c);
      result = c->range;
    } else {
      imgRangeClassFree(c, t);
    }
  }
  return result;
}

/** Function **
  Synopsis    [ Returns variable at top of functions  ]
  Description [ Traverses the vector of transition functions
  and returns the variable with minimal index found in their
  support ]
  SideEffects [ None ]
 */
static bdd
_TopVariable
(bdd_manager bddm,
 Atom vector)
{
  long min, measure;
  bdd result;
  result = vector->f;
  min = bdd_if_index(bddm, result);
  for (vector = vector->link; vector; vector = vector->link) {
    measure = bdd_if_index(bddm, vector->f);
    if (measure < min) {
      result = vector->f;
      min = measure;
    }
  }
  return result;
}

/** Function **
  Synopsis    [ Compute range by means of decomposition ]
  Description [ This routine determines a range restrictor
  for the current class, apply recursively the range computation
  on two smaller vectors obtained with this range restrictor, 
  and compose the results to obtain the range.  ]
  SideEffects [ Invokes BDD routines and checks for overflow,
  calling bdd_resize_manager while required. ]
  SeeAlso     [ imgRange _RangeCache ]
 */
static bdd
_RangeRecurse
(img_manager t,
 RangeClass c)
{
  bdd_manager bddm = t->bddm;
  RangeVector c0, c1;
  bdd r0 = 0,
      r1 = 0;
  Atom ptr;
  bdd not_function, function;

  function = _TopVariable(bddm, c->contents);
  not_function = bdd_not(bddm, function);

  c0 = imgRangeVectorNew(t);
  c1 = imgRangeVectorNew(t);
  for (ptr = c->contents; ptr; ptr = ptr->link) {
    imgRangeVectorAdd(t, c0, ptr->v,
                     imgRangeRestrict(bddm, ptr->f, not_function));
    imgRangeVectorAdd(t, c1, ptr->v,
                     imgRangeRestrict(bddm, ptr->f, function));
  }
  do {
    r0 = imgRange(t, c0);
    if (r0 == 0) {
      bdd_resize_manager(bddm);
      imgRangeVectorFree(t, c1);
      c0 = imgRangeVectorNew(t);
      c1 = imgRangeVectorNew(t);
      for (ptr = c->contents; ptr; ptr = ptr->link) {
        imgRangeVectorAdd(t, c0, ptr->v,
                         imgRangeRestrict(bddm, ptr->f, not_function));
        imgRangeVectorAdd(t, c1, ptr->v,
                         imgRangeRestrict(bddm, ptr->f, function));
      }
    }
  } while (r0 == 0);
  do {
    r1 = imgRange(t, c1);
    if (r1 == 0) {
      bdd_resize_manager(bddm);
      c1 = imgRangeVectorNew(t);
      for (ptr = c->contents; ptr; ptr = ptr->link) {
        imgRangeVectorAdd(t, c1, ptr->v,
                         imgRangeRestrict(bddm, ptr->f, function));
      }
    }
  } while (r1 == 0);
  bdd_free(bddm, not_function);

  do {
    bdd tmp = bdd_or(bddm, r0, r1);
    if (bdd_overflow(bddm)) {
      bdd_resize_manager(bddm);
    } else {
      bdd_free(bddm, r0);
      bdd_free(bddm, r1);
      return tmp;
    }
  } while (1);
}

