/** CFile **************************************************************

  FileName    [ img.c ]

  PackageName [ img ]

  Synopsis    [ Basic creation and destruction routines ]

  Description [ This file contains miscellaneous routines on
  img, including allocation and disallocation. ]

  SeeAlso     [ ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ 3.1 ]

***********************************************************************/

#include "imgInt.h"

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ img constructor ]
  Description [ Creates and initializes a img manager. 
  Multiple managers may exist at the same time. ]
  SideEffects [ Allocates memory. ]
  SeeAlso     [ img_Free ]
 */

img_manager
img_New
(bdd_manager bddm, 
 vbdd variables, 
 vbdd functions)
{
  img_manager t ;
  int nbvar ;

  nbvar = vbdd_length(variables) ;

  t = (img_manager) mem_get_block((SIZE_T) sizeof(img_manager_rec)) ;
  t->bddm = bddm;
  t->RangeClassMgr = mem_new_rec_mgr((SIZE_T) sizeof(RangeClassRec));
  t->RangeVectorMgr = mem_new_rec_mgr((SIZE_T) sizeof(RangeVectorRec));
  t->AtomMgr = mem_new_rec_mgr((SIZE_T) sizeof(AtomRec));
  t->cache = cache_1New(nbvar * nbvar, 
                        (hash_function) & imgRangeClassHash,
                        (eq_function) & imgRangeClassEq,
                        (del_function) & imgRangeClassFree,
                        (pointer) t, 
                        (float) 0, (float) 1);
  {
    bdd * v, * f;
    Atom a, p;
    v = vbdd_elements(variables); f = vbdd_elements(functions);
    a = mem_new_rec(t->AtomMgr);
    t->transitions = a;
    a->v = * v; a->f = * f; 
    for (++v, ++f; * v; ++v, ++f) { 
      p = a;
      a = mem_new_rec(t->AtomMgr);
      a->v = * v; a->f = * f; 
      p->link = a ;
    }
    a->link = 0;
  }
  {
    SIZE_T size;
    t->function_of_var_size = bdd_vars(bddm);
    size = (SIZE_T)  t->function_of_var_size * sizeof(bdd);
    t->function_of_var = (bdd *) mem_get_block(size);
    mem_zero((pointer) t->function_of_var, size);
  }
  imgBuildFunctionOfVar(t);
  {
    SIZE_T buffer_size;
    t->var_id_to_class_size = 0;
    t->var_id_to_class_maxsize = 2 * nbvar;
    buffer_size = sizeof(RangeClass) * t->var_id_to_class_maxsize;
    t->var_id_to_class = (RangeClass *) mem_get_block(buffer_size);
    mem_zero((pointer) t->var_id_to_class, buffer_size);
  }
  t->support = bdd_new_support(bddm);
  return t ;
}

/** Function **
  Synopsis    [ img destructor ]
  Description [ Dellocates the given img_manager and all memory
  associated with it. ]
  SideEffects [ required ]
  SeeAlso     [ optional ]
 */

void img_Free
(img_manager transition)
{
  bdd_manager bddm = transition->bddm;
  mem_free_rec_mgr(transition->RangeVectorMgr);
  mem_free_rec_mgr(transition->RangeClassMgr);
  mem_free_rec_mgr(transition->AtomMgr);
  cache_1Reset(transition->cache);
  bdd_free_support(bddm, transition->support);
  mem_free_block(transition) ;
}

/** Function **
  Synopsis    [ Library version number ]
  SideEffects [ None ]
 */
char * img_Version
(void)
{
  return "development version 3.1";
}

/** Function **
  Synopsis    [ Size of transition representation ]
  SideEffects [ None ]
 */
long
img_Size
(img_manager t)
{
  long sum;
  Atom a;
  bdd_manager bddm = t->bddm;
  for (a = t->transitions, sum = 0; 
       a;
       sum += bdd_size(bddm, a->f, 0), a = a->link);
  return sum;
}

/** Function **
  Synopsis    [ Flushes internal data structures ]
  Description [ This routine reset the internal caches
  of the given img manager. This usually frees a large
  number of BDD nodes. ]
  SideEffects [ Reference counters of internal BDDs are
  decremented ]
 */
void
img_Flush
(img_manager t)
{
  cache_1Reset(t->cache);
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Returns the transition function of a variable ]
  Description [ If v has an associated transition function in t,
  returns it, otherwise, returns 0. ]
  SideEffects [ none ]
 */
bdd
imgGetFunction
(img_manager t,
 bdd v)
{
  Atom a = t->transitions;
  while (a) {
    if (a->v == v) return a->f;
    a = a->link;
  }
  return 0;
}

/** Function **
  Synopsis    [ Generic list sort ]
  Description [ Sorts l by increasing order defined by lt.
  lt is a comparison predicate that takes as arguments three
  pointers, compares the first two using the third as evaluation
  environment and returns a non-negative value if the first is
  smaller than the first. ]
  SideEffects [ Modifies l ]
 */
Atom
imgSortAtomList
(Atom l,
 fn_compare lt,
 pointer env)
{
  if (l == 0) {
    return 0;
  }
  if (l->link == 0) {
    return l;
  } 
  return imgInsertAtomList(l, imgSortAtomList(l->link, lt, env), lt, env);
}

/** Function **
  Synopsis    [ Generic insertion in ordered list ]
  Description [ l is in increasing order, inserts a in l such that
  the result list is also ordered. the comparison predicate is as
  in imgSortAtomList ]
  SideEffects [ Modifies a and l ]
  SeeAlso     [ imgSortAtomList ]
 */
Atom
imgInsertAtomList
(Atom a,
 Atom l,
 fn_compare lt,
 pointer env)
{
  if (l == 0) {
    return a;
  }
  if ((* lt)(a, l, env)) {
    a->link = l;
    return a;
  }
  l->link = imgInsertAtomList(a, l->link, lt, env);
  return l;
}

/** Function **
  Synopsis    [ Deallocates list ]
  Description [ Deallocates l and its successors. ]
  SideEffects [ Returns memory. ]
 */
void
imgFreeAtomList
(img_manager t,
 Atom l)
{
  if (l == 0) {
    return;
  } else {
    imgFreeAtomList(t, l->link);
    mem_free_rec(t->AtomMgr, l);
  }
}

/** Function **
  Synopsis    [ Deallocates list ]
  Description [ Deallocates l and its successors. ]
  SideEffects [ Returns memory. ]
 */
void
imgBuildFunctionOfVar
(img_manager t)
{
  Atom a;
  for (a = t->transitions; a; a = a->link) {
    t->function_of_var[bdd_if_id(t->bddm, a->v)] = a->f;
  }
}

/** Function **
  Synopsis    [ ]
  Description [ ]
  SideEffects [ None ]
 */
bdd
imgFunctionOfVar
(img_manager t,
 long id)
{
  if (id >= t->function_of_var_size) {
    return 0;
  } else {
    return t->function_of_var[id];
  }
}

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

