/* ------------------------------------------------------------ */
/* File name:                                                   */
/*    graph.c                                                   */
/*                                                              */
/* Description:                                                 */
/*                                                              */
/* Project:                                                     */
/*    A symbolic model checker for VHDL                         */
/* Subproject:                                                  */
/*    A program that elaborates abstract machines from VHDL     */
/*    descriptions in the internal format.                      */
/*                                                              */
/* Author:                                                      */
/*    David Deharbe                                             */
/* Affiliation:                                                 */
/*    Carnegie Mellon University (Dept Computer Science)        */
/*                                                              */
/* ------------------------------------------------------------ */

/* ------------------------------------------------------------ */
/* File management information:                                 */
/* $Revision$ */
/* $Date$ */
/* $Locker$ */
/* $Log$ */
/* ------------------------------------------------------------ */

#include <stdio.h>
#include <stdlib.h>

#include <memuser.h>

#include "graph.h"

#define FREE (1 << 16)

/* ------------------------------------------------------------ */

tVertex graph_new_vertex
#if defined(__STDC__)
   (void * Data, long Arity, int Mark) 
#else
   (Data, Arity, Mark)
   void * Data ;
   long Arity ;
   int Mark ;
#endif
{
   tVertex Result ;
   Result = (tVertex) mem_get_block(sizeof(struct VertexCell));
   if (Arity) 
     Result->Edges = (tEdge *) mem_get_block(Arity*sizeof(tEdge));
   else 
     Result -> Edges = 0;
   Result -> Data = Data ;
   Result -> Arity = Arity ;
   Result -> Mark = Mark ;
   return Result;
}

void graph_free_vertex
#if defined(__STDC__)
   (tVertex Vertex)
#else
   (Vertex)
   tVertex Vertex ;
#endif
{
   mem_free_block((pointer)Vertex -> Edges) ;
   mem_free_block((pointer)Vertex) ;
}

void graph_free
#if defined(__STDC__)
   (tGraph Graph, int Free_Data, int Free_Label)
#else
   (Graph)
   tGraph Graph ;
#endif
{
   if (!graph_is_marked(Graph, FREE)) {
      long Cpt ;
      graph_mark_vertex(Graph, FREE) ;
      for (Cpt = 0 ; Cpt < Graph -> Arity ; Cpt += 1) {
         graph_free(Graph -> Edges[Cpt].Target, Free_Data, Free_Label) ;
         if (Free_Label) {
            mem_free_block((pointer)Graph -> Edges[Cpt].Label) ;
         }
      }
      mem_free_block((pointer)Graph -> Edges) ;
      if (Free_Data) {
         mem_free_block((pointer)Graph -> Data) ;
      }
      mem_free_block((pointer)Graph) ;
   }
}

/* ------------------------------------------------------------ */
long graph_get_arity
#if defined(__STDC__)
   (tVertex Vertex)
#else
   (Vertex)
   tVertex Vertex ;
#endif
{
   if (Vertex) {
      return(Vertex -> Arity) ;
   } else {
      fprintf(stderr, "graph_get_arity applied to a null vertex\n") ;
      return(0) ;
   }
}

void graph_set_arity
#if defined(__STDC__)
   (tVertex Vertex, long Arity)
#else
   (Vertex, Arity)
   tVertex Vertex ;
   long Arity ;
#endif
{
   if (Vertex == (tVertex) NULL) {
      fprintf(stderr, "graph_set_arity applied to a null vertex\n") ;
      return ;
   }
   Vertex -> Arity = Arity;
   if (Vertex -> Edges != (tEdge *) NULL) {
      fprintf(stderr, "graph_set_arity applied to a vertex with already existing edges\n") ;
   } else {
      Vertex -> Edges = (tEdge *) mem_get_block(Arity*sizeof(tEdge));
   }
}

void * graph_get_data
#if defined(__STDC__)
   (tVertex Vertex)
#else
   (Vertex)
   tVertex Vertex ;
#endif
{
   if (Vertex != (tVertex) NULL) {
      return(Vertex -> Data) ;
   } else {
      fprintf(stderr, "graph_get_data applied to a null vertex\n") ;
      return 0;
   }
}

void graph_set_data
#if defined(__STDC__)
   (tVertex Vertex, void * Data)
#else
   (Vertex, Data)
   tVertex Vertex ;
   void * Data ;
#endif
{
   if (Vertex == (tVertex) NULL) {
      fprintf(stderr, "graph_set_data applied to a null vertex\n") ;
      return ;
   }
   Vertex -> Data = Data;
}

int graph_get_mark
#if defined(__STDC__)
   (tVertex Vertex)
#else
   (Vertex)
   tVertex Vertex ;
#endif
{
   return(Vertex -> Mark) ;
}

void graph_mark_vertex
#if defined(__STDC__)
   (tVertex Vertex, int Mark)
#else
   (Vertex, Mark)
   tVertex Vertex ;
   int Mark ;
#endif
{
   Vertex -> Mark |= Mark ;
}

void graph_unmark_vertex
#if defined(__STDC__)
   (tVertex Vertex, int Mark)
#else
   (Vertex, Mark)
   tVertex Vertex ;
   int Mark ;
#endif
{
   Vertex -> Mark &= !Mark ;
}

int graph_is_marked
#if defined(__STDC__)
   (tVertex Vertex, int Mark)
#else
   (Vertex, Mark)
   tVertex Vertex ;
   int Mark ;
#endif
{
   return(Vertex -> Mark & Mark) ;
}

void graph_set_edge
#if defined(__STDC__)
   (tVertex Vertex, long EdgeNo, void * Label, tVertex Target)
#else
   (Vertex, EdgeNo, Label, Target)
   tVertex Vertex ;
   long EdgeNo ;
   void * Label ;
   tVertex Target ;
#endif
{
   (Vertex -> Edges[EdgeNo]).Label = Label ;
   (Vertex -> Edges[EdgeNo]).Target = Target ;
}

void graph_set_label
#if defined(__STDC__)
   (tVertex Vertex, long EdgeNo, void * Label)
#else
   (Vertex, EdgeNo, Label)
   tVertex Vertex ;
   long EdgeNo ;
   void * Label ;
#endif
{
   (Vertex -> Edges[EdgeNo]).Label = Label ;
}

void * graph_get_label
#if defined(__STDC__)
   (tVertex Vertex, long EdgeNo)
#else
   (Vertex, EdgeNo)
   tVertex Vertex ;
   long EdgeNo ;
#endif
{
   return ((Vertex -> Edges[EdgeNo]).Label) ;
}

void graph_set_target
#if defined(__STDC__)
   (tVertex Vertex, long EdgeNo, tVertex Target)
#else
   (Vertex, EdgeNo, Target)
   tVertex Vertex ;
   long EdgeNo ;
   tVertex Target ;
#endif
{
   (Vertex -> Edges[EdgeNo]).Target = Target ;
}

tVertex graph_get_target
#if defined(__STDC__)
   (tVertex Vertex, long EdgeNo)
#else
   (Vertex, EdgeNo)
   tVertex Vertex ;
   long EdgeNo ;
#endif
{
   return((Vertex -> Edges[EdgeNo]).Target) ;
}


