/** CFile **************************************************************

  FileName    [ misc.c ]

  PackageName [ cv ]

  Synopsis    [ Miscellaneous routines ]

  Description [ Optional ]

  SeeAlso     [ cv.h ]

  Author      [ David Deharbe ]

  Copyright   [ Copyright (C) 1996, Carnegie Mellon University.
                All rights reserved. ]

  Revision    [ $Id$ ]

***********************************************************************/

#include <Errors.h>
#include "cvInt.h"

static char rcsid [] = "$Id$";

/*--------------------------------------------------------------------*/
/* Structure declarations                                             */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Type declarations                                                  */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Variable declarations                                              */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Macro declarations                                                 */
/*--------------------------------------------------------------------*/

/**AutomaticStart******************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

/**AutomaticEnd********************************************************/

/*--------------------------------------------------------------------*/
/* Definition of exported functions                                   */
/*--------------------------------------------------------------------*/

/** Function **
  Synopsis    [ Computes the support of an expression]
  Description [ Given the node of a VHDL expression, this routine
  computes the set of signals and variables it depends on.
  If the parameter does not correspond to a covered case, or if
  the parameter is null, the value 0 is returned. No error message
  is generated. ]
  SideEffects [ Memory is allocated ]
  SeeAlso     [ cv_signal_support ]
 */
cvn
cv_expression_support
(cvn node)
{
  if (NullNode(node)) return 0;

  switch(qKind(node)) {
  case kIdentifier :
    if (IsA(qDeclaration(node), kSignalDeclaration) ||
        IsA(qDeclaration(node), kVariableDeclaration)) {
      return mList(0, 0, 0, cvn_copy(node));
    } else {
      return 0 ;
    }
    /* A unary operation, returns the sensitivity list of
       the operand */
  case kUnary :
    return cv_signal_support(qOperand(node));
    /* A binary operation: compute the supports of the left and the right
       operands; to do the union of the operands, use the support of the
       left operand as a basis, and traverse the support of the right
       operand; for each signal in the support of the right operand, check
       if it is in the support of the left. If yes, then remove it, otherwise,
       add it to the support of the left (i.e. the result). */
  case kBinary :
    return cv_list_union(cv_expression_support(qLeft(node)),
	                 cv_expression_support(qRight(node)),
                         & cv_eq_decl) ;
    /* Otherwise, returns the empty list. */
  default :
    return 0;
  }
}

/** Function **
  Synopsis    [ Computes the signal support ]
  Description [ Given the node of a VHDL expression or statement
  (or part of a statement, such as a select alternative), this 
  routine computes the set of signals it depends on.
  If the parameter does not correspond to a covered case, or if
  the parameter is null, the value 0 is returned. No error message
  is generated. ]
  SideEffects [ Memory is allocated ]
  SeeAlso     [ cv_sensitivity_list ]
 */
cvn
cv_signal_support
(cvn node)
{
  if (NullNode(node)) return 0;

  switch(qKind(node)) {
  case kWaitStatement :
    return cv_list_union(cv_signal_support(qSignals(node)),
		         cv_signal_support(qCondition(node)),
                         & cv_eq_decl);
  case kConditionalSignalAssignmentStatement :
  case kSignalAssignmentStatement : case kVariableAssignmentStatement :
    return cv_signal_support(qSource(node));
  case kWaveform:
    return cv_signal_support(qValue(node));
  case kConditionalWaveform: {
    cvn wave ;
    cvn support = 0;
    for (wave = node ; wave ; wave = qNext(wave)) {
      support = cv_list_union(support, cv_signal_support(qValue(wave)), & cv_eq_decl) ;
      support = cv_list_union(support, cv_signal_support(qCondition(wave)), & cv_eq_decl) ;
    }
    return support;
  }
  case kIfStatement : {
    cvn Branch ;
    cvn support = 0;
    for (Branch = qBranches(node) ; 
	 Branch ; Branch = qNext(Branch)) {
      support = cv_list_union(support, cv_signal_support(Branch), & cv_eq_decl) ;
    }
    return support;
  }
  case kSelectedSignalAssignmentStatement :
    return cv_list_union(cv_signal_support(qSelector(node)),
		         cv_signal_support(qAlternatives(node)), 
                         & cv_eq_decl);
  case kAlternative :
    return cv_signal_support(qStatements(node));
  case kBranch :
    return cv_list_union(cv_signal_support(qCondition(node)),
		         cv_signal_support(qStatements(node)), 
                         & cv_eq_decl);
    /* for a process statement, the function returns the explicit
       sensitivity list */
  case kProcessStatement :
    return cv_signal_support(qSensitivityList(node));
    /* returns the list of identifiers of signals present in the list */
  case kList : {
    cvn ptr;
    cvn res = 0 ;
    for (ptr = node ; ptr ; ptr = qNext(ptr)) {
      cvn id= qValue(ptr) ;
      if (IsA(qDeclaration(id), kSignalDeclaration)) { 
	res = mList(0, res, 0, cvn_copy(id));
      }
    }
    return res;
  }
    /* if the identifier is a signal, returns a one-element identifier
       list made of the declaration of this identifier, the empty list
       otherwise */
  case kIdentifier :
    if (IsA(qDeclaration(node), kSignalDeclaration)) {
      return mList(0, 0, 0, cvn_copy(node));
    } else {
      return 0 ;
    }
    /* A unary operation, returns the sensitivity list of
       the operand */
  case kUnary :
    return cv_signal_support(qOperand(node));
    /* A binary operation: compute the supports of the left and the right
       operands; to do the union of the operands, use the support of the
       left operand as a basis, and traverse the support of the right
       operand; for each signal in the support of the right operand, check
       if it is in the support of the left. If yes, then remove it, otherwise,
       add it to the support of the left (i.e. the result). */
  case kBinary :
    return cv_list_union(cv_signal_support(qLeft(node)),
	                 cv_signal_support(qRight(node)), 
                         & cv_eq_decl) ;
    /* Otherwise, returns the empty list. This has to be corrected as
       soon as the different forms of concurrent statements are accepted */
  default :
    return 0;
  }
}

/** Function **
  Synopsis    [ Computes the sensitivity list of a wait statement ]
  Description [ The argument kind must be kWaitStatement, otherwise
  an error message is printed to stdout, and 0 is returned.
  The routine builds a list of identifiers corresponding the set
  of signals this statement is sensitive to. ]
  SideEffects [ Sends output to stdout if an error occurs ]
  Allocates memory to store the result. ]
  SeeAlso     [ optional ]
 */

cvn
cv_wait_sensitivity_list
(const cvn stm)
{
  cvn sensitivity;
  if (qKind(stm) != kWaitStatement) {
    Message("cv_wait_sensitivity_list: parameter is not a wait statement",
            xxError, NoPosition);
    return 0;
  }
  sensitivity = qSignals(stm) ;
  if (NullNode(sensitivity)) {
    sensitivity = cv_signal_support(qCondition(stm));
  }
  return sensitivity;
}

/*--------------------------------------------------------------------*/
/* Definition of internal functions                                   */
/*--------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* Definition of static functions                                     */
/*--------------------------------------------------------------------*/

