#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <Errors.h>
#include <memuser.h>
#include "cvInt.h"

/* ----------------------------------- types ----------- */

/* table of intermediate format nodes */
typedef struct cvn_table_ {
  cvn * contents;
  unsigned size;
  unsigned max_size;
} cvn_table;

/* unit descriptor */
typedef struct unit_descriptor_ {
  char * library;
  char * primary;
  char * secondary;
  time_t timestamp;
  unsigned root;
  cvn_table graph;
} unit_desc;

/* table of unit descriptor */
typedef struct unit_desc_table_ {
  unit_desc * contents;
  unsigned size;
  unsigned max_size;
} unit_desc_table;

/* ----------------------------------------- constants -- */

static const unsigned MagicNumber = 512295;
static const unsigned RevisionNumber = 402;

const unsigned no_unit = (unsigned) -1;

/* ----------------------------------------- variables -- */

/* the table of descriptors of loaded units */
static unit_desc_table loaded_units_table;

/* the store for cv nodes */
static cvn_table * node_store;

/* record manager for intermediate format nodes */
static rec_mgr cvn_rec_mgr;

/* globally visible cvn representation of std.standard declarations */
cvn std_clause;
cvn integer_type, boolean_type, bit_type;
cvn true_literal, false_literal;
cvn bit0_literal, bit1_literal;

/* ----------------------------------------- routines -- */
static char * working_library (void);
static unsigned find_unit (char *, char *, char *, time_t);
unsigned unit_is_loaded (char * library, char * primary, char * secondary);
static char * build_filename (char *, char *, char *);
static void set_std_declarations(unsigned);

static FILE * open_file_r (char * filename);

/* ------------------------------------- cv_version --- */
char *
cv_version
(void)
{
  return "release-1.b.1";
}

/* ----------------------------------- cv_initialize --- */
void
cv_initialize
(void)
{
  /* initialize loaded_units_table */
  loaded_units_table.size = 1;
  loaded_units_table.max_size = 8;
  loaded_units_table.contents = (unit_desc *) mem_get_block((SIZE_T) 8 * sizeof(unit_desc));

  loaded_units_table.contents[0].library = 0;
  loaded_units_table.contents[0].primary = 0;
  loaded_units_table.contents[0].secondary = 0;
  loaded_units_table.contents[0].timestamp = 0;
  loaded_units_table.contents[0].root = 0;

  /* initialize node_store */
  node_store = & loaded_units_table.contents[0].graph;
  node_store->size = 0;
  node_store->max_size = 64;
  node_store->contents = (cvn *) mem_get_block((SIZE_T) 64 * sizeof(cvn));

  /* initialize cvn_rec_mgr */
  cvn_rec_mgr = mem_new_rec_mgr((SIZE_T) sizeof(cvn_rec));

  /* some global cvn nodes default value */
  integer_type = boolean_type = bit_type = 0;
  false_literal = true_literal = 0;
  bit0_literal = bit1_literal = 0;
}

/* ---------------------------------------- cv_reset_store --- */
void
cv_reset_store
(void)
{
  unsigned i;
  unit_desc * ptr;

  /* reset loaded_units_table */
  for (i = 1, ptr = loaded_units_table.contents; 
       i < loaded_units_table.size;
       ++i, ++ptr) {
    mem_free_block(ptr->library);
    mem_free_block(ptr->primary);
    mem_free_block(ptr->secondary);
    mem_free_block(ptr->graph.contents);
  }
  loaded_units_table.size = 1;

  /* reset node_store */
  node_store->size = 0;

  /* reset cvn_rec_mgr */
  mem_free_rec_mgr(cvn_rec_mgr);
  cvn_rec_mgr = mem_new_rec_mgr((SIZE_T) sizeof(cvn_rec));

  /* reset global cvn nodes */
  integer_type = boolean_type = bit_type = 0;
  false_literal = true_literal = 0;
  bit0_literal = bit1_literal = 0;
}

/* ----------------------------------- cv_save_store --- */
int
cv_save_store
(cvn root)
{
  tKind kind = qKind(root);
  char * filename, * library;
  FILE * file;

  if ((kind != kEntityDeclaration) && (kind != kArchitectureBody) &&
     (kind != kPackageDeclaration) && (kind != kPackageBody)) {
    return 0;
  }
  if ((library = working_library()) == 0) {
    return 0;
  }
  if ((kind == kEntityDeclaration) || (kind == kPackageDeclaration)) {
    filename = build_filename(library, qName(root), "");
  } else {
    filename = build_filename(library, qName(qPrimary(root)), qName(root));
  }
  if (filename == 0) { /* propagate error */
    return 0;
  }
  if ((file = fopen(filename, "w")) == 0) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("(libcv.a) cannot write to ") + strlen(filename) + 1)); 
    sprintf(message, "(libcv.a) cannot write to %s", filename);
    Message(message, xxError, NoPosition);
    return 0;
  }
  fprintf(stdout, "saving file %s.", filename);
  /* header */
  fprintf(file, "<%u %u %lu>\n",
          MagicNumber, RevisionNumber, (long unsigned) time(0));
  /* units references */
  {
    unsigned i;
    unit_desc * ptr;
    fprintf(file, "%u\n", (loaded_units_table.size - 1));
    for (i = 1, ptr = & loaded_units_table.contents[1]; 
         i < loaded_units_table.size;
         ++i, ++ptr) {
      fprintf(file,
              "<%lu %s "
              "%lu %s "
              "%lu %s>\n",
              strlen(ptr->library), ptr->library,
              strlen(ptr->primary), ptr->primary,
              strlen(ptr->secondary), ptr->secondary);
    }
  }
  /* graph */
  {
    unsigned i;
    cvn * ptr;
    fprintf(file, "%u\n", node_store->size);
    fprintf(file, "%u\n", root->header.position.offset);
    for (i = 0, ptr = node_store->contents; 
         i < node_store->size;
         ++i, ++ptr) {
      write_cvn(* ptr, file);
    }
  }
  fclose(file);
  fprintf(stdout, "done\n");
  return 1;
}

/* ------------------------------------ cv_load_unit --- */
cvn
cv_load_unit
(char * library,
 char * primary,
 char * secondary)
{
  unsigned i;
  unit_desc unit;
  if (strcasecmp("work", library) == 0) {
    if ((library = working_library()) == 0) {
      return 0;
    }
  }
  if ((i = find_unit(library, primary, secondary, 0)) ==
      loaded_units_table.size) {
    Message("(libcv.a) cannot load unit", xxError, NoPosition);
    return 0;
  }
  unit = loaded_units_table.contents[i];
  return unit.graph.contents[unit.root];
}


/* ----------------------------------------- cvn_new --- */
cvn
cvn_alloc
(const tKind kind)
{
  cvn result;
  result = mem_new_rec(cvn_rec_mgr);
  result->header.aKind = kind;
  return result;
}

/* ----------------------------------------- cvn_new --- */
/* returns a new cvn node.
 * this node is stored in the creation table.
 */
cvn
cvn_new
(const tKind kind)
{
  cvn result;
  if (node_store->size == node_store->max_size) {
    node_store->max_size *= 2;
    node_store->contents = (cvn *)
      mem_resize_block((pointer) node_store->contents,
                       (SIZE_T) node_store->max_size * sizeof(cvn));
  }
  result = cvn_alloc(kind);
  result->header.position.unit = 0;
  result->header.position.offset = node_store->size;
  node_store->contents[node_store->size++] = result;
  return result;
}

/* ----------------------------------------- cvn_copy --- */
cvn
cvn_copy
(const cvn n)
{
  cvn result;
  ref tmp;
  result = cvn_new(qKind(n));
  tmp = result->header.position;
  mem_copy(result, n, (SIZE_T) sizeof(cvn_rec));
  result->header.position = tmp;
  return result;
}

/* ----------------------------------------- cvn_to_ref --- */
/* 
 */
void
cvn_to_ref
(ref * r,
 const cvn n)
{
  if (n == 0) {
    r->unit = no_unit;
  } else {
    r->unit = n->header.position.unit;
    r->offset = n->header.position.offset;
  }
}

/* ----------------------------------------- ref_to_cvn --- */
/* 
 */
cvn
ref_to_cvn
(const ref r)
{
  if (r.unit == no_unit) {
    return 0;
  } else {
    return loaded_units_table.contents[r.unit].graph.contents[r.offset];
  }
}

/* -------------------------------- working_library -------- */
char *
working_library
(void)
{
  char * tmp;
  if ((tmp = getenv("work")) == 0) {
    Message("cannot get environment variable \042work\042", 
            xxError, NoPosition);
    return 0;
  } 
  return tmp;
}

/* ------------------------------------------ find_unit --- */
unsigned
find_unit
(char * library, char * primary, char * secondary, time_t date)
{
  char * filename;
  FILE * file;
  unsigned idx;
  time_t mydate;
  unsigned myindex;
  unsigned mymagicnumber;
  unsigned myrevisionnumber;
  unsigned nb_resources;
  unsigned * translation;

  /* if unit is there, do not bother to go further. builds and returns
     the root node */
  if ((idx = unit_is_loaded(library, primary, secondary)) != 0) {
    return idx;
  }

  /* open file */
  if ((filename = build_filename(library, primary, secondary)) == 0) {
    return loaded_units_table.size;
  }
  if ((file = open_file_r(filename)) == 0) {
    return loaded_units_table.size;
  }
  if (loaded_units_table.size == loaded_units_table.max_size) {
    loaded_units_table.max_size *= 2;
    loaded_units_table.contents =
      (unit_desc *) mem_resize_block(loaded_units_table.contents,
                                     (SIZE_T) sizeof(unit_desc) * loaded_units_table.max_size);
  }
 
  fscanf(file, "<%u %u %lu>\n", 
         & mymagicnumber, & myrevisionnumber, (time_t *) & mydate);
  if (mymagicnumber != MagicNumber) {
    Message("bad magic number", xxError, NoPosition);
    return loaded_units_table.size;
  }
  if (myrevisionnumber < RevisionNumber) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("library unit ") + strlen(filename) + 
       strlen(" is in an obsolete format") + 1));
    sprintf(message, "library unit %s is in an obsolete format", filename);
    Message(message, xxError, NoPosition);
    return loaded_units_table.size;
  }
  if (date && (mydate > date)) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("library unit ") + strlen(filename) + 
       strlen(" is obsolete") + 1));
    sprintf(message, "library unit %s is obsolete", filename);
    Message(message, xxError, NoPosition);
    return loaded_units_table.size;
  }
  {
    unsigned i;
    unsigned actual;
    fscanf(file, "%u\n", & nb_resources);
    translation = (unsigned *) mem_get_block((SIZE_T) (1 + nb_resources) * sizeof(unsigned));
    for (i = 1; i <= nb_resources; ++i) {
      unsigned long tmp;
      char * res_library;
      char * res_primary;
      char * res_secondary;
      fscanf(file, "<%lu ", & tmp);
      res_library = (char *) mem_get_block((SIZE_T) sizeof(char) * (tmp + 1));
      fscanf(file, "%s ", res_library);
      fscanf(file, "%lu ", & tmp);
      res_primary = (char *) mem_get_block((SIZE_T) sizeof(char) * (tmp + 1));
      fscanf(file, "%s ", res_primary);
      fscanf(file, "%lu ", & tmp);
      res_secondary = (char *) mem_get_block((SIZE_T) sizeof(char) * (tmp + 1));
      if (tmp == 0) {
        fscanf(file, " >\n");
        res_secondary[0] = 0;
      } else {
        fscanf(file, "%s>\n", res_secondary);
      }
      if ((actual = find_unit(res_library, res_primary, res_secondary, mydate))
           == loaded_units_table.size) {
        mem_free_block((pointer) translation);
        mem_free_block((pointer) res_library);
        mem_free_block((pointer) res_primary);
        mem_free_block((pointer) res_secondary);
        return loaded_units_table.size;
      } else {
        translation[i] = actual;
      }
    }
    translation[0] = myindex = loaded_units_table.size;
  }
  {
    unit_desc * unit;
    unsigned size, i;
    cvn * table, *ptr;
    ref current_position;
    unit = & loaded_units_table.contents[myindex];
    unit->library = library;
    unit->primary = primary;
    unit->secondary = secondary;
    fscanf(file, "%u\n%u\n", & unit->graph.size, & unit->root);
    size = unit->graph.max_size = unit->graph.size;
    table = unit->graph.contents = (cvn *) mem_get_block((SIZE_T) size * sizeof(cvn));
    current_position.unit = myindex;
    for (i = 0, ptr = table; i < size; ++i, ++ptr) {
      current_position.offset = i;
      * ptr = read_cvn(file, current_position);
      translate_cvn(* ptr, translation);
    }
  }
  if ((strcasecmp(library, "std") == 0) &&
      (strcasecmp(primary, "standard") == 0)) {
    set_std_declarations(myindex);
  }
  loaded_units_table.size += 1;
  return myindex;
}

/* -------------------------------------- unit_is_loaded--- */

/* unsigned unit_is_loaded(tString lib, tString primary, tString secondary)
 * lib is a library logical name (possibly work)
 * primary and secondary are the library unit primary and secondary
 * names.
 * Check whether the unit is in the loaded units table.
 * If it not in this table, it then returns 0. Otherwise, it
 * returns its position in the table. 
 */

unsigned unit_is_loaded
(char * library, char * primary, char * secondary)
{
  unsigned i;
  unit_desc * ptr;
  for(i = 1, ptr = & loaded_units_table.contents[1];
      i < loaded_units_table.size;
      ++i, ++ptr) {
    if ((strcasecmp(ptr->library, library) == 0) &&
        (strcasecmp(ptr->primary, primary) == 0) &&
        (strcasecmp(ptr->secondary, secondary) == 0)) {
      return i;
    }
  }
  return 0;
}

/* ---------------------------------- set_std_declarations --- */
void
set_std_declarations
(unsigned idx)
{
  unit_desc std;
  unsigned i, size;
  cvn * ptr;
  std = loaded_units_table.contents[idx];
  size = std.graph.size;
  for(i = 0, ptr = std.graph.contents; i < size; ++i, ++ptr) {
    if (qKind(* ptr) == kTypeDeclaration) {
      if (strcasecmp(qName(* ptr), "integer") == 0)
        integer_type = qBaseType(qSubtype(* ptr));
      if (strcasecmp(qName(* ptr), "boolean") == 0) {
        cvn tmp;
        boolean_type = qBaseType(qSubtype(* ptr));
        tmp = qElements(boolean_type);
        false_literal = qValue(tmp);
        true_literal = qValue(qNext(tmp));
      }
      if (strcasecmp(qName(* ptr), "bit") == 0) {
        cvn tmp;
        bit_type = qBaseType(qSubtype(* ptr));
        tmp = qElements(bit_type);
        bit0_literal = qValue(tmp);
        bit1_literal = qValue(qNext(tmp));
      }
    }
  }
  std_clause = qDeclarations(std.graph.contents[std.root]);
}

/* -------------------------------------- build_filename --- */
char *
build_filename
(char * library,
 char * primary,
 char * secondary)
{
  char * result, * directory, * lower, * tmp;
 
  lower = (char *) mem_get_block((SIZE_T) sizeof(char) * (strlen(library) + 1)); 
  sprintf(lower, "%s", library);
  for (tmp = lower; * tmp; ++tmp) {
    * tmp = tolower(* tmp);
  }
  if ((directory = getenv(lower)) == 0) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("(libcv.a) cannot find VHDL library ") + strlen(lower) + 1)); 
    sprintf(message, "(libcv.a) cannot find VHDL library %s", lower);
    Message(message, xxError, NoPosition);
    return 0;
  }
  free(lower);
  if (secondary && (strcasecmp(secondary, "") != 0)) {
    result  = (char *) mem_get_block((SIZE_T) sizeof(char) *
                                     (strlen(directory) + strlen(primary) + 
                                      strlen(secondary) + strlen(".cv") + 3));
    sprintf(result, "%s/%s-%s.cv", directory, primary, secondary);
  } else {
    result  = (char *) mem_get_block((SIZE_T) sizeof(char) *
                                     (strlen(directory) + strlen(primary) + strlen(".cv") + 2));
    sprintf(result, "%s/%s.cv", directory, primary);
  }
  return result;
}

/* -------------------------------------- open_file_r --- */

static FILE *
open_file_r
(char * filename)
{
  FILE * file;
  int result;
  struct stat buf;
  result = stat(filename, & buf);
  if (result == -1) {
    char * message, * reason;
    switch(errno) {
    case EACCES:
      reason = "permission denied";
      break;
    case EFAULT:
      reason = "memory fault";
      break;
    case EIO:
      reason = "i/o error";
      break;
    case ELOOP:
      reason = "too many symbolic links";
      break;
    case ENAMETOOLONG:
      reason = "name too long";
      break;
    case ENOENT:
      reason = "file does not exist";
      break;
    case ENOTDIR:
      reason = "path is not valid";
      break;
    default:
      reason = "";
      break;
    }
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("cannot read file ") + strlen(filename) + strlen(": ") +
       strlen(reason) + 1));
    sprintf(message, "cannot read file %s: %s", filename, reason);
    Message(message, xxError, NoPosition);
    return 0;
  }
  if (!S_ISREG(buf.st_mode)) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen(filename) + strlen(" is not a regular file") + 1));
    sprintf(message, "%s is not a regular file", filename);
    Message(message, xxError, NoPosition);
    return 0;
  }
  if ((file = fopen(filename, "r")) == 0) {
    char * message;
    message = (char *) mem_get_block((SIZE_T) sizeof(char) *
      (strlen("cannot open ") + strlen(filename) + 1));
    sprintf(message, "cannot open %s", filename);
    Message(message, xxError, NoPosition);
    return 0;
  }
  return file;
}
